//		Copyright (c) 1999 by
//		Advanced Visual Systems Inc.
//		All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/dbil/dbil_connect.hxx#1 $
//

//
// file: dbil_connect.hxx
// author: Patrick Li, Advanced Visual System
// purpose: Define class to establish and maintain a connection
//   to a spatial database.
//
#ifndef DBIL_CONNECT_DEFINED
#define DBIL_CONNECT_DEFINED 1

#include "dbil_driver.hxx"
#include "dbil_shape.hxx"

// Predefine classes as necesary
//
class DBIL_Transaction;

class DBIL_Connection_Entry {
public:
   DBIL_Connection_Entry( DBIL_Connection_Entry *previous, DBIL_Connection_Entry *next, DBIL_Connection *entry );
   virtual ~DBIL_Connection_Entry();

   DBIL_Connection *GetEntry() { return entry; }
   DBIL_Connection_Entry *GetNext() { return next; }
   DBIL_Connection_Entry *GetPrevious() { return previous; }

private:
   DBIL_Connection *entry;
   DBIL_Connection_Entry *previous;
   DBIL_Connection_Entry *next;

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Connection_Entry( const DBIL_Connection_Entry & ) {}
   DBIL_Connection_Entry &operator=(const DBIL_Connection_Entry &);
};

class DBIL_Connection_List {
public:
   DBIL_Connection_List();
   virtual ~DBIL_Connection_List();

   void Add( DBIL_Connection *entry );
   void Remove( DBIL_Connection *entry );

   void Shutdown();

private:
   DBIL_Connection_Entry *head;
   DBIL_Connection_Entry *tail;

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Connection_List( const DBIL_Connection_List & ) {}
   DBIL_Connection_List &operator=(const DBIL_Connection_List &);
};

// Define DBIL_Connection class
//
class DBIL_Connection: public DBIL_Error_Tool {
public:
   // Constructor and Destructor
   //
   DBIL_Connection();
   virtual ~DBIL_Connection();

   // Connect and Disconnect from server
   //
   DBIL_Status Connect( DBIL_Supported_APIs type, char *server, 
      char *instance, char *database, char *user, char *password );

   DBIL_Status Disconnect();

   // Get pointer to current driver
   //
   DBIL_Driver *GetDriver() { return m_driver; }
   static void GetAPIs( int *num_apis, char ***apis, int **ids );
   static void FreeAPIs( int num_apis, char **apis, int *ids );

   // Transaction Processing
   //
   DBIL_Status StartTransaction( DBIL_Transaction *transaction );
   const DBIL_Transaction *GetTransaction() { return current_transaction; }

   DBIL_Status CommitTransaction(int implicit=0);
   DBIL_Status RollbackTransaction(int implicit=0);
	DBIL_Status CancelTransaction(int implicit=0);

   // Insert, Update, and Delete processing
   //
   DBIL_Status Insert( char *table, DBIL_Shape *shape, 
                      int num_attrs, DBIL_Attribute *attrs, 
                      DBIL_Transaction *transaction = NULL );

   DBIL_Status Insert( char *table, DBIL_Shape_Array *shape, 
                      int num_attrs, DBIL_Attribute *attrs, int stopOnError,
                      DBIL_Transaction *transaction = NULL, DBIL_Status_Callback cb = NULL );

   DBIL_Status Update( char *table, char *where, char *shape_column,
                      DBIL_Shape *shape, int num_attrs, DBIL_Attribute *attrs,
                      DBIL_Shape *filter_shape, DBIL_FILTER_METHODS filter_method,
                      DBIL_Transaction *transaction = NULL, DBIL_Status_Callback cb = NULL );

   DBIL_Status Delete( char *table, char *where, char *shape, char *field, 
                      DBIL_Shape *filter_shape, DBIL_FILTER_METHODS filter_method,
                      DBIL_Transaction *transaction = NULL, DBIL_Status_Callback cb = NULL );

   // Misc SQL
   //
   DBIL_Status ProcessSQL( char *sql_statement, int use_transaction, 
                          DBIL_Transaction *transaction = NULL );

   // Data Management Functions
   //
	virtual DBIL_Status GetSysInfo( int option, char * table_name, char ** sql);
   virtual DBIL_Status GetUserTables( int permissions, int *num_tables, char ***tables );
   virtual DBIL_Status FreeUserTables( int num_tables, char **tables );

   virtual DBIL_Status DescribeTable( char *table, int *num_columns, DBIL_COLUMN_DEF **column_defs );
   virtual DBIL_Status FreeTableDescriptions( int num_columns, DBIL_COLUMN_DEF *column_defs );

   // Utilities
   //
   int IsConnected() const;
   const DBIL_Version *GetVersion() const;
   static void Shutdown() { connectlist.Shutdown(); }

private:
   DBIL_Driver *m_driver;
   DBIL_Supported_APIs m_type;

   int local_transaction;
   DBIL_Transaction *current_transaction;
   static DBIL_Connection_List connectlist;

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Connection( const DBIL_Connection & ) {}
   DBIL_Connection &operator=(const DBIL_Connection &) { return *this; }
};
#endif
