//		Copyright (c) 1999 by
//		Advanced Visual Systems Inc.
//		All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/dbil/dbil_driver.hxx#1 $
//

// file: dbil_driver.hxx
// author: Patrick Li, Advanced Visual Systems
// purpose: Provide low level driver for accessing a Database API
//
//   This is the base driver for any spatial engine. To create a new driver, simply
//   subclass from this, and fill in the appropriate methods. You then need to modify the
//   DBIL_Connect::Connect method to support instancing the new driver.
//
#ifndef DBIL_DRIVER_DEFINED
#define DBIL_DRIVER_DEFINED

#include "dbil_error.hxx"
#include "dbil_shp_arr.hxx"
#include "dbil_transaction.hxx"
#include "avs/dbil.h"

// Define structure to hold version info
//
typedef struct {
   int major;
   int minor;
   int bug;
   int release;
   char desc[64];
} DBIL_Version;

// Structure to hold cursor information
//
typedef struct DBIL_Cursor {
   short driver_type;                  // 0=unknown, 1=oracle, 2=odbc
   char *table;                        // Name of table for this query
   void *driver_info;                  // Abstract pointer to API specific info
   char *shape_name;                   // Name of shape for this query if any
   short shape_indicator;              // Indicator for shape
   int field_id;                       // Holds field id from initial query
   char *field_name;                   // Name of field for this query if any
   short field_indicator;              // Indicator for field
   int num_attrs;                      // Number of attributes to retrieve
   DBIL_Attribute *attrs;              // Attribute definitions
   long **indicators;                  // Indicators for attributes
//	short *value_len;							// Lengths of data value fetched 
//	short *row_rc;								// Row level return codes
	short has_long;							// Long or Long Raw columns in the select stmt
	int use_floating_type_only;
	int floating_type;
	int fixed_type;
   int rows_per_fetch;						// rows per fetch from server buffer in select stmt
	DB_OUTPUT_STAT output_stat;			// Statistics to be displayed in DB Kit
   // Fetch null value parameters
   //
   int           int_null_value;
   unsigned int  uint_null_value;
   float         float_null_value;
   double        double_null_value;
   char          *string_null_value;
   short         short_null_value;
   unsigned char byte_null_value;
} DBIL_CURSOR;

// Define DBIL_Driver class, this is the subclass for all spatial drivers
//   To define a new interface, subclass off of this, and overload the appropriate
//   methods.
//
class DBIL_Driver : public DBIL_Error_Tool {
public:
   // Constructor and Destructor
   //
   DBIL_Driver();
   virtual ~DBIL_Driver();

   // Connect and Disconnect from server
   //   (note: These methods should be called from methods that overload them)
   //
   virtual DBIL_Status Connect( char *server, char *instance, char *database, char *user, char *password );
   virtual DBIL_Status Disconnect();

   // Stream processing
   //
   virtual DBIL_Status OpenCursor( DBIL_CURSOR &cursor );
   virtual DBIL_Status CloseCursor( DBIL_CURSOR &cursor );

   // Transaction Processing
   //
   virtual DBIL_Status StartTransaction();
   virtual DBIL_Status CommitTransaction();
   virtual DBIL_Status RollbackTransaction();
   virtual DBIL_Status CancelTransaction();

   // Query and retrieve data from spatial engine, 
   //  overload as supported by a particular API.
   //   (note: These methods should NOT be called from methods that overload them)
   //
	virtual DBIL_Status PrepareStmt(char *sql_statement, int query_timeout_secs, int rows_per_fetch);
	virtual DBIL_Status ExecuteStmt(int iteration, int row_off);
   virtual DBIL_Status Define( DBIL_CURSOR &cursor);

   virtual DBIL_Status Query( DBIL_CURSOR &cursor, char *sql_statement );

   virtual DBIL_Status Query( DBIL_CURSOR &cursor, int num_tables, char **tables, char *where, 
                             char *shape, char *field, int num_attrs, char **attributes, 
                             DBIL_Shape *filter_shape, DBIL_FILTER_METHODS filter_method );

   virtual DBIL_Status Fetch( DBIL_CURSOR &cursor, DBIL_Shape *shape, 
                             DBIL_Attribute *attrs, int offset = 0 );

   virtual DBIL_Status Insert( char *table, DBIL_Shape *shape, 
                              int num_attrs, DBIL_Attribute *attrs );

   virtual DBIL_Status Insert( char *table, DBIL_Shape_Array *shape, int num_attrs, 
                              DBIL_Attribute *attrs, int stopOnError, DBIL_Status_Callback cb );

   virtual DBIL_Status Update( char *table, char *where, char *shape_column, DBIL_Shape *shape, 
                              int num_attrs, DBIL_Attribute *attrs, DBIL_Shape *filter_shape, 
                              DBIL_FILTER_METHODS filter_method, DBIL_Status_Callback cb );

   virtual DBIL_Status Delete( char *table, char *where, char *shape, char *field, 
                              DBIL_Shape *filter_shape, DBIL_FILTER_METHODS filter_method, 
                              DBIL_Status_Callback cb );

   // Misc SQL
   //
   virtual DBIL_Status ProcessSQL( char *sql_statement );

   // Data Management Functions
   //
	virtual DBIL_Status GetSysInfo( int option, char * table_name, char ** sql);

   virtual DBIL_Status GetUserTables( int permissions, int *num_tables, char ***tables );
   virtual DBIL_Status FreeUserTables( int num_tables, char **tables );

   virtual DBIL_Status DescribeTable( char *table, int *num_columns, DBIL_COLUMN_DEF **column_defs );
   virtual DBIL_Status FreeTableDescriptions( int num_columns, DBIL_COLUMN_DEF *column_defs );

   // Utilities
   //
   const DBIL_Version *GetVersion() { return &m_version; }

   // Alarm signal handling routine
   //
   void SigHandler();


protected:
   DBIL_Version m_version;

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Driver( const DBIL_Driver & ) {}
   DBIL_Driver &operator=(const DBIL_Driver &) { return *this; }
};
#endif
