//*-*-c++-*-*
/*
                        Copyright (c) 1996 by
                        Advanced Visual Systems Inc.
                        All Rights Reserved
        
        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.
        
        This file is under Perforce control
        $Id: //depot/express/fcs70/gis/xp_mods/modtrfm.cxx#1 $
*/

//
//	module:	GISProjection
//
//	Description:
//
//	Perform a map projection transformation on a
//	single lat/lon/alt point.
//
//	date		author		action
//	----		------		------
//	17 jan 96	Rob DeMillo	Original
//

#define XP_WIDE_API	/* Use Wide APIs */

#include "GISTrfm.hxx"

//
//	function: Index
//
//	Description:
//	Give a row, column and total number of columns, 
//	return the proper index into an array.
//
/* 64-bit porting. Directly Modified */
xp_long 
Index(xp_long row, xp_long col, xp_long ncols)
{
  return ((row * ncols) + col);
}

//
//	mfunction: project
//
//	This method is invoke by notifications from the input
//	lat/lon/alt ports. It performs a map projection from
//	lat/lon/alt space to XYZ space.

/* 64-bit porting. Only Modified Internally */
int
GIS_GISProjection::project(OMevent_mask event_mask, int seq_num)
{
  // Retrieve the pointer for the Projection class. It 
  // should have been instanced when this class was created.
  Projection *projPtr = (Projection *)ret_class_ptr("Projection");
  xp_long	 writePercent;
  xp_long	 percentDone;

  // The input vector length can only be 2 or 3
  // The outout vector length is forced here to be 
  // 3 at all times.

  if ((inLLA.veclen < 2) || (inLLA.veclen > 3))
    return 1;

  int ncols = inLLA.veclen;
  outXYZ.veclen = 3;

  // Retrieve the pointers for the arrays involved
  // First, retrieve the input array
  // Try forcing the datatype.

  double *inPtr = 
    (double *)inLLA.values.ret_typed_array_ptr(OM_GET_ARRAY_RD,
					       OM_TYPE_DOUBLE,
					       (xp_long *)NULL);
  if (inPtr == (double *)NULL)
    {
      return 1;
    }

  double *outPtr = (double *)
    outXYZ.values.ret_typed_array_ptr(OM_GET_ARRAY_WR,
				      OM_TYPE_DOUBLE,
				      (xp_long *)NULL);

  if (outPtr == (void *)NULL)
    {
      return 1;
    }


  // Set the projection type
  projPtr->setProjection((char *)Transform);

  // Set the global radius
  projPtr->setGlobeRadius(GlobalRadius);

  // Set the base latitude and long. The module defaults the
  // location to Boston
  projPtr->setBaseLatLon(baseLat, baseLon);

  // Set the first and second parallels
  projPtr->setFirstPar(firstPar);
  projPtr->setSecondPar(secondPar);

  // Set the longitude offset
  projPtr->setOffsetLon(offsetLon);

  // OK...loop through the LLA values projecting them to
  // XYZ values. If the input array only has a veclen of 2, 0 fill 
  // the altitude value.
  
  // write out a status bar every 10%...
  writePercent = (xp_long)(inLLA.nvals) / 10;
  percentDone = 0;
  for (xp_long i = 0; i < inLLA.nvals; i++)
    {
      if (writePercent != 0)
	{
	  if ((i % writePercent) == 0)
	    {
	      percentDone += 10;
	      OMstatus_check(percentDone, "GISMapTransform", NULL);
	    }
	}
      else
	{
	  percentDone = 100;
	  OMstatus_check(percentDone, "GISMapTransform", NULL);
	}

	
      // set location values...
      if (inLLA.veclen == 2)
	projPtr->setLLCoordinates(inPtr[Index(i,1,ncols)], 
				  inPtr[Index(i,0,ncols)], 
				  0.0);
      else
	projPtr->setLLCoordinates(inPtr[Index(i,1,ncols)], 
				  inPtr[Index(i,0,ncols)], 
				  inPtr[Index(i,2,ncols)]);
  
      // Perform the map transformation and place the results
      // into the XYZ output ports
      // Convert the point to the requested projection type
      switch (projPtr->fetchProjection())
	{
	case cartesian:
	  if (projPtr->Cartesian())
	    break;
	case mercator:
	  if (projPtr->Mercator())
	    break;
	case lambert_cyl:
	  if (projPtr->LambertCylindric())
	    break;
	case lambert_conical:
	  if (projPtr->LambertConical())
	    break;
	case gnomonic:
	  if (projPtr->Gnomonic())
	    break;
	case stereo:
	  if (projPtr->Stereo())
	    break;
	case ortho:
	  if (projPtr->Orthographic())
	    break;
	case postel:
	  if (projPtr->Postel())
	    break;
	case lambert_azi:
	  if (projPtr->LambertAzimuthal())
	    break;
	case albers:
	  if (projPtr->Albers())
	    break;	
	case cassini_soldner:
	  if (projPtr->CassiniSoldner())
	    break;
	case bonne:
	  if (projPtr->Bonne())
	    break;
	case werner:
	  if (projPtr->Werner())
	    break;
	case sanson_flamsteed:
	  if (projPtr->SansonFlamsteed())
	    break;
	case globe:
	  if (projPtr->Globe())
	    break;
	}
  
      // Fetch the stored XYZ
      double	x,
	        y,
	        z;

      projPtr->fetchWorldCoordinates(&x,&y,&z);
      outPtr[Index(i,0,ncols)] = x;
      outPtr[Index(i,1,ncols)] = y;
      outPtr[Index(i,2,ncols)] = z;
    }

  // Free the two pointers
  ARRfree(inPtr);
  ARRfree(outPtr);

  OMstatus_check(100, "GISMapTransform", NULL);
  // return 1 for success
  return(1);
}

