/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gmod/hsv2rgb.c#1 $
*/

/************************************************************************/
/* public functions: hsv_to_rgb, rgb_to_hsv, rgb_or_hsv                 */
/************************************************************************/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <avs/err.h>
#include <avs/om.h>

/*----------------------------------------------------------------------*/

static void hsvrgb(double,double,double,double *,double *,double *);
static void rgbhsv(double,double,double,double *,double *,double *);

static int set_hsv(OMobj_id, OMobj_id, OMobj_id, OMobj_id, OMobj_id, OMobj_id);
static int set_rgb(OMobj_id, OMobj_id, OMobj_id, OMobj_id, OMobj_id, OMobj_id);

/*----------------------------------------------------------------------*/

/* returning options */
/* some compilers may object to multiple return styles from a routine */
/* don't use this for now, since all routines here return 0 on error  */

#define RET_DONT   0
#define RET_VOID   1
#define RET_0      2
#define RET_1      3

#define RETURN(R) \
	switch( R ) {	case RET_VOID: return;    \
			case RET_0:    return(0); \
			case RET_1:    return(1); \
			case RET_DONT: break;     }

/*----------------------------------------------------------------------*/

/* error reporting options */

#define ERR_SILENT 0
#define ERR_REPORT 1

/*----------------------------------------------------------------------*/

/* get/set a real subobject value
 * object id ID, V subobject name "S", subobject value S, err flag E
 * note the C value variable must be the same as the V subobject name
 * the err flag controls reporting for undefined values
 * func_name variable must be defined in the calling routine
 */

#define GET_SUB_REAL_VAL(ID,S,E) \
	if (OMget_name_real_val(ID,OMstr_to_name(#S),&S) != OM_STAT_SUCCESS) { \
	   if (E == ERR_REPORT) { \
	   	char err_msg[80]; \
		sprintf( err_msg, "Error getting %s subobject value",#S); \
		ERRerror( func_name, 1, ERR_ORIG, err_msg ); \
	   } \
	   return(0); \
	}

#define SET_SUB_REAL_VAL(ID,S,E) \
	if (OMset_name_real_val(ID,OMstr_to_name(#S),S) != OM_STAT_SUCCESS) { \
	   if (E == ERR_REPORT) { \
	   	char err_msg[80]; \
		sprintf( err_msg, "Error setting %s subobject value",#S); \
		ERRerror( func_name, 1, ERR_ORIG, err_msg ); \
	   } \
	   return(0); \
	}

/*----------------------------------------------------------------------*/

/* get/set a real object value
 * V object name "S", object id S_id, object value S, err flag E
 * note the C value variable must be the same as the V object name
 * and the C object id variable must be that name with an '_id' suffix
 * the err flag, E, controls reporting for undefined values
 * func_name must be defined in the calling routine
 */

#define GET_REAL_VAL(S,E) \
	if (OMget_real_val(S##_id,&S) != OM_STAT_SUCCESS) { \
	   if (E == ERR_REPORT) { \
	   	char err_msg[80]; \
		sprintf( err_msg, "Error getting %s object value", #S); \
		ERRerror( func_name, 1, ERR_ORIG, err_msg ); \
	   } \
	   return(0); \
	}

#define SET_REAL_VAL(S,E) \
	if (OMset_real_val(S##_id,S) != OM_STAT_SUCCESS) { \
	   if (E == ERR_REPORT) { \
	   	char err_msg[80]; \
		sprintf( err_msg, "Error setting %s object value", #S); \
		ERRerror( func_name, 1, ERR_ORIG, err_msg ); \
	   } \
	   return(0); \
	}

/*----------------------------------------------------------------------*/

/* get a subobject id
 * object id ID, V subobject name "S", subobject id S_id, err flag E
 * note the C variable subobject id must be the V subobject name + '_id'
 * the err flag controls reporting for null subobject
 * func_name must be defined in the calling routine
 */

#define GET_SUB_ID(ID,S,E) \
	S##_id = OMfind_subobj( ID, OMstr_to_name(#S), OM_OBJ_RW ); \
	if (OMis_null_obj(S##_id)) { \
	   if (E == ERR_REPORT) { \
	   	char err_msg[80]; \
		sprintf( err_msg, "Error getting %s subobject id",#S); \
		ERRerror( func_name, 1, ERR_ORIG, err_msg ); \
	   } \
	   return(0); \
	}

/*----------------------------------------------------------------------*/

/* clamp a value to a range */

#define CLAMP(LO,VAL,HI) \
	if      (VAL < LO)  VAL = LO; \
	else if (VAL > HI)  VAL = HI;

/* find the maximum value from 3 inputs */

#define MAX3( V1,V2,V3) \
	(V1 > V2) ? ( (V1 > V3) ? V1 : V3 ) : ((V2 > V3) ? V2 : V3)

/* find the minimum value from 3 inputs */

#define MIN3(V1,V2,V3) \
	(V1 < V2) ? ( (V1 < V3) ? V1 : V3 ) : ((V2 < V3) ? V2 : V3);

/************************************************************************/
/* public functions                                                     */
/************************************************************************/

/*----------------------------------------------------------------------*/
/* hsv_to_rgb                                                           */
/*----------------------------------------------------------------------*/

int
hsv_to_rgb_proc(
    OMobj_id obj_id )
{
    char   func_name[] = "hsv_to_rgb";
    double h, s, v, r, g, b;

    /* get the hsv values */
    /* if they are undefined, silently return an error code */

    GET_SUB_REAL_VAL( obj_id, h, ERR_SILENT );
    GET_SUB_REAL_VAL( obj_id, s, ERR_SILENT );
    GET_SUB_REAL_VAL( obj_id, v, ERR_SILENT );

    hsvrgb( h, s, v, &r, &g, &b );

    /* set the rgb values */

    SET_SUB_REAL_VAL( obj_id, r, ERR_SILENT );
    SET_SUB_REAL_VAL( obj_id, g, ERR_SILENT );
    SET_SUB_REAL_VAL( obj_id, b, ERR_SILENT );

    return(1);
}

/*----------------------------------------------------------------------*/
/* rgb_to_hsv                                                           */
/*----------------------------------------------------------------------*/

int
rgb_to_hsv_proc(
    OMobj_id obj_id )
{
    char   func_name[] = "rgb_to_hsv";
    double h, s, v, r, g, b;

    /* get the RGB representation */
    /* if they are undefined, silently return an error code */

    GET_SUB_REAL_VAL( obj_id, r, ERR_SILENT );
    GET_SUB_REAL_VAL( obj_id, g, ERR_SILENT );
    GET_SUB_REAL_VAL( obj_id, b, ERR_SILENT );

    /* convert RGB to HSV */

    rgbhsv( r, g, b, &h, &s, &v );

    /* set the HSV representation */

    SET_SUB_REAL_VAL( obj_id, h, ERR_SILENT );
    SET_SUB_REAL_VAL( obj_id, s, ERR_SILENT );
    SET_SUB_REAL_VAL( obj_id, v, ERR_SILENT );

    return(1);
}

/*----------------------------------------------------------------------*/
/* rgb_or_hsv                                                           */
/*----------------------------------------------------------------------*/

int
rgb_or_hsv_proc(
    OMobj_id     obj_id,
    OMevent_mask ev,
    int          seq )
{
    char         func_name[] = "rgb_or_hsv";
    OMobj_id     r_id, g_id, b_id, h_id, s_id, v_id, mode_id;
    int          delta_rgb, delta_hsv, mode, ret;

    /* find all of the subobjects */

    GET_SUB_ID( obj_id, r, ERR_SILENT );
    GET_SUB_ID( obj_id, g, ERR_SILENT );
    GET_SUB_ID( obj_id, b, ERR_SILENT );

    GET_SUB_ID( obj_id, h, ERR_SILENT );
    GET_SUB_ID( obj_id, s, ERR_SILENT );
    GET_SUB_ID( obj_id, v, ERR_SILENT );

    GET_SUB_ID( obj_id, mode, ERR_SILENT );

    /* get the mode, default to zero */

    if (OMget_int_val(mode_id, &mode) != OM_STAT_SUCCESS)
       mode = 0;

    /* handle changes */

    delta_rgb = OMchanged( r_id, seq ) ||
                OMchanged( g_id, seq ) ||
                OMchanged( b_id, seq );

    delta_hsv = OMchanged( h_id, seq ) ||
                OMchanged( s_id, seq ) ||
                OMchanged( v_id, seq );

    if (delta_rgb && delta_hsv) {

       /* both changed, set according to mode */
       /* mode 0 means rgb has priority       */
       if (mode == 0) ret = set_rgb( r_id, g_id, b_id, h_id, s_id, v_id );
       else           ret = set_hsv( r_id, g_id, b_id, h_id, s_id, v_id );
    }
    else if (delta_rgb) {

       /* rgb changed, set hsv */
       ret = set_hsv( r_id, g_id, b_id, h_id, s_id, v_id );
    }
    else if (delta_hsv) {

       /* hsv changed, set rgb */
       ret = set_rgb( r_id, g_id, b_id, h_id, s_id, v_id );
    }
    else {

       /* no changes, someone called us without good reason */
       ret = 1;
    }

    return( ret );
}

/************************************************************************/
/* local functions                                                      */
/************************************************************************/

static int
set_rgb(
    OMobj_id r_id, OMobj_id g_id, OMobj_id b_id,
    OMobj_id h_id, OMobj_id s_id, OMobj_id v_id )
{
    char   func_name[] = "set_rgb";
    double r, g, b;
    double h, s, v;

    /* get HSV representation */

    GET_REAL_VAL( h, ERR_SILENT );
    GET_REAL_VAL( s, ERR_SILENT );
    GET_REAL_VAL( v, ERR_SILENT );

    /* convert to RGB */

    hsvrgb( h, s, v, &r, &g, &b );

    /* set RGB representation */

    SET_REAL_VAL( r, ERR_REPORT );
    SET_REAL_VAL( g, ERR_REPORT );
    SET_REAL_VAL( b, ERR_REPORT );

    return(1);
}

/*----------------------------------------------------------------------*/

static int
set_hsv(
    OMobj_id r_id, OMobj_id g_id, OMobj_id b_id,
    OMobj_id h_id, OMobj_id s_id, OMobj_id v_id )
{
    char   func_name[] = "set_hsv";
    double h, s, v;
    double r, g, b;

    /* get RGB representation */

    GET_REAL_VAL( r, ERR_SILENT );
    GET_REAL_VAL( g, ERR_SILENT );
    GET_REAL_VAL( b, ERR_SILENT );

    /* convert to HSV */

    rgbhsv( r, g, b, &h, &s, &v );

    /* set HSV representation */

    SET_REAL_VAL( h, ERR_REPORT );
    SET_REAL_VAL( s, ERR_REPORT );
    SET_REAL_VAL( v, ERR_REPORT );

    return(1);
}

/*----------------------------------------------------------------------*/

static void
rgbhsv(
   double r,  double g,  double b,
   double *h, double *s, double *v )
{
   double maxval, minval;

   CLAMP( 0.0, r, 1.0 );
   CLAMP( 0.0, g, 1.0 );
   CLAMP( 0.0, b, 1.0 );

   maxval = MAX3( r, g, b );
   minval = MIN3( r, g, b );

   *v = maxval;

   if (maxval != 0.0)
      *s = (maxval - minval) / maxval;
   else
      *s = 0.0;

   if (maxval == minval)
      *h = 0.0;
   else {
      if (r == maxval)
         *h = (g - b) / (6.0 * (maxval - minval));
      else if (g == maxval)
         *h = 1.0/3.0 + (b - r) / (6.0 * (maxval - minval));
      else
         *h = 2.0/3.0 + (r - g) / (6.0 * (maxval - minval));

      while (*h < 0) *h += 1.0;
   }
}

/*----------------------------------------------------------------------*/

static void
hsvrgb(
    double h,   double s,   double v,
    double *rr, double *rg, double *rb )
{
    int 	i;
    float	f, p, q, t;
    float	r, g, b;

    CLAMP( 0.0, h, 1.0 );
    CLAMP( 0.0, s, 1.0 );
    CLAMP( 0.0, v, 1.0 );

    if (v == 0.0) {
        r = 0.0;
	g = 0.0;
	b = 0.0;
    }
    else if (s == 0.0) {
        r = v;
	g = v;
	b = v;
    }
    else
    {
	h *= 6.0;
	if(h >= 6.0)
	    h = 0.0;

	i = h;
	f = h - i;

	p = v*(1.0-s);
	q = v*(1.0-s*f);
	t = v*(1.0-s*(1.0-f));

	switch(i)
	{
	    case 0: r = v; g = t; b = p; break;
	    case 1: r = q; g = v; b = p; break;
	    case 2: r = p; g = v; b = t; break;
	    case 3: r = p; g = q; b = v; break;
	    case 4: r = t; g = p; b = v; break;
	    case 5: r = v; g = p; b = q; break;
	}
    }

    /*
     * Due to floating point round-off errors, it is possible
     * for color components to be slightly out of the 0 .. 1 range.
     * Force them within range just to be safe.  Otherwise we could
     * end up affecting the next byte over.
     */

    CLAMP( 0.0, r, 1.0 );
    CLAMP( 0.0, g, 1.0 );
    CLAMP( 0.0, b, 1.0 );

    *rr = r;
    *rg = g;
    *rb = b;
}

/*----------------------------------------------------------------------*/
