/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gmod/journal.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/om.h>
#include <avs/om_type.h>
#include <avs/om_att.h>
#include <avs/strmio.h>
#include <avs/gmod.h>
#include "uci_gmod.h"

int OMjrn_hndlr_create_elem (OMobj_id);
int OMjrn_hndlr_destroy_elem (OMobj_id);
void OMjrn_hndlr_set_int_val (OMobj_id, int, int, int *);
void OMjrn_hndlr_set_long_val (OMobj_id, xp_long, int, int *);
int OMjrn_hndlr_set_str_val (OMobj_id, char *);
int OMjrn_hndlr_set_real_val (OMobj_id, double);
int OMjrn_hndlr_set_array (OMobj_id, int, char *, xp_long, int);
int OMjrn_hndlr_set_elem_val (OMobj_id, OMobj_id, int);
int OMjrn_hndlr_set_elem_prop (OMobj_id, OMobj_name, OMobj_id);
int OMjrn_hndlr_copy_elem (OMobj_id, OMobj_id, int);
void OMjrn_hndlr_push_elem_ctx (OMobj_id, int, int, int, int, int *);
void OMjrn_hndlr_pop_elem_ctx (OMobj_id, int, int *);
int OMjrn_hndlr_upd_subobj (OMobj_id, OMobj_id, OMobj_name, int);

int OMjrn_hndlr_add_elem_ref (OMobj_id, OMobj_id, int);
int OMjrn_hndlr_insert_obj_ref (OMobj_id, int, OMobj_id, int);
int OMjrn_hndlr_del_elem_ref (OMobj_id, OMobj_id, int);

static char *int_to_str (int ival, char *buf);
static char *long_to_str (xp_long lval, char *buf);
static char *double_to_str (double rval, char *buf);

OMtype_id OM_type_jrn_hndlr;

int OMjournal_init(void)
{
   OMtype_id type_id;

   OM_type_jrn_hndlr = type_id = OMcreate_type("jrn_hndlr",OM_type_obj,0);
   OMset_method(OM_METH_CREATE_OBJ,type_id,(OMpfi)OMjrn_hndlr_create_elem);
   OMset_method(OM_METH_DESTROY_OBJ,type_id,(OMpfi)OMjrn_hndlr_destroy_elem);
   OMset_method(OM_METH_SET_OBJ_VAL,type_id,(OMpfi)OMjrn_hndlr_set_elem_val);
   OMset_method(OM_METH_SET_OBJ_PROP,type_id,(OMpfi)OMjrn_hndlr_set_elem_prop);
   OMset_method(OM_METH_SET_INT_VAL,type_id,(OMpfi)OMjrn_hndlr_set_int_val);
   OMset_method(OM_METH_SET_LONG_VAL,type_id,(OMpfi)OMjrn_hndlr_set_long_val);
   OMset_method(OM_METH_SET_STR_VAL,type_id,(OMpfi)OMjrn_hndlr_set_str_val);
   OMset_method(OM_METH_SET_REAL_VAL,type_id,(OMpfi)OMjrn_hndlr_set_real_val);
   OMset_method(OM_METH_SET_ARRAY,type_id,(OMpfi)OMjrn_hndlr_set_array);
   OMset_method(OM_METH_COPY_OBJ,type_id,(OMpfi)OMjrn_hndlr_copy_elem);
   OMset_method(OM_METH_PUSH_OBJ_CTX,type_id,(OMpfi)OMjrn_hndlr_push_elem_ctx);
   OMset_method(OM_METH_POP_OBJ_CTX,type_id,(OMpfi)OMjrn_hndlr_pop_elem_ctx);
   OMset_method(OM_METH_ADD_OBJ_REF,type_id,(OMpfi)OMjrn_hndlr_add_elem_ref);
   OMset_method(OM_METH_UPD_SUBOBJ,type_id,(OMpfi)OMjrn_hndlr_upd_subobj);
   OMset_method(OM_METH_INSERT_OBJ_REF,type_id,
		(OMpfi)OMjrn_hndlr_insert_obj_ref);
   OMset_method(OM_METH_DEL_OBJ_REF,type_id,(OMpfi)OMjrn_hndlr_del_elem_ref);
   OMset_type_copy_props(type_id,0);
   OMset_type_copy_atts(type_id,0);
   return 1;
}

int
OMjournal_destroy(OMjournal *jrn, journal_Bitmask *jrn_mask)
{
   if (jrn->hndlr_active) {
      OMdel_global_msg_hndlr(jrn->hndlr_elem,OM_STATE_USR | OM_STATE_PROG,0);
      if (jrn->stream != NULL) IOstream_flush(jrn->stream);
   }
   return 1;
}

int
OMjournal_update(OMjournal *jrn, journal_Bitmask *jrn_mask)
{
   journal_Bitmask out_mask;

   /* Ok, john... help me get this the right way! */
   if (OMis_null_obj(jrn->elem_id)) {
      jrn->elem_id = OMcstruct_lookup_ptr((OMptr)jrn);
      jrn->hndlr_elem = OMcreate_obj(OMstr_to_name("hndl_elem"),
				      OM_type_jrn_hndlr,
				      jrn->elem_id,OMnull_obj,OMlocal_proc_id);
   }

   /*
    * If stop was set to nonzero, turn off recording.
    */
   if (jrn_mask->stop && jrn->stop && jrn->record) {
      memset((char *) &out_mask,0,sizeof(journal_Bitmask));
      out_mask.record = 1;
      jrn->record = 0;
      OMcstruct_update_om(jrn, &out_mask);
      jrn_mask->start_recording = 0;
      jrn_mask->record = 1;
   }

   if (jrn_mask->playback && jrn->playback && jrn->playback_filename != NULL) {
      if (OMis_null_obj(OMinst_obj)) {
	  ERRerror("OMjournal",0,ERR_ORIG,
		   "cannot playback without defined 'Application' element");
	  return(0);
      }
      /*
       * First make sure to run off recording!
       */
      memset((char *) &out_mask,0,sizeof(journal_Bitmask));
      out_mask.record = 1;
      jrn->record = 0;
      OMcstruct_update_om(jrn, &out_mask);
      jrn_mask->start_recording = 0;
      jrn_mask->record = 1;
   }

   if (jrn_mask->start_recording && jrn->start_recording &&
       jrn->record_filename != NULL) {

      if (OMis_null_obj(OMinst_obj)) {
	  ERRerror("OMjournal",0,ERR_ORIG,
		   "cannot record without defined 'Application' element");
	  return(0);
      }

      memset((char *) &out_mask,0,sizeof(journal_Bitmask));
      out_mask.record = 1;
      jrn->record = 1;
      OMcstruct_update_om(jrn, &out_mask);

      /*
       * Make the code below thing that this changed so that it reopens the
       * file
       */
      jrn_mask->record = 1;
      jrn_mask->record_filename = 1;
   }

   /*
    * If the record state changed to off and we do have a valid filename,
    * then we delete the message handler
    */
   if (jrn_mask->record && !jrn->record && jrn->hndlr_active) {
      OMdel_global_msg_hndlr(jrn->hndlr_elem,OM_STATE_USR | OM_STATE_PROG,0);
      if (jrn->stream) IOstream_flush(jrn->stream);
      jrn->hndlr_active = 0;
   }

   /*
    * If the file-name has changed since the last time, we must close
    * the old file.  We open the new file the first time that the record
    * guy is chosen.
    */
   if (jrn_mask->record_filename) {
      if (jrn->stream != NULL) {
	 IOstream_close(jrn->stream);
	 jrn->stream = NULL;
      }
   }

   if ((jrn_mask->record || jrn->stream == NULL) && jrn->record) {
      if (jrn->stream == NULL && jrn->record_filename != NULL) {
	 jrn->stream = IOstream_file_open(jrn->record_filename);
	 if (jrn->stream == NULL) {
	    ERRerror("OMjournal",1,ERR_ORIG,
		     "cannot open file: %s for journaling\n",
                     jrn->record_filename);
	    /*
	     * Make sure to delete this handler... they changed filenames
	     * on us while scripting and we couldn't open the file
	     */
	    if (jrn->hndlr_active) {
	       OMdel_global_msg_hndlr(jrn->hndlr_elem,
				      OM_STATE_USR | OM_STATE_PROG,0);
	       jrn->hndlr_active = 0;
	    }
	 }
      }
      if (jrn->stream != NULL && !jrn->hndlr_active) {
	 OMadd_global_msg_hndlr(jrn->hndlr_elem,OM_STATE_PROG | OM_STATE_USR,0);
	 jrn->hndlr_active = 1;
      }
   }

   /*
    * Make sure that this is done AFTER we have turned off recording or
    * else the commands that we read in will be recorded as well!
    */
   if (jrn_mask->playback && jrn->playback && jrn->playback_filename != NULL) {
      if (OMis_null_obj(OMinst_obj)) {
	  ERRerror("OMjournal",0,ERR_ORIG,
		   "cannot playback without defined 'Application' element");
	  return(0);
      }
      OMread_desc(OMinst_obj, jrn->playback_filename, OM_READ_REPLACE);
   }

   return(1);
}

OMjournal *
OM_journal_get_obj(void)
{
   OMobj_id journal_id;
   OMgroup_obj *group;
   OMjournal *jrn;

   /*
    * There are a few methods that we will be invoked for because the
    * the event occurred on our element rather than on the message
    * element.  In this case, we just return NULL and no one cares...
    */
   if (!OMstate.v.hndlr_elem) return(NULL);

   /* The handler element's parent must be a journal object... */
   OMget_obj_parent(OM_hndlr_obj,&journal_id);

   group = OMmap_id_to_obj_type(journal_id, OM_type_group, OMgroup_obj);
   jrn = (OMjournal *)group->struct_ptr;

   /* Reset this flag here to prevent all of the methods from having to...*/
   OMstate.v.hndlr_elem = 0;

   return(jrn);
}


int
OMjrn_hndlr_create_elem(OMobj_id elem_id)
{
   OMjournal *jrn = OM_journal_get_obj();
   OMtype_id type;
   char buf1[256];
   OMobj_id parent;
   char *dst_path;

   if (jrn == NULL) return(0);

   if (OMget_obj_type(elem_id,&type) != 1)
      return(0);

   if (OMget_obj_parent(elem_id,&parent) != 1 || OMis_null_obj(parent))
      return(-1);

   if (OMequal_objs(parent,OMinst_obj))
      IOstream_output(jrn->stream,type->name,
		      " ",OMret_obj_name(elem_id),";\n",NULL);
   else {
      dst_path = OMget_obj_path_to(parent,OMinst_obj,buf1,sizeof(buf1));

      /* Don't save out element creates that are not done in Instances */
      if (dst_path == NULL) return(0);

      IOstream_output(jrn->stream,
		   dst_path,
		   " {\n   ",
		   type->name,
		   " ",OMret_obj_name(elem_id),";\n",
		   "};\n",
		   NULL);
   }
   return(1);
}

int
OMjrn_hndlr_destroy_elem(OMobj_id elem_id)
{
   OMjournal *jrn = OM_journal_get_obj();
   char buf1[1024];
   char *path;

   path = OMget_obj_path_to(elem_id, OMinst_obj, buf1, sizeof(buf1));

   if (path == NULL) return(1);

   IOstream_output(jrn->stream,"-",path,";\n",NULL);

   return(1);
}

int
OMjrn_hndlr_set_real_val(OMobj_id elem_id, double val)
{
   char buf1[1024], buf2[128];
   OMjournal *jrn;
   char *path_to;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf1, sizeof(buf1));
   if (path_to == NULL) return(1);

   IOstream_output(jrn->stream,path_to," = ",double_to_str(val,buf2),";\n",
		   NULL);

   return(1);
}

/* 64-bit porting. Directly Modified */
int
OMjrn_hndlr_set_array(OMobj_id elem_id, int type, char *array,
                      xp_long len, int mode)
{
   char buf1[1024];
   OMjournal *jrn;
   char *path_to;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf1, sizeof(buf1));
   if (path_to == NULL) return(1);

   IOstream_output(jrn->stream,path_to," = ", NULL);
   /* Hack... should more consistently use IOstream package! */
   OMprint_array(jrn->stream->v.fs.fp,NULL,NULL,1,type,array,1,&len,0);
   IOstream_output(jrn->stream,";\n",NULL);

   return(1);
}

int
OMjrn_hndlr_set_str_val(OMobj_id elem_id, char *val)
{
   char buf1[1024];
   OMjournal *jrn;
   char *path_to;
   char *nval;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf1, sizeof(buf1));
   if (path_to == NULL) return(1);

   if (val != NULL)
      nval = OMoutput_string(val,"","",0);
   else
      nval = strdup("");

   IOstream_output(jrn->stream,path_to," = \"",nval,"\";\n",NULL);

   FREE(nval);

   return(1);
}

/* 64-bit porting. Only Modified Internally */
int
OMjrn_hndlr_set_elem_val(OMobj_id elem_id, OMobj_id val_id, int mode)
{
   char buf2[1024];
   OMjournal *jrn;
   char *path_to, *str, *fstr;
   xp_long len;

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf2, sizeof(buf2));
   if (path_to == NULL) return(1);

   if (OMis_null_obj(val_id)) {
      str = "";
      fstr = NULL;
   }
   else {
      str = NULL;
      if (OMsave_obj(val_id,elem_id,0,OM_SAVE_VAL,&str,&len) <= 0)
	 return(1);
      ALLOCN(fstr,char,len+1,"can't alloc string in journal set elem val");
      memcpy(fstr,str,len);
      fstr[len] = '\0';
      FREE(str);
      str = fstr;
   }

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   IOstream_output(jrn->stream,path_to," = ",str,";\n",NULL);
   if (fstr != NULL) FREE(fstr);

   return(1);
}

/* 64-bit porting. Only Modified Internally */
int
OMjrn_hndlr_set_elem_prop(OMobj_id elem_id, OMobj_name prop_name,
                          OMobj_id elem_prop)
{
   char buf2[4096];
   OMjournal *jrn;
   char *path_to, *str, *fstr;
   xp_long len;
   OMpfi set_func;
   int flags;

   jrn = OM_journal_get_obj();

   /*
    * NOTE: this routine must always return "1" so that we don't end up
    * adding this property to the element twice!
    */
   if (jrn == NULL) return(1);

   /*
    * This is a special case.... it specifies a string that we directly
    * output to the journal file
    */
   if (prop_name == OM_prop_journal_string) {
      IOstream_output(jrn->stream,
		      OMret_str_val(elem_prop,buf2,sizeof(buf2)),NULL);
      return(1);
   }

   /*
    * If this property is marked as not needing to be saved, we don't
    * journal it.
    */
   if (OMlookup_prop(prop_name,&flags,&set_func) == 1 &&
		     (flags & OM_PROP_FLAG_NOSAVE)) return(1);

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf2, sizeof(buf2));
   if (path_to == NULL) return(1);

   if (OMis_null_obj(elem_prop)) {
      str = "";
      fstr = NULL;
   }
   else {
      str = NULL;
      if (OMsave_obj(elem_prop,elem_id,0,OM_SAVE_VAL,&str,&len) <= 0)
	 return(1);
      ALLOCN(fstr,char,len+1,"can't alloc string in journal set elem val");
      memcpy(fstr,str,len);
      fstr[len] = '\0';
      FREE(str);
      str = fstr;
   }

   IOstream_output(jrn->stream,path_to,"<",OMname_to_str(prop_name),"=",
		   str,">;\n",NULL);
   if (fstr != NULL) FREE(fstr);

   return(1);
}

void
OMjrn_hndlr_set_int_val(OMobj_id elem_id, int val, int rpmode, int *stat)
{
   char buf1[128], buf2[1024];
   OMjournal *jrn;
   char *path_to;

   if (stat) *stat = 1;

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf2, sizeof(buf2));
   if (path_to == NULL) return;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return;

   IOstream_output(jrn->stream,path_to," = ",int_to_str(val,buf1),";\n",NULL);
}

/* 64-bit porting. Newly Introduced */
void
OMjrn_hndlr_set_long_val(OMobj_id elem_id, xp_long val, int rpmode, int *stat)
{
   char buf1[128], buf2[1024];
   OMjournal *jrn;
   char *path_to;

   if (stat) *stat = 1;

   /* Get the path-name to the element.  If we fail, we don't script this op */
   path_to = OMget_obj_path_to(elem_id, OMinst_obj, buf2, sizeof(buf2));
   if (path_to == NULL) return;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return;

   IOstream_output(jrn->stream,path_to," = ",long_to_str(val,buf1),";\n",NULL);
}

int
OMjrn_hndlr_copy_elem(OMobj_id src_id, OMobj_id dst_id, int mode)
{
   char buf1[1024], buf2[1024];
   OMjournal *jrn;
   OMobj_id dst_parent;
   char *dst_path;

   /* Don't do anything if this copy was for a derived sub-element or
      a virtual object... */
   if ((mode & OM_COPY_DERIVED_OBJ) || (mode & OM_COPY_VIRT_OBJ)) return(0);

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   if (OMget_obj_parent(dst_id,&dst_parent) != 1 ||
       OMis_null_obj(dst_parent)) return(-1);

   if (OMequal_objs(dst_parent,OMinst_obj))
      IOstream_output(jrn->stream,
		      OMret_obj_path_to(src_id,OMinst_obj,buf2,sizeof(buf2),
					OM_PATH_NO_VAL),
		      " ",OMret_obj_name(dst_id),";\n",NULL);
   else {
      dst_path = OMget_obj_path_to(dst_parent,OMinst_obj,buf1,sizeof(buf1));
      /*
       * If we are copying an element that is not inside of "instances"
       * blow it off...
       */
      if (dst_path == NULL) return(0);

      IOstream_output(jrn->stream,
		   dst_path, " {\n   ",
		   OMret_obj_path_to(src_id,OMinst_obj,buf2,sizeof(buf2),
				     OM_PATH_NO_VAL),
		   " ",OMret_obj_name(dst_id),";\n",
		   "};\n",
		   NULL);
   }

   return(0);
}

void
OMjrn_hndlr_push_elem_ctx(OMobj_id elem_id, int state_mode, int event_mode,
                          int child_event_mode, int rpmode, int *stat)
{
   OMjournal *jrn;

   if (stat) *stat = 0;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return;

   IOstream_trigger_output(jrn->stream,"$push",
		           state_mode & OM_STATE_USR ? " -usr\n" : "\n", NULL);
}

void
OMjrn_hndlr_pop_elem_ctx(OMobj_id elem_id, int rpmode, int *stat)
{
   OMjournal *jrn;

   if (stat) *stat = 0;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return;

   /*
    * If our trigger is still pending... we haven't changed anything
    * since the push_ctx for this guy so we don't have to save out anything
    */
   if (IOstream_trigger_cancel(jrn->stream) == 1) return;

   IOstream_output(jrn->stream,"$pop\n",NULL);
}

int
OMjrn_hndlr_upd_subobj (OMobj_id parent, OMobj_id child,
                        OMobj_name elem_name, int mode)
{
   OMjournal *jrn;
   char buf1[1024];
   char *path;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   if (!(mode & OM_UPD_NAME)) return(0);

   path = OMget_obj_path_to(child,OMinst_obj,buf1,sizeof(buf1));
   if (path == NULL) return(1);

   IOstream_output(jrn->stream,"$set_obj_name ",path," ",
		   OMname_to_str(elem_name),"\n", NULL);

   return(1);
}

int
OMjrn_hndlr_add_elem_ref (OMobj_id elem1, OMobj_id elem2, int mode)
{
   OMjournal *jrn;
   char buf1[1024], buf2[1024];
   char *src_path, *dst_path;
   char *comm;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   if (mode & OM_OBJ_REF_QUERY) return(0);

   dst_path = OMget_obj_path_to(elem1,OMinst_obj,buf1,sizeof(buf1));

   /*
    * This is often a name-element.  Deref the name element if so before
    * getting the path-name to the guy.
    */
   if (OMget_obj_atts(elem2,OM_atts_value)) {
      if (OMget_obj_pval(elem2,&elem2,OM_REF_MODE(mode)) != 1) return(0);
      comm = "$link ";
   }
   else comm = "$add_obj_ref ";
   src_path = OMget_obj_path_to(elem2,OMinst_obj,buf2,sizeof(buf2));

   if (dst_path == NULL || src_path == NULL) return(0);

   IOstream_output(jrn->stream,comm,dst_path," ",src_path, "\n",
		   NULL);

   return(1);
}

int
OMjrn_hndlr_insert_obj_ref (OMobj_id elem1, int ind, OMobj_id elem2, int mode)
{
   OMjournal *jrn;
   char buf1[1024], buf2[1024], buf3[64];
   char *src_path, *dst_path;
   char *comm;

   /*
    * Use this command instead since it prints out commands instead
    * of -666 in the file!
    */
   if (ind == OM_ARRAY_APPEND)
      return(OMjrn_hndlr_add_elem_ref(elem1, elem2, mode));

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   if (mode & OM_OBJ_REF_QUERY) return(0);

   dst_path = OMget_obj_path_to(elem1, OMinst_obj, buf1, sizeof(buf1));

   /*
    * This is often a name-element.  Deref the name element if so before
    * getting the path-name to the guy.
    */
   if (OMget_obj_atts(elem2,OM_atts_value)) {
      if (OMget_obj_pval(elem2,&elem2,OM_REF_MODE(mode)) != 1) return(0);
      comm = "$insert_link ";
   }
   else comm = "$insert_obj_ref ";

   src_path = OMget_obj_path_to(elem2,OMinst_obj,buf2,sizeof(buf2));

   if (dst_path == NULL || src_path == NULL) return(0);

   IOstream_output(jrn->stream,comm,dst_path," ",int_to_str(ind,buf3),
		   " ",src_path, "\n",NULL);

   return(1);
}

int
OMjrn_hndlr_del_elem_ref (OMobj_id elem1, OMobj_id elem2, int mode)
{
   OMjournal *jrn;
   char buf1[1024], buf2[1024];
   char *src_path, *dst_path;

   jrn = OM_journal_get_obj();
   if (jrn == NULL) return(0);

   dst_path = OMget_obj_path_to(elem1,OMinst_obj,buf1,sizeof(buf1));
   src_path = OMget_obj_path_to(elem2,OMinst_obj,buf2,sizeof(buf2));
   if (dst_path == NULL || src_path == NULL) return(0);

   IOstream_output(jrn->stream,"$del_elem_ref ",dst_path," ",src_path, "\n",
		   NULL);

   return(1);
}

static char *
int_to_str(int ival, char *buf)
{
   sprintf(buf, "%d", ival);
   return(buf);
}

static char *
long_to_str(xp_long lval, char *buf)
{
#ifdef WORDLENGTH_64
   sprintf(buf, "%ld", lval);
#else
   /* Really just an int on 32-bit platforms */
   sprintf(buf, "%d", lval);
#endif
   return(buf);
}

static char *
double_to_str(double rval, char *buf)
{
   sprintf(buf, "%.10g", rval);
   return(buf);
}
