/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gmod/loop.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/gmod.h>
#include "uci_gmod.h"
#include <avs/event.h>
#include <avs/om.h>

/* Private functions */
static void LoopDoRun (LoopStruct *, LoopBitmask *);
static void LoopDoRunBack (LoopStruct *, LoopBitmask *);
static void LoopDoStep (LoopStruct *, LoopBitmask *);
static void LoopDoStepBack (LoopStruct *, LoopBitmask *);

#define LOOP_DIR_FORWARD  0
#define LOOP_DIR_BACKWARD 1

#define LOOP_MODE_CYCLE  1
#define LOOP_MODE_BOUNCE 2

void LoopFunc(char *loop_arg)
{
        LoopStruct *loop = (LoopStruct *)loop_arg;
	LoopBitmask OutBmk;

	OMcstruct_resolve_ptr(loop, NULL);
	memset(&OutBmk, 0, sizeof(LoopBitmask));

	/* Process run request */
	if (loop->run)
		LoopDoRun(loop, &OutBmk);

	/* Process run backwards request */
	if (loop->run_back)
		LoopDoRunBack(loop, &OutBmk);

	OMcstruct_update_om(loop, &OutBmk);
}

int LoopUpdate(LoopStruct *loop, LoopBitmask *Bmk)
{
	LoopBitmask OutBmk;

	memset(&OutBmk, 0, sizeof(LoopBitmask));
	OMcstruct_resolve_ptr(loop, NULL);

	/* Process reset request.
	 * Be careful not to allow "reset" of 0 to trigger us again
	 * since this can cause infinite loops in some cases.
	 */
	if (Bmk->reset && loop->reset) {
		loop->direction = LOOP_DIR_FORWARD;
		loop->reset = 0;
		loop->done = 0;
		OutBmk.reset = 1;
		OutBmk.done = 1;
		loop->count = loop->start_val;
		OutBmk.count = 1;
	}
	/* Process reset back request. */
	if (Bmk->reset_back && loop->reset_back) {
		loop->direction = LOOP_DIR_FORWARD;
		loop->reset_back = 0;
		loop->done = 0;
		OutBmk.reset_back = 1;
		OutBmk.done = 1;
		loop->count = loop->end_val;
		OutBmk.count = 1;
	}

	/* Process stop request: if either run or run_back bits are set
	   but the flags are not AND the loop is active, stop it
        */
	if (((Bmk->run && !loop->run) || (Bmk->run_back && !loop->run_back)) &&
	    loop->is_active) {
		(void) EVdel_select(EV_POLL0, 0, LoopFunc, NULL,
				    (void *) loop,  0);
		loop->is_active = 0;
		loop->run = 0;
		loop->run_back = 0;
		OutBmk.run = 1;
		OutBmk.run_back = 1;
	}

	/* Process step request */
	if (Bmk->step && loop->step)
		LoopDoStep(loop, &OutBmk);

	/* Process step back request */
	if (Bmk->step_back && loop->step_back)
		LoopDoStepBack(loop, &OutBmk);

	/* Process run request */
	if (Bmk->run && loop->run) {
		if (!loop->set) {
			loop->set = 1;
			loop->count = loop->start_val;
			OutBmk.count = 1;
		}
		if (!loop->is_active) {
			EVadd_select(EV_POLL0, 0, LoopFunc, NULL,
				     (void *) loop,  0);
			loop->is_active = 1;
		}
		/* See if we are running backwards.  If we are,
		   clear run backwards to allow run to happen.
                */
		if (loop->run_back) {
			loop->run_back = 0;
			OutBmk.run_back = 1;
		}
	}

	/* Process run backwards request */
	if (Bmk->run_back && loop->run_back) {
		if (!loop->set) {
			loop->set = 1;
			loop->count = loop->end_val;
			OutBmk.count = 1;
		}
		if (!loop->is_active) {
			EVadd_select(EV_POLL0, 0, LoopFunc, NULL,
				     (void *) loop,  0);
			loop->is_active = 1;
		}
		/* See if we are runnning.  If we are, clear run
		   to allow run backwards to happen.
                */
		if (loop->run) {
			loop->run = 0;
			OutBmk.run = 1;
		}
	}

	OMcstruct_update_om(loop, &OutBmk);
	return(1);
}

int LoopDel(LoopStruct *loop, LoopBitmask *Bmk)
{
	if (loop->is_active)
		(void) EVdel_select(EV_POLL0, 0, LoopFunc, NULL,
				    (void *) loop,  0);
	loop->is_active = 0;
	return(1);
}


/* Private functions */

static void LoopDoRun(LoopStruct *loop, LoopBitmask *OutBmk)
{
	if (loop->incr > 0) {
		/* going from start to end - direction is
		   changed by bounce mode.
		*/
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count+loop->incr > loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->end_val ) {
						loop->count = loop->end_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run = 0;
					OutBmk->done = 1;
					OutBmk->run = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count += loop->incr;
				OutBmk->count = 1;
			}
		}
		/* going from end to start */
		else {
			if (loop->count-loop->incr < loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->start_val ) {
						loop->count = loop->start_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run = 0;
					OutBmk->done = 1;
					OutBmk->run = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count -= loop->incr;
				OutBmk->count = 1;
			}
		}
	}
	/* run request with increment is less than 0 */
	else {
		/* going from start to end - direction is
		   changed by bounce mode.
		*/
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count+loop->incr < loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->end_val ) {
						loop->count = loop->end_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run = 0;
					OutBmk->done = 1;
					OutBmk->run = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count += loop->incr;
				OutBmk->count = 1;
			}
		}
		/* going from end to start */
		else {
			if (loop->count-loop->incr > loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->start_val ) {
						loop->count = loop->start_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run = 0;
					OutBmk->done = 1;
					OutBmk->run = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count -= loop->incr;
				OutBmk->count = 1;
			}
		}
	}
}

static void LoopDoRunBack(LoopStruct *loop, LoopBitmask *OutBmk)
{
	if (loop->incr > 0) {
		/* going from end to start - direction is
		   changed by bounce mode.
		*/
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count-loop->incr < loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->start_val ) {
						loop->count = loop->start_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run_back = 0;
					OutBmk->done = 1;
					OutBmk->run_back = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count -= loop->incr;
				OutBmk->count = 1;
			}
		}
		/* going from start to end */
		else {
			if (loop->count+loop->incr > loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->end_val ) {
						loop->count = loop->end_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run_back = 0;
					OutBmk->done = 1;
					OutBmk->run_back = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count += loop->incr;
				OutBmk->count = 1;
			}
		}
	}
	/* run backwards request with increment is less than 0 */
	else {
		/* going from end to start - direction is
		   changed by bounce mode.
		*/
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count-loop->incr > loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->start_val ) {
						loop->count = loop->start_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run_back = 0;
					OutBmk->done = 1;
					OutBmk->run_back = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count -= loop->incr;
				OutBmk->count = 1;
			}
		}
		/* going from start to end */
		else {
			if (loop->count+loop->incr < loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					OutBmk->count = 1;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
					OutBmk->count = 1;
				}
				else {
					if( loop->count != loop->end_val ) {
						loop->count = loop->end_val;
						OutBmk->count = 1;
					}
					loop->done = 1;
					loop->run_back = 0;
					OutBmk->done = 1;
					OutBmk->run_back = 1;
					(void) EVdel_select(EV_POLL0,0,LoopFunc, NULL,
						    	(void *) loop,  0);
					loop->is_active = 0;
					loop->set = 0;
					loop->direction = LOOP_DIR_FORWARD;
				}
			}
			else {
				loop->count += loop->incr;
				OutBmk->count = 1;
			}
		}
	}
}

static void LoopDoStep(LoopStruct *loop, LoopBitmask *OutBmk)
{
	if (loop->is_active) {
		(void) EVdel_select(EV_POLL0, 0, LoopFunc, NULL,
			(void *) loop,  0);
		loop->is_active = 0;
	}
	loop->run = 0;
	loop->run_back = 0;
	loop->step = 0;
	OutBmk->run = 1;
	OutBmk->run_back = 1;
	OutBmk->step = 1;

	/* update count here */
	if (loop->incr > 0) {
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count+loop->incr > loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
				}
				else {
					loop->count = loop->end_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past end value in once mode");
				}
			}
			else loop->count += loop->incr;
		}
		else {
			if (loop->count-loop->incr < loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
				}
				else {
					loop->count = loop->start_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past start value in once mode");
				}
			}
			else loop->count -= loop->incr;
		}
	}
	else {
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count+loop->incr < loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
				}
				else {
					loop->count = loop->end_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past end value in once mode");
				}
			}
			else loop->count += loop->incr;
		}
		else {
			if (loop->count-loop->incr > loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
				}
				else {
					loop->count = loop->start_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past start value in once mode");
				}
			}
			else loop->count -= loop->incr;
		}
	}
	OutBmk->count = 1;
}

static void LoopDoStepBack(LoopStruct *loop, LoopBitmask *OutBmk)
{
	if (loop->is_active) {
		(void) EVdel_select(EV_POLL0, 0, LoopFunc, NULL,
			    (void *) loop,  0);
		loop->is_active = 0;
	}
	loop->run = 0;
	loop->run_back = 0;
	loop->step_back = 0;
	OutBmk->run = 1;
	OutBmk->run_back = 1;
	OutBmk->step_back = 1;

	/* update count here */
	if (loop->incr > 0) {
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count-loop->incr < loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
				}
				else {
					loop->count = loop->start_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past start value in once mode");
				}
			}
			else loop->count -= loop->incr;
		}
		else {
			if (loop->count+loop->incr > loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					loop->direction = LOOP_DIR_FORWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
				}
				else {
					loop->count = loop->end_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past end value in once mode");
				}
			}
			else loop->count += loop->incr;
		}
	}
	else {
		if (loop->direction == LOOP_DIR_FORWARD) {
			if (loop->count-loop->incr > loop->start_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->start_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->end_val;
				}
				else {
					loop->count = loop->start_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past start value in once mode");
				}
			}
			else loop->count -= loop->incr;
		}
		else {
			if (loop->count+loop->incr < loop->end_val) {
				if (loop->cycle == LOOP_MODE_BOUNCE) {
					loop->count = loop->end_val;
					loop->direction = LOOP_DIR_BACKWARD;
				}
				else if (loop->cycle == LOOP_MODE_CYCLE) {
					loop->count = loop->start_val;
				}
				else {
					loop->count = loop->end_val;
					ERRverror("LoopUpdate", ERR_WARNING,
						"Can't step past end value in once mode");
				}
			}
			else loop->count += loop->incr;
		}
	}
	OutBmk->count = 1;
}

