/*
                        Copyright (c) 1997 by
                        Advanced Visual Systems Inc.
                        All Rights Reserved

        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.

        This file is under Perforce control
        $Id: //depot/express/fcs70/include/avs/animator/LSpline.h#1 $
*/

#ifndef __LSpline_H
#define __LSpline_H

/*

	Object:			N-Dimensional Spline Object

	Author:			Kyle Lussier (lussier@inside.ch / klussier@avs.com)
					Enhanced Implementation for AVS/Express

					Original Implementation in the Manning/Prentice-Hall book:
					"POWER-3D: High-Speed 3D Graphics in Windows 95/NT"

	Description:	N-Dimensional Spline Object for use in AVS/Express Animator
					and Page Flip Book.

	Body:  

	This object uses a natural cubic spline solver to generate an "n"-dimensional
	time based interpolator.  The interpolator assumes that time is linearly mapped
	and that, in the data space, all given values are at regular, integral intervals,
	i.e., t=0,1,2,3,... and not t=0, 1.2, 3.1, ... .

	The LSpline class creates n-splines, one for each parameter that needs interpolation.
	For example, if you were spline interpolating a location in two-dimensional space,
	n would be equal to 2.  For three-dimensional space, n would be equal to 3.  To
	create an encapsulating spline that would move a camera around in three-dimensional
	space, you need to worry about the following quantities:

		Up<x,y,z>			// Spline 0, 1, 2
		Center<x,y,z>		// Spline 3, 4, 5
		Eye<x,y,z>			// Spline 6, 7, 8
		FOV (field of view)	// Spline 9

	This requires 10 separate time based splines, one for each component of the vectors
	and one for the field of view.
	
	In this manner, all values of a single object may be interpolated easily.  The user
	of this object must assign and use index values for the different parameters of each
	individual spline.
	
	The object accesses data in an external, read-only manner.  Therefore, the caller
	or initializer of this object must insure that the data values passed into the model
	will be valid at all times that the spline is being used.

	Date Started:	May 29, 1997

*/

#include "LInterp.h"

class LSpline  
{
private:
	int				m_LastType;		// Last Given Type
	unsigned		m_N;			// Number of Splines for this Object
	LInterpolator	**m_Set;		// Collection of splines
	unsigned		m_Interleaved;	// Given data sets are interleaved
	double * m_LastNumbers;
	double m_LastTime;
	int m_NeedsNormalization;
	float m_Magnitude;
	
protected:
	// Internal Initialization
	void p_Initialize();			// Initializes Data Space and m_Set Member

public:
	void event_SetData(float *data, unsigned N);
	float GetMagnitude(int time);
	void DestroyAllSplines();
	void ValidateAllSplines();
	void VerifySpline(unsigned N);
	void event_SetTension(int bi, double c);
	void event_SetData(double *data, unsigned N);
	void event_SetTypes(int t);
	void event_SetTimes(double *times, unsigned N);
	void event_SetInterleaved(unsigned t);
	void event_SetNumberSets(unsigned n);

	unsigned GetNumberSets();
	// Constructor/Destructor
	LSpline(unsigned params=0);	// Specify Number of Splines to Create,
									// for example, <x,y,z> would need 3 splines.
	virtual ~LSpline();

	// Setup/Initialization
//	bool SetupSpline(unsigned _param, double *_values, unsigned _number, LInterpType type=Linear, double *_times=NULL);

	// Access
	double GetValueAtTime(unsigned param, double time, double curvy=1.0);				// Retrieve Value

};

#endif
