//
//			Copyright (c) 1993, 1999 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/include/avs/cxxutl/list.hxx#1 $
//

// list.hxx - Header file for UtList and UtListIter classes

//----------------------------------------------------------------------

#ifndef CXXUTIL_LIST_HXX
#define CXXUTIL_LIST_HXX

#ifndef CXXUTIL_PORT_HXX
#include <avs/cxxutl/port.hxx>
#endif

// Forward class declarations

template <class TYPE> class UtList;
template <class TYPE> class UtListIter;

//----------------------------------------------------------------------
// Protected class to store one list object
//----------------------------------------------------------------------
template <class TYPE>
class UtListObj 
//----------------------------------------------------------------------
{
    friend class UtList<TYPE>;
    friend class UtListIter<TYPE>;

  protected:
    UtListObj( ) : _next(0), _prev(0) {}
  private:
    UtListObj<TYPE> *_next, *_prev;
    TYPE _element;
};

//----------------------------------------------------------------------
// Double pointed list of any type
//----------------------------------------------------------------------
template <class TYPE>
class UtList 
//----------------------------------------------------------------------
{
    friend class UtListIter<TYPE>;

private:

    typedef UtList<TYPE> self;
    typedef UtListObj<TYPE>   node_type;
    typedef UtListObj<TYPE> * node_ptr;

public:

    typedef TYPE value_type;
    typedef value_type* pointer;
    typedef const value_type* const_pointer;
    typedef value_type& reference;
    typedef const value_type& const_reference;
    typedef size_t size_type;

    typedef UtListIter<TYPE> iterator;
    typedef const UtListIter<TYPE> const_iterator;

    // default constructor
    UtList() : _node(0) { create_null_node(); }

    UtList( size_t n ) : _node(0) { 
        create_null_node();
        for( size_t i = 0; i < n; ++i ) {
            node_ptr new_node = new UtListObj<TYPE>;
            insert_node( _node->_prev, new_node, _node );
        }
    }

    UtList( size_t n, TYPE t ) : _node(0) { 
        create_null_node();
        for( size_t i = 0; i < n; ++i ) {
            push_back( t );
        }
    }

    // destructor
    ~UtList( ) {
        clear( );
        delete _node;
        _node = 0;
    }

    // copy constructor
    UtList( const self& x ) {
        _node = 0;
        create_null_node();
        iterator beg = x.begin( );
        iterator end = x.end( );
        for( ; beg != end; ++ beg ) push_back( *beg );
    }

    // assignment operator
    self& operator=( const self& x ) {
        clear( );
        iterator iter = x.begin( );
        iterator iend = x.end( );
        for( ; iter != iend; ++iter ) push_back( *iter );
        return *this;
    }

    // iterator methods
    iterator begin() { return iterator( _node->_next ); }
    const_iterator begin() const { return iterator( _node->_next ); }
    iterator end() { return iterator( _node ); }
    const_iterator end() const { return iterator( _node ); }

    void push_back ( const TYPE& element ) {
        node_ptr new_node = new UtListObj<TYPE>;
        new_node->_element = element;
        insert_node( _node->_prev, new_node, _node ); 
    }

    void push_front ( const TYPE& element ) {
        node_ptr new_node = new UtListObj<TYPE>;
        new_node->_element = element;
        insert_node( _node, new_node, _node->_next ); 
    }

    void pop_front( ) { 
        node_ptr node = _node->_next;
        if( node != _node ) {
            join_nodes( node->_prev, node->_next );
            delete node;
        }
    }

    void pop_back( ) { 
        node_ptr node = _node->_prev;
        if( node != _node ) {
            join_nodes( node->_prev, node->_next );
            delete node;
        }
    }

    TYPE& front() { return _node->_next->_element; }
    const TYPE& front() const { return _node->_next->_element; }
    TYPE& back() { return _node->_prev->_element; }
    const TYPE& back() const  { return _node->_prev->_element; }

    void clear( ) { 
        node_ptr ptr = _node->_next;
        while( ptr != _node ) {
            ptr = ptr->_next;
            delete ptr->_prev;
        }
        _node->_prev = _node;
        _node->_next = _node;
    };

    bool empty() const { return (_node->_prev == _node) && (_node->_next == _node); }

    size_type size() const {
        size_t result = 0;
        node_ptr ptr = _node->_next;
        while( ptr != _node ) {
            ++result;
            ptr = ptr->_next;
        }
        return result;
    }

    void erase( iterator iter ) {
        node_ptr node = iter._node;
        if( node != _node ) {
            join_nodes( node->_prev, node->_next );
            delete node;
        }
    }

    void insert( iterator iter, const TYPE& element ) {
        node_ptr new_node = new UtListObj<TYPE>;
        new_node->_element = element;
        insert_node( iter._node->_prev, new_node, iter._node );
    }

  private:

    // Create the special empty node that is a part of every list
    void create_null_node( ) {
        _node = new UtListObj<TYPE>;
        _node->_next = _node;
        _node->_prev = _node;
    }

    void insert_node( node_ptr prev, node_ptr node, node_ptr next ) {
        prev->_next = node;

        node->_prev = prev;
        node->_next = next;

        next->_prev = node;
    }

    void join_nodes( node_ptr prev, node_ptr next ) {
        prev->_next = next;
        next->_prev = prev;
    }

    // Orginal methods before STL conversion
    // Search list for given element and remove first (if any) which is found 
    //void removeFirst (TYPE element);

    node_ptr _node;
};

#if 0
template <class TYPE> 
bool operator=( const Utlist<TYPE>& list1, const Utlist<TYPE>& list1 )
{
    // This is lame.  To know for sure, you would have to check
    // the entire list, a potentially expensive operation.
    return( (list1.front( ) == list2.front( )) &&
            (list1.back( ) == list2.back( )) )
}
#endif

//----------------------------------------------------------------------
// Iterator for UtList
//----------------------------------------------------------------------
template <class TYPE> 
class UtListIter 
//----------------------------------------------------------------------
{
  typedef UtListIter<TYPE> self;

  friend class UtList<TYPE>;

  public:
    UtListIter() : _node(0) {}

    bool operator==( const self& x ) const { 
      return _node == x._node;
    }

    bool operator!=( const self& x ) const {
      return _node != x._node;
    }

    TYPE& operator*( ) const {
      return _node->_element;
    }

    self& operator++( ) {
      _node = _node->_next;
      return *this;
    }

    self operator++( int ) {
      self temp = *this;
      _node = _node->_next;
      return temp;
    }

    self& operator--( ) {
      _node = _node->_prev;
      return *this;
    }

    self operator--( int ) {
      self temp = *this;
      _node = _node->_prev;
      return temp;
    }

  private:
    typedef UtListObj<TYPE> * node_ptr;

    // Constructor for use by the begin and end methods of UtList
    UtListIter( node_ptr node ) : _node(node) {}

    node_ptr _node;
};
    
#endif
