//			Copyright (c) 1998 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/include/avs/cxxutl/vector.hxx#1 $
//

// Header file for UtVector template class (no source file)

//----------------------------------------------------------------------

#ifndef CXXUTIL_VECTOR_HXX
#define CXXUTIL_VECTOR_HXX

#ifndef CXXUTIL_PORT_HXX
#include <avs/cxxutl/port.hxx>
#endif

// Turn off used-before-initialized warning
// I'd like to turn it off at the end of this file, but then it has no effect!
#if defined( __sgi ) && ! ( defined( _ABIO32 ) && ( _MIPS_SIM == _ABIO32 ) )
#pragma set woff 1551
#endif

//----------------------------------------------------------------------
// Vector template class patterned after the STL version.
//
// Should be able to swap in the real STL version when all
// platforms/compilers support the STL.
//
//----------------------------------------------------------------------
template <class TYPE>
class UtVector
//----------------------------------------------------------------------
{
  typedef UtVector<TYPE> self;

  public:

    // This is all you need to define the iterator.
    // Since STL iterators act like pointers, in some situations
    // it works just to use an actual pointer.
    typedef TYPE * iterator;
    typedef const TYPE * const_iterator;

    // default constructor
    UtVector() : _vector(0), _size(0), _capacity(0) {
        alloc_no_copy( 1 );
    }

    // constructor
    UtVector(size_t n) : _vector(0), _size(0), _capacity(0) {
        alloc_no_copy( (n < 1) ? 1 : n );
        _size = n;
    }

    UtVector(size_t n, const TYPE& t) : _vector(0), _size(0), _capacity(0) {
        // Always allocate at least one
        alloc_no_copy( (n < 1) ? 1 : n );
        _size = n;
        for( size_t i = 0; i < _size; ++i ) _vector[i] = t;
    }

    // destructor
    ~UtVector() {
        delete [] _vector;
        _vector = 0;
        _size = 0;
        _capacity = 0;
    }

    // copy constructor
    UtVector(const self& vector) {
        _size = 0;
        _capacity = 0;
        _vector = 0;
        alloc_no_copy( vector.size() );
        _size = vector.size();
        for( size_t i = 0; i < _size; ++i ) _vector[i] = vector[i];
    }

    // assignment operator
    self& operator=(const self& vector) {
        if( this != &vector ) {
            if( _capacity < vector.size() ) {
                alloc_no_copy( vector.size() );
            } else {
                destroy_elements( vector.size(), _size );
            }
            _size = vector.size();
            for( size_t i = 0; i < _size; ++i ) _vector[i] = vector[i];
        }
        return *this;
    }

    // iterator methods
    iterator begin() { return &_vector[0]; }
    const_iterator begin() const { return &_vector[0]; }
    iterator end() { return &_vector[_size]; }
    const_iterator end() const { return &_vector[_size]; }

    void push_back(const TYPE& element) {
        if( _size == _capacity ) alloc_copy( _capacity * 2 );
        _vector[ _size ] = element;
        ++_size;
    }

    void pop_back() {
        if( _size > 0 ) {
            destroy_elements( _size - 1, _size );
            --_size;
        }
    }

    TYPE& front() { return _vector[ 0 ]; }
    const TYPE& front() const { return _vector[ 0 ]; }
    TYPE& back() { return (_size == 0) ? _vector[ 0 ] : _vector[ _size-1 ]; }
    const TYPE& back() const { return (_size == 0) ? _vector[ 0 ] : _vector[ _size-1 ]; }

    void clear() {
        destroy_elements( 0, _size );
        _size = 0;
    }

    bool empty() const { return _size == 0; }
    size_t size() const { return _size; }
    size_t capacity() const { return _capacity; }

    void reserve( size_t new_capacity ) {
        if( new_capacity > _capacity ) alloc_copy( new_capacity );
    }

    TYPE& operator[]( size_t index ) { return _vector[ index ]; }
    const TYPE& operator[] ( size_t index ) const { return _vector[ index ]; }

    void swap( UtVector<TYPE>& x ) {
        TYPE * tmp_vector   = x._vector;
        size_t tmp_size     = x._size;
        size_t tmp_capacity = x._capacity;
        x._vector   = _vector;
        x._size     = _size;
        x._capacity = _capacity;
        _vector   = tmp_vector;
        _size     = tmp_size;
        _capacity = tmp_capacity;
    }

    void erase( iterator iter ) {
        iterator iend = end( );
        if( iter == iend ) return;
        --iend;
        for( ; iter != iend; ++iter ) {
            *iter = *(iter + 1);
        }
        {   // Erase the element at the end.
            // See comments below for destroy_elements
            TYPE tmp;  // Some compilers will generate a warning here.
            *iend = tmp;
        }
        --_size;
    }

    void insert( iterator iter, const TYPE& element ) {
        if( _size == _capacity ) {
            // Oops, we need to worry about invalidating the iterator!
            long index = iter - begin();
            alloc_copy( _capacity * 2 );
            iter = begin() + index;
        }

        iterator iend = end( );
        if( iter == iend ) {
            // The easy case, just pop the new one on the end
            *iter = element;
            ++_size;
            return;
        }

        // Iterate backwards from the end towards the insertion point.
        iterator ii = iend;
        // Make room by shifting up;
        for( ; ii != iter; --ii ) {
            *ii = *(ii-1);
        }
        *iter = element;
        ++_size;
    }

  private:

    void destroy_elements( size_t start, size_t stop ) {
        if( start < stop ) {
           // Create a empty element and copy it over the destroyed elements.
           // This doesn't accomplish much with a primitive type such as int
           // or float, but may free up some space with a class that internally
           // allocates memory such as strings.
           TYPE tmp; // Some compilers will generate a warning here.
           for( size_t i = start; i < stop; ++i ) {
                _vector[ i ] = tmp; // Copy it over possibly non empty elements.
           }
       }
    }

    void alloc_copy( size_t new_capacity ) {
        TYPE * new_vector = new TYPE[ new_capacity ];

        size_t i,j;
        for( i = 0, j = 0;
             (i < _size) && (j < new_capacity);
             ++i, ++j )
          new_vector[ j ] = _vector[ i ];

        delete [] _vector;

        _capacity = new_capacity;
        _vector   = new_vector;
        if( _size > _capacity ) _size = _capacity;
    }

    void alloc_no_copy( size_t new_capacity ) {
        TYPE * new_vector = new TYPE[ new_capacity ];

        delete [] _vector;

        _capacity = new_capacity;
        _vector   = new_vector;
        _size     = 0;
    }

    // data

    TYPE * _vector;
    size_t _size;
    size_t _capacity;
 };

template <class TYPE>
inline void swap( UtVector<TYPE>& x, UtVector<TYPE>& y ) { x.swap( y ); }

// The following sort code taken from YACL.

/*
 *
 *          Copyright (C) 1995, M. A. Sridhar
 *
 *
 *     This software is Copyright M. A. Sridhar, 1995. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */

template <class TYPE>
inline void UtSort( TYPE * left, TYPE * right )
{
    if ( !left || !right ) return;
    UtSort1( left, right-1 );
}

template <class TYPE>
inline void UtSort1( TYPE * left, TYPE * right )
{
    // The sort code is basically stolen from Bentley's "Programming
    // Pearls" book.
    TYPE * i;

    for (i = left; i < right; ++i) { // Scan to see if already sorted
        if (UtCompare (*i, *(i+1)) > 0)
            break;
    }

    if (i >= right) // It's already sorted
        return;

    while (left < right) {
        TYPE tmp;
        TYPE * r;
        TYPE * m;

        // r = random (right - left - 1) + left + 1;
        r = left + 1;        // Pick partition/pivot element

        tmp   = *r;          // Swap partition element to left end
        *r    = *left;
        *left = tmp;

        m = left;            // Division point between left and right
        for (i = left+1; i <= right; ++i) {
            if (UtCompare (*i, *left) < 0) {
                m++;         // Move division point one step right
                tmp = *m;    // Swap element to left side
                *m  = *i;
                *i  = tmp;
            }
        }
        tmp   = *m;          // Restore partition element from left end
        *m    = *left;
        *left = tmp;

        // Now recurse on the smaller half, and loop back for the
        // bigger half so as to minimize stack depth
        if (m-left < right-m) {
            UtSort1( left, m-1 );
            left = m+1;
        }
        else {
            UtSort1( m+1, right );
            right = m-1;
        }
    }
    return;
}


#endif

