//
//                      Copyright (c) 1996 by
//                      Advanced Visual Systems Inc.
//                      All Rights Reserved
//
//      This software comprises unpublished confidential information of
//      Advanced Visual Systems Inc. and may not be used, copied or made
//      available to anyone, except in accordance with the license
//      under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/include/avs/gis/demread.h#1 $
//
//
//      module:         DEMread.cxx
//
//      description:
//
//      Class library to read in a USGS Digital Elevation Model (DEM)
//      file. This class will start out reading the 30'x30' and
//      1d x 1d DEM files. The 7.5' file ingestion will be 
//      deferred until later.
//

#ifndef DEMREAD_HEADER
#define DEMREAD_HEADER

#ifdef XP_STD_IOSTREAM
#include <iostream>
using std::ostream;
using std::ios;
#include <fstream>
using std::ifstream;
#else
#include <iostream.h>
#include <fstream.h>
#endif

#ifdef _AIX
// This is a hack to get around some IBM goofiness
#undef _ABS
#endif

#include <math.h>

#ifndef CXXUTIL_STRING_HXX
#include <avs/cxxutl/string.hxx>
#endif

#ifndef CXXUTIL_DATETIME_HXX
#include <avs/cxxutl/datetime.hxx>
#endif

//
//      DEM Classes
//
//      These three class will ingest a USGS Digital Elevation Map file and store the 
//      pertinenant information. Initialially, only the 30'x30' and 1d x 1d data
//      will be supported.
//
//      The ingestion member function will read in the header information and
//      fill up the fields below. It will leave the C file reading pointer at
//      the start of the data
//
//
//      For full explanation of these header values, refer to the DEM Data Users' Guide
//

//      Header class. Corresponds to logical record A, plus file information
class DEM_LRA{
  // private instances
private:
  // Class information
  ifstream      fIn;                    // Standard C++ file stream
  UtString      buffer;                 // Temporary for reading data files

  // LOGICAL RECORD A
  //    OLD DEM information starts here
  UtString      Filename;               // Fully qualified filename location
  UtString      Quadrangle;             // Name of the DEM quadrangle
  UtString      Info;                   // Information from Free-Format field
  int           ProcessCode;            // 1=GPM, 2=Manual Profile, 3=DLG2DEM, 4=DCASS
  UtString      SectionalIndicator;     // specific to 30' DEMs
  UtString      MCorigin;               // Mapping Center origin. EMC, WMC, MCMC, RMMC, FS, GPM2
  int           DEMlevel;               // 1,2,3
  int           ElevationPattern;       // 1 = regular, 2 = random
  int           GroundPlan;             // ground plan reference system 0=geographic, 1=UTM, 2=state
  int           GroundZone;             // zone in GroundPlan. 0 if GroundPlan == 0
  double        MPparameters[15];       // Map projection parameters. Mostly ignored for all three 
  int           Units;                  // 0 = radians, 1= feet, 2=meters, 3=arc-seconds
  int           ElevationUnits;         // 1 = feet, 2 = meters
  int           nSides;                 // number of sides of polygon defining the coverage region
  double        GroundCoords[4][2];     // Coordinates of the quadrangle corners in clockwise order
  double        ElevMinMax[2];          // minimum/maximum elevation values
  double        ReferenceAngle;         // CC angle from primary axis of GroundPlan to primary
                                        // axis of the DEM reference system.
  int           AccuracyCode;           // 0 = unknown, 1=information given in record type C
  double        Rez[3];                 // Spatial resolution (x,y,z)
  int           RowsCols[2];            // Number of rows/colums
  //    OLD DEM information stops here
  //    NEW DEM information starts here
  int           LargestInterval;        // Largest primary contour interval
  int           LIUnits;                // Largest contour interval units 0=N/A, 1=feet, 2=meters
  int           SmallestInterval;       // Smallest primary contour interval
  int           SIUnits;                // Smallest contour interval units. (same)
  UtDate        SourceDate;             // Date of source
  UtDate        Revision;               // Revision/Inspection date
  char          RIFlag;                 // "I" inspection, "R" revision
  int           Validation;             // 0 = no validation, 1=TESTDEM, 2=Water body edit
                                        // 3 = DES no TESDEM, 4 = DES with record C added,
                                        // 5 = DES and TESTDEM
  int           Suspect;                // 0=none, 1=suspect areas, 2=void areas, 3=both
  int           VertDatum;              // 1 = local mean sea level, 2 = National Geodetic
                                        // vertical datum 1929 NGVD 29, 3 = North American
                                        // Vertical datum 1988 NAVD 88
  int           HorzDatum;              // 1 = North American Datum 1927 NAD 27
                                        // 2 = World Geodetic System 1972 WGS 72
                                        // 3 = WGS 84, 4=NAD 83, 5 = Old Hawaii Datum,
                                        // 6 = Puerto Rico Datum, 7 = NAD 83 Provisional
  int           DataEdition;            // 01-99 Primarily a DMA specific field
  int           PercentVoid;            // percentage of void nodes
  //    NEW DEM information stops here

  // protected instances
protected:
  // Various flags
  int           recordA;                // Has record A been ingested?


  //    Byte count information - automagically avoid bytes 1021-1024 of
  //    any logical record! No error checking here, so don't allow users
  //    access to these functions!
  int           ByteCount;      // Cannot let the BYTE count get out of hand.

  void          moveToNextBlock() {
    if (fIn.rdbuf()->is_open())
      {
        fIn.seekg((1024-ByteCount), ios::cur);
        ByteCount = 0;
      }
  }

  void          setByteCount(int value) {
    if (value >= 1021)
      moveToNextBlock();
    else
       ByteCount = value;
  }
  int           fetchByteCount() {
    return ByteCount;
  }
  int           incByteCount() {
    if ((ByteCount+1) >= 1021)
      moveToNextBlock();
    else
      ByteCount++;
    return ByteCount;
  }
  int           addByteCount(int value) {
    if ((ByteCount+value) >= 1021)
      moveToNextBlock();
    else
      ByteCount += value;
    return ByteCount;
  }
  void          resetByteCount() {
    ByteCount = 0;
  }



  // file i/o
  int           readFile(UtString *buffer, 
                         int size);     // Return a character array from file
  int           readFile(int *result, 
                         int size);     // Return an integer from the file that 
                                        // has been stored in a string of size "size"
  int           readFile(double *result,// Return a double from the file that
                         int size);     // has been stored in a string of size "size"
  int           readFile(UtDate *buffer,
                         int size);     // Read in a buffer as a date field.
  int           readFile(char *result); // Suck in a single character                 
  int           skipFile(int places);   // Skip "places" bytes in the file

  ifstream&     fetchIFStream() {       // Return a file pointer
    return fIn;
  }

  // public instances
public:
  // Constructors
  DEM_LRA();                            // Default. No file name
  DEM_LRA(const UtString& fname);       // Full pathname of the file

  // Destructors
  ~DEM_LRA();

  int           setFN(UtString fname); // Set file name
  int           closeFN();              // Close it up
  int           ingest();               // Read in logical record A. Filepointer left at end.
                                        // File is ASCII, not binary

  // Has this record been read?
  void          setLRAIngest(int flag) {
    recordA = flag;
  }

  int           fetchLRAIngest() {
    return recordA;
  }

  // Users are not allowed to enter information directly into the 
  // class, so I've just provided fetch routines here. They are all
  // self-explanatory

  UtString      fetchFileName() {
    return Filename;
  }

  UtString      fetchQuadrangle() {
    return Quadrangle; 
  }

  UtString      fetchInfo() {
    return Info;
  }

  int           fetchProcessCode() {
    return ProcessCode;
  }

  UtString      fetchSI() {
    return SectionalIndicator;
  }

  UtString      fetchMCorigin() {
    return MCorigin;
  }

  int           fetchDEMLevel() {
    return DEMlevel;
  }
  
  int           fetchElvPat() {
    return ElevationPattern;
  }

  int           fetchGroundPlan() {
    return GroundPlan;
  }

  int           fetchGroundZone() {
    return GroundZone;
  }

  double        *fetchMPParameters() {
    return &MPparameters[0];
  }

  int           fetchUnits() {
    return Units;
  }
  
  int           fetchElvUnits() {
    return ElevationUnits;
  }

  int           fetchNSides() {
    return nSides;
  }

  double*       fetchGroundCoords() {
    return &GroundCoords[0][0];
  }

  double        fetchElvMin() {
    return ElevMinMax[0];
  }

  double        fetchElvMax() {
    return ElevMinMax[1];
  }

  double        fetchRefAngle() {
    return ReferenceAngle;
  }
  
  int           fetchAccCode() {
    return AccuracyCode;
  }

  double*       fetchRez() {
    return &Rez[0];
  }

  int           fetchRows() {
    return RowsCols[0];
  }

  int           fetchColumns() {
    return RowsCols[1];
  }

  int           fetchLCI() {
    return LargestInterval;
  }

  int           fetchLCIUnits() {
    return LIUnits;
  }

  int           fetchSCI() {
    return SmallestInterval;
  }

  int           fetchSCIUnits() {
    return SIUnits;
  }

  UtDate        fetchSourceData() {
    return SourceDate;
  }

  UtDate        fetchRevisionDate(char *Flag) {
    *Flag = RIFlag;
    return Revision;
  }

  int           fetchValidation() {
    return Validation;
  }

  int           fetchSuspect() {
    return Suspect;
  }

  int           fetchVertDatum() {
    return VertDatum;
  }

  int           fetchHorzDatum() {
    return HorzDatum;
  }

  int           fetchDE() {
    return DataEdition;
  }

  int           fetchPercentVoid() {
    return      PercentVoid;
  }

#ifndef XP_STD_IOSTREAM
  // Misc.
  friend        ostream &operator<<(ostream &out, DEM_LRA &recA);
  friend        ostream &operator<<(ostream &out, DEM_LRA *recA);
#endif
};


//
//      Class corresponding to logical record B
//      Consult the DEM Data User's Guide for details
//

class DEM_LogicalRecord: public DEM_LRA {
private:

  // LOGICAL RECORD B
  int           RowColID[2];    // Row/Column id number of this profile record
  int           elevRowCol[2];  // number of rows and columns in this elev profile
  double        baseXY[2];      // The ground plaimetric coords (x,y) of first elev.
  double        localDatum;     // Elevation of local datum. Units are in DEM_LRA::fetchElvUnits()
  double        profElvMinMax[2]; // min/max elevations for this profile
  int           *Elevations;    // the m x n elevations for this profile. See the 
                                // user's guide for an explaination on converting these
                                // to true elevations
  

protected:
  // set and fetch flags

public:
  // Constructor
  DEM_LogicalRecord();                            // Make sure we have a valid working file
  DEM_LogicalRecord(UtString fname);

  // Destructor
  ~DEM_LogicalRecord();

  // file i/o
  int           setFN(UtString fName); // Overload the filename set.
  int           readRecord();           // read the next logical record B

  // Provide access to the information above.
  int           fetchRowID() {
    return RowColID[0];
  }

  int           fetchColID() {
    return RowColID[1];
  }

  int           fetchElevRows() {
    return elevRowCol[0];
  }

  int           fetchElevCols() {
    return elevRowCol[1];
  }

  double        fetchBaseX() {
    return baseXY[0];
  }

  double        fetchBaseY() {
    return baseXY[1];
  } 

  double        fetchLocalDatum() {
    return localDatum;
  }

  double        fetchProfMinElev() {
    return profElvMinMax[0];
  }

  double        fetchProfMaxElev() {
    return profElvMinMax[1];
  }

  int*  fetchElevSet() {
    return Elevations;
  }

  int   fetchElevation(int m, int n) {          // Return a specific elevation
    return (*(Elevations + ElevIndex(m,n)));
  }

  // Misc.
  int           ElevIndex(int m, int n){        // Returns true index into linear array
    return ((m*fetchElevCols()) + n);
  }

#ifndef XP_STD_IOSTREAM
  // output operators
  friend        ostream &operator<<(ostream &out, DEM_LogicalRecord &recB);
  friend        ostream &operator<<(ostream &out, DEM_LogicalRecord *recB);
#endif
};

#endif

