//
//                      Copyright (c) 1996 by
//                      Advanced Visual Systems Inc.
//                      All Rights Reserved
//
//      This software comprises unpublished confidential information of
//      Advanced Visual Systems Inc. and may not be used, copied or made
//      available to anyone, except in accordance with the license
//      under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/include/avs/gis/latlon.h#1 $
//

#ifndef LATLON_HEADER
#define LATLON_HEADER

#ifdef XP_STD_IOSTREAM
#include <iostream>
using std::ostream;
#else
#include <iostream.h>
#endif

#ifdef _AIX
// This is a hack to get around some IBM goofiness
#undef _ABS
#endif

#include <math.h>

//----------------
//      Useful globals
//----------------

#ifndef TORAD
// Convert to radians from degrees 
#define TORAD(x)        (((x) / 180.0) * M_PI)
#endif
#ifndef TODEG
// convert to degrees from RADS
#define TODEG(x)        (((x) / M_PI) * 180.0)
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif


#define TOTAL_DIRECTIONS 8

enum direction {
    north,
    northeast,
    east,
    southeast,
    south,
    southwest,
    west,
    northwest,
    identical
};



/////////////////////////////////////////////////////////////////////////////
// LatLon
//

class LatLon
{
  // private instances
private:
  // Information for the lat/lon are always stored in decimal
  double        Latitude;
  double        Longitude;

  // Altitude is stored as a double, but the units are application dependent
  double        Altitude;

  // Error check
  int           checkValues(double *lat, double *lon);

  // Convert from decimal to DMS 
  void  latToDMS(double *D, 
                 double *M,
                 double *S);
  void  lonToDMS(double *D, 
                 double *M,
                 double *S);
  void  latFromDMS(double D, 
                   double M,
                   double S);
  void  lonFromDMS(double D, 
                   double M,
                   double S);


  // public instances
public:
  // contructors - enter the information either as DMS or as decimal - 2D
  // LatLon();
  // Decimal
  LatLon(double lat = 0.0, double lon = 0.0);
  // DMS arrays
  LatLon(double *, double *);
  // DMS single floats
  LatLon(double, double, double, 
         double, double, double);
  // Data entered in Seconds
  LatLon(long, long);

  // contructors - enter the information either as DMS or as decimal - 3D
  // Decimal
  LatLon(double lat, double lon, double alt);
  // DMS arrays
  LatLon(double *, double *, double alt);
  // DMS single floats
  LatLon(double, double, double, 
         double, double, double,
         double);
  // Data entered in Seconds
  LatLon(long, long, double);

  // Destructor
  ~LatLon() {}

  // Convert the currently stored lat lon to whatever we need
  double        fetchLat()              { return Latitude; }
  void          fetchLat(double *lat)   { *lat = Latitude; }
  void          fetchLat(double *D, 
                         double *M, 
                         double *S)     { latToDMS(D, M, S); }
  void          fetchLat(long *lat)     { *lat = ((long)(Latitude * 3600.0)); }

  double        fetchLon()              { return Longitude; }
  void          fetchLon(double *lon)   { *lon = Longitude; }
  void          fetchLon(double *D, 
                         double *M, 
                         double *S)     { lonToDMS(D, M, S); }
  void          fetchLon(long *lon)     { *lon = ((long)(Longitude * 3600.0)); }

  double        fetchAltitude() { return Altitude; }

  // Change indivual lats and lons
  void  setLat(double);
  void  setLat(double, double, double);
  void  setLat(double *);
  void  setLat(long);

  void  setLon(double);
  void  setLon(double, double, double);
  void  setLon(double *);
  void  setLon(long);

  void  setAlt(double alt) { Altitude = alt; }


  // Overload the = operator for a nice assignment constructor
  LatLon&       operator=(const LatLon& rval);

  // Be able to add and subtract
  LatLon&       operator+=(const LatLon &rval);
  LatLon&       operator-=(const LatLon &rval);

    // Compare and contrast lat/lon points
  friend int operator==(LatLon, LatLon);
  friend int operator!=(LatLon, LatLon);

    // Return a direction relative to the current lat/lon point
  direction     whichWay(LatLon &inQuestion);

  // Returns a distance in secs - does not take altitude into account!
  double        distanceSQ(LatLon);     // Distance squared (in secs)

  // Print a nice, neat lat/lon
  friend ostream &operator<<(ostream &out, LatLon &LL);
};

#endif
