/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/include/avs/wt_type.h#1 $
*/

/* Header file containing type definitions for the WT subsystem */

/*---------------------------------------------------------------------*/

/* Prevent the contents of this file from being included more than once */

#ifndef XP_WT_TYPE_INCLUDED
#define XP_WT_TYPE_INCLUDED

/*---------------------------------------------------------------------*/

/* Header files */

#include <avs/port.h>
#include <avs/locale.h>
#include <avs/cstring.h>

#ifdef WIN32

#ifdef __cplusplus
#include <afxwin.h>
#include <afxext.h>
#else
#include <windows.h>
#undef min
#undef max
#endif
#include <windowsx.h>

#else

#include <Xm/Xm.h>

#endif

/*--------------------------------------------------------------------*/
/*                      Constants and macros                          */
/*--------------------------------------------------------------------*/

/* Modes for moving, translating, or scaling windows */

#define WT_COORD_ABSOLUTE  0
#define WT_COORD_RELATIVE  1

/* Modes for window visibility */

#define WT_VIS_CLEAR       0
#define WT_VIS_HIDDEN      1
#define WT_VIS_OVERLAPPED  2

/* Attenuation factors for determining bevel colors.  Note that flat surfaces
   are colored using a fraction of the actual color. */

#define WT_BEVEL_FLAT_SCALE    0.6
#define WT_BEVEL_TOP_SCALE     1.0
#define WT_BEVEL_LEFT_SCALE    1.0
#define WT_BEVEL_BOTTOM_SCALE  0.2
#define WT_BEVEL_RIGHT_SCALE   0.2

/* Graphics margin and bevel thickness for button windows */

#define WT_BUTTON_MARGIN      4.0
#define WT_BUTTON_BEVEL_SIZE  2.0

/* Height of dropdown lists for combo boxes */

#define WT_COMBO_BOX_HEIGHT  300

/* Flags for creating popup menus */

#ifdef WIN32

#define WT_MF_STRING     MF_STRING
#define WT_MF_SEPARATOR  MF_SEPARATOR
#define WT_MF_GRAYED     MF_GRAYED

#else

#define WT_MF_STRING     0x1
#define WT_MF_SEPARATOR  0x2
#define WT_MF_GRAYED     0x4

#endif

/* Chunk size for reallocing arrays */

#define WT_ALLOC_CHUNK_SIZE  10

/* Number of pixels that the mouse must move from its button-press location
   before it is actually considered to have moved */

#define WT_PT_TOL  5

/* Maximum interval (in milliseconds) between clicks for a double click */

#define WT_DCLICK_TIME  400

/* Button timer ID (for Windows) */

#define WT_BUTTON_TIMER_ID  1

/*
 * These are the timer delays (ms) that we use in scrolling.   The first number
 * is the delay after we scroll just one guy before going into continuous
 * scrolling mode.  The second guy is the delay between each guy.  We're
 * of course assuming that machines will require delays as they get faster
 * and faster!
 */

#define WT_SCROLL_START_TIME  500
#define WT_SCROLL_INCR_TIME   30

/* Macro for determining if two mouse events are "in the same location" */

#define WT_EQUAL_PTS(x1, y1, x2, y2) \
        (UTABS((x1) - (x2)) < WT_PT_TOL && UTABS((y1) - (y2)) < WT_PT_TOL)

/* Floating point epsilon */

#define WT_FEPS  1.0e-4

/* Macro for determining if two floats are "equal" */

#define WT_FEQ(a, b)  (UTABS(a - b) < WT_FEPS)

/* Macro for determining the size of an array */

#define WT_ARRAY_SIZE(array)  ((unsigned int) (sizeof(array)/sizeof(array[0])))

/* Mask bits for font attributes */

#define WT_FONT_ATTS_NONE       0x0
#define WT_FONT_ATTS_FAMILY     0x1
#define WT_FONT_ATTS_HEIGHT     0x2
#define WT_FONT_ATTS_WEIGHT     0x4
#define WT_FONT_ATTS_SLANT      0x8
#define WT_FONT_ATTS_SET_WIDTH  0x10
#define WT_FONT_ATTS_USER_FONT  0x20
#define WT_FONT_ATTS_ALL        0xffffffff

/* Mask bits for pixmap file access */

#define WT_PIXMAP_FILE_ACCESS_NONE       0x0
#define WT_PIXMAP_FILE_ACCESS_READ       0x1
#define WT_PIXMAP_FILE_ACCESS_WRITE      0x2
#define WT_PIXMAP_FILE_ACCESS_READ_WRITE \
    (WT_PIXMAP_FILE_ACCESS_READ | WT_PIXMAP_FILE_ACCESS_READ_WRITE)

/*---------------------------------------------------------------------*/

/* Event types */

#ifdef WIN32

#define WT_EVENT_BUTTON_PRESS    1
#define WT_EVENT_BUTTON_RELEASE  2
#define WT_EVENT_BUTTON_MOTION   3
#define WT_EVENT_ENTER           4  /* Not implemented */
#define WT_EVENT_LEAVE           5  /* Not implemented */
#define WT_EVENT_EXPOSE          6
#define WT_EVENT_RESIZE          7
#define WT_EVENT_VIS             8  /* Not implemented */
#define WT_EVENT_TIMER           9

#else

#define WT_EVENT_BUTTON_PRESS    ButtonPress
#define WT_EVENT_BUTTON_RELEASE  ButtonRelease
#define WT_EVENT_BUTTON_MOTION   MotionNotify
#define WT_EVENT_ENTER           EnterNotify
#define WT_EVENT_LEAVE           LeaveNotify
#define WT_EVENT_EXPOSE          Expose
#define WT_EVENT_RESIZE          ConfigureNotify
#define WT_EVENT_VIS             VisibilityNotify

/* Timer events are not available in X, so we use LASTEvent, which is one
   more than the highest event type index, to represent it */

#define WT_EVENT_TIMER           LASTEvent

#endif

/*---------------------------------------------------------------------*/

/* Event masks.  Note that, on X, PointerMotionHintMask is used in
   combination with ButtonMotionMask.  This means that XQueryPointer must
   be used to get the actual pointer position for a motion event. */

#ifdef WIN32

#define WT_EVENT_BUTTON_PRESS_MASK    0x001
#define WT_EVENT_BUTTON_RELEASE_MASK  0x002
#define WT_EVENT_BUTTON_MOTION_MASK   0x004
#define WT_EVENT_ENTER_MASK           0x008
#define WT_EVENT_LEAVE_MASK           0x010
#define WT_EVENT_EXPOSE_MASK          0x020
#define WT_EVENT_RESIZE_MASK          0x040
#define WT_EVENT_VIS_MASK             0x080

#else

#define WT_EVENT_BUTTON_PRESS_MASK   ButtonPressMask
#define WT_EVENT_BUTTON_RELEASE_MASK ButtonReleaseMask
#define WT_EVENT_BUTTON_MOTION_MASK  (ButtonMotionMask | PointerMotionHintMask)
#define WT_EVENT_ENTER_MASK          EnterWindowMask
#define WT_EVENT_LEAVE_MASK          LeaveWindowMask
#define WT_EVENT_EXPOSE_MASK         ExposureMask
#define WT_EVENT_RESIZE_MASK         StructureNotifyMask
#define WT_EVENT_VIS_MASK            VisibilityChangeMask

#endif

/*--------------------------------------------------------------------*/
/*                    Coordinate mapping macros                       */
/*--------------------------------------------------------------------*/

/* The following macros may be used for converting positions (the x and y
   macros) and dimensions (the w and h macros) */

/*
 * Convert from WT window coords to X/MS window pixel coords (transformed):
 *
 *     int WTmap_wind_x (WTwind *wind, WTcoord x)
 *     int WTmap_wind_y (WTwind *wind, WTcoord y)
 *     int WTmap_wind_w (WTwind *wind, WTcoord w)
 *     int WTmap_wind_h (WTwind *wind, WTcoord h)
 */

#define WTmap_wind_x(wind, x) \
    (!(wind) ? (int) (x) : (int) (((x) + (wind)->r.offx) * (wind)->r.scale))

#define WTmap_wind_y(wind, y) \
    (!(wind) ? (int) (y) : (int) (((y) + (wind)->r.offy) * (wind)->r.scale))

#define WTmap_wind_w(wind, w) \
    (!(wind) ? (int) (w) : (int) ((w) * (wind)->r.scale))

#define WTmap_wind_h(wind, h) \
    (!(wind) ? (int) (h) : (int) ((h) * (wind)->r.scale))

/*--------------------------------------------------------------------*/

/*
 * Convert from X/MS window pixel coords (transformed) to VPG window coords:
 *
 *     WTcoord WTunmap_wind_x (WTwind *wind, int x)
 *     WTcoord WTunmap_wind_y (WTwind *wind, int y)
 *     WTcoord WTunmap_wind_w (WTwind *wind, int w)
 *     WTcoord WTunmap_wind_h (WTwind *wind, int h)
 */

#define WTunmap_wind_x(wind, x) \
    (!(wind) ? (WTcoord) (x) : \
               (((WTcoord) (x)/(wind)->r.scale - (wind)->r.offx)))

#define WTunmap_wind_y(wind, y) \
    (!(wind) ? (WTcoord) (y) : \
               (((WTcoord) (y)/(wind)->r.scale - (wind)->r.offy)))

#define WTunmap_wind_w(wind, x) \
    (!(wind) ? (WTcoord) (x) : ((WTcoord) (x)/(wind)->r.scale))

#define WTunmap_wind_h(wind, y) \
    (!(wind) ? (WTcoord) (y) : ((WTcoord) (y)/(wind)->r.scale))

/*--------------------------------------------------------------------*/

/*
 * Convert a value given in a container window's coordinate system to a value
 * in the coordinate system of the GIVEN CHILD of the container window.
 *
 *     WTcoord WTmap_parent_x (WTwind *wind, WTcoord x)
 *     WTcoord WTmap_parent_y (WTwind *wind, WTcoord y)
 *     WTcoord WTmap_parent_w (WTwind *wind, WTcoord w)
 *     WTcoord WTmap_parent_h (WTwind *wind, WTcoord h)
 */

#define WTmap_parent_x(c, X) \
        WTunmap_wind_x((c),  \
        WTmap_wind_x((c)->parent, X) - WTmap_wind_x((c)->parent, (c)->r.x))

#define WTmap_parent_y(c, Y) \
        WTunmap_wind_y((c),  \
        WTmap_wind_y((c)->parent, Y) - WTmap_wind_y((c)->parent, (c)->r.y))

#define WTmap_parent_w(c, w)  (((c)->parent->r.scale * (w)) / (c)->r.scale)

#define WTmap_parent_h(c, h)  (((c)->parent->r.scale * (h)) / (c)->r.scale)

/*--------------------------------------------------------------------*/

/*
 * Convert a value given in the coordinate system of the GIVEN CHILD of a
 * container window to a value in the container window's coordinate system.
 *
 *     WTcoord WTmap_to_parent_x (WTwind *wind, WTcoord x)
 *     WTcoord WTmap_to_parent_y (WTwind *wind, WTcoord y)
 *     WTcoord WTmap_to_parent_w (WTwind *wind, WTcoord w)
 *     WTcoord WTmap_to_parent_h (WTwind *wind, WTcoord h)
 */

#define WTmap_to_parent_x(c, X)     \
        WTunmap_wind_x((c)->parent, \
        WTmap_wind_x(c, X) + WTmap_wind_x((c)->parent, (c)->r.x))

#define WTmap_to_parent_y(c, Y)     \
        WTunmap_wind_y((c)->parent, \
        WTmap_wind_y(c, Y) + WTmap_wind_y((c)->parent, (c)->r.y))

#define WTmap_to_parent_w(c, w) ((c)->r.loc_scale * (w))

#define WTmap_to_parent_h(c, h) ((c)->r.loc_scale * (h))

/* Return a window's current scale relative to its parent */

#define WTwind_scale(w) (w->r.loc_scale)

/*--------------------------------------------------------------------*/
/*                         Enumerated types                           */
/*--------------------------------------------------------------------*/

/* Text drawing modes */

typedef enum {
    WT_TEXT_LEFT,
    WT_TEXT_CENTER,
    WT_TEXT_RIGHT
} WTtext_justification;

/* Pixmap type */

typedef enum {
    WT_PIXMAP_BMX,		/* ASCII version of .bmp files (DEPRECATED) */
    WT_PIXMAP_AVS_IMAGE,	/* AVS .x image files */
    WT_PIXMAP_X_BITMAP,		/* X11 .bmp files */
    WT_PIXMAP_RESOURCE,		/* Windows only */
    WT_PIXMAP_BMP,		/* Windows-style .bmp files */
    WT_PIXMAP_PPM,		/* Portable Pixmap .ppm files */
    WT_PIXMAP_UNSPECIFIED	/* WT guesses the type from the file ext */
} WTpixmap_type;

/* Button types */

typedef enum {
   WT_BUTTON_LEFT_ARROW,
   WT_BUTTON_RIGHT_ARROW,
   WT_BUTTON_UP_ARROW,
   WT_BUTTON_DOWN_ARROW
} WTbutton_type;

/* Callback reasons */

typedef enum {
    WT_REASON_TYPEIN_ACTIVATE,
    WT_REASON_TYPEIN_CANCEL,
    WT_REASON_POPUP_MENU_ACTIVATE,
    WT_REASON_POPUP_MENU_DESTROY,
    WT_REASON_BUTTON_ARM,
    WT_REASON_BUTTON_TIMER,
    WT_REASON_BUTTON_ACTIVATE,
    WT_REASON_SCROLLBAR_SCROLL,
    WT_REASON_SCROLLBAR_DONE,
    WT_REASON_OPTION_MENU_ACTIVATE
} WTcallback_reason;

/* Function return status */

typedef enum {
    WT_ERROR,
    WT_SUCCESS
} WTstatus;

/* Font weights */

typedef enum {
    WT_FONT_WEIGHT_MEDIUM,
    WT_FONT_WEIGHT_BOLD
} WTfont_weight;

/* Font slants */

typedef enum {
    WT_FONT_SLANT_REGULAR,
    WT_FONT_SLANT_OBLIQUE
} WTfont_slant;

/* Font set widths */

typedef enum {
    WT_FONT_SET_WIDTH_NORMAL,
    WT_FONT_SET_WIDTH_NARROW,
    WT_FONT_SET_WIDTH_SEMICONDENSED
} WTfont_set_width;

/*--------------------------------------------------------------------*/
/*                           Type definitions                         */
/*--------------------------------------------------------------------*/

/* Forward references to data structure types */

typedef struct _WTcallback       WTcallback;
typedef struct _WTdisp           WTdisp;
typedef union  _WTevent          WTevent;
typedef struct _WTevent_handler  WTevent_handler;
typedef struct _WTudata          WTudata;
typedef struct _WTwind           WTwind;

/* Opaque pointer types */

typedef struct _WTpixmap            *WTpixmap_ptr;
typedef struct _WTdecor             *WTdecor_ptr;
typedef struct _WTdecor_class_rec   *WTdecor_class;
typedef struct _WTcursor_class_rec  *WTcursor_class;
typedef struct _WTfontset           *WTfontset_ptr;

/*--------------------------------------------------------------------*/

/*  Window-system independent display pointer */

#ifdef WIN32
typedef HINSTANCE  WTsys_disp;
#else
typedef Display   *WTsys_disp;
#endif

/* Window-system independent window pointer */

#ifdef WIN32
typedef HWND    WTsys_wind;
#else
typedef Widget  WTsys_wind;
#endif

/* Window-system independent color palette */

#ifdef WIN32
typedef HPALETTE WTsys_pal;
#else
typedef Colormap WTsys_pal;
#endif

/* Window-system independent popup menu pointer */

#ifdef WIN32
typedef HMENU   WTpopup_menu;
#else
typedef Widget  WTpopup_menu;
#endif

/* Coordinate value */

typedef float  WTcoord;

/* X, Y point value */

typedef WTcoord  WTvec2[2];

/* Rectangle structure */

typedef struct _WTrect {
    WTcoord  x, y, w, h;
} WTrect;

/* RGB color value (component range: 0.0 to 1.0) */

typedef float  WTcolor[3];

/* ARGB value from AVS image file (component range: 0 to 255) */

typedef unsigned char  WTicolor[4];

/* Window system pixel value */

#ifdef WIN32
typedef COLORREF  WTpixel;
#else
typedef Pixel     WTpixel;
#endif

/*--------------------------------------------------------------------*/

/* Window event handler function */

typedef void (*WTevent_handler_func) (
    WTwind   *wind,
    WTevent  *event,
    void     *arg);

/* Window callback function */

typedef void (*WTcallback_func) (
    WTwind             *wind,
    WTcallback_reason  reason,
    void               *client_data,
    void               *call_data);

/* Display context pixel function */

typedef WTpixel (*WTpixel_func) (
    WTdisp   *disp,
    WTcolor  color,
    void     *data);

/*--------------------------------------------------------------------*/

/* Window method prototypes: */

/* Method for initializing a window */

typedef void (*WTinit_wind_func) (
    WTwind  *wind);

/* Method for handling window destruction */

typedef void (*WTdestroy_wind_func) (
    WTwind  *wind);

/* Method for handling system window destruction */

typedef void (*WTdestroy_sys_wind_func) (
    WTwind  *wind);

/* Method for handling window realization */

typedef void (*WTrealize_wind_func) (
    WTwind  *wind);

/* Method for handling window configures */

typedef void (*WTreshape_wind_func) (
    WTwind  *wind);

/*--------------------------------------------------------------------*/

/* Event types */

/* Mouse button event */

typedef struct _WTbutton_event {
    int           type;        /* Must be first */
    int           xi, yi;
    WTcoord      x, y;
    WTcoord      press_x;     /* Coordinates of the "press" */
    WTcoord      press_y;     /* Coordinates of the "press" */
    time_t  	 press_time;  /* Time in milliseconds */
    int           press_xi;    /* Press integer coordinates     */
    int           press_yi;
    WTcoord      last_x;      /* Last motion event coordinates */
    WTcoord      last_y;
    int           last_xi;     /* Last integer coordinates     */
    int           last_yi;
    int           button;      /* Button that was pressed */
    time_t		  time;        /* Time in milliseconds */
    unsigned int  shift:1;     /* 1 if the shift key was down */
    unsigned int  control:1;   /* 1 if the control key was down */
    unsigned int  dclick:1;    /* 1 if this event is a double-click release */
} WTbutton_event;

/* Graphics expose event */

typedef struct _WTexpose_event {
    int  type;  /* Must be first */
} WTexpose_event;

/* Window resize event */

typedef struct _WTresize_event {
    int           type;         /* Must be first */
    int           xi, yi;
    unsigned int  wi, hi;
    WTcoord      x, y, w, h;
} WTresize_event;

/* Window visibility change event */

typedef struct _WTvis_event {
    int  type;  /* Must be first */
    int  vis;
} WTvis_event;

/* Union of all event types */

union _WTevent {
    int              type;    /* Must be first */
    WTbutton_event  button;
    WTexpose_event  expose;
    WTresize_event  resize;
    WTvis_event     vis;
};

/*--------------------------------------------------------------------*/

/* Member of a linked list of pending rubberband boxes */

typedef struct _WTpend_box {

    int                  hide_count;
    WTwind              *window;
    WTcolor             color;
    WTcoord             x;
    WTcoord             y;
    WTcoord             width;
    WTcoord             height;
    struct _WTpend_box  *next;

} WTpend_box;

/*--------------------------------------------------------------------*/

/* Display data structure.  This corresponds to an X display or Windows
   instance handle. */

struct _WTdisp {

    WTsys_disp      sys_disp;

#ifdef WIN32
    char            *class_name;
#else
    Screen          *screen;
    Colormap        colormap;
    int             depth;
    int             bits_per_pixel;
    Visual          *visual;
    GC              gc;
    GC              xorgc;
#endif

    WTwind          **roots;      /* List of root WTwinds */
    int             num_roots;
    int             num_roots_alloced;
    WTpend_box      *pend_list;   /* Head of pending rubberband box list */

    /*
     * Here we cache information about the pointer state... this allows 
     * the event handler to provide user interaction history to the
     * guy that is handling these events.
     */

    WTbutton_event  last_press;
    WTbutton_event  last_motion;
    time_t          dclick_time;   /* -1 or the time of the way last press */
    unsigned int    press_valid:1;
    unsigned int    press_moved:1; /* True if this press has moved */
    WTwind          *press_window; /* Window for current press */

    WTpixel_func    pixel_func;
    void            *pixel_func_data;

    WTwind          *pointer_window;

    struct _WTdisp  *next;
};

/*--------------------------------------------------------------------*/

/* "Region" structure.  Generally used to define the position, size, and
   transformation for a window. */

typedef struct _WTregion {
    WTcoord  offx, offy;  /* Offset to add to coordinates */
    WTcoord  x, y, w, h;  /* Location and size in parent's coordinates */
    float     loc_scale;   /* Local scale (relative to our parent) */
    float     scale;       /* Absolute scale (to pixels) */
} WTregion;

/* "User data" to be stored with a window */

struct _WTudata {
    char      id[4];
    void      *uptr;
    WTudata  *next;
};

/* Entry in an option menu */

typedef struct _WToption {
    char         *string;
    WTsys_wind  sys_wind;
} WToption;

/*--------------------------------------------------------------------*/

/* Class record for a window */

typedef struct _WTwind_type {

    WTinit_wind_func         init_wind;
    WTdestroy_wind_func      destroy_wind;
    WTdestroy_sys_wind_func  destroy_sys_wind;
    WTrealize_wind_func      realize_wind;
    WTreshape_wind_func      reshape_wind;
    int                      use_decor_handler;

} WTwind_type;

/*--------------------------------------------------------------------*/

/* Data structure for a window */

struct _WTwind {
    WTwind_type      *type;           /* Pointer to class record */
    WTsys_wind       sys_wind;        /* System window pointer */
    WTsys_wind       child_control;   /* Optional subwindow */
    WTregion         r;               /* Region defining our coord system */
    WTdecor_ptr      decor;           /* Pointer to decoration object */
    WTdisp           *disp;           /* Pointer to display context   */
    WTwind           *parent;         /* Back pointer to parent window */
    WTsys_wind       parent_sys_wind; /* Parent widget if no parent window */
    WTwind           **children;      /* Array of children */
    int              num_children;
    int              num_children_alloced;

    unsigned int     mapped:1;        /* If the window is mapped */
    unsigned int     highlighted:1;   /* If the window is highlighted */
    unsigned int     created:1;       /* If the window was "created"
                                         (not inited) */
    unsigned int     visibility:2;    /* WT_VIS_CLEAR, WT_VIS_HIDDEN,
                                          WT_OVERLAPPED */
    unsigned int     cleared:1;
    unsigned int     typein_modified:1;
    unsigned int     typein_multiline:1;
    unsigned int     typein_scroll:1;
    unsigned int     being_destroyed:1;
    unsigned int     delivering_event:1;
    unsigned int     calling_callback:1;
    unsigned int     button_in:1;
    unsigned int     scroll_horiz:1;
    unsigned int     scroll_pressed:1;
    unsigned int     scrolling:1;
    unsigned int     fixed_stacking_position:1;
    unsigned int     enabled:1;

    unsigned long    bg_pixel;
    WTpixmap_ptr     bg_pixmap;
#ifdef WIN32
    HBRUSH           bg_hbrush;
#else
    Pixmap           bg_X_pixmap;
#endif
    WTudata          *udata;
    WTevent_handler  *event_handlers;
    WTcallback       *callbacks;

    unsigned int     timer_interval;
    WTevent_handler_func  timer_func;
    void             *timer_arg;
#ifdef WIN32
    WNDPROC          orig_wnd_proc;
    WNDPROC          orig_child_wnd_proc;
    UINT             timer;
#else
    int              timer;
#endif
    char             *typein_string;
    WTfontset_ptr    typein_fontset;

    WTbutton_type    button_type;

    int              scroll_position;
    int              scroll_page_incr;
    int              scroll_line_incr;
    int              scroll_min;
    int              scroll_max;
    int              scroll_slider_size;

    WToption        *options;
    int              num_options;
    int              current_option;
#ifndef WIN32
    WTsys_wind       option_pulldown;
    void             *xmfontlist;
#endif
};

/*--------------------------------------------------------------------*/

/* Event handler structure.  Member of a linked list. */

struct _WTevent_handler {
    unsigned int              mask;
    WTevent_handler_func     func;
    void                      *arg;
    int                       pending;
    struct _WTevent_handler  *next;
};

/*--------------------------------------------------------------------*/

/* Callback structure.  Member of a linked list. */

struct _WTcallback {
    WTcallback_reason   reason;
    WTcallback_func     func;
    void                 *arg;
    int                  pending;
    struct _WTcallback  *next;
};

/*--------------------------------------------------------------------*/

/* Data structure for font attributes */

typedef struct _WTfont_attributes {
    int               family;
    int               height;
    WTfont_weight     weight;
    WTfont_slant      slant;
    WTfont_set_width  set_width;
    char              *user_font;
} WTfont_attributes;

/*--------------------------------------------------------------------*/
/*                       External references                          */
/*--------------------------------------------------------------------*/

#ifdef __cplusplus  /* Do not leave open across includes */
extern "C" {        /* For C++ V2.0 */
#endif

#ifdef XP_INTERNAL_GLOBALS
#define WT_EXTERN IMPORT
#else
#define WT_EXTERN DLL_IMPORT
#endif

/* References to available window types */

WT_EXTERN WTwind_type  *WT_container_wind_type, *WT_primitive_wind_type,
  		       *WT_typein_wind_type, *WT_button_wind_type,
  		       *WT_scrollbar_wind_type, *WT_option_menu_wind_type;

/* References to available decor types */

WT_EXTERN WTdecor_class  WT_decor_window_bevel, WT_decor_selected_bevel,
  			 WT_decor_custom_bevel;

/* References to available cursor types */

WT_EXTERN WTcursor_class  WT_hourglass_cursor, WT_rotation_cursor,
			  WT_translation_cursor, WT_merge_cursor,
			  WT_copy_cursor, WT_reparent_cursor,
  			  WT_connect_cursor;

#ifdef __cplusplus
}  /* End extern "C" */
#endif

/*--------------------------------------------------------------------*/

#endif /* _WT_TYPE_H_INCLUDED */
/* DON'T ADD ANYTHING AFTER THIS #endif */
