/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/advect.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>

#define ERR_RETURN(A) ERRerror("advec",1,ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define ALLOC_SIZE 256

int FUNCadvec (OMobj_id in, double time, double start_time,
               double release_interval, int dir, OMobj_id out);

int DVadvect_update(OMobj_id mod_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id in, out;
	double  time, start_time, release_interval;
	int dir;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out"), OM_OBJ_RW);
	if (OMget_name_real_val(mod_id, OMstr_to_name("time"), &time) != OM_STAT_SUCCESS)
		return(0);
	if (OMget_name_real_val(mod_id, OMstr_to_name("start_time"), &start_time) != OM_STAT_SUCCESS)
		start_time = 0.0;
	if (OMget_name_real_val(mod_id, OMstr_to_name("release_interval"), &release_interval) != OM_STAT_SUCCESS)
		release_interval = 0.0;
	if (OMget_name_int_val(mod_id, OMstr_to_name("direction"), &dir) != OM_STAT_SUCCESS)
		return(0);

	return
		(FUNCadvec(in, time, start_time, release_interval, dir, out));
}

/* 64-bit porting. Only Modified Internally */
int FUNCadvec (OMobj_id in, double time, double start_time,
               double release_interval, int dir, OMobj_id out)
{
	xp_long i_w, cell, node, nnodes, out_nnodes;
	int   i, rdir, nspace, data_type, veclen, ncomp;
	xp_long *node_list, *out_node_list, ncells, size;
	int nsets, cs;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	float vel[3], vel1[3], vel2[3], del[3], dt, rt, t, mag, len, *coord, *out_coord, *node_data, *out_data;
	OMobj_id   cell_set;
	xp_long ntimes, tidx, out_size;
	double *times, rel_time;


	/************************************************************/
	/*  if streamlines are 0-length clean the output and return */
	/************************************************************/
	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (nnodes == 0) {
		if (FLDset_nnodes (out, 0) != 1) {
			ERR_RETURN("Error setting nnodes");
		}
		if (FLDset_ncell_sets(out, 0) != 1) {
			ERR_RETURN("cannot set nsets");
		}
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}

		return(1);
	}

	if (dir)
		rdir = 1;
	else
		rdir = -1;

	/*****************************************/
	/* get stremalines coordinates and data  */
	/*****************************************/
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (ncomp == 0) {
		ERR_RETURN("advec: no time  component in the input field");
	}
	if (FLDget_node_data_veclen(in, 0, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}
	data_type = DTYPE_FLOAT;
	if (FLDget_node_data(in, 0, &data_type, (char **)(&node_data), &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_coord(in, &coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates");
	}

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	out_data = out_coord = NULL;
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coord");
	}

	/**********************************/
	/*   Pre-allocated output arrays  */
	/**********************************/
	out_size = ALLOC_SIZE;
	out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, nspace*out_size, NULL);
	if (out_coord == NULL) {
		ERR_RETURN("cannot allocate coordinates");
	}
	out_data = (float *)ARRalloc(NULL, DTYPE_FLOAT, veclen*out_size, NULL);
	if (out_coord == NULL) {
		ERR_RETURN("cannot allocate node data");
	}
	out_nnodes=0;

	/**********************************************************/
	/* compute number of times at which to release particles  */
	/**********************************************************/
	if (release_interval != 0.0) {
		ntimes = (time - start_time) / release_interval + 1.0001;
		if (ntimes < 1)
			return(0);
	}
	else
		ntimes = 1;

	/********************************************************/
	/* compute array of times at which to release particles */
	/********************************************************/
	times = (double *)ARRalloc(NULL, DTYPE_DOUBLE, ntimes, NULL);
	if (times == NULL) {
		ERR_RETURN("cannot allocate time array");
	}
	for( rel_time = time, i_w = ntimes; i_w > 0; rel_time -= release_interval, i_w-- ) {
		if (rel_time < 0.0)
			rel_time = 0.0;
		times[i_w-1] = rel_time;
	}


	/**************************************************************/
	/* for each streamline (polyline) compute particles locations */
	/**************************************************************/
	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	for (cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_npolys(cell_set, &ncells) != 1) {
			ERR_RETURN("cannot get ncells");
		}
		if (FLDget_poly_connect(cell_set, &node_list,
					&size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}
		for (cell=0; cell<ncells; cell++) {
			for(t=0.0, tidx=0, node=node_list[cell*2]; node<node_list[cell*2+1]; node++) {
				vel1[0]=vel1[1]=vel1[2]=0.0;
				vel2[0]=vel2[1]=vel2[2]=0.0;
				memcpy(vel1, node_data+node*veclen,
				       veclen*sizeof(float));
				memcpy(vel2, node_data+(node+1)*veclen,
				       veclen*sizeof(float));

				/******************************************************************/
				/* compute average velocity along line segment, and mag(velocity) */
				/******************************************************************/
				for (i=0; i<3; i++)
					vel[i] = 0.5*(vel1[i]+vel2[i]);
				for (mag=0.0, i=0; i<veclen; i++)
					mag += vel[i]*vel[i];
				mag = sqrt((double)mag);

				/**************************************/
				/* compute the length of line segment */
				/**************************************/
				for (i=0; i<nspace; i++)
					del[i] = coord[node*nspace+i]-
						 coord[(node+1)*nspace+i];
				for (len=0.0, i=0; i<nspace; i++)
					len += del[i]*del[i];
				len = sqrt((double)len);

				/**************************************************/
				/* compute delta-time represented by line segment */
				/**************************************************/
				if (mag != 0.0)
					dt =len/mag;
				else
					dt = 0;
				t += dt;

				/*******************************************************/
				/* keep walking streamline until t > next release time */
				/*******************************************************/

				if ( (tidx < ntimes) && (t < times[tidx]) )
					continue;

				while( (tidx < ntimes) && (t >= times[tidx]) ) {
					/**********************************************/
					/* compute particle location for each release */
					/**********************************************/
					rt = rdir*(dt-t+times[tidx]);
					if (out_nnodes >= out_size) {
						out_size += ALLOC_SIZE;
						out_coord = (float *)ARRrealloc(out_coord, DTYPE_FLOAT, nspace*out_size, NULL);
						out_data =  (float *)ARRrealloc(out_data,  DTYPE_FLOAT, veclen*out_size, NULL);
					}
					for (i=0; i<nspace; i++) {
						out_coord[nspace*out_nnodes+i] =
							coord[nspace*node+i]+ rt*vel[i];
					}
					for (i=0; i<veclen; i++) {
						out_data[veclen*out_nnodes+i] = vel[i];
					}
					out_nnodes++;

					/* next release */
					tidx++;
				}
			}
		}
		if (node_list)
			ARRfree(node_list);
	}
	/********************/
	/*** OUTPUT Field ***/
	/********************/
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (out_nnodes) {
		if (FLDset_coord(out, out_coord, nspace*out_nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting coord");
		}
	}
	else if (out_coord != NULL)
		ARRfree(out_coord);

	/*********************************/
	/*  put VELOCITY into Node_Data  */
	/*********************************/
	if (FLDset_node_data_ncomp (out, 1) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDget_node_data_units(in, 0, units, MAX_NAME_SIZE) != 1) {
		strcpy(units, "");
	}
	if (FLDget_node_data_label(in, 0, label, MAX_NAME_SIZE) != 1) {
		strcpy(label, "");
	}
	if (FLDset_node_data_comp (out, 0, veclen, label, units) != 1) {
		ERR_RETURN("Error setting node component");
	}
	if (out_nnodes) {
		if (FLDset_node_data(out, 0, (char *)out_data, DTYPE_FLOAT,
				     veclen*out_nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDcopy_node_minmax(in, out, 0, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, 0, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
	}
	else if (out_data != NULL)
	      	ARRfree(out_data);

	/*---------------*/
	/*   OUT CELLS   */
	/*---------------*/
	if (FLDset_ncell_sets(out, 1) != 1) {
		ERR_RETURN("cannot set nsets");
	}
	if (FLDget_cell_set(out, 0, &cell_set) != 1) {
		ERR_RETURN("Error getting cell set");
	}

	if (FLDset_cell_set(cell_set, "Point") != 1) {
		ERR_RETURN("Error setting cell type");
	}

	if (FLDset_ncells(cell_set, out_nnodes) != 1) {
		ERR_RETURN("Error setting ncells");
	}
	if (out_nnodes) {
		if (FLDget_node_connect(cell_set, &out_node_list,
					&size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}
		for (i_w=0; i_w<out_nnodes; i_w++)
			out_node_list[i_w] = i_w;
		ARRfree(out_node_list);
	}

	if (times)
		ARRfree(times);
	if (node_data)
		ARRfree(node_data);
	if (coord)
		ARRfree(coord);
	return(1);
}
