/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cnct_xfm.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/gd.h>

#define MAX_NAME_SIZE 1024
#define NPOINTS  1024

static float IXFORM[16] = {1.0, 0.0, 0.0, 0.0,
			   0.0, 1.0, 0.0, 0.0,
			   0.0, 0.0, 1.0, 0.0,
			   0.0, 0.0, 0.0, 1.0};

#define ERR_RETURN(A) ERRerror("copy_xform", 0, ERR_ORIG, A); return(0);

int DVcopy_xform_update(OMobj_id obj_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id in, out;
	int seq_in, seq_out;
	int stat;

	in = OMfind_subobj(obj_id, OMstr_to_name("xform_in"), OM_OBJ_RD);
	out= OMfind_subobj(obj_id, OMstr_to_name("xform_out"), OM_OBJ_RW);

	if (OMis_null_obj(in)) {
		ERR_RETURN("cannot get xform_in");
	}
	if (OMis_null_obj(out)) {
		ERR_RETURN("cannot get xform_out");
	}

	seq_in = OMget_obj_seq(in, OMnull_obj, OM_SEQ_VAL);
	seq_out = OMget_obj_seq(out, OMnull_obj, OM_SEQ_VAL);

	if (seq_in > seq_out) {
		stat = OMset_obj_val(out, in, 0);
		if (stat != OM_STAT_SUCCESS) {
			ERR_RETURN("cannot set xform_out");
		}
	}
	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("concat_xform", 0, ERR_ORIG, A); return(0);

int DVconcat_xform_update(OMobj_id  elem_id)
{
	OMobj_id elm0, elm1, elm2;
	float rmat[16], mat[16], xlate[3], center[3];
	float mat1[16], mat2[16];
	int i;

	elm0 = OMfind_subobj(elem_id, OMstr_to_name("xform_in1"), OM_OBJ_RD);
	elm1 = OMfind_subobj(elem_id, OMstr_to_name("xform_in2"), OM_OBJ_RD);
	elm2 = OMfind_subobj(elem_id, OMstr_to_name("xform_out"), OM_OBJ_RW);

	if (OMis_null_obj(elm0)) {
		ERR_RETURN("cannot get xform_in1");
	}
	if (OMis_null_obj(elm1)) {
		ERR_RETURN("cannot get xform_in2");
	}
	if (OMis_null_obj(elm2)) {
		ERR_RETURN("cannot get xform_out");
	}

	GDxform_get_comp_matrix(elm0, mat1);
	GDxform_get_comp_matrix(elm1, mat2);

	MATmat4_multiply((Matr4 *)mat1, (Matr4 *)mat2, (Matr4 *)mat);

	memcpy(rmat, IXFORM, 16*sizeof(float));

	for (i=0; i<3;i++)
		memcpy(rmat+4*i, mat+4*i, 3*sizeof(float));

	memcpy(xlate, mat+12, 3*sizeof(float));

	center[0] = center[1] = center[2] = 0.0;

	GDxform_set_matrix(elm2, (Matr4 *)rmat);
	GDxform_set_xlate(elm2, xlate);
	GDxform_set_center(elm2, center);

	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("invert_xform", 0, ERR_ORIG, A); return(0);

int DVinvert_xform_update(OMobj_id  elem_id)
{
	OMobj_id in, out;
	float rmat[16], mat[16], t_mat[16], xlate[3], center[3];
	int i;

	in  = OMfind_subobj(elem_id, OMstr_to_name("xform_in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("xform_out"), OM_OBJ_RW);

	if (OMis_null_obj(in)) {
		ERR_RETURN("cannot get xform_in");
	}
	if (OMis_null_obj(out)) {
		ERR_RETURN("cannot get xform_out");
	}

	GDxform_get_comp_matrix(in, mat);

	MATmat4_inverse((Matr4 *)t_mat, (Matr4 *)mat);

	memcpy(rmat, IXFORM, 16*sizeof(float));
	for (i=0; i<3;i++)
		memcpy(rmat+4*i, t_mat+4*i, 3*sizeof(float));
	memcpy(xlate, t_mat+12, 3*sizeof(float));

	center[0]=center[1]=center[2]=0;

	GDxform_set_matrix(out, (Matr4 *)rmat);
	GDxform_set_xlate(out, xlate);
	GDxform_set_center(out, center);

	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("rotate_xform", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVrotate_xform_update(OMobj_id  elem_id)
{
	OMobj_id out;
	int i, axis, stat;
	double angle;
	float rmat[4][4], mat[4][4], xlate[3], center[3], *pcenter;
	xp_long size;

	if (OMget_name_real_val(elem_id, OMstr_to_name("angle"), &angle) != 1)
		angle = 0.0;
	if (OMget_name_int_val(elem_id, OMstr_to_name("axis"), &axis) != 1)
		axis = 0;
	stat = FLDget_array_float(elem_id, "center", &pcenter, &size, OM_GET_ARRAY_RD);
	if (stat != 1 || stat == 0 || pcenter == NULL)
		center[0]=center[1]=center[2]=0;
	else
		memcpy(center, pcenter, 3*sizeof(float));

	out = OMfind_subobj(elem_id, OMstr_to_name("xform_out"), OM_OBJ_RW);

	memcpy(rmat, IXFORM, 16*sizeof(float));
	switch (axis) {
	      case 0:
		rmat[1][1] = cos(angle);
		rmat[1][2] = sin(angle);
		rmat[2][1] = -1*sin(angle);
		rmat[2][2] = cos(angle);
		break;
	      case 1:
		rmat[0][0] = cos(angle);
		rmat[0][2] = -1*sin(angle);
		rmat[2][0] = sin(angle);
		rmat[2][2] = cos(angle);
		break;
	      case 2:
		rmat[0][0] = cos(angle);
		rmat[0][1] = sin(angle);
		rmat[1][0] = -1*sin(angle);
		rmat[1][1] = cos(angle);
		break;
	      default:
		break;
	}

	MATmat4_translate(mat, -center[0], -center[1], -center[2]);
	MATmat4_multiply(mat, rmat, mat);

	MATmat4_translate(rmat, center[0], center[1], center[2]);
	MATmat4_multiply(mat, rmat, mat);

	center[0]=center[1]=center[2]=0;
	memcpy(xlate, mat[3], 3*sizeof(float));
	memcpy(rmat, IXFORM, 16*sizeof(float));
	for (i=0; i<3;i++)
		memcpy(rmat[i], mat[i], 3*sizeof(float));

	GDxform_set_matrix(out, rmat);
	GDxform_set_xlate(out, xlate);
	GDxform_set_center(out, center);

	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("xform_coord", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVxform_coord_update(OMobj_id  elem_id)
{
	OMobj_id in;
	int stat, nspace, *comp, trans;
	xp_long nnodes, i, n, size, ncomp;
	float *in_xyz, *out_xyz, point[3], xform[4][4];

	stat = FLDget_array_int(elem_id, "comp", &comp, &ncomp, OM_GET_ARRAY_RD);
	if (stat != 1 || ncomp == 0 || comp == NULL) {
		OMset_name_int_val(elem_id, OMstr_to_name("coord_nspace"), 0);
		FLDset_array_float (elem_id, "coord", 0, NULL, OM_SET_ARRAY_FREE);
		return(0);
	}

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);

	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_coord(in, &in_xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates");
	}

	stat = FLDget_xform(in, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for in");
	}
	else if (stat == 0 || MATmat_is_identity((float *)xform, 4))
		trans = 0;
	else
		trans = 1;

	OMset_name_int_val(elem_id, OMstr_to_name("coord_nspace"), (int)ncomp);
	stat = FLDget_array_float (elem_id, "coord", &out_xyz, &size, OM_GET_ARRAY_RW);
	if (stat != 1) {
		ERR_RETURN("cannot get out coord");
	}
	for (n=0; n<nnodes; n++) {
		point[0]=point[1]=point[2]=0.0;
		memcpy(point, in_xyz+n*nspace, nspace*sizeof(float));
		if (trans)
			MATvec3_mat4_multiply(point, (Matr4 *)xform);
		for(i=0; i<ncomp; i++)
			out_xyz[n*ncomp+i] = point[comp[i]];
	}
	ARRfree(in_xyz);
	ARRfree(out_xyz);
	ARRfree(comp);
	return(1);
}
