/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cont_sol.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("solid_contour", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define TEST_ID(ID,N) \
	if (OMis_null_obj(ID)) { \
	  sprintf( err_msg, "Error getting subobject id for %s", N ); \
	  ERRerror( "solid_contour", 0, ERR_ORIG, err_msg ); \
	  return METHOD_FAILURE; }

#define TEST_STAT(S,N) \
	if (S != OM_STAT_SUCCESS) { \
	  sprintf( err_msg, "Error getting subobject value for %s", N ); \
	  ERRerror( "solid_contour", 0, ERR_ORIG, err_msg ); \
	  return METHOD_FAILURE; }

#define MAX_NAME_SIZE 1024

#define MAX_NSETS 6

static char *cell_name[]= {"Tet", "Prism", "Pyr", "Hex", "Tri", "Quad"};

/* Arguments reordered to get around a IA64 compiler bug (gcc-2.96) */
int FUNCsolid_contour (OMobj_id in, OMobj_id out, OMobj_id out_line,
                       int ncont, float *level, int contour_comp,
                       int contour_lines, int color_lines);

/* 64-bit porting. Only Modified Internally */
int DVsolid_contour_update(OMobj_id elem_id)
{
    OMobj_id in, out, out_line, level_id, contour_comp_id;
    float    *level;
    xp_long  ncont;
    int      contour_comp;
    int      stat, type;
    char     err_msg[128];
    int      contour_lines, color_lines;

    in  = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    out_line = OMfind_subobj(elem_id, OMstr_to_name("lines"), OM_OBJ_RW);
    TEST_ID( out_line, "lines" );
    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);
    TEST_ID( level_id, "level" );

    type  = OM_TYPE_FLOAT;
    level = NULL;
    ncont = 0;

    stat = OMget_array_sz( level_id, &type, (char **)(&level),
                           &ncont, OM_GET_ARRAY_RD);
    TEST_STAT( stat, "level array" );

    contour_comp_id = OMfind_subobj(elem_id, OMstr_to_name("contour_comp"),
                                    OM_OBJ_RD);
    TEST_ID( contour_comp_id, "contour_comp" );

    stat = OMget_int_val(contour_comp_id, &contour_comp);
    TEST_STAT( stat, "contour_comp" );

    if (OMget_name_int_val(elem_id, OMstr_to_name("contour_lines"), &contour_lines) != 1)
        contour_lines = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("color_lines"), &color_lines) != 1)
        color_lines = 0;

    stat = FUNCsolid_contour(in, out, out_line, ncont, level, contour_comp, contour_lines, color_lines);

    if (level) ARRfree(level);

    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRsolid_contour_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, out_arr, out_line_arr;
    xp_long i, num_fields;

    OMobj_id in, out, out_line, level_id, contour_comp_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    float    *level;
    xp_long  ncont;
    int      contour_comp;
    int      stat, type;
    char     err_msg[128];
    int      contour_lines, color_lines;

    in_arr  = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    out_line_arr = OMfind_subobj(elem_id, OMstr_to_name("lines"), OM_OBJ_RW);

    stat = OMset_array_size( out_line_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);

    type  = OM_TYPE_FLOAT;
    level = NULL;
    ncont = 0;

    stat = OMget_array_sz( level_id, &type, (char **)(&level),
                           &ncont, OM_GET_ARRAY_RD);
    TEST_STAT( stat, "level array" );

    contour_comp_id = OMfind_subobj(elem_id, OMstr_to_name("contour_comp"),
                                    OM_OBJ_RD);
    TEST_ID( contour_comp_id, "contour_comp" );

    stat = OMget_int_val(contour_comp_id, &contour_comp);
    TEST_STAT( stat, "contour_comp" );

    if (OMget_name_int_val(elem_id, OMstr_to_name("contour_lines"), &contour_lines) != 1)
        contour_lines = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("color_lines"), &color_lines) != 1)
        color_lines = 0;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr,  i, &in,  OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_line_arr, i, &out_line, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCsolid_contour(in, out, out_line, ncont, level, 
                                 contour_comp, contour_lines, color_lines);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "solid_contour", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif
        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(in,  xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(out, xform_name, OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    if (level) ARRfree(level);

    return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCsolid_contour (OMobj_id in, OMobj_id out, OMobj_id out_line,
		       int ncont, float *level, int contour_comp,
		       int contour_lines, int color_lines)
{
	int   i, nl, nsets, nspace, veclen, ncomp, null_flag;
	xp_long j, nl_w, nnodes, out_ncells[MAX_NSETS], out_nnodes, size, conn_size[MAX_NSETS]; 
	xp_long *out_nlist[MAX_NSETS], *n_out_nnodes;
	xp_long *min_node_list, *max_node_list;
	xp_long *old_nlist, new_nnodes, old_nnodes ;
	int   stat, dtype;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	float  *out_coord, **r_out_coord;
	char   *node_data, *cell_data, *list;
	char   *mesh_info;
	double min, max, l1, l2, null_value = 0;
	OMobj_id   cell_set;

	int   nline_sets;
	xp_long nline1 = 0, nline2 = 0;
	xp_long *line_conn1, *line_conn2;

        /********************************/
        /*   Test arguments             */
        /********************************/

	if (ncont < 2) {
		ERR_RETURN("number of contours should be greater than 1, contour is not performed");
	}

        /********************************/
        /*   Alloc & init local arrays  */
        /********************************/

	node_data = cell_data = list =NULL;
	min_node_list = max_node_list = old_nlist = NULL;

	r_out_coord  = (float **)malloc((ncont-1)*sizeof(float*));
        if (!r_out_coord) {
                ERR_RETURN("Memory allocation failed");
        }

	n_out_nnodes = (xp_long *)malloc((ncont-1)*sizeof(xp_long));
        if (!n_out_nnodes) {
                free( r_out_coord );
                ERR_RETURN("Memory allocation failed");
        }

        for (i=0; i<ncont-1; i++) {
            r_out_coord[i]  = NULL;
            n_out_nnodes[i] = 0;
        }

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	out_coord = NULL;
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	if (FLDset_ncell_sets (out_line, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}

	if (nnodes == 0 || ncomp == 0) {
		return METHOD_SUCCESS;
	}
	if (FLDget_node_data_veclen(in, contour_comp, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}

	if (veclen != 1) {
		ERR_RETURN("component is not scalar, contour is not performed");
	}

	if (FLDget_node_data(in, contour_comp, &dtype, &node_data,
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, contour_comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}

	if (FLDget_node_data_units(in, contour_comp,
				   units, MAX_NAME_SIZE) != 1) {
		strcpy(units, "");
	}
	if (FLDget_node_data_label(in, contour_comp,
				   label, MAX_NAME_SIZE) != 1) {
		strcpy(label, "");
	}
	if (FLDget_node_data_minmax(in, contour_comp, (char *)&min, (char *)&max) != 1) {
		ERR_RETURN("Error getting node minmax");
	}

	out_nnodes = 0;
	nsets = 0;
	nline_sets = 0;

	for (nl=0; nl<ncont-1; nl++) {

		l1 = (double) level[nl];
		l2 = (double) level[nl+1];

		if (l1 > l2) continue;

		if (contour_lines) {
			if (nl == 0) {
				nline1 = 1;
				nline2 = 1;
			}
			else {
				nline1 = 0;
				nline2 = 1;
			}
		}

                line_conn1 = line_conn2 = NULL;

		OMpush_status_range((int)nl*100/(ncont-1), (int)(nl+1)*100/(ncont-1));
		stat = UTILcontour(mesh_info, node_data, dtype,
				   null_flag, (char *)&null_value,
				   l1, l2,  &old_nnodes, &old_nlist,
				   &new_nnodes, &min_node_list, &max_node_list,
				   &list, out_ncells, out_nlist, conn_size,
				   nspace, &r_out_coord[nl],
				   &nline1, &line_conn1, &nline2, &line_conn2);
		OMpop_status_range();

		if (min_node_list)
			ARRfree(min_node_list);
		if (max_node_list)
			ARRfree(max_node_list);
		if (list)
			ARRfree(list);
		if (old_nlist)
			free(old_nlist);

		if (stat != 1) {
			if (node_data)
				ARRfree(node_data);
			FLDfree_mesh_info(mesh_info);
			ERR_RETURN("no contour created");
		}

		for (i=0; i<MAX_NSETS; i++) {
			if (out_ncells[i]) {
				if (nl)
					for (j=0;j<conn_size[i];j++)
						out_nlist[i][j] += out_nnodes;
				if (FLDadd_cell_set(out, cell_name[i]) != 1) {
					ERR_RETURN("Error adding cell set");
				}
				if (FLDget_cell_set(out, nsets, &cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDset_ncells(cell_set, out_ncells[i]) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDset_node_connect(cell_set, out_nlist[i], conn_size[i],
							OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell connect list");
				}
				if (FLDset_cell_data_ncomp(cell_set, 1) != 1) {
					ERR_RETURN("Error getting ncomp");
				}
				if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
					ERR_RETURN("Error setting node component");
				}
				if (FLDget_cell_data(cell_set, 0, &dtype, &cell_data,
						     &size, OM_GET_ARRAY_WR) != 1) {
					ERR_RETURN("Error setting cell data");
				}
				UTILinit_array(cell_data, out_ncells[i],
					       (double)(l1+l2)/2.0, dtype);
				if (FLDset_cell_data_minmax(cell_set, 0,
							    (char *)&min, (char *)&max, dtype) != 1) {
					ERR_RETURN("Error setting cell minmax");
				}
				nsets++;
				if (cell_data)
					ARRfree(cell_data);
			}
		}
		/*** Lines ****/
		if (contour_lines) {
			if (nline1 && line_conn1) {
				if (nl)
					for (j=0;j<nline1*2;j++)
						line_conn1[j] += out_nnodes;
				if (FLDadd_cell_set(out_line, "Line") != 1) {
					ERR_RETURN("Error adding cell set");
				}
				if (FLDget_cell_set(out_line, nline_sets, &cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDset_ncells(cell_set, nline1) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDset_node_connect(cell_set, line_conn1, 2*nline1,
							OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell connect list");
				}
				if (color_lines) {
					if (FLDset_cell_data_ncomp(cell_set, 1) != 1) {
						ERR_RETURN("Error getting ncomp");
					}
					if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
						ERR_RETURN("Error setting node component");
					}
					if (FLDget_cell_data(cell_set, 0, &dtype, &cell_data, 
							     &size, OM_GET_ARRAY_WR) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					UTILinit_array(cell_data, nline1, (double)l1, dtype);
					if (FLDset_cell_data_minmax(cell_set, 0, 
								    (char *)&min, (char *)&max, dtype) != 1) {
						ERR_RETURN("Error setting cell minmax");
					}
					if (cell_data)
						ARRfree(cell_data);
				}
				else {
					if (FLDset_cell_data_ncomp(cell_set, 0) != 1) {
						ERR_RETURN("Error getting ncomp");
					}
				}
				nline_sets++;
			}
			else if( line_conn1 ) {
				ARRfree( line_conn1 );
			}

			if (nline2 && line_conn2) {
				if (nl)
					for (j=0;j<nline2*2;j++)
						line_conn2[j] += out_nnodes;
				if (FLDadd_cell_set(out_line, "Line") != 1) {
					ERR_RETURN("Error adding cell set");
				}
				if (FLDget_cell_set(out_line, nline_sets, &cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDset_ncells(cell_set, nline2) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDset_node_connect(cell_set, line_conn2, 2*nline2,
							OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell connect list");
				}
				if (color_lines) {
					if (FLDset_cell_data_ncomp(cell_set, 1) != 1) {
						ERR_RETURN("Error getting ncomp");
					}
					if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
						ERR_RETURN("Error setting node component");
					}
					if (FLDget_cell_data(cell_set, 0, &dtype, &cell_data, 
							     &size, OM_GET_ARRAY_WR) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					UTILinit_array(cell_data, nline2, (double)l2, dtype);
					if (FLDset_cell_data_minmax(cell_set, 0, 
								    (char *)&min, (char *)&max, dtype) != 1) {
						ERR_RETURN("Error setting cell minmax");
					}
					if (cell_data)
						ARRfree(cell_data);
				}
				else {
					if (FLDset_cell_data_ncomp(cell_set, 0) != 1) {
						ERR_RETURN("Error getting ncomp");
					}
				}
				nline_sets++;
			}
			else if( line_conn2 ) {
				ARRfree( line_conn2 );
			}
		}

		n_out_nnodes[nl] = old_nnodes + new_nnodes;
		out_nnodes += n_out_nnodes[nl];
	}

	if (FLDset_ncell_sets (out, nsets) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_ncell_sets (out_line, nline_sets) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (out_nnodes) {
		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}
		for (nl_w=0,i=0; i< ncont-1; i++) {
			if (n_out_nnodes[i]) {
				memcpy(out_coord+nl_w, r_out_coord[i],
				       n_out_nnodes[i]*nspace*sizeof(float));
				nl_w += n_out_nnodes[i]*nspace;
			}
			if (r_out_coord[i])
		           ARRfree(r_out_coord[i]);
		}
		if (out_coord)
			ARRfree(out_coord);
	}
	if (node_data)
		ARRfree(node_data);
	FLDfree_mesh_info(mesh_info);
	free(n_out_nnodes);
	free(r_out_coord);
	return METHOD_SUCCESS;
}
