/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/contour.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("contour", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024
#define MAX_NSETS 6

static char *cell_name[]= {"Tet", "Prism", "Pyr", "Hex", "Tri", "Quad"};

int FUNCcontour (OMobj_id in, double level1, double level2,
                 int contour_comp, int nmap_comp, int *map_comp, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVcontour_update(OMobj_id elem_id)
{
    OMobj_id in, out, level_id, out_comp_id, contour_comp_id;
    double  level1, level2;
    xp_long out_ncomp;
    int *out_comp, contour_comp;
    int stat, type;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level_min"), OM_OBJ_RD);
    OMget_real_val(level_id, &level1);

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level_max"), OM_OBJ_RD);
    OMget_real_val(level_id, &level2);

    out_comp_id = OMfind_subobj(elem_id, OMstr_to_name("map_comp"),
                                OM_OBJ_RD);
    contour_comp_id = OMfind_subobj(elem_id, OMstr_to_name("contour_comp"),
                                    OM_OBJ_RD);

    OMget_int_val(contour_comp_id, &contour_comp);
    type = OM_TYPE_INT;
    out_ncomp = 0;
    out_comp = (int *)NULL;
    stat = OMget_array_sz(out_comp_id, &type, (char **)(&out_comp), 
                          &out_ncomp, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS) {
        out_ncomp = 0;
        out_comp = NULL;
    }

    stat = FUNCcontour(in, level1, level2, contour_comp,
                       (int)out_ncomp, out_comp, out);
    if (out_comp)
        ARRfree(out_comp);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int 
DV_ARRcontour_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, out_arr;
    xp_long i, num_fields;

    OMobj_id in, out, level_id, out_comp_id, contour_comp_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    double  level1, level2;
    xp_long out_ncomp;
    int *out_comp, contour_comp;
    int stat, type;

    in_arr = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level_min"), OM_OBJ_RD);
    OMget_real_val(level_id, &level1);

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level_max"), OM_OBJ_RD);
    OMget_real_val(level_id, &level2);

    out_comp_id = OMfind_subobj(elem_id, OMstr_to_name("map_comp"),
                                OM_OBJ_RD);
    contour_comp_id = OMfind_subobj(elem_id, OMstr_to_name("contour_comp"),
                                    OM_OBJ_RD);

    OMget_int_val(contour_comp_id, &contour_comp);
    type = OM_TYPE_INT;
    out_ncomp = 0;
    out_comp = (int *)NULL;
    stat = OMget_array_sz(out_comp_id, &type, (char **)(&out_comp),
                          &out_ncomp, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS) {
        out_ncomp = 0;
        out_comp = NULL;
    }

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr, i, &in, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCcontour(in, level1, level2, contour_comp,
                           (int)out_ncomp, out_comp, out);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "contour", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(in,  xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(out, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    if (out_comp)
        ARRfree(out_comp);

    return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCcontour (OMobj_id in, double level1, double level2,
                 int contour_comp, int nmap_comp, int *map_comp, OMobj_id out)
{
	int   i, nspace, data_type, veclen, ncomp, null_flag;
	xp_long nnodes, out_ncells1[MAX_NSETS], out_nnodes, size, conn_size1[MAX_NSETS];
	xp_long *out_nlist1[MAX_NSETS];
	xp_long *min_node_list, *max_node_list;
	int   stat, dtype;
	int   comp, ns;
	xp_long *old_nlist, old_nnodes, new_nnodes;
	int   data_id;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	float  *out_coord;
	char   *node_data, *in_node_data, *out_node_data, *list;
	char   *mesh_info;
	double null_value = 0;
	OMobj_id   cell_set;
	xp_long nline1=0, nline2=0, *line_conn1=NULL, *line_conn2=NULL;


	node_data=in_node_data=out_node_data=list=NULL;
	min_node_list=max_node_list=old_nlist=NULL;
	out_coord = NULL;

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	out_coord = NULL;
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	/*****************/
	/** Check input **/
	/*****************/
	if (level1 >= level2)
		return(1);

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}

	if (nnodes == 0 || ncomp == 0) {
		return(1);
	}
	for (i=0; i<nmap_comp; i++) {
		if (map_comp[i] >= ncomp) {
			ERR_RETURN("requested component does not exist, cut is not performed");
		}
	}

	if (FLDget_node_data_veclen(in, contour_comp, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}

	if (veclen != 1) {
		ERR_RETURN("component is not scalar, contour is not performed");
	}

	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}

	if (FLDget_node_data(in, contour_comp, &dtype, &node_data, 
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, contour_comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}

	stat = UTILcontour(mesh_info, node_data, dtype, 
			   null_flag, (char *)&null_value,
			   (double)level1, (double)level2,
			   &old_nnodes, &old_nlist,
			   &new_nnodes, &min_node_list, &max_node_list,
			   &list, out_ncells1, out_nlist1, conn_size1, nspace, &out_coord,
			   &nline1, &line_conn1, &nline2, &line_conn2);

	if (stat != 1) {
		if (node_data)
			ARRfree(node_data);
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no contour created");
	}

	for (ns=0,i=0; i<MAX_NSETS; i++) {
		if (out_ncells1[i]) {
			if (FLDadd_cell_set(out, cell_name[i]) != 1) {
				ERR_RETURN("Error adding cell set");
			}
			if (FLDget_cell_set(out, ns++, &cell_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (FLDset_ncells(cell_set, out_ncells1[i]) != 1) {
				ERR_RETURN("Error setting ncells");
			}
			if (FLDset_node_connect(cell_set, out_nlist1[i], conn_size1[i],
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
		}
	}
	/*** OUTPUT FIELD ***/
	out_nnodes = old_nnodes + new_nnodes;
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_coord(out, out_coord, nspace*out_nnodes, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	if (FLDset_node_data_ncomp (out, nmap_comp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (out_nnodes) {
		for (comp=0; comp<nmap_comp; comp++) {
			if (FLDget_node_data_units(in, map_comp[comp], 
						   units, MAX_NAME_SIZE) != 1) {
				strcpy(units, "");
			}
			if (FLDget_node_data_label(in, map_comp[comp], 
						   label, MAX_NAME_SIZE) != 1) {
				strcpy(label, "");
			}
			if (FLDget_node_data_veclen(in, map_comp[comp], &veclen) != 1) {
				ERR_RETURN("Error getting veclen");
			}
			if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
				ERR_RETURN("Error setting node component");
			}
			if (FLDget_node_data_id(in, map_comp[comp], &data_id) == 1)
				FLDset_node_data_id(out, comp, data_id);

			if (FLDcopy_node_minmax(in, out, map_comp[comp], comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDcopy_node_minmax_vec(in, out, map_comp[comp], comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}

			if (FLDget_node_data(in, map_comp[comp], &data_type, &in_node_data, 
					     &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get node data");
			}

			if (FLDget_node_data(out, comp, &data_type, &out_node_data, 
					     &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error setting node data");
			}
			null_value = 0;
			size = UTILget_data_list(mesh_info, data_type, veclen, in_node_data, 
						 old_nnodes, old_nlist, out_node_data);

			if (FLDget_node_null_data(in, map_comp[comp], &null_flag, (char *)&null_value) != 1) {
				ERR_RETURN("cannot get null data");
			}
			UTILeval_contour(nnodes, new_nnodes, 0, dtype, node_data, level1, level2, 
					 min_node_list, max_node_list, list, 
					 veclen, data_type, null_flag, (char *)&null_value,
					 0, in_node_data, out_node_data+size);
			if (null_flag) {
				if (FLDset_node_null_data(out, comp, (char *)&null_value, data_type) != 1) {
					ERR_RETURN("Error setting null value");
				}
			}
			else {
				if (FLDset_node_null_flag(out, comp, 0) != 1) {
					ERR_RETURN("Error setting null flag");
				}
			}
			if (in_node_data)
				ARRfree(in_node_data);
			if (out_node_data)
				ARRfree(out_node_data);
		}
	}
	if (node_data)
		ARRfree(node_data);
	FLDfree_mesh_info(mesh_info);

	if (min_node_list)
		ARRfree(min_node_list);
	if (max_node_list)
		ARRfree(max_node_list);
	if (list)
		ARRfree(list);
	if (old_nlist)
		free(old_nlist);
	return METHOD_SUCCESS;
}
