/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cut.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("cut", 0 , ERR_ORIG, A); \
        if (min_node_list) \
		ARRfree(min_node_list); \
	if (max_node_list) \
		ARRfree(max_node_list); \
	if (t) \
		free(t); \
	if (old_nlist) \
		free(old_nlist); \
        return(0);

#define MAX_NAME_SIZE 1024

/* See the list of enums in mod_util/cut_fnc.c */
#define MAX_NSETS 7
static const char *cell_name[MAX_NSETS]= {"Tet", "Prism", "Pyr", "Hex", "Tri", "Quad", "Line"};

/* modules/celldata.c */
int FUNCcopy_cell_data (OMobj_id in, OMobj_id in_set, OMobj_id out_set, 
                        xp_long ncells, xp_long *cell_list, 
                        int ncomp, int *comp, xp_long start);

int FUNCcut (OMobj_id in, double level, int above, int cut_comp,
             int nmap_comp, int *map_comp, int ncell_data, int *cell_data,
             OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVcut_update(OMobj_id elem_id)
{
	OMobj_id in, out, level_id, out_comp_id, cut_comp_id, above_id;
	double  level;
	int stat, *out_comp, cut_comp, above;
	xp_long out_ncomp;
	int type, *cell_data;
	xp_long ncell_data;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);
	OMget_real_val(level_id, &level);

	out_comp_id = OMfind_subobj(elem_id, OMstr_to_name("map_comp"),
				     OM_OBJ_RW);
	cut_comp_id = OMfind_subobj(elem_id, OMstr_to_name("cut_comp"),
				     OM_OBJ_RW);
	above_id = OMfind_subobj(elem_id, OMstr_to_name("above"),
				      OM_OBJ_RW);

	OMget_int_val(cut_comp_id, &cut_comp);
	OMget_int_val(above_id, &above);
	type = OM_TYPE_INT;
	out_ncomp = 0;
	out_comp = (int *)NULL;
	stat = OMget_array_sz(out_comp_id, &type, (char **)(&out_comp), 
			      &out_ncomp, OM_GET_ARRAY_RD);
	if (stat != 1) {
		out_ncomp = 0;
		out_comp = NULL;
	}

	cell_data = (int *)OMret_name_array_ptr(elem_id, OMstr_to_name("cell_data"),
						OM_GET_ARRAY_RD, &ncell_data, NULL);
	if (ncell_data == 0)
		cell_data = NULL;
	if (cell_data == NULL)
		ncell_data = 0;

	stat = FUNCcut(in, level, above, cut_comp, (int)out_ncomp, out_comp, 
		       (int)ncell_data, cell_data, out);
	if (out_comp)
		ARRfree(out_comp);
	if (cell_data)
		ARRfree(cell_data);
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FUNCcut (OMobj_id in, double level, int above, int cut_comp,
             int nmap_comp, int *map_comp, int ncell_data, int *cell_data,
             OMobj_id out)
{
	xp_long nnodes, out_ncells1[MAX_NSETS], out_nnodes, size, conn_size1[MAX_NSETS];
	int   i, nspace, data_type, veclen, ncomp, null_flag;
	xp_long   *out_nlist1[MAX_NSETS], *min_node_list, *max_node_list;
	int   comp, ns, dir, data_id, stat;
	xp_long *old_nlist, new_nnodes, old_nnodes;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	float  *out_coord, *t;
	char   *node_data, *out_node_data;
	char   *mesh_info;
	double null_value;
	int   set, nset;
	xp_long *ncells, **cell_ind, count;
	OMobj_id   cell_set, in_cell_set;

	t = NULL;
	old_nlist = NULL;
	min_node_list = NULL;
	max_node_list = NULL;
	node_data=out_node_data = NULL;
	out_coord = NULL;

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	out_coord = NULL;
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting out_coord");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	/*****************/
	/** Check input **/
	/*****************/

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("Error getting nnodes");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (nnodes == 0 || ncomp == 0) {
		return(1);
	}
	for (i=0; i<nmap_comp; i++) {
		if (map_comp[i] >= ncomp) {
			ERR_RETURN("requested component does not exist, cut is not performed");
		}
	}

	if (FLDget_node_data_veclen(in, cut_comp, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}
	if (veclen != 1) {
		ERR_RETURN("component is not scalar, cut is not performed");
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}

	if (FLDget_node_data(in, cut_comp, &data_type, &node_data, 
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, cut_comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}

	if (above)
		dir = 1;
	else
		dir = -1;

	if (ncell_data) {
		if (FLDget_ncell_sets (in, &nset) != 1) {
			ERR_RETURN("Error getting ncell_sets");
		}
		ncells = (xp_long *)malloc(MAX_NSETS*nset*sizeof(xp_long));
		cell_ind = (xp_long **)malloc(MAX_NSETS*nset*sizeof(xp_long *));
	}
	else {
		cell_ind = NULL;
		ncells = NULL;
	}

	OMpush_status_range(0, 90);

	stat = UTILcut(mesh_info, node_data, data_type,
		       null_flag, (char *)&null_value,
		       (double)level, dir,
		       &old_nnodes, &old_nlist,
		       &new_nnodes, &min_node_list, &max_node_list,
		       out_ncells1, out_nlist1, conn_size1, ncells, cell_ind);
	OMpop_status_range();

	if (stat != 1) {
		if (node_data)
			ARRfree(node_data);
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no cut performed");
	}

	for (ns=0,i=0; i<MAX_NSETS; i++) {
		if (out_ncells1[i]) {
			if (FLDadd_cell_set(out, cell_name[i]) != 1) {
				ERR_RETURN("Error adding cell set");
			}
			if (FLDget_cell_set(out, ns++, &cell_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (FLDset_ncells(cell_set, out_ncells1[i]) != 1) {
				ERR_RETURN("Error setting ncells");
			}
			if (FLDset_node_connect(cell_set, out_nlist1[i], conn_size1[i],
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}

			if (ncell_data) {
				if (FLDset_cell_set(cell_set, "Cell_Data_Set") != 1) {
					ERR_RETURN("Error setting cell type");
				}
				if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
					ERR_RETURN("Error setting cell data ncomp");
				}
				if (FLDset_cell_data_ncomp (cell_set, ncell_data) != 1) {
					ERR_RETURN("Error setting cell data ncomp");
				}
				for (count=0,set=0; set<nset; set++) {
					if (FLDget_cell_set(in, set, &in_cell_set) != 1) {
						ERR_RETURN("Error getting cell set");
					}
					if (cell_ind[i+MAX_NSETS*set] == NULL)
					    continue;
					stat = FUNCcopy_cell_data (in, in_cell_set, cell_set, 
								   ncells[i+MAX_NSETS*set], cell_ind[i+MAX_NSETS*set], 
								   ncell_data, cell_data, count);
					if (stat != 1) {
						if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
							ERR_RETURN("Error setting cell data ncomp");
						}
						for(count=0; count<(xp_long)nset*MAX_NSETS; count++)
							if (cell_ind[count])
								ARRfree(cell_ind[count]);
						break;
					}
					count += ncells[i+MAX_NSETS*set];
					if (cell_ind[i+MAX_NSETS*set])
						ARRfree(cell_ind[i+MAX_NSETS*set]);
				}
			}
		}
		else {
			if (ncell_data) {
				for (set=0; set<nset; set++) {
					if (cell_ind[i+MAX_NSETS*set])
						ARRfree(cell_ind[i+MAX_NSETS*set]);
				}
			}
		}
	}
	if (ncells)
		free(ncells);
	if (cell_ind)
		free(cell_ind);
	/*** OUTPUT FIELD ***/
	out_nnodes = old_nnodes + new_nnodes;
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (out_nnodes) {
		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}
	}
	if (old_nnodes) {
		UTILget_coord_list(mesh_info, old_nnodes, old_nlist, out_coord);
	}
	if (new_nnodes) {
		t = (float *)malloc(new_nnodes * sizeof(float));
		if (t==NULL) {
			ERR_RETURN("Error allocating t parameter");
		}
		UTILget_param(new_nnodes, data_type, node_data,
			      level, min_node_list, max_node_list, t);
		UTILeval_coord(mesh_info, new_nnodes, 
			       min_node_list, max_node_list, t, out_coord+nspace*old_nnodes);

	}
	if (node_data)
		ARRfree(node_data);
	if (out_nnodes) {
		ARRfree(out_coord);
	}

	OMstatus_check(95,NULL,&stat);

	if (stat) {
		if (FLDset_nnodes (out, 0) != 1) {
			ERR_RETURN("Error setting nnodes");
		}
		for (ns=0,i=0; i<MAX_NSETS; i++)
			if (out_ncells1[i])
				ARRfree(out_nlist1[i]);
		if (min_node_list)
			ARRfree(min_node_list);
		if (max_node_list)
			ARRfree(max_node_list);
		if (t)
			free((char *)t);
		if (old_nlist)
			free(old_nlist);
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no cut performed");
	}

	if (FLDset_node_data_ncomp (out, nmap_comp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (out_nnodes) {
		for (comp=0; comp<nmap_comp; comp++) {
			if (FLDget_node_data_units(in, map_comp[comp], 
						   units, MAX_NAME_SIZE) != 1) {
				strcpy(units, "");
			}
			if (FLDget_node_data_label(in, map_comp[comp], 
						   label, MAX_NAME_SIZE) != 1) {
				strcpy(label, "");
			}
			if (FLDget_node_data_veclen(in, map_comp[comp], &veclen) != 1) {
				ERR_RETURN("Error getting veclen");
			}
			if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
				ERR_RETURN("Error setting node component");
			}
			if (FLDget_node_data_id(in, map_comp[comp], &data_id) == 1)
				FLDset_node_data_id(out, comp, data_id);

			if (FLDcopy_node_minmax(in, out, map_comp[comp], comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDcopy_node_minmax_vec(in, out, map_comp[comp], comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}

			if (FLDget_node_data(in, map_comp[comp], &data_type, &node_data, 
					     &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get node data");
			}

			if (FLDget_node_data(out, comp, &data_type, &out_node_data, 
					     &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error setting node data");
			}
			null_value = 0;
			if (FLDget_node_null_data(in, map_comp[comp], &null_flag, (char *)&null_value) != 1) {
				ERR_RETURN("cannot get null data");
			}
			size = UTILget_data_list(mesh_info, data_type, veclen, node_data, 
						 old_nnodes, old_nlist, out_node_data);
			if (map_comp[comp] == cut_comp) {
				UTILinit_array(out_node_data+size, new_nnodes*veclen, level, data_type);
			}
			else {
				UTILeval_param(new_nnodes, veclen, data_type, node_data, null_flag, (char *)&null_value,
					       min_node_list, max_node_list, t, out_node_data+size);
			}
			if (null_flag) {
				if (FLDset_node_null_data(out, comp, (char *)&null_value, data_type) != 1) {
					ERR_RETURN("Error setting null value");
				}
			}
			else {
				if (FLDset_node_null_flag(out, comp, 0) != 1) {
					ERR_RETURN("Error setting null flag");
				}
			}
			if (node_data)
				ARRfree(node_data);
			if (out_node_data)
				ARRfree(out_node_data);
		}
	}

	OMstatus_check(100,NULL,&stat);

	FLDfree_mesh_info(mesh_info);

	if (min_node_list)
		ARRfree(min_node_list);
	if (max_node_list)
		ARRfree(max_node_list);
	if (t)
		free(t);
	if (old_nlist)
		free(old_nlist);
	return(1);
}
