/*
			Copyright (c) 1995 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cyl_plot.c#1 $
*/

#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/err.h>
#include <avs/math.h>
#include <avs/dtype.h>
#include <avs/data_utils.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd_def.h>

#define ERR_RETURN(A) ERRerror("cyl_plot", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024

#define ALLOC_SIZE 128

#define CLEAN_AND_RETURN(A) \
        if (h.xyz) \
            ARRfree(h.xyz); \
        if (h.quad_conn) \
            ARRfree(h.quad_conn); \
        if (h.tri_conn) \
            ARRfree(h.tri_conn); \
        ARRfree(dims); \
        ERRerror("cyl_plot", 0, ERR_ORIG, A); \
	return(0);

typedef struct _OUTmesh {
	xp_long nnodes;
	float *xyz;
	xp_long size;
	xp_long nquad;
	xp_long *quad_conn;
	xp_long quad_size;
	xp_long ntri;
	xp_long *tri_conn;
	xp_long tri_size;
} OUTmesh;

int FUNCcyl_plot (OMobj_id in, OMobj_id loc, int comp, int vertical,
		  int scale_height, int scale_rad, int col_sides,
		  double rad, double scale,
		  int nsegments, double start_ang, float *rgb, OMobj_id out);

static int make_cyl_rad (OUTmesh *h, float *center, float rad, float height,
			 float ang0, float ang1, xp_long nseg, xp_long bot, xp_long top);

static int make_cyl_vert (OUTmesh *h, float *center, float rad, float *height,
			  xp_long ncyl, xp_long nseg, xp_long bot, xp_long top);

/* 64-bit porting. Only Modified Internally */
int DVcyl_plot_update(OMobj_id elem)
{
	OMobj_id src_id, dst_id, e_id, loc_id;
	int stat, comp, scale_height, scale_rad, col_sides, nseg, vert;
	xp_long size;
	double scale, rad, ang;
	float *rgb;

	src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
	loc_id = OMfind_subobj(elem,OMstr_to_name("in_locations"),OM_OBJ_RD);
	dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

	e_id = OMfind_subobj(elem,OMstr_to_name("component"),OM_OBJ_RD);
	OMget_int_val(e_id, &comp);

	e_id = OMfind_subobj(elem,OMstr_to_name("scale_height"),OM_OBJ_RD);
	OMget_int_val(e_id, &scale_height);

	e_id = OMfind_subobj(elem,OMstr_to_name("scale_radius"),OM_OBJ_RD);
	OMget_int_val(e_id, &scale_rad);

	e_id = OMfind_subobj(elem,OMstr_to_name("color_sides"),OM_OBJ_RD);
	OMget_int_val(e_id, &col_sides);

	e_id = OMfind_subobj(elem,OMstr_to_name("scale"),OM_OBJ_RD);
	OMget_real_val(e_id, &scale);

	e_id = OMfind_subobj(elem,OMstr_to_name("radius"),OM_OBJ_RD);
	OMget_real_val(e_id, &rad);

	e_id = OMfind_subobj(elem,OMstr_to_name("angle"),OM_OBJ_RD);
	OMget_real_val(e_id, &ang);

	e_id = OMfind_subobj(elem,OMstr_to_name("nseg"),OM_OBJ_RD);
	OMget_int_val(e_id, &nseg);

	e_id = OMfind_subobj(elem,OMstr_to_name("vertical"),OM_OBJ_RD);
	OMget_int_val(e_id, &vert);

	rgb = (float *)OMret_name_array_ptr(elem, OMstr_to_name("default_rgb"),
					    OM_GET_ARRAY_RD, &size, NULL);

	stat = FUNCcyl_plot(src_id, loc_id, comp, vert, scale_height, scale_rad, col_sides,
			    rad, scale, nseg, ang*acos(-1)/180., rgb, dst_id);
	ARRfree(rgb);

	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FUNCcyl_plot (OMobj_id in, OMobj_id loc, int comp, int vertical,
		  int scale_height, int scale_rad, int col_sides,
		  double rad, double scale,
		  int nsegments, double start_ang, float *rgb, OMobj_id out)
{
	OUTmesh  h;
	OMobj_id cell_set;
	xp_long i_w, j_w, *dims, m, size;
	xp_long rdims[4], rmin[4], rmax[4];
	xp_long ncyl, nseg, bot, top, nloc;
	int i, j, ndim, n, ncomp, nspace, null_flag;
	int set, set_xform, dims_size;
	int veclen, dtype, type;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	double null_value;
	double two_pi;
	float  ang0, ang1, f, sum, sum_max, *centers, center[3], crad, cheight, xform[4][4];
	float  *data, *data_slice, *height, min, max;

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	h.xyz = NULL;
	h.quad_conn = NULL;
	data = NULL;

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, h.xyz, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coord");
	}

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (ndim != 2) {
		ERR_RETURN("ndim must be equal 2");
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (ncomp == 0) {
		ERR_RETURN("No node data");
	}
	if (comp >= ncomp) {
		ERRerror("cyl_plot", 0, ERR_ORIG, 
			 "data component is out of range, assume 0-th");
		comp = 0;
	}
	if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
		ERR_RETURN("Error getting veclen");
	}
	if (veclen > 1) {
		ERR_RETURN("node data must be scalar");
	}
	if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}
	if (FLDget_node_data_type(in, comp, &type) != 1) {
		ERR_RETURN("Error getting veclen");
	}
	if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
		strcpy(units, "");
	}
	if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
		strcpy(label, "");
	}

	h.nnodes = 0;
	h.size = ALLOC_SIZE;
	h.xyz = (float *)ARRalloc(NULL, DTYPE_FLOAT, 3*h.size, NULL);

	h.nquad = 0;
	h.quad_size = 0;
	h.quad_conn = NULL;

	h.ntri = 0;
	h.tri_size = 0;
	h.tri_conn = NULL;

	ncyl = dims[0];
	nseg = dims[1];
	dtype = DTYPE_FLOAT;
	two_pi = 2*acos(-1);
	set = 0;
	min = 0;
	max = 1;

	data_slice = (float *)malloc(nseg*veclen*sizeof(float));
	if (vertical)
		height = (float *)malloc(nseg*sizeof(float));

	for (i=0; i<ndim; i++)
		rdims[i+1] = dims[i];

	if (FLDget_coord(loc, &centers, &size, OM_GET_ARRAY_RD) != 1) {
		if (FLDget_nspace (in, &nspace) != 1) {
			ERR_RETURN("Error getting nspace");
		}
		nloc = ncyl;
		centers = (float *)ARRalloc(NULL, DTYPE_FLOAT, ncyl*nspace, NULL);
		rdims[0] = nspace;
		rmin[0] = 0;
		rmax[0] = nspace;
		rmin[1] = 0;
		rmax[1] = ncyl;
		rmin[2] = nseg/2;
		rmax[2] = rmin[2]+1;
		if (FLDget_sub_coord(in, ndim+1, rdims, rmin, rmax, centers) != 1) {
			CLEAN_AND_RETURN("cannot get coordinates");
		}
		if (FLDget_xform(in, (float *)xform) != 1)
			set_xform = 0;
		else
			set_xform = 1;
	}
	else {
		if (FLDget_nspace (loc, &nspace) != 1) {
			ERR_RETURN("Error getting nspace");
		}
		if (FLDget_nnodes (loc, &nloc) != 1) {
			ERR_RETURN("Error getting nnodes");
		}
		if (FLDget_xform(loc, (float *)xform) != 1)
			set_xform = 0;
		else
			set_xform = 1;
	}
	if (nloc < 1) {
		ERR_RETURN("No location for cylinders is specified");
	}
	rdims[0] = veclen;
	rmin[0] = 0;
	rmax[0] = veclen;
	rmin[2] = 0;
	rmax[2] = nseg;

	if (col_sides) {
		for (i_w=0; i_w<ncyl; i_w++) {
			rmin[1] = i_w;
			rmax[1] = i_w+1;
			if (FLDget_sub_node_data(in, comp, &dtype, ndim+1, rdims, rmin, rmax, (char *)data_slice) != 1) {
				CLEAN_AND_RETURN("cannot get node data");
			}
			for (sum=0.0, j_w=0; j_w<nseg; j_w++)
				sum += data_slice[j_w];
			if (i_w==0)
				sum_max = sum;
			else if (sum > sum_max)
				sum_max = sum;
		}
	}
	else
		sum_max = 1;

	for (i_w=0; i_w<ncyl; i_w++) {
		rmin[1] = i_w;
		rmax[1] = i_w+1;
		if (FLDget_sub_node_data(in, comp, &dtype, ndim+1, rdims, rmin, rmax, (char *)data_slice) != 1) {
			CLEAN_AND_RETURN("cannot get node data");
		}
		for (sum=0.0, j_w=0; j_w<nseg; j_w++)
			sum += data_slice[j_w];

		if (sum == 0.0)
			continue;
		if (scale_height)
			cheight = scale*sum;
		else
			cheight = scale;
		if (scale_rad)
			crad = rad*sum;
		else
			crad = rad;
		if (i_w < nloc) {
			for (j=0; j<nspace; j++)
				center[j] = centers[i_w*nspace+j];
			for (; j<3; j++)
				center[j] = 0.0;
		}
		else {
			if (nloc < 2)
				center[0] = 0.0;
			else {
				center[0] = (i_w/nloc)*(centers[(nloc-1)*nspace]-
						      centers[j])/(nloc-1) +
							      centers[(i_w%nloc)*nspace];
			}
			for (j=1; j<nspace; j++)
				center[j] = centers[(i_w%nloc)*nspace+j];
			for (; j<3; j++)
				center[j] = 0.0;
		}

		for (j=0; j<3; j++)
			h.xyz[h.nnodes*3+j] = center[j];
		h.xyz[h.nnodes*3+3] = center[0];
		h.xyz[h.nnodes*3+4] = center[1];
		h.xyz[h.nnodes*3+5] = center[2]+cheight;

		bot = h.nnodes;
		top = h.nnodes+1;
		h.nnodes += 2;

		if (vertical) {
			for (j_w=0; j_w<nseg; j_w++) {
				if (null_flag) {
					UTILtype_to_float(&f, (char *)&null_value, type);
					if (f == data_slice[j_w])
						height[j_w] = 0.0;
					else
						height[j_w] = cheight*data_slice[j_w]/sum;
				}
				else
					height[j_w] = cheight*data_slice[j_w]/sum;
			}
			h.nquad = 0;
			h.quad_size = nseg*nsegments;
			h.quad_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 4*h.quad_size, NULL);

			h.ntri = 0;
			h.tri_size = 2*nsegments;
			h.tri_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 3*h.tri_size, NULL);

			make_cyl_vert (&h, center, crad, height, nseg , nsegments, bot, top);

			if (h.nquad) {
				if (FLDadd_cell_set(out, "Quad") != 1) {
					ERR_RETURN("Error setting cell type");
				}
				if (FLDget_cell_set(out, set++, &cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDset_ncells(cell_set, h.nquad) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDset_node_connect(cell_set, h.quad_conn, 4*h.nquad,
							OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell connect list");
				}

				if (FLDset_cell_data_ncomp (cell_set, 1) != 1) {
					ERR_RETURN("Error setting ncell_data");
				}
				if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
					ERR_RETURN("Error setting cell component");
				}
				data = (float *)ARRalloc(NULL, DTYPE_FLOAT, h.nquad, NULL);
				for (j_w=0; j_w<nseg; j_w++)
					UTILinit_array(data+j_w*nsegments, nsegments, (double)(j_w/(nseg-1.0)),
						       DTYPE_FLOAT);
				if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
						     h.nquad, OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell data");
				}
				if (FLDset_cell_data_minmax(cell_set, 0, (char *)&min, (char *)&max, 
							    DTYPE_FLOAT) != 1) {
					ERR_RETURN("Error setting cell data minmax");
				}
			}
			else {
				ARRfree(h.quad_conn);
			}
			if (h.ntri) {
				if (FLDadd_cell_set(out, "Tri") != 1) {
					ERR_RETURN("Error setting cell type");
				}
				if (FLDget_cell_set(out, set++, &cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDset_ncells(cell_set, h.ntri) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDset_node_connect(cell_set, h.tri_conn, 3*h.ntri,
							OM_SET_ARRAY_FREE) != 1) {
					ERR_RETURN("Error setting cell connect list");
				}

				if (FLDset_cell_data_ncomp (cell_set, 1) != 1) {
					ERR_RETURN("Error setting ncell_data");
				}
				if (col_sides) {
					if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
						ERR_RETURN("Error setting cell component");
					}

					data = (float *)ARRalloc(NULL, DTYPE_FLOAT, h.ntri, NULL);
					UTILinit_array(data, h.ntri, (double)(sum/sum_max), DTYPE_FLOAT);

					if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
							     h.ntri, OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					if (FLDset_cell_data_minmax(cell_set, 0, (char *)&min, (char *)&max, 
								    DTYPE_FLOAT) != 1) {
						ERR_RETURN("Error setting cell data minmax");
					}
				}
				else {
					if (FLDset_cell_data_comp (cell_set, 0, 3, label, units) != 1) {
						ERR_RETURN("Error setting cell component");
					}
					data = (float *)ARRalloc(NULL, DTYPE_FLOAT, 3*h.ntri, NULL);
					for (m=0; m<h.ntri; m++) {
						data[3*m]   = rgb[0];
						data[3*m+1] = rgb[1];
						data[3*m+2] = rgb[2];
					}
					if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
							     3*h.ntri, OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					if (FLDset_cell_data_id(cell_set, 0, GD_COLOR_DATA_ID) != 1) {
						ERR_RETURN("Error setting cell data id");
					}
				}
			}
			else {
				ARRfree(h.tri_conn);
			}
		}
		else {
			for (ang0=start_ang, j_w=0; j_w<nseg; j_w++) {
				if (data_slice[j_w] == 0.0)
					continue;
				if (null_flag) {
					UTILtype_to_float(&f, (char *)&null_value, type);
					if (f == data_slice[j_w])
						continue;
				}
				h.nquad = 0;
				h.quad_size = ALLOC_SIZE;
				h.quad_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 4*h.quad_size, NULL);

				h.ntri = 0;
				h.tri_size = ALLOC_SIZE;
				h.tri_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 3*h.tri_size, NULL);

				f = two_pi*data_slice[j_w]/sum;
				n = nsegments*data_slice[j_w]/sum;
				if (n<1)
					n=1;
				ang1 = ang0 + f;
				make_cyl_rad (&h, center, crad, cheight, ang0, ang1, n, bot, top);
				ang0 += f;
				if (h.nquad) {
					if (FLDadd_cell_set(out, "Quad") != 1) {
						ERR_RETURN("Error setting cell type");
					}
					if (FLDget_cell_set(out, set++, &cell_set) != 1) {
						ERR_RETURN("Error getting cell set");
					}
					if (FLDset_ncells(cell_set, h.nquad) != 1) {
						ERR_RETURN("Error setting ncells");
					}
					if (FLDset_node_connect(cell_set, h.quad_conn, 4*h.nquad,
								OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("Error setting cell connect list");
					}

					if (FLDset_cell_data_ncomp (cell_set, 1) != 1) {
						ERR_RETURN("Error setting ncell_data");
					}
					if (col_sides) {
						if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
							ERR_RETURN("Error setting cell component");
						}
						data = (float *)ARRalloc(NULL, DTYPE_FLOAT, h.nquad, NULL);
						UTILinit_array(data, h.nquad, (double)(sum/sum_max), DTYPE_FLOAT);

						if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
								     h.nquad, OM_SET_ARRAY_FREE) != 1) {
							ERR_RETURN("Error setting cell data");
						}
						if (FLDset_cell_data_minmax(cell_set, 0, (char *)&min, (char *)&max, 
									    DTYPE_FLOAT) != 1) {
							ERR_RETURN("Error setting cell data minmax");
						}
					}
					else {
						if (FLDset_cell_data_comp (cell_set, 0, 3, label, units) != 1) {
							ERR_RETURN("Error setting cell component");
						}
						data = (float *)ARRalloc(NULL, DTYPE_FLOAT, 3*h.nquad, NULL);
						for (m=0; m<h.nquad; m++) {
							data[3*m]   = rgb[0];
							data[3*m+1] = rgb[1];
							data[3*m+2] = rgb[2];
						}
						if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
								     3*h.nquad, OM_SET_ARRAY_FREE) != 1) {
							ERR_RETURN("Error setting cell data");
						}
						if (FLDset_cell_data_id(cell_set, 0, GD_COLOR_DATA_ID) != 1) {
							ERR_RETURN("Error setting cell data id");
						}
					}
				}
				else {
					ARRfree(h.quad_conn);
				}
				if (h.ntri) {
					if (FLDadd_cell_set(out, "Tri") != 1) {
						ERR_RETURN("Error setting cell type");
					}
					if (FLDget_cell_set(out, set++, &cell_set) != 1) {
						ERR_RETURN("Error getting cell set");
					}
					if (FLDset_ncells(cell_set, h.ntri) != 1) {
						ERR_RETURN("Error setting ncells");
					}
					if (FLDset_node_connect(cell_set, h.tri_conn, 3*h.ntri,
								OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("Error setting cell connect list");
					}

					if (FLDset_cell_data_ncomp (cell_set, 1) != 1) {
						ERR_RETURN("Error setting ncell_data");
					}
					if (FLDset_cell_data_comp (cell_set, 0, 1, label, units) != 1) {
						ERR_RETURN("Error setting cell component");
					}

					data = (float *)ARRalloc(NULL, DTYPE_FLOAT, h.ntri, NULL);
					UTILinit_array(data, h.ntri, (double)(j_w/(nseg-1.0)), DTYPE_FLOAT);

					if (FLDset_cell_data(cell_set, 0, (char *)data, DTYPE_FLOAT, 
							     h.ntri, OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					if (FLDset_cell_data_minmax(cell_set, 0, (char *)&min, (char *)&max, 
								    DTYPE_FLOAT) != 1) {
						ERR_RETURN("Error setting cell data minmax");
					}
				}
				else {
					ARRfree(h.tri_conn);
				}
			}
		}
	}
	if (vertical)
		free(height);

	if (FLDset_ncell_sets (out, set) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, 3) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, h.nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (h.nnodes)
		if (FLDset_coord(out, h.xyz, 3*h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting coord");
		}
	if (FLDset_xform (out, (float *)xform) != 1) {
		ERR_RETURN("Error setting xform");
	}
	ARRfree(dims);
	free(data_slice);
	ARRfree(centers);
	return(1);
}

/* 64-bit porting. Directly Modified */
static int make_cyl_rad (OUTmesh *h, float *center, float rad, float height,
			 float ang0, float ang1, xp_long nseg, xp_long bot, xp_long top)
{
	xp_long n, m, nnew, i;
	double ang, h_ang;

	nnew = (nseg+1)*2;
	if (h->nnodes+nnew+2 >= h->size) {
		n = nnew/ALLOC_SIZE+1;
		h->size += n*ALLOC_SIZE;
		h->xyz = (float *)ARRrealloc(h->xyz, DTYPE_FLOAT, 3*h->size, NULL);
	}
	if (h->nquad+nseg >= (h->quad_size)) {
		n = nseg/ALLOC_SIZE+1;
		h->quad_size += n*ALLOC_SIZE;
		h->quad_conn = (xp_long *)ARRrealloc(h->quad_conn, DTYPE_LONG, 4*h->quad_size, NULL);
	}
	if (h->ntri+2*nseg >= (h->tri_size)) {
		n = 2*nseg/ALLOC_SIZE+1;
		h->tri_size += n*ALLOC_SIZE;
		h->tri_conn = (xp_long *)ARRrealloc(h->tri_conn, DTYPE_LONG, 3*h->tri_size, NULL);
	}
	h_ang = (ang1-ang0)/nseg;

	for (n=h->nnodes*3, ang = ang0, i=0; i<=nseg; i++) {
		h->xyz[n]   = center[0] + rad*cos(ang);
		h->xyz[n+1] = center[1] + rad*sin(ang);
		h->xyz[n+2] = center[2];

		h->xyz[n+3] = h->xyz[n];
		h->xyz[n+4] = h->xyz[n+1];
		h->xyz[n+5] = h->xyz[n+2]+height;
		ang += h_ang;
		n += 6;
	}

	/** quads **/

	if (height >= 0.0) {
		for (m=h->nnodes, n=4*h->nquad, i=0; i<nseg; i++) {
			h->quad_conn[n]   = m;
			h->quad_conn[n+1] = m+2;
			h->quad_conn[n+2] = m+3;
			h->quad_conn[n+3] = m+1;
			(h->nquad)++;
			n += 4;
			m += 2;
		}
	}
	else {
		for (m=h->nnodes, n=4*h->nquad, i=0; i<nseg; i++) {
			h->quad_conn[n]   = m;
			h->quad_conn[n+1] = m+1;
			h->quad_conn[n+2] = m+3;
			h->quad_conn[n+3] = m+2;
			(h->nquad)++;
			n += 4;
			m += 2;
		}
	}

	/** tris **/

	if (height >= 0.0) {
		for (n=3*h->ntri, m=h->nnodes, i=0; i<nseg; i++) {
			h->tri_conn[n]   = m;
			h->tri_conn[n+1] = bot;
			h->tri_conn[n+2] = m+2;

			h->tri_conn[n+3] = m+3;
			h->tri_conn[n+4] = top;
			h->tri_conn[n+5] = m+1;
			h->ntri += 2;
			m += 2;
			n += 6;
		}
	}
	else {
		for (n=3*h->ntri, m=h->nnodes, i=0; i<nseg; i++) {
			h->tri_conn[n]   = m;
			h->tri_conn[n+1] = m+2;
			h->tri_conn[n+2] = bot;

			h->tri_conn[n+3] = m+3;
			h->tri_conn[n+4] = m+1;
			h->tri_conn[n+5] = top;
			h->ntri += 2;
			m += 2;
			n += 6;
		}
	}
	(h->nnodes) += (nseg+1)*2;

	return(1);
}

/* 64-bit porting. Directly Modified */
static int make_cyl_vert (OUTmesh *h, float *center, float rad, float *height,
			  xp_long ncyl, xp_long nseg, xp_long bot, xp_long top)
{
	xp_long cyl, n, m, nnew, i;
	float ht, h_ang;
	double ang;

	nnew = (nseg+1)*(ncyl+1);
	if (h->nnodes+nnew+2 >= h->size) {
		n = nnew/ALLOC_SIZE+1;
		h->size += n*ALLOC_SIZE;
		h->xyz = (float *)ARRrealloc(h->xyz, DTYPE_FLOAT, 3*h->size, NULL);
	}

	h_ang = 2*acos(-1.0)/nseg;
	for (m=h->nnodes, cyl=0; cyl<=ncyl; cyl++) {
		if (cyl)
			ht += height[cyl-1];
		else
			ht = 0.0;
		for (n=m*3, ang=0, i=0; i<=nseg; i++) {
			h->xyz[n]   = center[0] + rad*cos(ang);
			h->xyz[n+1] = center[1] + rad*sin(ang);
			h->xyz[n+2] = center[2] + ht;
			ang += h_ang;
			n += 3;
		}
		m += (nseg+1);
	}

	/** quads **/

	if (height[0] >= 0.0) {
		for (cyl=0; cyl<ncyl; cyl++) {
			m = h->nnodes+cyl*(nseg+1);
			for (n=4*h->nquad, i=0; i<nseg; i++) {
				h->quad_conn[n]   = m;
				h->quad_conn[n+1] = m+1;
				h->quad_conn[n+2] = m+nseg+2;
				h->quad_conn[n+3] = m+nseg+1;
				(h->nquad)++;
				m++;
				n += 4;
			}
		}
	}
	else {
		for (cyl=0; cyl<ncyl; cyl++) {
			m = h->nnodes+cyl*(nseg+1);
			for (n=4*h->nquad, i=0; i<nseg; i++) {
				h->quad_conn[n]   = m+1;
				h->quad_conn[n+1] = m;
				h->quad_conn[n+2] = m+nseg+1;
				h->quad_conn[n+3] = m+nseg+2;
				(h->nquad)++;
				m++;
				n += 4;
			}
		}
	}

	/** tris **/
	nnew = (nseg+1)*ncyl;

	if (height[0] >= 0.0) {
		for (n=3*h->ntri, i=0; i<nseg; i++) {
			m = h->nnodes+i;
			h->tri_conn[n]   = m;
			h->tri_conn[n+1] = bot;
			h->tri_conn[n+2] = m+1;

			m += nnew;
			h->tri_conn[n+3] = m+1;
			h->tri_conn[n+4] = top;
			h->tri_conn[n+5] = m;
			h->ntri += 2;
			n += 6;
		}
	}
	else {
		for (n=3*h->ntri, i=0; i<nseg; i++) {
			m = h->nnodes+i;
			h->tri_conn[n]   = m+1;
			h->tri_conn[n+1] = bot;
			h->tri_conn[n+2] = m;

			m += nnew;
			h->tri_conn[n+3] = m;
			h->tri_conn[n+4] = top;
			h->tri_conn[n+5] = m+1;
			h->ntri += 2;
			n += 6;
		}
	}
	(h->nnodes) += (nseg+1)*(ncyl+1);

	return(1);
}
