/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/ext_edge.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("ext_edge", 0, ERR_ORIG, A); return(0);
#define MAX_CELL_SETS 256

static int corner_nnodes[]={0,1,2,3,4,0,3,0,4};

int FUNCext_edge (OMobj_id in, double angle, OMobj_id out);

int DVext_edge_update(OMobj_id elem_id)
{
	OMobj_id in, out, angle_id;
	double   angle;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out_field"), OM_OBJ_RW);
	angle_id = OMfind_subobj(elem_id, OMstr_to_name("angle"), OM_OBJ_RD);
	OMget_real_val(angle_id, &angle);
	if (FUNCext_edge(in, angle, out)) {
		return(1);
	}
	else return(0);
}

/* 64-bit porting. Only Modified Internally */
int FUNCext_edge (OMobj_id in, double angle, OMobj_id out)
{
	int   cs, os, ns, nsets, cell_nnodes, cell_corner_nnodes, stat;
	xp_long ncells, nnodes;
	int   nspace, ndim, cell_nfaces, fnl_size;
	xp_long size, out_edges;
	xp_long *node_list, *face_node_list, *out_list;
	int   *face_nnodes, *face_corner_nnodes;
	int   out_nsets, out_cell_nnodes[MAX_CELL_SETS];
 	xp_long out_ncells[MAX_CELL_SETS], *out_cells[MAX_CELL_SETS],
	        *out_lists[MAX_CELL_SETS], out_list_sizes[MAX_CELL_SETS];
	float *xyz;

	OMobj_id   cell_set;
	/*
	 * clean the output
	 */
	if (FLDset_nnodes(out, 0) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	FLDset_node_data_ncomp (out, 0);

	if (FLDget_nnodes(in, &nnodes) != 1 || nnodes == 0) {
		/*
		 * this could be a case when module disconnected 
		 * clean the output and return
		 */
		return(0);
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDget_coord(in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates");
	}

	/*--------------------------------------*/
	/*  Make out to be the same as in       */
	/*--------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != 1) {
		ERR_RETURN("Error setting field reference");
	}

	/*----------------------------------------*/
	/*   REMOVE ALL 3D and 2D cells from out  */
	/*----------------------------------------*/
	if (FLDget_ncell_sets(out, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	for (cs=nsets-1; cs>=0; cs--) {
		if (FLDget_cell_set(out, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_cell_ndim(cell_set, &ndim) != 1) {
			ERR_RETURN("cannot get ndim");
		}

		if (ndim > 1) {
			if (FLDdel_cell_set(out, cell_set) != 1) {
				ERR_RETURN("cannot delete cell");
			}
		}
	}
	/*-------------------------------------*/
	/*  PROCESS all 3D and 2D cells in in  */
	/*-------------------------------------*/
	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	if (FLDget_ncell_sets(out, &ns) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	for (cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_cell_ndim(cell_set, &ndim) != 1) {
			ERR_RETURN("cannot get ndim");
		}
		if (ndim < 2)
			continue;

		if (FLDget_ncells(cell_set, &ncells) != 1) {
			ERR_RETURN("cannot get ncells");
		}

		if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
			ERR_RETURN("cannot get cell nnodes");
		}
		if (FLDget_cell_corner_nnodes(cell_set,  &cell_corner_nnodes) != 1) {
			ERR_RETURN("cannot get cell nnodes");
		}

		if (FLDget_node_connect(cell_set, &node_list, 
					    &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}
		if (ndim < 3) {
			OMpush_status_range((int)(cs*100/nsets),
					    (int)((cs+1)*100/nsets));

			stat = UTILexternal_edges (nnodes, nspace, xyz, ncells, 
						   cell_nnodes, cell_corner_nnodes,
						   node_list, 
						   &out_edges, &out_list, angle);

			OMpop_status_range();
			if (stat != 1) {
				ARRfree(node_list);
				if (FLDset_ncell_sets (out, 0) != 1) {
					ERR_RETURN("Error setting ncell_sets");
				}
				ERR_RETURN("edges are not extracted");
			}

			if (out_edges == 0)
				continue;

			if (FLDadd_cell_set(out, "Line") != 1) {
				ARRfree(node_list);
				ERR_RETURN("Error setting cell type");
			}
			if (FLDget_cell_set(out, ns, &cell_set) != 1) {
				ARRfree(node_list);
				ERR_RETURN("Error getting cell set");
			}

			if (FLDset_ncells(cell_set, out_edges) != 1) {
				ARRfree(node_list);
				ERR_RETURN("Error setting ncells");
			}

			if (FLDset_node_connect(cell_set, out_list, 2*out_edges, 
						    OM_SET_ARRAY_FREE) != 1) {
				ARRfree(node_list);
				ERR_RETURN("Error setting cell connect list");
			}
			ns++;
			ARRfree(node_list);
			continue;
		}

		/*------------------------------------------*/
		/*   3D case: GET EXTERNAL faces            */
		/*------------------------------------------*/
		OMpush_status_range((int)(cs*100/nsets),
				    (int)((cs+1)*100/nsets));

		OMpush_status_range(0, 50);

		if (FLDget_cell_faces(cell_set, &cell_nfaces, &face_nnodes, 
				      &face_corner_nnodes, &face_node_list, &fnl_size) != 1) {
			ERR_RETURN("cannot get cell nfaces");
		}

		stat = UTILexternal_faces(nnodes, ncells, cell_nnodes, node_list, 
					  cell_nfaces, face_nnodes, face_corner_nnodes, 
					  face_node_list,
					  MAX_CELL_SETS, &out_nsets, out_ncells, 
					  out_cell_nnodes, out_cells,
					  out_lists, out_list_sizes);
		OMpop_status_range();

		if (stat != 1) {
			OMpop_status_range();
			ARRfree(node_list);
			ARRfree(face_nnodes);
			ARRfree(face_corner_nnodes);
			ARRfree(face_node_list);
			if (FLDset_ncell_sets (out, 0) != 1) {
				ERR_RETURN("Error setting ncell_sets");
			}
			ERR_RETURN("faces are not extracted");
		}
		/*-----------------------------------*/
		/*  EXTRACT EDGES                    */
		/*-----------------------------------*/
		OMpush_status_range(50, 100);

		for (os=0; os<out_nsets; os++) {
			if (out_cells[os]) {
				ARRfree(out_cells[os]);
			}
			if (out_list_sizes[os] == 0 || out_ncells[os] == 0) {
				if (out_lists[os]) {
					ARRfree(out_lists[os]);
				}
				continue;
			}

			OMpush_status_range((int)(os*100/out_nsets),
				    (int)((os+1)*100/out_nsets));

			stat = UTILexternal_edges (nnodes, nspace, xyz, 
						   out_ncells[os], out_cell_nnodes[os],
						   corner_nnodes[out_cell_nnodes[os]],
						   out_lists[os],
						   &out_edges, &out_list, angle);
			OMpop_status_range();
			if (stat != 1) {
				OMpop_status_range();
				OMpop_status_range();
				ARRfree(node_list);
				ARRfree(face_nnodes);
				ARRfree(face_corner_nnodes);
				ARRfree(face_node_list);
				for (cs=os; cs<out_nsets; cs++) {
					if (out_cells[cs]) {
						ARRfree(out_cells[cs]);
					}
					if (out_lists[cs]) {
						ARRfree(out_lists[cs]);
					}

					if (FLDset_ncell_sets (out, 0) != 1) {
						ERR_RETURN("Error setting ncell_sets");
					}
				}
				ERR_RETURN("edges are not extracted");
			}
			if (out_lists[os]) {
				ARRfree(out_lists[os]);
			}

			if (out_edges == 0)
				continue;

			if (FLDadd_cell_set(out, "Line") != 1) {
				ARRfree(node_list);
				ARRfree(face_nnodes);
				ARRfree(face_corner_nnodes);
				ARRfree(face_node_list);
				ERR_RETURN("Error setting cell type");
			}
			if (FLDget_cell_set(out, ns++, &cell_set) != 1) {
				ARRfree(node_list);
				ARRfree(face_nnodes);
				ARRfree(face_corner_nnodes);
				ARRfree(face_node_list);
				ERR_RETURN("Error getting cell set");
			}

			if (FLDset_ncells(cell_set, out_edges) != 1) {
				ARRfree(node_list);
				ARRfree(face_nnodes);
				ARRfree(face_corner_nnodes);
				ARRfree(face_node_list);
				ERR_RETURN("Error setting ncells");
			}

			if (FLDset_node_connect(cell_set, out_list, 2*out_edges, OM_SET_ARRAY_FREE) != 1) {
				ARRfree(node_list);
				ARRfree(face_nnodes);
				ARRfree(face_corner_nnodes);
				ARRfree(face_node_list);
				ERR_RETURN("Error setting cell connect list");
			}
		}
		OMpop_status_range();
		OMpop_status_range();
		ARRfree(node_list);
		ARRfree(face_nnodes);
		ARRfree(face_corner_nnodes);
		ARRfree(face_node_list);
	}
	ARRfree(xyz);
	return(1);
}

