/*
			Copyright (c) 1995 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/extrude.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("DVextrude_cells_update", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_LEN 256
#define ALLOC_BLOCK 1024

/* 64-bit porting. Only Modified Internally */
int DVextrude_cells_update(OMobj_id mod_id)
{
	OMobj_id in, out, in_set, out_set;
	int comp, sides, col_sides, flip;
	double scale;
	int nsets, out_nsets, set, cell_nnodes;
	xp_long ncells, nnodes;
	int nspace, out_nspace;
	xp_long *node_conn;
	int j, n, veclen, dtype;
	xp_long i, k, n_w, m, ind, size, out_nnodes;
	float *in_coord, *out_coord, *quad_data, height;
	char *in_data, name[MAX_NAME_LEN], units[MAX_NAME_LEN];
	int poly_flag, *poly_nnodes;
	xp_long count, npoly, *poly_conn, quad_ncells, *quad_conn, quad_size;


	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);

	if (OMget_name_int_val(mod_id, OMstr_to_name("height_comp"), &comp) != 1)
		comp = 0;

	if (OMget_name_real_val(mod_id, OMstr_to_name("scale"), &scale) != 1)
		scale = 0;

	if (OMget_name_int_val(mod_id, OMstr_to_name("draw_sides"), &sides) != 1)
		sides = 0;

	if (OMget_name_int_val(mod_id, OMstr_to_name("color_sides"), &col_sides) != 1)
		col_sides = 0;

	if (OMget_name_int_val(mod_id, OMstr_to_name("flip_sides_normals"), &flip) != 1)
		flip = 0;

	/*
	 * clean the output
	 */
	if (FLDset_nnodes(out, 0) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDget_nnodes(in, &nnodes) != 1 || nnodes == 0) {
		/*
		 * this could be a case when module disconnected
		 * clean the output and return
		 */
		return(0);
	}

	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (nspace == 3)
		out_nspace = 3;
	else
		out_nspace = nspace+1;

	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	if (sides) {
		out_nsets = nsets+1;
		out_nnodes = 2*nnodes;
	}
	else {
		out_nsets = nsets;
		out_nnodes = nnodes;
	}

	if (FLDset_ncell_sets(out, out_nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}

	if (FLDset_nnodes(out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}

	if (FLDget_coord(in, &in_coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	for (i=0; i<nnodes; i++)
		memcpy(out_coord+i*out_nspace, in_coord+i*nspace, nspace*sizeof(float));
	if (out_nspace > nspace)
		for (i=0; i<nnodes; i++)
			out_coord[i*out_nspace+nspace] = 0.0;
	if (sides) {
		for (i=0; i<nnodes; i++)
			memcpy(out_coord+(i+nnodes)*out_nspace, in_coord+i*nspace, nspace*sizeof(float));
		if (out_nspace > nspace)
			for (i=0; i<nnodes; i++)
				out_coord[(i+nnodes)*out_nspace+nspace] = 0.0;
		quad_size = ALLOC_BLOCK;
		quad_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 4*quad_size, NULL);
		if (col_sides)
			quad_data = (float *)ARRalloc(NULL, DTYPE_FLOAT, quad_size, NULL);
		quad_ncells = 0;
	}

	for (set=0; set<nsets; set++) {
		if (FLDget_cell_set(in, set, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}

		if (FLDget_cell_set_name(in_set, name, MAX_NAME_LEN) != 1) {
			ERR_RETURN("Error getting cell set name");
		}
		if (FLDset_cell_set(out_set, name) != 1) {
			ERR_RETURN("Error setting cell set");
		}
		/*--------------------------------------------------*/
		/*  Make out cell set to be the same as in set      */
		/*--------------------------------------------------*/
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDcopy_subelem_ref(out_set, in_set) != 1) {
			ERR_RETURN("Error setting field reference");
		}
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_cell_data_veclen(out_set, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDget_cell_data(out_set, comp, &dtype, &in_data,
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get cell data");
		}
		if (FLDget_poly_flag(out_set, &poly_flag) != 1) {
			poly_flag = 0;
		}

		if (poly_flag == 0) {
			if (FLDget_ncells(out_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			if (FLDget_cell_set_nnodes(out_set,  &cell_nnodes) != 1) {
				ERR_RETURN("cannot get cell nnodes");
			}
			if (FLDget_node_connect(out_set, &node_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get cell connectivity");
			}
			for (i=0; i<ncells; i++) {
				UTILtype_to_float(&height, in_data+i*veclen*DTYPEtype_size[dtype], dtype);
				m = i*cell_nnodes;
				for (n=0; n<cell_nnodes; n++)
					out_coord[(node_conn[m+n]+1)*out_nspace-1] += scale*height;

			}
			if (sides) {
				if (quad_ncells+ncells*cell_nnodes >= quad_size) {
					quad_size += (ncells*cell_nnodes/ALLOC_BLOCK+1)*ALLOC_BLOCK;
					quad_conn = (xp_long *)ARRrealloc(quad_conn, DTYPE_LONG, 4*quad_size, NULL);
					if (col_sides)
						quad_data = (float *)ARRrealloc(quad_data, DTYPE_FLOAT, quad_size, NULL);
				}
				for (i=0; i<ncells; i++)
					for (j=0; j<cell_nnodes; j++) {
						n_w = node_conn[i*cell_nnodes+j];
						if (j == cell_nnodes-1)
							m = node_conn[i*cell_nnodes];
						else
							m = node_conn[i*cell_nnodes+j+1];
						ind = 4*quad_ncells;
						if (flip) {
							quad_conn[ind] = n_w;
							quad_conn[ind+1] = m;
							quad_conn[ind+2] = m+nnodes;
							quad_conn[ind+3] = n_w+nnodes;
						}
						else {
							quad_conn[ind] = m;
							quad_conn[ind+1] = n_w;
							quad_conn[ind+2] = n_w+nnodes;
							quad_conn[ind+3] = m+nnodes;
						}
						if (col_sides)
							quad_data[quad_ncells] = height;
						quad_ncells++;
					}
			}
			ARRfree(node_conn);
		}
		else {
			if (FLDget_npolys(out_set, &npoly) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			if (FLDget_poly_nnodes(out_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly_nnodes");
			}
			if (FLDget_poly_connect(out_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly connectivity");
			}
			for (count=0,i=0; i<npoly; i++) {
				UTILtype_to_float(&height, in_data+i*veclen*DTYPEtype_size[dtype], dtype);

				if (poly_nnodes[i] == 2) {  /* polyline or polytri */
					cell_nnodes = (int)(poly_conn[2*i+1]-poly_conn[2*i]+1);
					for (j=0; j<cell_nnodes; j++) {
						m = poly_conn[2*i]+j;
						out_coord[(m+1)*out_nspace-1] += scale*height;
					}
				}
				else {
					cell_nnodes = poly_nnodes[i];
					if (sides) {
						if (quad_ncells+cell_nnodes >= quad_size) {
							quad_size += (cell_nnodes/ALLOC_BLOCK+1)*ALLOC_BLOCK;
							quad_conn = (xp_long *)ARRrealloc(quad_conn, DTYPE_LONG, 4*quad_size, NULL);
							if (col_sides)
								quad_data = (float *)ARRrealloc(quad_data, DTYPE_FLOAT, quad_size, NULL);
						}
					}
					ind =  poly_conn[count];
					for (j=0; j<cell_nnodes; j++) {
						n_w = poly_conn[count];
						out_coord[(n_w+1)*out_nspace-1] += scale*height;
						if (sides) {
							if (j == cell_nnodes-1)
								m = ind;
							else
								m = poly_conn[count+1];
							k = 4*quad_ncells;
							if (flip) {
								quad_conn[k] = n_w;
								quad_conn[k+1] = m;
								quad_conn[k+2] = m+nnodes;
								quad_conn[k+3] = n_w+nnodes;
							}
							else {
								quad_conn[k] = m;
								quad_conn[k+1] = n_w;
								quad_conn[k+2] = n_w+nnodes;
								quad_conn[k+3] = m+nnodes;
							}

							if (col_sides)
								quad_data[quad_ncells] = height;
							quad_ncells++;
						}
						count++;
					}
				}
			}
			ARRfree(poly_conn);
			ARRfree(poly_nnodes);
		}
		ARRfree(in_data);
	}
	if (sides) {
		if (FLDget_cell_set(out, out_nsets-1, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_cell_set(out_set, "Quad") != 1) {
			ERR_RETURN("Error setting cell set");
		}
		if (FLDset_ncells(out_set, quad_ncells) != 1) {
			ERR_RETURN("cannot set ncells");
		}
		if (FLDset_node_connect(out_set, quad_conn, quad_ncells*4, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("cannot set quad connectivity");
		}
		if (col_sides) {
			if (FLDset_cell_data_ncomp(out_set, 1) != 1) {
				ERR_RETURN("Error setting ncomp");
			}
			strcpy(units, "");
			strcpy(name, "");
			if (FLDset_cell_data_comp (out_set, 0, 1, name, units) != 1) {
				ERR_RETURN("Error setting node component");
			}
			if (FLDset_cell_data(out_set, 0, (char *)quad_data, DTYPE_FLOAT, quad_ncells, OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("cannot set quad data");
			}
		}
		else {
			if (FLDset_cell_data_ncomp(out_set, 0) != 1) {
				ERR_RETURN("Error setting ncomp");
			}
		}
	}

	ARRfree(in_coord);
	ARRfree(out_coord);
	return(1);
}
