/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/g_glyph.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd_def.h>
#include <avs/arr.h>
#include <avs/mat.h>
#include <avs/dtype.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("geo_glyph", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define STATUS_REPORT   256

#define BLOCK_SIZE   1024
#define BLOCK_GROW   1.5

int FUNCgeo_glyph (OMobj_id in, int nglyph, OMobj_id *glyphs, double scale, 
                   float *color, int norm_ind, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVgeo_glyph_update(OMobj_id elem_id)
{
	OMobj_id in, out, glyph, *glyphs, scale_id, col_id, norm_id;
	float *color;
	double scale;
	int norm, stat, type;
	xp_long i, nglyph, size;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	glyph = OMfind_subobj(elem_id, OMstr_to_name("glyph"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	scale_id = OMfind_subobj(elem_id, OMstr_to_name("scale"), OM_OBJ_RD);
	OMget_real_val(scale_id, &scale);
	norm_id = OMfind_subobj(elem_id, OMstr_to_name("normalize"), OM_OBJ_RD);
	OMget_int_val(norm_id, &norm);

	col_id = OMfind_subobj(elem_id, OMstr_to_name("color"), OM_OBJ_RD);
	type = OM_TYPE_FLOAT;
	size = 0;
	color = (float *)NULL;
	if (OMget_array_sz(col_id, &type, (char **)(&color), 
		    &size, OM_GET_ARRAY_RD) != 1) color = NULL;

	stat = OMget_array_size(glyph, &nglyph);
	if (stat != 1) {
		return(1);
	}
	glyphs = (OMobj_id *)malloc(nglyph*sizeof(OMobj_id));
	if (glyphs == NULL) {
		ERR_RETURN("cannot allocate glyphs");
	}
	for (i=0; i<nglyph; i++) {
		if ((stat = OMget_array_val (glyph, i, &glyphs[i], OM_OBJ_RD)) != 1) {
			ERR_RETURN("Error getting glyphs");
		}
	}

	stat = FUNCgeo_glyph(in, (int)nglyph, glyphs, scale, color, norm, out);

	free(glyphs);

	if (color != NULL) ARRfree(color);

	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FUNCgeo_glyph (OMobj_id in, int nglyph, OMobj_id *glyphs, double scale, 
                   float *color, int norm_ind, OMobj_id out)
{
	xp_long nnodes, nnodes_g, out_nnodes;
	int   dtype, stat;
	xp_long size, ncount;
	int   poly_flag, polyhedron, *poly_nnodes, *out_poly_nnodes;
	xp_long npoly, *poly_conn, *out_poly_conn;
	int   out_veclen, glyph_ncomp, data_id, out_dtype, null_flag, pick_type;
	xp_long *pick_data;
	int   j, k, l, r_out_veclen, r_glyph_ncomp, r_data_id, r_out_dtype;
	xp_long i, j_w, k_w, m, n, out_n, count;
	xp_long *node_list, *out_node_list, ncells;
	int   nsets, cs, cell_nnodes;
	int   nspace, nspace_g, nspace_out;
	float  *coord, *coord_g, *out_coord, trans[3], point[3];
	float  *xfm, xform[4][4], *in_xfm, in_xform[4][4];
	char   *node_data, *glyph_data, *out_data;
	xp_long *index_data, ind, nsize;
	int gl, ncomp, veclen, out_nsets;
	OMobj_id   glyph, cell_set, out_cell_set;
	char  name[MAX_NAME_SIZE];
	xp_long  per_count, block_size = BLOCK_SIZE;
	int interrupt;
	float per;
	double  min, max, dmin, dmax, null_value, dnull_value;

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDset_ncell_sets(out, 0) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	out_coord = NULL;
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting out_coord");
	}

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (nnodes == 0) {
		return(1);
	}

	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_coord(in, &coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ncomp = 0;
	}
	if (ncomp) {
		if (FLDget_node_data_veclen(in, 0, &veclen) !=1) {
			ERR_RETURN("Error getting veclen");
		}
		if (veclen != 1) {
			ncomp = 0;
			ERRerror("geo_glyph",1,ERR_ORIG, "index data is not a scalar, ignored");
		}
	}
	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else {
		in_xfm = (float *)in_xform;
	}

	per_count =0;
	per = 100.0/nnodes;
	if (FLDset_node_data_ncomp (out, 2) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDset_node_data_veclen(out, 1, 1) !=1) {
		ERR_RETURN("Error setting veclen");
	}
	if (FLDset_node_data_id(out, 1, GD_PICK_DATA_ID) != 1) {
		ERR_RETURN("Error setting data id");
	}
	if (ncomp == 0 || nglyph == 1) {
		glyph = glyphs[0];

		if (FLDget_node_data_ncomp(glyph, &glyph_ncomp) != 1 || glyph_ncomp==0) {
			glyph_ncomp = 0;
			data_id = 667;
			out_dtype = DTYPE_FLOAT;
			out_veclen = 3;
		}
		else {
			if (FLDget_node_data_veclen(glyph, 0, &out_veclen) !=1) {
				ERR_RETURN("Error getting veclen");
			}
			if (FLDget_node_data_id(glyph, 0, &data_id) != 1)
				data_id = 0;
			if (FLDget_node_data(glyph, 0, &out_dtype, (char **)&glyph_data, 
					     &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get node data");
			}
		}
		if (FLDset_node_data_veclen(out, 0, out_veclen) != 1) {
			ERR_RETURN("Error setting data veclen");
		}
		if (data_id) {
			if (FLDset_node_data_id(out, 0, data_id) != 1) {
				ERR_RETURN("Error setting data veclen");
			}
		}
		if (FLDget_nnodes(glyph, &nnodes_g) != 1) {
			ERR_RETURN("cannot get nnodes");
		}
		if (FLDget_coord(glyph, &coord_g, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get coordinates");
		}
		if (FLDget_nspace(glyph, &nspace_g) != 1) {
			ERR_RETURN("cannot get nspace");
		}
		stat = FLDget_xform(glyph, (float *)xform);
		if (stat < 0) {
			ERR_RETURN("cannot get xform for probe");
		}
		else if (stat == 0) {
			xfm = (float *)0;
		}
		else if (MATmat_is_identity((float *)xform, 4))
			xfm = (float *)0;
		else
			xfm = (float *)xform;

		/*** OUTPUT GRID ***/
		nspace_out = (nspace > nspace_g) ? nspace : nspace_g;
		if (nspace_out == 2 && in_xfm)
			MATmat4_xform_dim(in_xfm, &nspace_out);


		if (FLDset_nspace (out, nspace_out) != 1) {
			ERR_RETURN("Error setting nspace");
		}
		if (FLDset_nnodes (out, nnodes*nnodes_g) != 1) {
			ERR_RETURN("Error setting nnodes");
		}
		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}

		if (FLDget_node_data(out, 0, &out_dtype, (char **)&node_data, 
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		pick_type = DTYPE_LONG;
		if (FLDget_node_data(out, 1, &pick_type, (char **)&pick_data, 
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		/*------------------*/
		/*   COORD and DATA */
		/*------------------*/
		for (count=0,out_n=0,i=0; i<nnodes; i++) {
			if (per_count == STATUS_REPORT) {
				OMstatus_check((int)(i*per),NULL,&interrupt);
				if (interrupt) {
					if (FLDset_nnodes (out, 0) != 1) {
						ERR_RETURN("Error setting nnodes");
					}
					if (FLDset_node_data_ncomp (out, 0) != 1) {
						ERR_RETURN("Error setting nnode_data");
					}
					ARRfree(node_data);
					ARRfree(coord_g);
					ARRfree(out_coord);
					ARRfree(coord);
					return(0);
				}
				per_count =0;
			}
			else
				per_count++;


			n = nspace*i;
			for (j=0; j<nspace; j++)
				trans[j] = coord[n+j];
			for (k=j; k<3; k++)
				trans[k] = 0.0;
			if (in_xfm)
				MATvec3_mat4_multiply(trans, (Matr4 *)in_xfm);
			for (j_w=0; j_w<nnodes_g; j_w++) {
				for (k=0; k<nspace_g; k++)
					point[k] = coord_g[j_w*nspace_g+k]*scale;
				for (l=k; l<3; l++)
					point[l] = 0.0;
				if (xfm)
					MATvec3_mat4_multiply(point, (Matr4 *)xfm);
				for (k=0; k<nspace_out; k++)
					out_coord[out_n++] = point[k]+trans[k];
				if (glyph_ncomp)
					memcpy(node_data+count*out_veclen*DTYPEtype_size[out_dtype], 
					       glyph_data+j_w*out_veclen*DTYPEtype_size[out_dtype],
					       out_veclen*DTYPEtype_size[out_dtype]);
				else if (color != NULL)
					memcpy(node_data+count*3*sizeof(float), color, 3*sizeof(float));
				pick_data[count] = i;
				count++;
			}
		}

		/*---------------*/
		/*   OUT CELLS   */
		/*---------------*/
		if (FLDset_ncell_sets(out, 0) != 1) {
			ERR_RETURN("cannot get nsets");
		}

		if (FLDget_ncell_sets(glyph, &nsets) != 1) {
			ERR_RETURN("cannot get nsets");
		}
		for (cs=0; cs<nsets; cs++) {
			if (FLDget_cell_set(glyph, cs, &cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			if (FLDget_cell_set_name(cell_set, name, MAX_NAME_SIZE) != 1) {
				ERR_RETURN("cannot get cell set name");
			}
			if (FLDadd_cell_set(out, name) != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDget_cell_set(out, cs, &out_cell_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}

			if (FLDget_poly_flag(cell_set, &poly_flag) != 1) {
				poly_flag = 0;
			}
			if (poly_flag == 0) {
				if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
					ERR_RETURN("cannot get cell nnodes");
				}
				if (FLDget_ncells(cell_set, &ncells) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				if (FLDset_ncells(out_cell_set, ncells*nnodes) != 1) {
					ERR_RETURN("Error setting ncells");
				}
				if (FLDget_node_connect(cell_set, &node_list, 
							&size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get cell connectivity");
				}
				if (FLDget_node_connect(out_cell_set, &out_node_list, 
							&size, OM_GET_ARRAY_WR) != 1) {
					ERR_RETURN("cannot get cell connectivity");
				}
				m = cell_nnodes*ncells;
				for (i=0; i<nnodes; i++) {
					k_w = i*m;
					n = i*nnodes_g;
					for (j_w=0; j_w<m; j_w++)
						out_node_list[k_w+j_w] = node_list[j_w] + n;
				}
				ARRfree(node_list);
				ARRfree(out_node_list);
			}
			else {
				if (FLDget_npolys(cell_set, &npoly) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				if (FLDset_npolys(out_cell_set, npoly*nnodes) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				if (FLDget_poly_nnodes(cell_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get poly_nnodes");
				}
				if (FLDget_poly_connect(cell_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get poly connectivity");
				}
				if (FLDget_poly_nnodes(out_cell_set, &out_poly_nnodes, &size, OM_GET_ARRAY_RW) != 1) {
					ERR_RETURN("cannot get poly_nnodes");
				}
				for (m=0,polyhedron=0,i=0; i<npoly; i++) {
					m += poly_nnodes[i];
					if (poly_nnodes[i] != 2)
						polyhedron=1;
				}
				if (polyhedron == 0) {
					if (FLDget_poly_connect(out_cell_set, &out_poly_conn, &size, OM_GET_ARRAY_RW) != 1) {
						ERR_RETURN("cannot get poly connectivity");
					}
					m = 2*npoly;
					for (i=0; i<nnodes; i++) {
						k_w = i*m;
						n = i*nnodes_g;
						for (j_w=0; j_w<m; j_w++)
							out_poly_conn[k_w+j_w] = poly_conn[j_w] + n;
					}
					ARRfree(out_poly_conn);
				}
				else {
					out_poly_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, m*nnodes, NULL);
					for (ncount=0, i=0; i<nnodes; i++) {
						for (j_w=0; j_w<npoly; j_w++)
							out_poly_nnodes[ncount++] = poly_nnodes[j_w];
						k_w = i*m;
						n = i*nnodes_g;
						for (j_w=0; j_w<m; j_w++)
							out_poly_conn[k_w+j_w] = poly_conn[j_w] + n;
					}
					if (FLDset_poly_connect(out_cell_set, out_poly_conn, m*nnodes, OM_SET_ARRAY_FREE) != 1) {
						ERR_RETURN("cannot get poly connectivity");
					}
				}
				ARRfree(poly_conn);
				ARRfree(poly_nnodes);
				ARRfree(out_poly_nnodes);
			}
		}
		ARRfree(node_data);
		ARRfree(pick_data);
		ARRfree(coord_g);
		ARRfree(out_coord);
		ARRfree(coord);
		if (glyph_ncomp)
			ARRfree(glyph_data);
		return(1);
	}

	/*-----------------------*/
	/*  Multiple glyphs case */
	/*-----------------------*/

	nspace_out = 3;
	null_flag = 0;
	if (FLDset_nspace (out, nspace_out) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (norm_ind) {
		if (FLDget_node_data(in, 0, &dtype, (char **)&node_data, 
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_data_minmax(in, 0, (char *)&min, (char *)&max) != 1) {
			ERR_RETURN("cannot get minmax");
		}
		UTILtype_to_double(&dmin, (char *)&min, dtype);
		UTILtype_to_double(&dmax, (char *)&max, dtype);
		if (FLDget_node_null_data(in, 0, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag)
			UTILtype_to_double(&dnull_value, (char *)&null_value, dtype);
	}
	else {
		dtype = DTYPE_INT;
		if (FLDget_typed_node_data(in, 0, &dtype, (char **)&node_data, 
					   &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get node data");
		}
		index_data = (xp_long *)node_data;
		if (FLDget_node_null_data(in, 0, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag)
			UTILtype_to_double(&dnull_value, (char *)&null_value, dtype);
	}

	out_nnodes = 0;
	count = 0;

	nsize = block_size;
	out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, nsize*nspace_out, NULL);

	glyph = glyphs[0];
	if (FLDget_node_data_ncomp(glyph, &glyph_ncomp) != 1 || glyph_ncomp==0) {
		glyph_ncomp = 0;
		data_id = 667;
		out_dtype = DTYPE_FLOAT;
		out_veclen = 3;
	}
	else {
		if (FLDget_node_data_veclen(glyph, 0, &out_veclen) !=1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDget_node_data_id(glyph, 0, &data_id) != 1)
			data_id = 0;
		if (FLDget_node_data(glyph, 0, &out_dtype, (char **)&glyph_data, 
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get node data");
		}
	}
	if (FLDset_node_data_veclen(out, 0, out_veclen) != 1) {
		ERR_RETURN("Error setting data veclen");
	}
	if (data_id) {
		if (FLDset_node_data_id(out, 0, data_id) != 1) {
			ERR_RETURN("Error setting data veclen");
		}
	}
	out_data = (char *)ARRalloc(NULL, out_dtype, nsize*out_veclen, NULL);
	pick_data = (xp_long *)ARRalloc(NULL, DTYPE_LONG, nsize, NULL);

	for (gl=0; gl<nglyph; gl++) {
		glyph = glyphs[gl];

		if (gl) {
			if (FLDget_node_data_ncomp(glyph, &r_glyph_ncomp) != 1 || r_glyph_ncomp==0) {
				r_glyph_ncomp = 0;
				r_data_id = 667;
				r_out_dtype = DTYPE_FLOAT;
				r_out_veclen = 3;
			}
			else {
				if (FLDget_node_data_veclen(glyph, 0, &r_out_veclen) !=1) {
					ERR_RETURN("Error getting veclen");
				}
				if (FLDget_node_data_id(glyph, 0, &r_data_id) != 1)
					r_data_id = 0;
				if (FLDget_node_data(glyph, 0, &r_out_dtype, (char **)&glyph_data, 
						     &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get node data");
				}
			}
			if (glyph_ncomp != r_glyph_ncomp || data_id != r_data_id || out_dtype != r_out_dtype ||
			    out_veclen != r_out_veclen) {
				ERR_RETURN("different glyphs have incompatible data");
			}
		}

		if (FLDget_nnodes(glyph, &nnodes_g) != 1) {
			ERR_RETURN("cannot get nnodes");
		}
		if (FLDget_coord(glyph, &coord_g, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get coordinates");
		}
		if (FLDget_nspace(glyph, &nspace_g) != 1) {
			ERR_RETURN("cannot get nspace");
		}
		stat = FLDget_xform(glyph, (float *)xform);
		if (stat < 0) {
			ERR_RETURN("cannot get xform for probe");
		}
		else if (stat == 0) {
			xfm = (float *)0;
		}
		else if (MATmat_is_identity((float *)xform, 4))
			xfm = (float *)0;
		else
			xfm = (float *)xform;

		for (out_n=0, i=0; i<nnodes; i++) {
			if (null_flag) {
				UTILtype_to_double(&min, (char *)node_data+i*DTYPEtype_size[dtype], dtype);
				if (min == dnull_value)
					continue;
			}
			if (norm_ind) {
				UTILtype_to_double(&min, (char *)node_data+i*DTYPEtype_size[dtype], dtype);
				ind = (min-dmin)/(dmax-dmin)*(nglyph);
			}
			else
				ind = index_data[i];

			if (ind < 0)
				ind = 0;
			if (ind > (xp_long)(nglyph-1))
				ind = nglyph-1;
			if (ind != (xp_long)gl)
				continue;

			if (per_count == STATUS_REPORT) {
				OMstatus_check((int)(i*per),NULL,&interrupt);
				if (interrupt) {
					if (FLDset_nnodes (out, 0) != 1) {
						ERR_RETURN("Error setting nnodes");
					}
					if (FLDset_node_data_ncomp (out, 0) != 1) {
						ERR_RETURN("Error setting nnode_data");
					}
					if (FLDset_ncell_sets(out, 0) != 1) {
						ERR_RETURN("cannot get nsets");
					}
					ARRfree(node_data);
					ARRfree(coord_g);
					ARRfree(coord);
					return(0);
				}
				per_count =0;
			}
			else
				per_count++;

			n = nspace*i;
			for (j=0; j<nspace; j++)
				trans[j] = coord[n+j];
			for (k=j; k<3; k++)
				trans[k] = 0.0;
			if (in_xfm)
				MATvec3_mat4_multiply(trans, (Matr4 *)in_xfm);

			if (out_nnodes+nnodes_g >= nsize) {
				while (out_nnodes+nnodes_g >= nsize) {
					nsize += block_size;
					block_size *= BLOCK_GROW;
				}
				out_coord = (float *)ARRrealloc(out_coord, DTYPE_FLOAT,
								nspace_out*nsize, NULL);
				out_data = (char *)ARRrealloc(out_data, out_dtype,
								out_veclen*nsize, NULL);
				pick_data = (xp_long *)ARRrealloc(pick_data, DTYPE_LONG,
							      nsize, NULL);
			}
			for (j_w=0; j_w<nnodes_g; j_w++) {
				for (k=0; k<nspace_g; k++)
					point[k] = coord_g[j_w*nspace_g+k]*scale;
				for (l=k; l<3; l++)
					point[l] = 0.0;
				if (xfm)
					MATvec3_mat4_multiply(point, (Matr4 *)xfm);
				for (k=0; k<nspace_out; k++)
					out_coord[out_nnodes*nspace_out+k] = point[k]+trans[k];
				if (glyph_ncomp)
					memcpy(out_data+out_nnodes*out_veclen*DTYPEtype_size[out_dtype], 
					       glyph_data+j_w*out_veclen*DTYPEtype_size[out_dtype],
					       out_veclen*DTYPEtype_size[out_dtype]);
				else if (color != NULL)
					memcpy(out_data+out_nnodes*3*sizeof(float), color, 3*sizeof(float));
				pick_data[out_nnodes] = i;
				out_nnodes++;
			}
			out_n++;
		}
		if (out_n) {
			/*---------------*/
			/*   OUT CELLS   */
			/*---------------*/
			if (FLDget_ncell_sets(glyph, &nsets) != 1) {
				ERR_RETURN("cannot get nsets");
			}
			if (FLDget_ncell_sets(out, &out_nsets) != 1) {
				ERR_RETURN("cannot get nsets");
			}
			for (cs=0; cs<nsets; cs++) {
				if (FLDget_cell_set(glyph, cs, &cell_set) != 1) {
					ERR_RETURN("cannot get cell set");
				}
				if (FLDget_cell_set_name(cell_set, name, MAX_NAME_SIZE) != 1) {
					ERR_RETURN("cannot get cell set name");
				}
				if (FLDadd_cell_set(out, name) != 1) {
					ERR_RETURN("Error setting cell type");
				}
				if (FLDget_cell_set(out, out_nsets+cs, &out_cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDget_poly_flag(cell_set, &poly_flag) != 1) {
					poly_flag = 0;
				}
				if (poly_flag == 0) {
					if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
						ERR_RETURN("cannot get cell nnodes");
					}
					if (FLDget_ncells(cell_set, &ncells) != 1) {
						ERR_RETURN("cannot get ncells");
					}
					if (FLDset_ncells(out_cell_set, ncells*out_n) != 1) {
						ERR_RETURN("Error setting ncells");
					}
					if (FLDget_node_connect(cell_set, &node_list, 
								&size, OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("cannot get cell connectivity");
					}
					if (FLDget_node_connect(out_cell_set, &out_node_list, 
								&size, OM_GET_ARRAY_WR) != 1) {
						ERR_RETURN("cannot get cell connectivity");
					}
					m = cell_nnodes*ncells;
					for (i=0; i<out_n; i++) {
						k_w = i*m;
						n = i*nnodes_g+count;
						for (j_w=0; j_w<m; j_w++)
							out_node_list[k_w+j_w] = node_list[j_w] + n;
					}
					ARRfree(node_list);
					ARRfree(out_node_list);
				}
				else {
					if (FLDget_npolys(cell_set, &npoly) != 1) {
						ERR_RETURN("cannot get ncells");
					}
					if (FLDset_npolys(out_cell_set, npoly*out_n) != 1) {
						ERR_RETURN("cannot get ncells");
					}
					if (FLDget_poly_nnodes(cell_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("cannot get poly_nnodes");
					}
					if (FLDget_poly_connect(cell_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("cannot get poly connectivity");
					}
					if (FLDget_poly_nnodes(out_cell_set, &out_poly_nnodes, &size, OM_GET_ARRAY_RW) != 1) {
						ERR_RETURN("cannot get poly_nnodes");
					}
					for (m=0,polyhedron=0,i=0; i<npoly; i++) {
						m += poly_nnodes[i];
						if (poly_nnodes[i] != 2)
							polyhedron=1;
					}
					if (polyhedron == 0) {
						if (FLDget_poly_connect(out_cell_set, &out_poly_conn, &size, OM_GET_ARRAY_RW) != 1) {
							ERR_RETURN("cannot get poly connectivity");
						}
						m = 2*npoly;
						for (i=0; i<out_n; i++) {
							k_w = i*m;
							n = i*nnodes_g+count;
							for (j_w=0; j_w<m; j_w++)
								out_poly_conn[k_w+j_w] = poly_conn[j_w] + n;
						}
						ARRfree(out_poly_conn);
					}
					else {
						out_poly_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, m*out_n, NULL);
						for (ncount=0, i=0; i<out_n; i++) {
							for (j_w=0; j_w<npoly; j_w++)
								out_poly_nnodes[ncount++] = poly_nnodes[j_w];
							k_w = i*m;
							n = i*nnodes_g+count;
							for (j_w=0; j_w<m; j_w++)
								out_poly_conn[k_w+j_w] = poly_conn[j_w] + n;
						}
						if (FLDset_poly_connect(out_cell_set, out_poly_conn, m*out_n, OM_SET_ARRAY_FREE) != 1) {
							ERR_RETURN("cannot get poly connectivity");
						}
					}
					ARRfree(poly_conn);
					ARRfree(poly_nnodes);
					ARRfree(out_poly_nnodes);
				}
			}
		}
		ARRfree(coord_g);
		count = out_nnodes;
	}
	if (FLDset_nspace (out, nspace_out) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, out_coord, out_nnodes*nspace_out, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	if (FLDset_node_data(out, 0, (char *)out_data, out_dtype, out_veclen*out_nnodes, 
			     OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("cannot get node data");
	}
	out_dtype = DTYPE_INT;
	if (FLDset_node_data(out, 1, (char *)pick_data, out_dtype, out_nnodes, 
			     OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("cannot get node data");
	}
	ARRfree(node_data);
	ARRfree(coord);

	return(1);
}
