/*
			Copyright (c) 1996 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/image/libpbm/iface.c#1 $
*/

#include "pbm.h"

/*-------------------------------------------------------------------------*/
void *pbmOpen(char *filename, int access)
{
  pbm_t *pbm;

  if (!(pbm = (pbm_t *) malloc(sizeof(pbm_t))))
  {
    ERRerror("pbmOpen", 0, ERR_ORIG, "Failure allocating pbm struct");
    return(NULL);
  }

  /* init */
  pbm->image = pbm->argb_data = NULL;

  if (access & DV_IMAGE_FILE_ACCESS_READ)
  {
    if ((pbm->fp = (FILE *) FILEfopen(filename, SIO_R_BIN)) == NULL)
    {
      ERRerror("pbmOpen", 0, ERR_ORIG, "Cannot open file for input");
      free(pbm);
      return(NULL);
    }

    if (ReadpbmHeader(pbm) != pbmSUCCESS)
    {
      ERRerror("pbmOpen", 0, ERR_ORIG, "Failure reading image data");
      fclose(pbm->fp);
      free(pbm);
      return(NULL);
    }
  }
  else if (access & DV_IMAGE_FILE_ACCESS_WRITE)
  {
    if ((pbm->fp = (FILE *) FILEfopen(filename, SIO_W_BIN)) == NULL)
    {
      ERRerror("pbmOpen", 0, ERR_ORIG, "Cannot open file for output");
      free(pbm);
      return(NULL);
    }
  }
  else
  {
    ERRerror("pbmOpen", 0, ERR_ORIG, "File access type is invalid");
    free(pbm);
    return(NULL);
  }

  pbm->access = access;

  return(pbm);
}

/*-------------------------------------------------------------------------*/
void pbmClose(void *pbm)
{
  if (pbm)
  {
    pbm_t *pt = (pbm_t *) pbm;

    /* if writing to it on close */
    if (pt->access & DV_IMAGE_FILE_ACCESS_WRITE)
    {
      if (Writepnm(pt) == pbmFAILURE) {
        ERRerror("pbmClose", 0, ERR_ORIG, "Error while writing pixmap file");
      }

      /* copy of caller's ptr: have him do the free, not us */
      pt->argb_data = NULL;
    }

    if (pt->image != NULL)
      free(pt->image);

    if (pt->argb_data != NULL)
      free(pt->argb_data);

    if (pt->fp != NULL)
      fclose(pt->fp);

    free(pt);
  }
}

/*----------------------  G E T  F U N C T I O N S  -----------------------*/

/*-------------------------------------------------------------------------*/
int pbmGetWidth(void *pbm, int *width)
{
  pbm_t *pt = (pbm_t *) pbm;
  *width = pt->width;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmGetHeight(void *pbm, int *height)
{
  pbm_t *pt = (pbm_t *) pbm;
  *height = pt->height;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmGetARGBImage(void *pbm, pbmBYTE **data)
{
  pbm_t *pt = (pbm_t *) pbm;
  int i, npixels;
  pbmBYTE *src, *dst;

  if ((ReadpbmImage(pbm)) == pbmFAILURE)
    return(0);

  src = pt->image;
  npixels = pt->width * pt->height;

  if (!(pt->argb_data = (pbmBYTE *) malloc(npixels * 4)))
  {
    ERRerror("pbmGetARGBImage", 0, ERR_ORIG,
	     "Failure allocating space for ARGB data");
    return(pbmFAILURE);
  }

  dst = pt->argb_data;
  if (pt->colortype == DV_IMAGE_COLORTYPE_RGB)
  {
    /* we have 3 byte RGB, expand into ARGB */
    for (i = 0; i < npixels; i++)
    {
      *dst++ = 255;  *dst++ = *src++;  *dst++ = *src++;  *dst++ = *src++;
    }
  }
  else
  {
    /* 1 byte greyscale, expand into ARGB */
    for (i = 0; i < npixels; i++)
    {
      *dst++ = 255;  *dst++ = *src;  *dst++ = *src;  *dst++ = *src++;
    }
  }

  *data = pt->argb_data;
  return(pbmSUCCESS);
}

/*----------------------  S E T  F U N C T I O N S  -----------------------*/

/*-------------------------------------------------------------------------*/
int pbmSetWidth(void *pbm, int width)
{
  pbm_t *pt = (pbm_t *) pbm;

  if (width <= 0)
  {
    ERRerror("pbmSetWidth", 1, ERR_ORIG, "Width (%d) is unreasonable?", width);
    return(pbmFAILURE);
  }

  pt->width = width;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetHeight(void *pbm, int height)
{
  pbm_t *pt = (pbm_t *) pbm;

  if (height <= 0)
  {
    ERRerror("pbmSetHeight", 1, ERR_ORIG, "Height (%d) is unreasonable?", height);
    return(pbmFAILURE);
  }

  pt->height = height;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetFileType(void *pbm, int type)
{
  pbm_t *pt = (pbm_t *) pbm;

  if (type < 0 || type > DV_IMAGE_FILE_TYPE_MAX)
  {
    ERRerror("pbmSetFileType", 1, ERR_ORIG, "Type (%d) is invalid", type);
    return(pbmFAILURE);
  }

  pt->type = type;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetDepth(void *pbm, int depth)
{
  /* N/A, set via colortype */
  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetColorType(void *pbm, int type)
{
  pbm_t *pt = (pbm_t *) pbm;

  if (type < 0 || type > DV_IMAGE_COLORTYPE_MAX)
  {
    ERRerror("pbmSetColorType", 1, ERR_ORIG, "Type (%d) is invalid", type);
    return(pbmFAILURE);
  }

  pt->colortype = type;

  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetCompressionType(void *pbm, int type, int quality)
{
  /* N/A, always uncompressed */
  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetReductionType(void *pbm, int type)
{
  /* N/A */
  return(pbmSUCCESS);
}

/*-------------------------------------------------------------------------*/
int pbmSetARGBImage(void *pbm, unsigned char *data, int flip)
{
  pbm_t *pt = (pbm_t *) pbm;

  pt->argb_data = data;
  pt->flip = flip;

  return(pbmSUCCESS);
}
