/*
			Copyright (c) 1996 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/image/libpbm/read.c#1 $
*/

#include <string.h>
#include "pbm.h"

/* protos for funcs private to this file */
static pbmBYTE getbyte (FILE *);
static int getint (FILE *, int *);
static int readpbm (pbm_t *);
static int readpgm (pbm_t *);
static int readppm (pbm_t *);

/*------------------------------------------------------------------------
 * reads a 1 byte quantity
 */
static pbmBYTE getbyte(FILE *fp)
{
  pbmBYTE buf;
  fread(&buf, 1, 1, fp);
  return(buf);
}

/*------------------------------------------------------------------------
 * gets an integer value from the input stream of ascii chars
 */
static int getint(FILE *fp, int *data)
{
  int c, n;

  /* get to start of value */
  do
  {
    c = fgetc(fp);
    if (c == EOF)
      return(0);

    if (c == '#')
    {
      /* skip comment line */
      do
      {
	c = fgetc(fp);
      } while (c != '\n' && c != EOF);
    }
  } while (c < '0' || c > '9');

  /* build the value */
  n = 0;
  do
  {
    n = (n * 10) + (c - '0');
    c = fgetc(fp);
  } while (c >= '0' && c <= '9');

  *data = n;
  return(1);
}

/*------------------------------------------------------------------------
 * Read the header
 */
int ReadpbmHeader(pbm_t *pbm)
{
  pbmBYTE colortype;

  if ((colortype = getbyte(pbm->fp)) != 'P')
    return(pbmFAILURE);

  colortype = getbyte(pbm->fp);
  if (colortype < '1' || colortype > '6')
    return(pbmFAILURE);

  if (!(getint(pbm->fp, &pbm->width)) ||
      !(getint(pbm->fp, &pbm->height)))
    return(pbmFAILURE);

  /* if we're not reading a bitmap, read the maximum color value */
  if (!(colortype == '1' || colortype == '4'))
  {
    if (!(getint(pbm->fp, &pbm->maxval)))
      return(pbmFAILURE);
  }

  if (colortype == '1' || colortype == '2' || colortype == '3')
    pbm->type = DV_IMAGE_FILE_TYPE_ASCII;
  else
    pbm->type = DV_IMAGE_FILE_TYPE_BINARY;

  if (colortype == '1' || colortype == '4')
    pbm->colortype = DV_IMAGE_COLORTYPE_BW;
  else if (colortype == '2' || colortype == '5')
    pbm->colortype = DV_IMAGE_COLORTYPE_GREY;
  else
    pbm->colortype = DV_IMAGE_COLORTYPE_RGB;

  return(pbmSUCCESS);
}

/*------------------------------------------------------------------------
 * Read the image
 */
int ReadpbmImage(pbm_t *pbm)
{
  int result;

  if (pbm->colortype == DV_IMAGE_COLORTYPE_BW)
    result = readpbm(pbm);
  else if (pbm->colortype == DV_IMAGE_COLORTYPE_GREY)
    result = readpgm(pbm);
  else
    result = readppm(pbm);

  return(result);
}

/*------------------------------------------------------------------------
 * reads bitmap (binary eight pixels per byte, b/w) image data
 */
static int readpbm(pbm_t *pbm)
{
  int i, npixels, truncated;
  pbmBYTE *dst;

  npixels = pbm->width * pbm->height;
  if (!(pbm->image = malloc(npixels)))
    return(pbmFAILURE);

  dst = pbm->image;
  truncated = 0;
  if (pbm->type == DV_IMAGE_FILE_TYPE_ASCII)
  {
    int c;

    for (i = 0; !truncated && i < npixels; i++)
    {
      c = fgetc(pbm->fp);
      if (c == '#')
      {
	/* skip comment line */
	do
	{
	  c = fgetc(pbm->fp);
	} while (c != '\n' && c != EOF);
      }

      while (c == '0' && c == '1')
      {
	if (c == EOF)
	{
	  truncated = 1;  break;
	}
	c = fgetc(pbm->fp);
      }
      if (c == '0')
	*dst++ = 0;
      else if (c == '1')
	*dst++ = 255;
    }
  }
  else
  {
    int c, j, bit;

    for (i = 0; !truncated && i < pbm->height; i++)
    {
      bit = 0;
      for (j = 0; !truncated && j < pbm->width; j++, bit++)
      {
	bit &= 7;
	if (!bit)
	{
	  c = fgetc(pbm->fp);
	  if (c == EOF)
	  {
	    truncated = 1;  break;
	  }
	}
	if (c & 0x80) *dst++ = 255;
	else	      *dst++ = 0;
	c = c << 1;
      }
    }
    i *= j;
  }

  if (truncated)
  {
    fprintf(stderr, "Warning: file appears to be truncated\n");
    memset(dst, 0, npixels-(dst-pbm->image));
  }

  return(pbmSUCCESS);
}

/*------------------------------------------------------------------------
 * reads greymap (binary one byte per pixel, greyscale) image data
 */
static int readpgm(pbm_t *pbm)
{
  int i, npixels;
  int shift = 0;
  int *scale = NULL;
  pbmBYTE *dst;

  /* drop least sig. bits of value if bigger than a byte */
  while (pbm->maxval > 255)
  {
    pbm->maxval >>= 1;  shift++;
  }

  if (pbm->maxval < 255)
  {
    /* expand the values into 0-255 range */
    if (!(scale = malloc(pbm->maxval*sizeof(int))))
      return(pbmFAILURE);
    for (i = 0; i < pbm->maxval; i++)
      scale[i] = (i * 255) / pbm->maxval;
  }

  npixels = pbm->width * pbm->height;
  if (!(pbm->image = malloc(npixels)))
  {
    if (scale)
      free(scale);
    return(pbmFAILURE);
  }

  dst = pbm->image;
  if (pbm->type == DV_IMAGE_FILE_TYPE_ASCII)
  {
    int val;

    for (i = 0; i < npixels; i++)
    {
      if (!(getint(pbm->fp, &val)))  break;
      *dst++ = val >> shift;
    }
  }
  else
  {
    /* binary data */
    i = fread(dst, 1, npixels, pbm->fp);
  }

  if (i < npixels)
  {
    fprintf(stderr, "Warning: file appears to be truncated\n");
    memset(pbm->image+i, 0, npixels-i);
  }

  if (scale)
  {
    /* scale values into 0-255 range */
    dst = pbm->image;
    for (i = 0; i < npixels; i++) {
      *dst = scale[*dst];
      dst++;
    }
    free(scale);
  }

  return(pbmSUCCESS);
}

/*------------------------------------------------------------------------
 * reads pixmap (binary three bytes per pixel, rgb) image data
 */
static int readppm(pbm_t *pbm)
{
  int i, npixels;
  int shift = 0;
  int *scale = NULL;
  pbmBYTE *dst;

  /* drop least sig. bits of value if bigger than a byte */
  while (pbm->maxval > 255)
  {
    pbm->maxval >>= 1;  shift++;
  }

  if (pbm->maxval < 255)
  {
    /* expand the values into 0-255 range */
    if (!(scale = malloc(pbm->maxval*sizeof(int))))
      return(pbmFAILURE);
    for (i = 0; i < pbm->maxval; i++)
      scale[i] = (i * 255) / pbm->maxval;
  }

  npixels = pbm->width * pbm->height * 3;
  if (!(pbm->image = malloc(npixels)))
  {
    if (scale)
      free(scale);
    return(pbmFAILURE);
  }

  dst = pbm->image;
  if (pbm->type == DV_IMAGE_FILE_TYPE_ASCII)
  {
    int val;

    for (i = 0; i < npixels; i++)
    {
      if (!(getint(pbm->fp, &val)))  break;
      *dst++ = val >> shift;
    }
  }
  else
  {
    /* binary data */
    i = fread(dst, 1, npixels, pbm->fp);
  }

  if (i < npixels)
  {
    fprintf(stderr, "Warning: file appears to be truncated\n");
    memset(pbm->image+i, 0, npixels-i);
  }

  if (scale)
  {
    /* scale values into 0-255 range */
    dst = pbm->image;
    for (i = 0; i < npixels; i++)
      *dst++ = scale[*dst];
    free(scale);
  }

  return(pbmSUCCESS);
}
