/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/int_data.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/mat.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("interp_data", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define NPOINTS  1024
#define MAX_DATA 64

int FUNCinterp_data (OMobj_id in, OMobj_id probe, int ncomp, int *comps,
                     OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVinterp_data_update(OMobj_id elem_id)
{
	OMobj_id in, out, probe, e1, e2, c1, c2;
	int stat, *comps;
	xp_long ncomp;
	float xform[4][4];

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out_nd"), OM_OBJ_RD);

	probe = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);

	/* pli 5/5/1999 CFS 15324 - Add a check for identity 
		transformation matrix
	*/
	stat = FLDget_xform(probe, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (MATmat_is_identity(xform, 4)) {
		if (!OMis_null_obj(in) && !OMis_null_obj(probe)) {
			c1 = OMfind_subobj(in, OMstr_to_name("coordinates"), OM_OBJ_RD);
			c2 = OMfind_subobj(probe, OMstr_to_name("coordinates"), OM_OBJ_RD);
			if (OMis_null_obj(c1) || OMis_null_obj(c2))
				return(0);
			if ((stat=OMget_obj_val(c1, &e1)) != 1 ||
				(stat=OMget_obj_val(c2, &e2)) != 1 )
				return (stat);
			if (OMequal_objs(e1, e2)) {
				if (FLDcopy_subelem_ref(out, in) != 1) {
					ERR_RETURN("cannot in into out ");
				}
				return(1);
			}
		}
	}
	if (FLDget_array_int (elem_id, "comps", &comps, &ncomp, OM_GET_ARRAY_RD) != 1)
		return(0);
	if (ncomp == 0) {
		xp_long nprobes, old_nnodes;
		if (FLDget_nnodes(out, &old_nnodes) != 1) {
			/* If the output is unset, its probably best just
			 * to leave it untouched.
			 */
		return(0);
		}
		/* The output mesh is just a merged version of the input
		 * probe.  Thus its possible to generate an inconsistent output
		 * field if we just return here without updating the output
		 * node data to be consistent with the mesh.  CFS PR 30461.
		 */
		if (FLDget_nnodes(probe, &nprobes) == 1) {
			if (FLDset_nnodes(out, nprobes) != 1) {
				ERR_RETURN("cannot set nnodes");
			}
			/* These lines here are a judgement call.  If the user
			 * is playing around with the UI component checkboxes,
			 * you don't want to see error messages from downstream
			 * modules when, for a moment, no boxes are checked.
			 * However, if the number of nodes change, the output
			 * node data could be seriously bogus, so remove it.
			 */
			if (nprobes != old_nnodes) {
				if (FLDset_node_data_ncomp (out, 0) != 1) {
					ERR_RETURN("Error setting nnode_data");
				}
			}
			/* not an error to have zero ncomp */
			return(1);
		}
		/* Couldn't get in.nnodes .... unset out.nnodes ? */
		return(0);
	}

	stat = FUNCinterp_data(in, probe, (int)ncomp, comps, out);

	if (comps) ARRfree(comps);

	return(stat == 1);
}


/* 64-bit porting. Only Modified Internally */
int FUNCinterp_data (OMobj_id in, OMobj_id probe, int ncomp, int *comps,
                     OMobj_id out)
{
	int   nspace, in_ncomp, stat, i, j, data_id;
	xp_long size, nprobes;
	int   veclen[MAX_DATA];
	xp_long count, nleft, nreq, dims[2], min_rng[2], max_rng[2];
	float probe_xyz[3*NPOINTS], xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];
	char  *node_data[MAX_DATA], *out_node_data[MAX_DATA], *out_node_data_ptr[MAX_DATA];
	char  *block_table, *mesh_info;
	int    probe_nspace, data_type, dtype, in_null_flag[MAX_DATA], out_null_flag[MAX_DATA];
	double min[MAX_DATA], max[MAX_DATA], null_value[MAX_DATA];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];

	/** Free pre-allocated data **/

	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_node_data_ncomp(in, &in_ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDget_nnodes(probe, &nprobes) != 1) {
		ERR_RETURN("cannot get nnodes for probe");
	}
	if (FLDget_nspace(probe, &probe_nspace) != 1) {
		ERR_RETURN("cannot get nspace for probe");
	}
	for (i=0; i<ncomp; i++) {
		if (comps[i] >= in_ncomp) {
			if (FLDset_nnodes(out, nprobes) != 1) {
				ERR_RETURN("cannot set nnodes");
			}
			if (FLDset_node_data_ncomp (out, 0) != 1) {
				ERR_RETURN("Error setting nnode_data");
			}
			ERR_RETURN("requested component does not exist, interpolation is not performed");
		}
		if (FLDget_node_data_veclen(in, comps[i], &(veclen[i])) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDget_node_data(in, comps[i], &dtype, &(node_data[i]), &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (i > 1) {
			if (data_type != dtype) {
				ERR_RETURN("components must be of the same type");
			}
		}
		else
			data_type = dtype;
		if (FLDget_node_null_data(in, comps[i], &(in_null_flag[i]), 
					  (char *)&(null_value[i])) != 1) {
			ERR_RETURN("cannot get null data");
		}
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}

	stat = FLDinterp_init(in, mesh_info, &block_table);
	if (stat != 1) {
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no block table created");
	}

	stat = FLDget_xform(probe, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0) {
		xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)xform, 4))
		xfm = (float *)0;
	else
		xfm = (float *)xform;
	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else
		in_xfm = (float *)in_xform;
	if (in_xfm) {
		MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
		if (xfm)
			MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
		else
			xfm = (float *)t_xform;
	}

	/**** OUT DATA ***/
	if (FLDset_nnodes(out, nprobes) != 1) {
		ERR_RETURN("cannot set nnodes");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (i=0; i<ncomp; i++) {
		if (FLDget_node_data_units(in, comps[i], units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comps[i], label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDset_node_data_comp (out, i, veclen[i], label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comps[i], &data_id) == 1)
			FLDset_node_data_id(out, i, data_id);

		if (FLDcopy_node_minmax(in, out, comps[i], i) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comps[i], i) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDget_node_data_minmax(in, comps[i], (char *)&(min[i]), 
					    (char *)&(max[i])) != 1) {
			ERR_RETURN("Error getting node minmax");
		}

		if (FLDget_node_data(out, i, &dtype, (char **)&(out_node_data[i]), 
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
	}
	stat = 1;
	count = 0;
	dims[0] = probe_nspace;
	dims[1] = nprobes;
	min_rng[0] = 0;
	max_rng[0] = probe_nspace;
	for (i=0; i<ncomp; i++) {
		out_node_data_ptr[i] = out_node_data[i];
		out_null_flag[i] = 0;
	}
	/* Otherwise, you might get the "building block table" message. */
	OMstatus_check( 0.0, "interp_data", NULL );	
	while (stat) {
		nleft = nprobes - count;
		if (nleft > 0) {
			if (nleft > NPOINTS)
				nreq = NPOINTS;
			else {
				nreq = nleft;
				stat = 0;
			}
		}
		else
			break;

		min_rng[1] = count;
		max_rng[1] = count + nreq;
		if (FLDget_sub_coord(probe, 2, dims, min_rng, max_rng, probe_xyz) != 1) {
			ERR_RETURN("cannot get coordinates for probe");
		}

		OMpush_status_range((int)(count*100/nprobes), (int)((count+nreq)*100/nprobes));

		stat = UTILinterpolate_data(nreq, probe_xyz, probe_nspace, xfm,
					    mesh_info, block_table,
					    ncomp, dtype, veclen, node_data,
					    min, max, in_null_flag, null_value,
					    out_node_data_ptr);
		OMpop_status_range();

		if (stat != 1) {
			for (j=0; j<i; j++) {
				ARRfree(node_data[j]);
				if (out_node_data[j] != NULL)
				   ARRfree(out_node_data[j]);
			}
			if (FLDset_node_data_ncomp (out, 0) != 1) {
				ERR_RETURN("Error setting nnode_data");
			}
			FLDfree_mesh_info (mesh_info);
			FLDinterp_end(block_table);
			ERR_RETURN("interpolation is not performed");
		}

		count += nreq;
		for (i=0; i<ncomp; i++) {
			out_node_data_ptr[i] = out_node_data[i]+
				veclen[i]*count*DTYPEtype_size[dtype];
			if (in_null_flag[i] == 1)
				out_null_flag[i] = 1;
		}
	}
	for (i=0; i<ncomp; i++) {
		if (out_null_flag[i]) {
			if (FLDset_node_null_data(out, i, (char *)&null_value[i], dtype) != 1) {
				ERR_RETURN("cannot set null data");
			}
		}
		else {
			if (FLDset_node_null_flag(out, i, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data[i]);
		/*
		 * If there are 0 values, get_array is returning NULL.
		 */
		if (out_node_data[i] != NULL)
		   ARRfree(out_node_data[i]);
	}
	FLDfree_mesh_info (mesh_info);
	FLDinterp_end(block_table);
	return(1);
}

