/*
			Copyright (c) 2004 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/interp_to_unif.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/mat.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("interp_data_to_unif", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#define MAX_NAME_SIZE 1024
#define NPOINTS  2048
#define MAX_DATA 64

int FUNCinterp_data_to_unif (OMobj_id in, OMobj_id probe,
        int ncomp, int *comps, float *null_values, int algorithm,
        OMobj_id out);

int FUNCinterp_cell_data_to_unif (OMobj_id in, OMobj_id probe,
        int ncomp, int comp, float null_value, int algorithm,
        OMobj_id out);

/*  mod_util/finterp_unif.m */
int UTILinterpolate_data_to_unif( int nprobes, float *probe_xyz,
        int probe_nspace, float *xfm, void *field_table, void *block_table,
        int ncomp, int dtype, int *veclen, char **node_data,
        int *null_flag, double *null_value, int first_only,
        char **out_node_data );

int UTILinterpolate_cell_data_to_unif(int nprobes, float *probe_xyz,
        int probe_nspace, float *xfm, void *field_table, void *block_table,
        int *sets, int dtype, int veclen, char **cell_data,
        int *null_flag, char *null_value, int first_only,
        char *out_node_data);

/* mod_util/blk_tbl_unif.c */
int UTILinterp_init2(char *, float *, int, xp_long *,
                     int, float *, float *, int, char **);
int UTILinterp_init_struct2(char *, float *, int, xp_long *,
                            int, float *, float *, int, char **);
int UTILinterp_end2 (void *block_tbl);

int DVinterp_data_to_unif_update(OMobj_id elem_id, OMevent_mask mask, int seq)
{
    OMobj_id in, out, probe;
    int stat, *comps, algo;
    xp_long ncomp, nnull;
    float *null_values;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out_nd"), OM_OBJ_RD);

    probe = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);

    if (FLDget_array_int (elem_id, "comps", &comps, &ncomp, OM_GET_ARRAY_RD) != 1) {
        ERR_RETURN("Cannot get components");
    }

    if (FLDget_array_float (elem_id, "null_values", &null_values, &nnull,
                            OM_GET_ARRAY_RD) != 1) {
        null_values = NULL;
    }
    else {
        if( nnull != 0 && nnull != ncomp ) {
            ERR_RETURN("Number of nulls must match number of components");
        }
    }

    if (OMget_name_int_val(elem_id, OMstr_to_name("algorithm"), &algo)
        != OM_STAT_SUCCESS)
        algo = 0;

    if (ncomp == 0) {
        xp_long nprobes, old_nnodes;
        if (FLDget_nnodes(out, &old_nnodes) != 1) {
            /* If the output is unset, its probably best just
             * to leave it untouched.
             */
            return(METHOD_FAILURE);
        }
        /* The output mesh is just a merged version of the input
         * probe.  Thus its possible to generate an inconsistent output
         * field if we just return here without updating the output
         * node data to be consistent with the mesh.
         */
        if (FLDget_nnodes(probe, &nprobes) == 1) {
            if (FLDset_nnodes(out, nprobes) != 1) {
                ERR_RETURN("cannot set nnodes");
            }
            /* These lines here are a judgement call.  If the user
             * is playing around with the UI component checkboxes,
             * you don't want to see error messages from downstream
             * modules when, for a moment, no boxes are checked.
             * However, if the number of nodes change, the output
             * node data could be seriously bogus, so remove it.
             */
            if (nprobes != old_nnodes) {
                if (FLDset_node_data_ncomp (out, 0) != 1) {
                    ERR_RETURN("Error setting nnode_data");
                }
            }
            /* not an error to have zero ncomp */
            return(METHOD_SUCCESS);
        }
        /* Couldn't get in.nnodes .... unset out.nnodes ? */
        return(METHOD_FAILURE);
    }

    stat = FUNCinterp_data_to_unif(in, probe, ncomp, comps, null_values, algo, out);

    if (comps) ARRfree(comps);
    if (null_values) ARRfree(null_values);

    return(stat == 1);
}


/*
 * algorithm -
 *  0 - for each probe point, grab first cell that passes a simple
 *      bounding box test.  Then loop over the nodes of the cell and
 *      do a distance test to determine which node value to use.
 *
 *  1 - For each probe point, build a list of cells that pass a
 *      simple bounding box test.  Then, use a more complex algoritm
 *      to determine if the probe point is inside a given cell.
 *      results should be very similar to the standard interp_data module.
 */
int FUNCinterp_data_to_unif (OMobj_id in, OMobj_id probe,
        int ncomp, int *comps, float *null_value_param, int algorithm,
        OMobj_id out)
{
    int   nspace, in_ncomp, stat, i, j, data_id;
    int   veclen[MAX_DATA], count, nleft, nreq;
    xp_long  size, dims[2], min_rng[2], max_rng[2];
    float probe_xyz[3*NPOINTS], probe_min[3], probe_max[3];
    float xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];
    float xform_rev[4][4], *xfm_rev = NULL;
    char  *node_data[MAX_DATA], *out_node_data[MAX_DATA], *out_node_data_ptr[MAX_DATA];
    char  *block_table, *mesh_info;
    int   data_type, dtype, in_null_flag[MAX_DATA], out_null_flag[MAX_DATA];
    double null_value[MAX_DATA];
    char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
    xp_long  nprobes, *probe_dims;
    int   probe_nspace, probe_ndim, first_only;

    /** Free pre-allocated data **/

    if (FLDset_node_data_ncomp (out, 0) != 1) {
        ERR_RETURN("Error setting nnode_data");
    }

    if (FLDget_nspace(in, &nspace) != 1) {
        ERR_RETURN("cannot get nspace");
    }
    if (FLDget_node_data_ncomp(in, &in_ncomp) != 1) {
        ERR_RETURN("Error getting ncomp");
    }
    if (FLDget_nnodes(probe, &nprobes) != 1) {
        ERR_RETURN("cannot get nnodes for probe");
    }
    if (FLDget_nspace(probe, &probe_nspace) != 1) {
        ERR_RETURN("cannot get nspace for probe");
    }
    /* Iterate over components of input field. */
    for (i=0; i<ncomp; i++) {
        if (comps[i] >= in_ncomp) {
            if (FLDset_nnodes(out, nprobes) != 1) {
                ERR_RETURN("cannot set nnodes");
            }
            if (FLDset_node_data_ncomp (out, 0) != 1) {
                ERR_RETURN("Error setting nnode_data");
            }
            ERR_RETURN("requested component does not exist, interpolation is not performed");
        }
        if (FLDget_node_data_veclen(in, comps[i], &(veclen[i])) != 1) {
            ERR_RETURN("Error getting veclen");
        }
        node_data[i] = NULL;
        if (FLDget_node_data(in, comps[i], &dtype, &(node_data[i]), &size, OM_GET_ARRAY_RD) != 1) {
            ERR_RETURN("cannot get node data");
        }
        if (i > 1) {
            if (data_type != dtype) {
                ERR_RETURN("components must be of the same type");
            }
        }
        else
            data_type = dtype;
        if (FLDget_node_null_data(in, comps[i], &(in_null_flag[i]),
                                  (char *)&(null_value[i])) != 1) {
            ERR_RETURN("cannot get null data");
        }
        if (in_null_flag[i] == 0) {
            /* Conjure up some null_values if we don't already have them
             * for this component.
             */
            if (null_value_param != NULL) {
                /* Supplied as a module parameter */
                UTILfloat_to_type((char *)&(null_value[i]),	/* dest */
                                  null_value_param[i], 		/* src */
                                  dtype);
            }
            else {
                double min, max;	/* holds any prim */
                if (FLDget_node_data_minmax(in, comps[i], (char *)&min,
                                            (char *)&max) != 1) {
                    ERR_RETURN("Error getting node minmax");
                }

                /* With an eye towards volume rendering, use 0 for the
                 * null value for byte/short data if it has not been supplied
                 * by some other method. Otherwise, use the max+1 technique
                 * used by the regular interp_data.
                 */
                switch (dtype) {
                case DTYPE_BYTE:
                case DTYPE_CHAR:
                  *((char *)&(null_value[i]))  = 0;
                  break;
                case DTYPE_SHORT:
                  *((short *)&(null_value[i])) = 0;
                  break;
                case DTYPE_INT:
                  /* Also use the veclen test? */
                  *((int *)&(null_value[i]))    = *((int *)&max) + 1;
                  break;
                case DTYPE_FLOAT:
                  *((float *)&(null_value[i]))  = *((float *)&max) + 1;
                  break;
                case DTYPE_DOUBLE:
                  *((double *)&(null_value[i])) = max + 1;
                }
            }
        }
    }
    stat = FLDget_mesh_info(in, &mesh_info);
    if (!stat) {
        ERR_RETURN("cannot create cell table");
    }

    if (FLDget_coord_extent(probe, probe_min, probe_max) != 1) {
        ERR_RETURN("cannot get probe extents");
    }
    if (probe_nspace < 3) { probe_min[2] = probe_max[2] = 0.0; }
    if (probe_nspace < 2) { probe_min[1] = probe_max[1] = 0.0; }

    stat = FLDget_xform(probe, (float *)xform);
    if (stat < 0) {
        ERR_RETURN("cannot get xform for probe");
    }
    else if (stat == 0) {
        xfm = (float *)0;
    }
    else if (MATmat_is_identity((float *)xform, 4))
        xfm = (float *)0;
    else
        xfm = (float *)xform;
    stat = FLDget_xform(in, (float *)in_xform);
    if (stat < 0) {
        ERR_RETURN("cannot get xform for field");
    }
    else if (stat == 0) {
        in_xfm = (float *)0;
    }
    else if (MATmat_is_identity((float *)in_xform, 4))
        in_xfm = (float *)0;
    else
        in_xfm = (float *)in_xform;
    if (in_xfm) {
        /* Invert in_field xform */
        MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
        if (xfm) {
            /* probe xfrom divided by in_field xform
             *
             * This allows us to map a probe point back into
             * the "coordinate space" of the in_field.
             */
            MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
        }
        else {
            xfm = (float *)t_xform;
        }
    }
    /* If the fields have the same transform, even if its a non-identity
     * transform, you don't need to map the coordinates back and forth.
     */
    {
        int same = 1;
        for( i = 0; i < 16; ++i ) {
            if( ((float *)in_xform)[i] != ((float *)xform)[i] ) {
               same = 0;
               break;
            }
        }
        if( same == 1 ) xfm = NULL;
    }
    if (xfm) {
       /* This allows us to map a xyz point from in_field into
        * the "coordinate space" of the probe field.
        */
        MATmat4_inverse(xform_rev, (Matr4 *)xfm);
        xfm_rev = (float *)xform_rev;
    }
    else xfm_rev = NULL;

    if( algorithm == 0 ) first_only = 1;
    else first_only = 0;

    /* New style of block table that has dimensions that
     * exactly match the uniform probe field.
     *
     * To support different sized 'in' and probe fields (e.g. the probe
     * field is smaller, essentially doing a crop), the min/max coordinates
     * should be from the probe field, not the 'in' field.
     */
    stat = FLDget_dims(probe, &probe_dims, &probe_ndim);
    if (stat == OM_STAT_SUCCESS) {
        int test_ndim;
        stat = FLDget_ndim( in, &test_ndim );
        if( stat != OM_STAT_SUCCESS || test_ndim == 0 ) {
            /* in field is unstructured */
            stat = UTILinterp_init2(mesh_info, xfm_rev,
                                    probe_ndim, probe_dims,
                                    probe_nspace, probe_min, probe_max,
                                    first_only, &block_table);
        }
        else {
            /* in field is structured */
            stat = UTILinterp_init_struct2(mesh_info, xfm_rev,
                                    probe_ndim, probe_dims,
                                    probe_nspace, probe_min, probe_max,
                                    first_only, &block_table);
        }
        ARRfree( probe_dims );

    }
    if (stat != OM_STAT_SUCCESS) {
        FLDfree_mesh_info(mesh_info);
        ERR_RETURN("no block table created");
    }

    /**** OUT DATA ***/
    /*
     * Copy over node data information from 'in' to 'out' and then
     * grab pointers to the output node data arrays.
     */

    /* 'out' is Node_Data without a mesh, so you have to
     * explicitly tell 'out' how many nodes ... the node data
     * arrays won't be valid unless you do this.
     */
    if (FLDset_nnodes(out, nprobes) != 1) {
        ERR_RETURN("cannot set nnodes");
    }
    if (FLDset_node_data_ncomp (out, ncomp) != 1) {
        ERR_RETURN("Error setting nnode_data");
    }
    for (i=0; i<ncomp; i++) {
        if (FLDget_node_data_units(in, comps[i], units, MAX_NAME_SIZE) != 1) {
            strcpy(units, "");
        }
        if (FLDget_node_data_label(in, comps[i], label, MAX_NAME_SIZE) != 1) {
            strcpy(label, "");
        }
        if (FLDset_node_data_comp (out, i, veclen[i], label, units) != 1) {
            ERR_RETURN("Error setting node component");
        }
        if (FLDget_node_data_id(in, comps[i], &data_id) == 1)
            FLDset_node_data_id(out, i, data_id);

        if (FLDcopy_node_minmax(in, out, comps[i], i) != 1) {
            ERR_RETURN("Error copying node minmax");
        }
        if (FLDcopy_node_minmax_vec(in, out, comps[i], i) != 1) {
            ERR_RETURN("Error copying node minmax");
        }

        /* Output node data */
        out_node_data[i] = NULL;
        if (FLDget_node_data(out, i, &dtype, (char **)&(out_node_data[i]),
                             &size, OM_GET_ARRAY_WR) != 1) {
            ERR_RETURN("cannot get out node data");
        }
    }

    stat = 1;
    count = 0;
    dims[0] = probe_nspace;
    dims[1] = nprobes;
    min_rng[0] = 0;
    max_rng[0] = probe_nspace;
    for (i=0; i<ncomp; i++) {
        out_node_data_ptr[i] = out_node_data[i];
        out_null_flag[i] = 0;
    }
    /* Otherwise, you might get the "building block table" message. */
    OMstatus_check( 0, "interp_data_to_unif", NULL );

    while (stat) {
        nleft = nprobes - count;
        if (nleft > 0) {
            if (nleft > NPOINTS)
                nreq = NPOINTS;
            else {
                nreq = nleft;
                stat = 0;
            }
        }
        else
            break;

        /* Request next chunk of coords */
        min_rng[1] = count;
        max_rng[1] = count + nreq;
        /* Actually, if we can assume that the block table has been set up
         * to exactly match the probe field, we should not have to bother
         * with coordinates at all - just use indicies all the way through.
         */
        if (FLDget_sub_coord(probe, 2, dims, min_rng, max_rng, probe_xyz)
            != OM_STAT_SUCCESS) {
            ERR_RETURN("cannot get coordinates for probe");
        }

        OMpush_status_range(count*100/nprobes, (count+nreq)*100/nprobes);

        stat = UTILinterpolate_data_to_unif( nreq, probe_xyz, probe_nspace,
            xfm, mesh_info, block_table,
            ncomp, dtype, veclen, node_data, in_null_flag, null_value,
            first_only, out_node_data_ptr );

        OMpop_status_range();

        if (stat != 1) {
            for (j=0; j<i; j++) {
                ARRfree(node_data[j]);
                if (out_node_data[j] != NULL)
                    ARRfree(out_node_data[j]);
            }
            if (FLDset_node_data_ncomp (out, 0) != 1) {
                ERR_RETURN("Error setting nnode_data");
            }
            FLDfree_mesh_info (mesh_info);
            UTILinterp_end2(block_table);
            ERR_RETURN("interpolation is not performed");
        }

        count += nreq;
        for (i=0; i<ncomp; i++) {
            out_node_data_ptr[i] = out_node_data[i] +
                veclen[i]*count*DTYPEtype_size[dtype];
            if (in_null_flag[i] == 1)
                out_null_flag[i] = 1;
        }
    }
    for (i=0; i<ncomp; i++) {
        if (out_null_flag[i]) {
            if (FLDset_node_null_data(out, i, (char *)&null_value[i], dtype) != 1) {
                ERR_RETURN("cannot set null data");
            }
        }
        else {
            if (FLDset_node_null_flag(out, i, 0) != 1) {
                ERR_RETURN("Error setting null flag");
            }
        }
        ARRfree(node_data[i]);
        /*
         * If there are 0 values, get_array is returning NULL.
         */
        if (out_node_data[i] != NULL)
            ARRfree(out_node_data[i]);
    }
    FLDfree_mesh_info (mesh_info);
    UTILinterp_end2(block_table);
    return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) {ERRerror("interp_cell_data_to_unif", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

/* Basically a copy of DVinterp_cell_data_to_unif_update */
int DVinterp_cell_data_to_unif_update(OMobj_id elem_id, OMevent_mask mask, int seq)
{
    OMobj_id in, out, probe;
    int stat, ncomp, comp, algo;
    double null_value;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out_nd"), OM_OBJ_RD);

    probe = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);

    if (OMget_name_int_val (elem_id, OMstr_to_name("component"), &comp)
        != OM_STAT_SUCCESS) {
        ERR_RETURN("Cannot get component");
    }
    ncomp = 1;

    if (OMget_name_real_val (elem_id, OMstr_to_name("null_value"),
        &null_value) != OM_STAT_SUCCESS) {
        /* Hmmm, need a unset flag ? */
        null_value = 0.0;
    }

    if (OMget_name_int_val(elem_id, OMstr_to_name("algorithm"), &algo)
        != OM_STAT_SUCCESS)
        algo = 0;

    stat = FUNCinterp_cell_data_to_unif(in, probe, ncomp, comp, null_value,
                                        algo, out);

    return(stat == 1);
}


int FUNCinterp_cell_data_to_unif (OMobj_id in, OMobj_id probe,
        int ncomp, int comp, float _null_value, int algorithm,
        OMobj_id out)
{
    int   stat, cs, *sets, cell_ncomp, nspace, first;
    int   dtype, in_dtype, null_flag, in_null_flag, out_null_flag;
    int   nsets, veclen, in_veclen;
    xp_long   size, nnodes, nprobes;
    int   count, nleft, nreq;
    xp_long   dims[2], min_rng[2], max_rng[2];
    char  label[MAX_NAME_SIZE], units[MAX_NAME_SIZE];
    char  **cell_data, *node_data, *out_node_data;
    double null_value, dnull_value, rnull_value, min, max;
    float probe_xyz[3*NPOINTS], probe_min[3], probe_max[3];
    float xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];
    float xform_rev[4][4], *xfm_rev = NULL;
    OMobj_id cell_set;
    char  *block_table, *mesh_info;
    xp_long   *probe_dims;
    int   probe_nspace, probe_ndim, first_only;

    if (FLDget_nnodes(in, &nnodes) != 1) {
        ERR_RETURN("cannot get nnodes");
    }
    if (nnodes == 0) {
        /* Clean output and return */
        if (FLDset_nnodes (out, 0) != 1) {
            ERR_RETURN("Error setting nnodes");
        }
        if (FLDset_ncell_sets(out, 0) != 1) {
            ERR_RETURN("cannot set nsets");
        }
        if (FLDset_node_data_ncomp (out, 0) != 1) {
            ERR_RETURN("Error setting nnode_data");
        }
        return(1);
    }
    if (FLDget_nspace(in, &nspace) != 1) {
        ERR_RETURN("cannot get nspace");
    }
    if (FLDget_ncell_sets(in, &nsets) != 1) {
        return(0);
    }
    stat = FLDget_mesh_info(in, &mesh_info);
    if (!stat) {
        ERR_RETURN("cannot create field info");
    }

    if (FLDget_nnodes(probe, &nprobes) != 1) {
        ERR_RETURN("cannot get nnodes for probe");
    }
    if (FLDget_nspace(probe, &probe_nspace) != 1) {
        ERR_RETURN("cannot get nspace for probe");
    }

    if (FLDget_coord_extent(probe, probe_min, probe_max) != 1) {
        ERR_RETURN("cannot get probe extents");
    }
    if (probe_nspace < 3) { probe_min[2] = probe_max[2] = 0.0; }
    if (probe_nspace < 2) { probe_min[1] = probe_max[1] = 0.0; }

    stat = FLDget_xform(probe, (float *)xform);
    if (stat < 0) {
        ERR_RETURN("cannot get xform for probe");
    }
    else if (stat == 0) {
        xfm = (float *)0;
    }
    else if (MATmat_is_identity((float *)xform, 4))
        xfm = (float *)0;
    else
        xfm = (float *)xform;
    stat = FLDget_xform(in, (float *)in_xform);
    if (stat < 0) {
        ERR_RETURN("cannot get xform for field");
    }
    else if (stat == 0) {
        in_xfm = (float *)0;
    }
    else if (MATmat_is_identity((float *)in_xform, 4))
        in_xfm = (float *)0;
    else
        in_xfm = (float *)in_xform;
    if (in_xfm) {
        MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
        if (xfm)
            MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
        else
            xfm = (float *)t_xform;
    }
    {
        int i, same = 1;
        for( i = 0; i < 16; ++i ) {
            if( ((float *)in_xform)[i] != ((float *)xform)[i] ) {
               same = 0;
               break;
            }
        }
        if( same == 1 ) xfm = NULL;
    }
    if (xfm) {
       /* This allows us to map a xyz point from in_field into
        * the "coordinate space" of the probe field.
        */
        MATmat4_inverse(xform_rev, (Matr4 *)xfm);
        xfm_rev = (float *)xform_rev;
    }
    else xfm_rev = NULL;

    if( algorithm == 0 ) first_only = 1;
    else first_only = 0;

    /* New style of block table that has dimensions that
     * exactly match the uniform probe field.
     */
    stat = FLDget_dims(probe, &probe_dims, &probe_ndim);
    if (stat == OM_STAT_SUCCESS) {
          /* With cell_data we are not worried much about
           * the in_field being structured.
           */
          stat = UTILinterp_init2(mesh_info, xfm_rev,
                                  probe_ndim, probe_dims,
                                  probe_nspace, probe_min, probe_max,
                                  first_only, &block_table);
          ARRfree( probe_dims );
    }
    if (stat != OM_STAT_SUCCESS) {
        FLDfree_mesh_info(mesh_info);
        ERR_RETURN("no block table created");
    }

    if (FLDset_nnodes(out, nprobes) != 1) {
        ERR_RETURN("cannot set nnodes");
    }
    if (FLDset_node_data_ncomp (out, 1) != 1) {
        ERR_RETURN("Error setting nnode_data");
    }

    if (FLDget_cell_data_minmax_total(in, comp,
                                      (char *)&min, (char *)&max) != 1) {
        ERR_RETURN("Error getting cell minmax");
    }

    sets = (int *)malloc(nsets*sizeof(int));
    cell_data = (char **)malloc(nsets*sizeof(char *));

    for (first=1,cs=0; cs<nsets; cs++) {
        if (FLDget_cell_set(in, cs, &cell_set) != 1) {
            ERR_RETURN("cannot get cell set");
        }
        if (FLDget_cell_data_ncomp(cell_set, &cell_ncomp) != 1) {
            ERR_RETURN("Error getting ncomp");
        }
        if (comp >= cell_ncomp) {
            ERRerror("interp_cell_data", 2, ERR_ORIG, "cell set %d has no data component %d, skipping", cs, comp);
            sets[cs] = -1;
            continue;
        }
        else
            sets[cs] = cs;

        if (FLDget_cell_data(cell_set, comp, &in_dtype, &cell_data[cs],
                             &size, OM_GET_ARRAY_RD) != 1) {
            ERR_RETURN("Error setting cell data");
        }

        if (FLDget_cell_null_data(cell_set, comp, &in_null_flag, (char *)&null_value) != 1) {
            ERR_RETURN("cannot get null data");
        }
        if (FLDget_cell_data_veclen(cell_set, comp, &in_veclen) != 1) {
            ERR_RETURN("Error getting cell veclen");
        }
        if (first) {
            first = 0;
            null_flag = in_null_flag;
            veclen = in_veclen;
            dtype = in_dtype;
            if (null_flag)
                UTILtype_to_double(&dnull_value, (char *)&null_value, dtype);
            stat = FLDget_cell_data_label(cell_set, comp, label, MAX_NAME_SIZE);
            if (stat < 0)
                return(stat);
            else if (stat == 0)
                strcpy(label, "Interp Cell Data");
            stat = FLDget_cell_data_units(cell_set, comp, units, MAX_NAME_SIZE);
            if (stat < 0)
                return(stat);
            else if (stat == 0)
                strcpy(units, "");
            if (FLDset_node_data_comp (out, 0, veclen, label, units) != 1) {
                ERR_RETURN("Error setting node data component");
            }
            if (FLDget_node_data(out, 0, &dtype, &node_data, &size, OM_GET_ARRAY_WR) != 1) {
                ERR_RETURN("Error getting node data for writing");
            }
        }
        else {
            if (in_null_flag) {
                null_flag = 1;
                UTILtype_to_double(&rnull_value, (char *)&null_value, dtype);
                if (rnull_value != dnull_value) {
                    ERRerror("interp_cell_data", 2, ERR_ORIG, "cell set %d has different null data for component %d, skipping", cs, comp );
                    sets[cs] = -1;
                    continue;
                }
            }
            if (in_veclen != veclen) {
                ERRerror( "interp_cell_data", 2, ERR_ORIG, "cell set %d has different veclen for component %d, skipping", cs, comp );
                sets[cs] = -1;
                continue;
            }
            if (in_dtype != dtype) {
                ERRerror( "interp_cell_data", 2, ERR_ORIG, "cell set %d has different data type for component %d, skipping", cs, comp );
                sets[cs] = -1;
                continue;
            }
        }
    }

    stat = 1;
    count = 0;
    dims[0] = probe_nspace;
    dims[1] = nprobes;
    min_rng[0] = 0;
    max_rng[0] = probe_nspace;
    out_null_flag = 0;
    out_node_data = node_data;

    while (stat) {
        nleft = nprobes - count;
        if (nleft > 0) {
            if (nleft > NPOINTS)
                nreq = NPOINTS;
            else {
                nreq = nleft;
                stat = 0;
            }
        }
        else
            break;

        min_rng[1] = count;
        max_rng[1] = count + nreq;
        if (FLDget_sub_coord(probe, 2, dims, min_rng, max_rng, probe_xyz) != 1) {
            ERR_RETURN("cannot get coordinates for probe");
        }
        if (UTILinterpolate_cell_data_to_unif(nreq, probe_xyz, probe_nspace, xfm,
                                      mesh_info, block_table,
                                      sets, dtype, veclen, cell_data,
                                      &null_flag, (char *)&null_value,
                                      first_only, out_node_data) != 1) {
            ARRfree(node_data);
            if (FLDset_node_data_ncomp (out, 0) != 1) {
                ERR_RETURN("Error setting nnode_data");
            }
            FLDfree_mesh_info (mesh_info);
            UTILinterp_end2 (block_table);
            ERR_RETURN("interpolation is not performed");
        }

        count += nreq;
        out_node_data = node_data+veclen*count*DTYPEtype_size[dtype];
        if (null_flag == 1)
            out_null_flag = 1;
    }

    ARRfree(node_data);

    if (out_null_flag) {
        if (FLDset_node_null_data(out, 0, (char *)&null_value, dtype) != 1) {
            ERR_RETURN("Error setting null value");
        }
    }
    else {
        if (FLDset_node_null_flag(out, 0, 0) != 1) {
            ERR_RETURN("Error setting null flag");
        }
    }

    if (FLDset_node_data_minmax(out, 0, (char *)&min, (char *)&max, dtype) != 1) {
        ERR_RETURN("Error setting node minmax");
    }
    FLDfree_mesh_info(mesh_info);
    UTILinterp_end2(block_table);
    free(sets);
    for (cs=0; cs<nsets; cs++) ARRfree(cell_data[cs]);
    free(cell_data);

    return 1;
}
