/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/isotrace.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/mat.h>

#define ERR_RETURN(A) ERRerror("isotrace", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024

int FUNCisotrace (OMobj_id in, OMobj_id probe, double *level,
                  OMobj_id out, OMobj_id param);

int DVisotrace_update(OMobj_id elem_id)
{
	OMobj_id in, xyz_id, out, level_id, param;
	double  level;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	xyz_id = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	param = OMfind_subobj(elem_id, OMstr_to_name("nparam"), OM_OBJ_RW);
	level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);

	if (FUNCisotrace(in, xyz_id, &level, out, param)) {
		OMset_real_val(level_id, level);
		return(1);
	}
	else return(0);
}

/* 64-bit porting. Only Modified Internally */
int FUNCisotrace (OMobj_id in, OMobj_id probe, double *level,
                  OMobj_id out, OMobj_id param)
{
	xp_long nnodes, out_ncells, out_nnodes, size, conn_size;
	int   probe_nspace, data_type, veclen, ncomp;
	xp_long *out_nlist, *min_node_list, *max_node_list;
	int   null_flag, stat;
	char  units[MAX_NAME_SIZE];
	float  *out_coord, *t, point[3];
	char   *node_data;
	char   *mesh_info, *block_table;
	double null_value;
	OMobj_id   cell_set;
	float      *xyz, xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_int (param, "nnodes", 0) != 1) {
		ERR_RETURN("Error setting out nnodes");
	}
	min_node_list = max_node_list = NULL;
	if (FLDset_array_long (param, "min_node", 0, min_node_list,
			      OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting min_node_list");
	}
	if (FLDset_array_long (param, "max_node", 0, max_node_list,
			      OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting max_node_list");
	}

	/**  Check the input **/

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (nnodes == 0 || ncomp == 0) {
		return(1);
	}

	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}
	stat = FLDinterp_init(in, mesh_info, &block_table);
	if (stat != 1) {
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no block table created");
	}

	out_nnodes = 0;
	if (FLDget_nspace(probe, &probe_nspace) != 1) {
		ERR_RETURN("cannot get nspace for probe");
	}
	stat = FLDget_xform(probe, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0) {
		xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)xform, 4))
		xfm = (float *)0;
	else
		xfm = (float *)xform;
	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else
		in_xfm = (float *)in_xform;
	if (in_xfm) {
		MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
		if (xfm)
			MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
		else
			xfm = (float *)t_xform;
	}

	if (FLDget_node_data_veclen(in, 0, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}

	if (veclen != 1) {
		if (FLDset_nnodes(out, 0) != 1) {
			ERR_RETURN("cannot set nnodes");
		}
		if (FLDset_ncell_sets(out, 0) != 1) {
			ERR_RETURN("cannot set nsets");
		}
		ERR_RETURN("component is not scalar, isotrace is not performed");
	}

	if (FLDget_node_data(in, 0, &data_type, &node_data,
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, 0, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}

	if (FLDget_coord(probe, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting coordinate array");
	}
	point[0]=point[1]=point[2]=0.0;
	memcpy(point, xyz, probe_nspace*sizeof(float));
	ARRfree(xyz);
	if (xfm)
		MATvec3_mat4_multiply(point, (Matr4 *)xfm);

	stat = UTILiso_trace(mesh_info, block_table, node_data, data_type,
			     null_flag, (char *)&null_value, point, level,
			     &out_nnodes, &min_node_list, &max_node_list,
			     &out_ncells, &out_nlist, &conn_size);

	if (stat != 1) {
		if (FLDset_ncell_sets (out, 0) != 1) {
			ERR_RETURN("Error setting ncell_sets");
		}
		if (node_data)
			ARRfree(node_data);
		FLDinterp_end(block_table);
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("No isosurface created");
	}


	/*** OUTPUT FIELD ***/

	if (FLDset_nspace (out, 3) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (out_nnodes) {
		if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
			if (FLDset_coord_units (out, units) != 1) {
				ERR_RETURN("Error setting units");
			}
		}
		if (FLDset_ncell_sets (out, 1) != 1) {
			ERR_RETURN("Error setting ncell_sets");
		}
		if (FLDget_cell_set(out, 0, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_cell_set(cell_set, "Tri") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDset_ncells(cell_set, out_ncells) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}
		if (FLDset_node_connect(cell_set, out_nlist, conn_size,
					    OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}

		if (FLDset_long (param, "nnodes", out_nnodes) != 1) {
			ERR_RETURN("Error setting out nnodes");
		}
		if (FLDset_array_long (param, "min_node", out_nnodes,
				      min_node_list, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting min_node_list");
		}
		if (FLDset_array_long (param, "max_node", out_nnodes,
				      max_node_list, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting max_node_list");
		}
		if (FLDget_array_float (param, "tpar", &t, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting t array");
		}
		UTILget_param(out_nnodes, data_type, node_data,
			      *level, min_node_list, max_node_list, t);
		UTILeval_coord(mesh_info, out_nnodes,
			       min_node_list, max_node_list, t, out_coord);

		ARRfree(t);
		ARRfree(out_coord);
	}
	else {
		if (FLDset_ncell_sets (out, 0) != 1) {
			ERR_RETURN("Error setting ncell_sets");
		}
		ARRfree(out_nlist);
		ARRfree(min_node_list);
		ARRfree(max_node_list);
	}
	ARRfree(node_data);
	FLDinterp_end(block_table);
	FLDfree_mesh_info(mesh_info);
	return(1);
}
