/*
			Copyright (c) 1997 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/minmax.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/arr.h>
#include <avs/fld.h>
#include <avs/data_utils.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("reset_extent", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

int DVreset_extent_update(OMobj_id mod_id)
{
	OMobj_id  in, out;
	int grid_type;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_grid"), OM_OBJ_RW);
		
	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	/*----------------------------------------------*/
	/*  Reset extent to local minmax of coordinates */
	/*  or points based on grid type                */
	/*----------------------------------------------*/
	
	if (FLDget_grid_type(in, &grid_type) != OM_STAT_SUCCESS)
		grid_type=0;
	if (grid_type != 0) {
		if (FLDreset_point_extent(out) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting coordinates");
		}
	}
	else {
		if (FLDreset_coord_extent(out) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting coordinates");
		}
	}
	return METHOD_SUCCESS;
}

#undef  ERR_RETURN
#define ERR_RETURN(A) {ERRerror("reset_minmax", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

/* 64-bit porting. Only Modified Internally */
int DVreset_minmax_update(OMobj_id mod_id)
{
	OMobj_id  in, out;
	int *comp;
	xp_long n, ncomp;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_nd"), OM_OBJ_RW);

	comp = (int *)OMret_name_array_ptr(mod_id, OMstr_to_name("comp"),
					   OM_GET_ARRAY_RD, &ncomp, NULL);
	if (comp == NULL)
		ncomp = 0;

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}
	
	/*----------------------------------------------*/
	/*  Reset minmax to local minmax of node data   */
	/*----------------------------------------------*/
	for (n=0; n<ncomp; n++) {
		if (FLDreset_node_minmax(out, comp[n]) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting node minmax");
		}
	}
	return METHOD_SUCCESS;
}

#undef  ERR_RETURN
#define ERR_RETURN(A) {ERRerror("reset_minmax_cell", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

/* 64-bit porting. Only Modified Internally */
int DVreset_minmax_cell_update(OMobj_id mod_id)
{
	OMobj_id  in, out, in_set_id, set_id;
	int *comp, nset, set;
	xp_long ncomp, n;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_cd"), OM_OBJ_RW);

	comp = (int *)OMret_name_array_ptr(mod_id, OMstr_to_name("comp"),
					   OM_GET_ARRAY_RD, &ncomp, NULL);
	if (comp == NULL)
		ncomp = 0;

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}
	
	/*----------------------------------------------*/
	/*  Reset minmax to local minmax of cell data   */
	/*----------------------------------------------*/
	if (FLDget_ncell_sets (in, &nset) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;
	for (set=0; set < nset; set++) {
		if (FLDget_cell_set (in, set, &in_set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting cell set");
		}
		if (FLDget_cell_set (out, set, &set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting cell set");
		}
                if (!OMequal_objs(in_set_id, set_id) &&
		    FLDcopy_subelem_ref(set_id, in_set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error setting set reference");
		}
		for (n=0; n<ncomp; n++) {
			if (FLDreset_cell_minmax(set_id, comp[n]) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error resetting cell minmax");
			}
		}
	}
	return METHOD_SUCCESS;
}

#undef  ERR_RETURN
#define ERR_RETURN(A) {ERRerror("set_minmax", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

int DVset_minmax_update(OMobj_id mod_id)
{
	OMobj_id  in, out;
	int comp, dtype;
	double min, max, rmin, rmax;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_nd"), OM_OBJ_RW);

	if (OMget_name_int_val(mod_id, OMstr_to_name("comp"), &comp) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;

	if (OMget_name_real_val(mod_id, OMstr_to_name("min"), &min) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;
	if (OMget_name_real_val(mod_id, OMstr_to_name("max"), &max) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}
	
	/*----------------------------------------------*/
	/*  Set minmax of node data                     */
	/*----------------------------------------------*/
	if (FLDget_node_data_type(out, comp, &dtype) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error getting data type");
	}
	UTILdouble_to_type(&rmin, min, dtype);
	UTILdouble_to_type(&rmax, max, dtype);
	if (FLDset_node_data_minmax(out, comp, (char *)&rmin, (char *)&rmax, dtype) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting node data minmax");
	}
	return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRset_minmax_update(OMobj_id mod_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, out_arr;
    xp_long i, num_fields;
    int stat;

    OMobj_id  in, out;
    int comp, dtype;
    double min, max, rmin, rmax;

    in_arr  = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(mod_id, OMstr_to_name("out_nd"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    if (OMget_name_int_val(mod_id, OMstr_to_name("comp"), &comp) != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    if (OMget_name_real_val(mod_id, OMstr_to_name("min"), &min) != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    if (OMget_name_real_val(mod_id, OMstr_to_name("max"), &max) != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        xp_long tmp_nnodes;
        int tmp_num;
        stat = OMget_array_val( in_arr,  i, &in,  OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        /*---------------------------------------------*/
        /*  Make common sub-objects of 'out'  to be    */
        /*  the references to sub-objects of 'in'      */
        /*---------------------------------------------*/
        if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
            ERR_RETURN("Error setting field reference");
        }

        /* Skip fields with no node data */
        stat = FLDget_nnodes( in, &tmp_nnodes );
        if( stat != OM_STAT_SUCCESS || (tmp_nnodes == 0))
            continue;
        stat = FLDget_node_data_ncomp( in, &tmp_num );
        if( (stat != OM_STAT_SUCCESS) || (tmp_num == 0) )
            continue;

        /*----------------------------------------------*/
        /*  Set minmax of node data                     */
        /*----------------------------------------------*/
        if (FLDget_node_data_type(out, comp, &dtype) != OM_STAT_SUCCESS) {
            ERR_RETURN("Error getting data type");
        }
        UTILdouble_to_type(&rmin, min, dtype);
        UTILdouble_to_type(&rmax, max, dtype);

        if (FLDset_node_data_minmax(out, comp, (char *)&rmin, (char *)&rmax, dtype) != OM_STAT_SUCCESS) {
            ERR_RETURN("Error setting node data minmax");
        }
    }

    return METHOD_SUCCESS;
}

#undef  ERR_RETURN
#define ERR_RETURN(A) {ERRerror("set_minmax_cell", 0, ERR_ORIG, A); \
                       return OM_STAT_FAILURE;}

int DVset_minmax_cell_update(OMobj_id mod_id)
{
	OMobj_id  in, out, set_id, in_set_id;
	int nset, set, comp, dtype;
	double min, max, rmin, rmax;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out_cd"), OM_OBJ_RW);

	if (OMget_name_int_val(mod_id, OMstr_to_name("comp"), &comp) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;

	if (OMget_name_real_val(mod_id, OMstr_to_name("min"), &min) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;
	if (OMget_name_real_val(mod_id, OMstr_to_name("max"), &max) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;


	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}
	
	if (FLDget_ncell_sets (in, &nset) != OM_STAT_SUCCESS)
		return METHOD_FAILURE;
	for (set=0; set < nset; set++) {
		if (FLDget_cell_set (in, set, &in_set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting cell set");
		}
		if (FLDget_cell_set (out, set, &set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error resetting cell set");
		}
                if (!OMequal_objs(in_set_id, set_id) &&
		    FLDcopy_subelem_ref(set_id, in_set_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error setting set reference");
		}
		/*----------------------------------------------*/
		/*  Set minmax of cell data                     */
		/*----------------------------------------------*/
		if (FLDget_cell_data_type(set_id, comp, &dtype) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error getting data type");
		}
		UTILdouble_to_type(&rmin, min, dtype);
		UTILdouble_to_type(&rmax, max, dtype);
		if (FLDset_cell_data_minmax(set_id, comp, (char *)&rmin, (char *)&rmax, dtype) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error setting cell data minmax");
		}
	}
	return METHOD_SUCCESS;
}
