/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_fld2.c#1 $
*/

/*
       Read FIELD for time dependent data
                N.Matsumoto
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/math.h>
#include <avs/dtype.h>
#include <avs/fld.h>
#include <avs/f_utils.h>
#include <avs/rf.h>
#include "rf/rf_int.h"

struct RFparse_results2 RFglobal2;

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A)	{ ERRerror("Read_Field", 0, ERR_ORIG, A); return METHOD_FAILURE; }

#define MSG_LENGTH 8192

/* prototypes for functions private to this file */
static int cleanup_local (OMobj_id id);
static int get_struct (struct RFparse_results2 *s_ptr);
static void free_ptr (struct RFparse_results2 *s_ptr);
static int set_struct (struct RFparse_results2 **n_ptr);
static int step_count (FILE *fp, struct RFparse_results2 *s_ptr);
static void step_alloc (struct RFparse_results2 **n_ptr);

/* ---------------------------------------------------------------------
 *
 * The module update public entry point
 *
 */
int DVread_field_update(OMobj_id id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id file_id, sub_id, out, temp, out_all, temp_all;
    int      store_all, store_changed, file_changed, param_changed;
    char     *filename;
    FILE     *fp;
    int      portable, swap, flip;
    int      nsteps, stepno;
    char     timestring[TIMESTR_MAXLEN];
    struct RFparse_results2 *RF_ptr = NULL;
    int      stat = 0;

    if (event_mask & OM_EVENT_INST)
    {
      /* init all the dynamically allocated ptrs at inst */
      RFglobal2.min_ext = NULL;
      RFglobal2.max_ext = NULL;
      RFglobal2.min_val = NULL;
      RFglobal2.max_val = NULL;
      RFglobal2.labels = NULL;
      RFglobal2.name_list_count = NULL;
      RFglobal2.units = NULL;
      RFglobal2.unit_list_count = NULL;
      RFglobal2.flag = NULL;
      RFglobal2.data = NULL;
      RFglobal2.data_in_file = NULL;
      RFglobal2.points = NULL;
      RFglobal2.points_in_file = NULL;
      RFglobal2.data_ptr_file = NULL;
      RFglobal2.points_ptr_file = NULL;
      RFglobal2.time_t = NULL;
      RFglobal2.time_in_file = NULL;
    }

    /* check for store_all changed */
    sub_id = OMfind_subobj(id, OMstr_to_name("store_all"), OM_OBJ_RD);
    if (OMis_null_obj(sub_id))
      ERR_RETURN("Can't get 'store_all' object id.")
    store_changed = 0;
    if (OMchanged(sub_id, seq_num))
      store_changed = 1;

    /* get store_all value */
    if (OMget_name_int_val(id, OMstr_to_name("store_all"), &store_all) != 1)
      store_all = 0;

    /* check for filename changed */
    file_id = OMfind_subobj(id, OMstr_to_name("filename"), OM_OBJ_RD);
    if (OMis_null_obj(file_id))
      ERR_RETURN("Can't get 'filename' object id.")
    file_changed = 0;
    if (OMchanged(file_id, seq_num))
    {
      file_changed = 1;

      /* when file changes ensure store_all is (re)set to 0,
       * note: we don't want to make the store_all toggle visible here
       *       because that's only appropriate for a multistep file
       *       and we don't know if we have one yet...
       */
      OMset_name_int_val(id, OMstr_to_name("store_all"), 0);
      OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 0);
      store_all = 0;
    }

    /* when toggling store_all from ON to OFF set the toggle invisible:
     * the user only gets one chance - to protect the uninitiated against
     * misuse of the toggle
     */
    if (store_changed && !store_all)
      OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 0);

    /* (re-)read data if file changes or store_all button is toggled */
    if (file_changed || store_changed)
    {
      /* reset stepno and nsteps */
      stepno = 1;
      OMset_name_int_val(id, OMstr_to_name("stepno"), 0);
      OMset_name_int_val(id, OMstr_to_name("nsteps"), 0);

      /* get filename */
      if (OMget_str_val(file_id, &filename, 0) != 1)
	ERR_RETURN("Can't get filename.")

      /* get output field ids */
      out = OMfind_subobj(id, OMstr_to_name("out"), OM_OBJ_RW);
      if (OMis_null_obj(out))
	ERR_RETURN("Can't get 'out' object id.")
      out_all = OMfind_subobj(id, OMstr_to_name("out_all"), OM_OBJ_RW);
      if (OMis_null_obj(out_all))
	ERR_RETURN("Can't get 'out_all' object id.")

      /* open file, but first cleanup after any previous one... */
      cleanup_local(id);
      if ((fp = FILEfopen(filename, SIO_R_BIN)) == NULL) {
	if (filename) free(filename);
	ERR_RETURN("Can't open file")
      }

      /* get/set number of timesteps to allocate memory */
      step_count(fp, &RFglobal2);

      OMget_name_int_val(id, OMstr_to_name("portable"), &portable);
      OMget_name_int_val(id, OMstr_to_name("swap"), &swap);
      OMget_name_int_val(id, OMstr_to_name("flip"), &flip);

      stat = RFread_field2(id, &temp, &temp_all,
			   &fp, filename, portable, swap, flip,
			   &nsteps, stepno, timestring, 1, store_all);

      if (filename)  free(filename);

      if (stat != OM_STAT_SUCCESS) {
         cleanup_local(id);
         return METHOD_FAILURE;
      }
     
      if (nsteps < 1)  nsteps = 0;

      OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), fp, 0);
      OMset_name_int_val(id, OMstr_to_name("nsteps"), nsteps);
      OMset_name_int_val(id, OMstr_to_name("stepno"), stepno);
      OMset_name_str_val(id, OMstr_to_name("time_string"), timestring);
      /* Copy RFglobal2 to "DVread_field.RF_ptr" */
      set_struct(&RF_ptr);
      OMset_name_ptr_val(id, OMstr_to_name("RF_ptr"), RF_ptr, 0);

      /* for a just opened multistep file, set the store_all toggle visible */
      if (file_changed && nsteps > 1)
	OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 1);

      if (stat == OM_STAT_SUCCESS)
      {
	stat = OMset_obj_ref_mode(out, OM_OBJ_REF);
	if (stat == OM_STAT_SUCCESS)
	  stat = OMset_obj_ref(out, temp, 0);

        if (store_all)
	{
          stat = OMset_obj_ref_mode(out_all, OM_OBJ_REF);
          if (stat == OM_STAT_SUCCESS)
	    stat = OMset_obj_ref(out_all, temp_all, 0);
        }
      }

      return(stat);
    }

    /* file or store_all hasn't changed, see if something else did... */
    fp = NULL;
    OMget_name_ptr_val(id, OMstr_to_name("file_ptr"), (void *) &fp, 0);
    if (fp == NULL) {
      /* file not open yet */
      return METHOD_FAILURE;
    }

    /* retrieve current multistep info */
    if (OMget_name_int_val(id, OMstr_to_name("nsteps"), &nsteps) != 1)
      nsteps = 0;
    if (OMget_name_int_val(id, OMstr_to_name("stepno"), &stepno) != 1)
      stepno = 1;
    OMget_name_ptr_val(id, OMstr_to_name("RF_ptr"), (void *) &RF_ptr, 0);

    /* see if any of "portable", "swap" or "flip" have changed */
    param_changed = 0;
    sub_id = OMfind_subobj(id, OMstr_to_name("flip"), OM_OBJ_RD);
    if (!OMis_null_obj(sub_id) && OMchanged(sub_id, seq_num))
      param_changed = 1;
    else
    {
      sub_id = OMfind_subobj(id, OMstr_to_name("portable"), OM_OBJ_RD);
      if (!OMis_null_obj(sub_id) && OMchanged(sub_id, seq_num))
	param_changed = 1;
      else
      {
	sub_id = OMfind_subobj(id, OMstr_to_name("swap"), OM_OBJ_RD);
	if (!OMis_null_obj(sub_id) && OMchanged(sub_id, seq_num))
	  param_changed = 1;
      }
    }

    /* for multistep data only... */
    if (!param_changed && nsteps > 1)
    {
      /* check for "forward" pushed */
      sub_id = OMfind_subobj(id, OMstr_to_name("forward"), OM_OBJ_RD);
      if (OMis_null_obj(sub_id))
        ERR_RETURN("Can't get 'forward' object id.")
      if (OMchanged(sub_id, seq_num))
      {
	stepno++;
	if (stepno > nsteps)
	  stepno = 1;
	param_changed = 1;
      }

      /* check for "backward" pushed */
      sub_id = OMfind_subobj(id, OMstr_to_name("backward"), OM_OBJ_RD);
      if (OMis_null_obj(sub_id))
        ERR_RETURN("Can't get 'backward' object id.")
      if (OMchanged(sub_id, seq_num))
      {
	stepno--;
	if (stepno < 1)
	  stepno = nsteps;
	param_changed = 1;
      }

      if (!param_changed)
      {
	/* check for the "stepno" _parameter_ changed */
	sub_id = OMfind_subobj(id, OMstr_to_name("stepno"), OM_OBJ_RD);
	if (OMis_null_obj(sub_id))
	  ERR_RETURN("Can't get 'stepno' object id.")
	if (OMchanged(sub_id, seq_num))
        {
	  if (stepno < 1)
	    stepno = 1;
	  else if (stepno > nsteps)
	    stepno = nsteps;
	  param_changed = 1;
	}
      }
    }

    if (param_changed)
    {
      /* get filename */
      if (OMget_str_val(file_id, &filename, 0) != 1)
	ERR_RETURN("Can't get filename.")

      /* get output field ids */
      out = OMfind_subobj(id, OMstr_to_name("out"), OM_OBJ_RW);
      if (OMis_null_obj(out))
	ERR_RETURN("Can't get 'out' object id.")
      out_all = OMfind_subobj(id, OMstr_to_name("out_all"), OM_OBJ_RW);
      if (OMis_null_obj(out_all))
	ERR_RETURN("Can't get 'out_all' object id.")

      OMget_name_int_val(id,  OMstr_to_name("portable"), &portable);
      OMget_name_int_val(id,  OMstr_to_name("swap"), &swap);
      OMget_name_int_val(id,  OMstr_to_name("flip"), &flip);

      if (RF_ptr) {
        /* Copies the parse results stored in RF_ptr (retrieved from 
         * the V variable "DVread_field.RF_ptr") into the variable RFglobal2.
         * It also frees the contents of RF_ptr, which can cause unpleasant 
         * surprises when cleaning up. So make sure we can't access the
         * contents of RF_ptr or "DVread_field.RF_ptr".  CFS PR 30969.
         */
	get_struct(RF_ptr);
        OMset_name_ptr_val(id, OMstr_to_name("RF_ptr"), NULL, 0);
        RF_ptr = NULL;
      }

      /* pass in the referenced FLD for all if all steps have been stored */
      if (store_all)
      {
	if (OMget_obj_ref(out_all, &temp_all, 0) != 1)
	  ERR_RETURN("Can't get 'out_all' object id.")
      }

      stat = RFread_field2(id, &temp, &temp_all,
			   &fp, filename, portable, swap, flip,
			   &nsteps, stepno, timestring, 0, store_all);

      if (filename)
	free(filename);

      OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), fp, 0);
      OMset_name_int_val(id, OMstr_to_name("stepno"), stepno);
      OMset_name_str_val(id, OMstr_to_name("time_string"), timestring);

      if (stat != OM_STAT_SUCCESS) {
         /* Hack to leave the main file open in case we can just skip
          * over an error in a single time step.
          */
         OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), NULL, 0);
         cleanup_local(id);
         OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), fp, 0);
         return METHOD_FAILURE;
      }

      /* Copy RFglobal2 to "DVread_field.RF_ptr" */
      set_struct(&RF_ptr);
      /* RF_ptr is now valid again */
      OMset_name_ptr_val(id, OMstr_to_name("RF_ptr"), RF_ptr, 0);

      if (stat == OM_STAT_SUCCESS)
      {
	stat = OMset_obj_ref_mode(out, OM_OBJ_REF);
	if (stat == OM_STAT_SUCCESS)
	  stat = OMset_obj_ref(out, temp, 0);
      }
    }

    return(stat);
}

/* ---------------------------------------------------------------------
 *
 * The module delete public entry point
 *
 */
int DVread_field_delete(OMobj_id id)
{
  int i;

  cleanup_local(id);

  if (RFglobal2.min_ext) free(RFglobal2.min_ext);
  if (RFglobal2.max_ext) free(RFglobal2.max_ext);
  if (RFglobal2.min_val) free(RFglobal2.min_val);
  if (RFglobal2.max_val) free(RFglobal2.max_val);
  if (RFglobal2.labels) {
    for(i = 0; i < MAXVECLEN; i++)
      if (RFglobal2.labels[i]) free(RFglobal2.labels[i]);
    free(RFglobal2.labels);
  }
  if (RFglobal2.name_list_count ) free(RFglobal2.name_list_count);
  if (RFglobal2.units) {
    for(i = 0; i < MAXVECLEN; i++)
      if (RFglobal2.units[i]) free(RFglobal2.units[i]);
    free(RFglobal2.units);
  }
  if (RFglobal2.unit_list_count) free(RFglobal2.unit_list_count);
  if (RFglobal2.flag) free(RFglobal2.flag);
  if (RFglobal2.data) free(RFglobal2.data);
  if (RFglobal2.data_in_file) free(RFglobal2.data_in_file);
  if (RFglobal2.points) free(RFglobal2.points);
  if (RFglobal2.points_in_file) free(RFglobal2.points_in_file);
  if (RFglobal2.data_ptr_file) free(RFglobal2.data_ptr_file);
  if (RFglobal2.points_ptr_file) free(RFglobal2.points_ptr_file);
  if (RFglobal2.time_t) free(RFglobal2.time_t);
  if (RFglobal2.time_in_file) free(RFglobal2.time_in_file);

  RFglobal2.min_ext = NULL;
  RFglobal2.max_ext = NULL;
  RFglobal2.min_val = NULL;
  RFglobal2.max_val = NULL;
  RFglobal2.labels = NULL;
  RFglobal2.name_list_count = NULL;
  RFglobal2.units = NULL;
  RFglobal2.unit_list_count = NULL;
  RFglobal2.flag = NULL;
  RFglobal2.data = NULL;
  RFglobal2.data_in_file = NULL;
  RFglobal2.points = NULL;
  RFglobal2.points_in_file = NULL;
  RFglobal2.data_ptr_file = NULL;
  RFglobal2.points_ptr_file = NULL;
  RFglobal2.time_t = NULL;
  RFglobal2.time_in_file = NULL;

  return METHOD_SUCCESS;
}

/* ---------------------------------------------------------------------
 *
 * Local functions to deal with private ptr data
 *
 */
static int cleanup_local(OMobj_id id)
{
  FILE *fp;
  struct RFparse_results2 *RF_ptr;
  float *reserv;

  OMget_name_ptr_val(id, OMstr_to_name("file_ptr"), (void *) &fp, 0);
  if (fp != NULL)
  {
    /* a file was already open, close it */
    fclose(fp);  fp = NULL;
    OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), fp, 0);
  }

  OMget_name_ptr_val(id, OMstr_to_name("RF_ptr"), (void *) &RF_ptr, 0);
  if (RF_ptr)
  {
    free_ptr(RF_ptr);  RF_ptr = NULL;
    OMset_name_ptr_val(id, OMstr_to_name("RF_ptr"), RF_ptr, 0);
  }

  OMget_name_ptr_val(id, OMstr_to_name("coord_ptr"), (void *) &reserv, 0);
  if (reserv)
  {
    free(reserv);  reserv = NULL;
    OMset_name_ptr_val(id, OMstr_to_name("coord_ptr"), reserv, 0);
  }

  return(1);
}

/* 64-bit porting. Only Modified Internally */
static int get_struct(struct RFparse_results2 *s_ptr)
{
  int i,j;

  if( s_ptr == NULL ) return(0);

  RFglobal2.uniform = s_ptr->uniform;
  RFglobal2.ndim = s_ptr->ndim;
  RFglobal2.nspace = s_ptr->nspace;
  RFglobal2.veclen = s_ptr->veclen;
  RFglobal2.size =  s_ptr->size;
  RFglobal2.type =  s_ptr->type;
  RFglobal2.portable = s_ptr->portable;
  RFglobal2.nstep = s_ptr->nstep;
  RFglobal2.stepno = s_ptr->stepno;
  RFglobal2.time_value = s_ptr->time_value;
  RFglobal2.file_open = s_ptr->file_open;
  RFglobal2.maxtime = s_ptr->maxtime;

  if( RFglobal2.dims != 0 )
    free(RFglobal2.dims);
  UALLOCN(RFglobal2.dims, xp_long, RFglobal2.ndim,
	  "DVread_field_update: Failure allocating ndims array");
  memcpy(RFglobal2.dims, s_ptr->dims, sizeof(xp_long) * RFglobal2.ndim);

  RF_alloc(RFglobal2.maxtime);

  for(i=0; i<MAXVECLEN; i++) {
    if( s_ptr->labels[i] )
      RFglobal2.labels[i] = strdup(s_ptr->labels[i]);
    else RFglobal2.labels[i] = NULL;

    if( s_ptr->units[i] )
      RFglobal2.units[i]  = strdup(s_ptr->units[i]);
    else RFglobal2.units[i]  = NULL;
  }

  for(j = 0; j < RFglobal2.maxtime; j++) {

    RFglobal2.flag[j] =  s_ptr->flag[j];
    RFglobal2.name_list_count[j] = s_ptr->name_list_count[j];
    RFglobal2.unit_list_count[j] = s_ptr->unit_list_count[j];
    RFglobal2.data_in_file[j] = s_ptr->data_in_file[j];
    RFglobal2.points_in_file[j] = s_ptr->points_in_file[j];
    RFglobal2.data_ptr_file[j] = s_ptr->data_ptr_file[j];
    RFglobal2.points_ptr_file[j] = s_ptr->points_ptr_file[j];
    RFglobal2.time_t[j].type = s_ptr->time_t[j].type;
    RFglobal2.time_t[j].dtype = s_ptr->time_t[j].dtype;
    RFglobal2.time_t[j].skip = s_ptr->time_t[j].skip;
    RFglobal2.time_t[j].stride = s_ptr->time_t[j].stride;
    RFglobal2.time_t[j].offset = s_ptr->time_t[j].offset;
    RFglobal2.time_t[j].tclose = s_ptr->time_t[j].tclose;
    if( RFglobal2.time_t[j].time_ptr != 0 )
      RFglobal2.time_t[j].time_ptr = s_ptr->time_t[j].time_ptr;
    else
      RFglobal2.time_t[j].time_ptr = 0;
    RFglobal2.time_t[j].time_ptr_end = s_ptr->time_t[j].time_ptr_end;

    if( s_ptr->time_t[j].filename != 0 )
      RFglobal2.time_t[j].filename = strdup( s_ptr->time_t[j].filename);
    else
      RFglobal2.time_t[j].filename = 0;

    if( s_ptr->time_t[j].timestring[0] != 0 )
      strcpy(RFglobal2.time_t[j].timestring, s_ptr->time_t[j].timestring);
    else
      RFglobal2.time_t[j].timestring[0] = 0;

    RFglobal2.time_t[j].time_value = s_ptr->time_t[j].time_value;

    for(i=0; i<MAXVECLEN; i++) {
      RFglobal2.min_val[j*MAXVECLEN+i] = s_ptr->min_val[j*MAXVECLEN+i];
      RFglobal2.max_val[j*MAXVECLEN+i] = s_ptr->max_val[j*MAXVECLEN+i];
      RFglobal2.data[j*MAXVECLEN+i].stride =s_ptr->data[j*MAXVECLEN+i].stride;
      RFglobal2.data[j*MAXVECLEN+i].skip = s_ptr->data[j*MAXVECLEN+i].skip;
      RFglobal2.data[j*MAXVECLEN+i].offset = s_ptr->data[j*MAXVECLEN+i].offset;
      RFglobal2.data[j*MAXVECLEN+i].type = s_ptr->data[j*MAXVECLEN+i].type;
      RFglobal2.data[j*MAXVECLEN+i].tclose = s_ptr->data[j*MAXVECLEN+i].tclose;
      RFglobal2.data[j*MAXVECLEN+i].file_ptr_end = s_ptr->data[j*MAXVECLEN+i].file_ptr_end;
      RFglobal2.data[j*MAXVECLEN+i].flag = s_ptr->data[j*MAXVECLEN+i].flag;

      if( s_ptr->data[j*MAXVECLEN+i].filename != 0 )
	RFglobal2.data[j*MAXVECLEN+i].filename =
          strdup( s_ptr->data[j*MAXVECLEN+i].filename );
      else
        RFglobal2.data[j*MAXVECLEN+i].filename = 0;
    }

    for(i=0; i<MAXDIM; i++) {
      RFglobal2.min_ext[j*MAXDIM+i] = s_ptr->min_ext[j*MAXDIM+i];
      RFglobal2.max_ext[j*MAXDIM+i] = s_ptr->max_ext[j*MAXDIM+i];
      RFglobal2.points[j*MAXDIM+i].flag = s_ptr->points[j*MAXDIM+i].flag;
      RFglobal2.points[j*MAXDIM+i].stride = s_ptr->points[j*MAXDIM+i].stride;
      RFglobal2.points[j*MAXDIM+i].skip = s_ptr->points[j*MAXDIM+i].skip;
      RFglobal2.points[j*MAXDIM+i].offset = s_ptr->points[j*MAXDIM+i].offset;
      RFglobal2.points[j*MAXDIM+i].type = s_ptr->points[j*MAXDIM+i].type;
      RFglobal2.points[j*MAXDIM+i].tclose = s_ptr->points[j*MAXDIM+i].tclose;
      RFglobal2.points[j*MAXDIM+i].file_ptr_end =
        s_ptr->points[j*MAXDIM+i].file_ptr_end;

      if( s_ptr->points[j*MAXDIM+i].filename != 0 )
        RFglobal2.points[j*MAXDIM+i].filename =
          strdup( s_ptr->points[j*MAXDIM+i].filename);
      else
        RFglobal2.points[j*MAXDIM+i].filename = 0;
    }
  }

  free_ptr(s_ptr);
  return(1);
}

static void free_ptr( struct RFparse_results2 *s_ptr)
{
  int i,j;

  if (s_ptr->dims)
    free(s_ptr->dims);

  for(i=0; i<MAXVECLEN; i++) {
    if (s_ptr->labels[i]) free(s_ptr->labels[i]);
    if (s_ptr->units[i])  free(s_ptr->units[i]);
  }
  if (s_ptr->labels) free(s_ptr->labels);
  if (s_ptr->units)  free(s_ptr->units);

  for(j = 0; j < s_ptr->maxtime; j++) {
    if( s_ptr->time_t[j].filename != 0 )
      free(s_ptr->time_t[j].filename);

    for(i=0; i<MAXVECLEN; i++) {
      if( s_ptr->data[j*MAXVECLEN+i].filename != 0 )
	free(s_ptr->data[j*MAXVECLEN+i].filename);
    }

    for(i=0; i<MAXDIM; i++) {
      if( s_ptr->points[j*MAXDIM+i].filename != 0 )
	free( s_ptr->points[j*MAXDIM+i].filename);
    }
  }

  if (s_ptr->min_ext) free(s_ptr->min_ext);
  if (s_ptr->max_ext) free(s_ptr->max_ext);
  if (s_ptr->min_val) free(s_ptr->min_val);
  if (s_ptr->max_val) free(s_ptr->max_val);
  if (s_ptr->name_list_count ) free(s_ptr->name_list_count);
  if (s_ptr->unit_list_count ) free(s_ptr->unit_list_count);
  if (s_ptr->flag ) free(s_ptr->flag);
  if (s_ptr->data ) free(s_ptr->data);
  if (s_ptr->data_in_file ) free(s_ptr->data_in_file);
  if (s_ptr->points ) free(s_ptr->points);
  if (s_ptr->points_in_file ) free(s_ptr->points_in_file);
  if (s_ptr->data_ptr_file ) free(s_ptr->data_ptr_file);
  if (s_ptr->points_ptr_file ) free(s_ptr->points_ptr_file);
  if (s_ptr->time_t ) free(s_ptr->time_t);
  if (s_ptr->time_in_file ) free(s_ptr->time_in_file);

  free(s_ptr);
}

/* 64-bit porting. Only Modified Internally */
static int set_struct(struct RFparse_results2 **n_ptr)
{
  struct RFparse_results2 *ptr;
  int i,j;

  UALLOCN(ptr, struct RFparse_results2, 1,
	  "DVread_field_update: Failure allocating results struct");

  ptr->uniform = RFglobal2.uniform;
  ptr->ndim = RFglobal2.ndim;
  ptr->dims = NULL;
  if (RFglobal2.ndim > 0 && RFglobal2.dims != NULL) {
    UALLOCN(ptr->dims, xp_long, RFglobal2.ndim,
	    "DVread_field_update: Failure allocating ndims array");
    memcpy(ptr->dims, RFglobal2.dims, sizeof(xp_long) * RFglobal2.ndim);

    free(RFglobal2.dims);
    RFglobal2.dims = NULL;
  }
  ptr->nspace =  RFglobal2.nspace;
  ptr->veclen =  RFglobal2.veclen;
  ptr->size = RFglobal2.size ;
  ptr->type = RFglobal2.type ;
  ptr->portable =RFglobal2.portable;
  ptr->nstep = RFglobal2.nstep;
  ptr->stepno = RFglobal2.stepno;
  ptr->portable = RFglobal2.portable;
  ptr->time_value = RFglobal2.time_value;
  ptr->file_open = RFglobal2.file_open;
  ptr->maxtime = RFglobal2.maxtime;

  step_alloc(&ptr);

  for(i=0; i<MAXVECLEN; i++) {
    if( RFglobal2.labels[i] )
      ptr->labels[i] = strdup(RFglobal2.labels[i]);
    else ptr->labels[i] = NULL;
    if( RFglobal2.units[i] )
      ptr->units[i]  = strdup(RFglobal2.units[i]);
    else ptr->units[i]  = NULL;
  }

  for(j = 0; j < RFglobal2.maxtime; j++) {

    ptr->flag[j]= RFglobal2.flag[j];
    ptr->name_list_count[j] = RFglobal2.name_list_count[j];
    ptr->unit_list_count[j] = RFglobal2.unit_list_count[j];
    ptr->data_in_file[j] =  RFglobal2.data_in_file[j];
    ptr->points_in_file[j] =  RFglobal2.points_in_file[j];
    ptr->data_ptr_file[j]= RFglobal2.data_ptr_file[j];
    ptr->points_ptr_file[j] =  RFglobal2.points_ptr_file[j];
    ptr->time_t[j].type = RFglobal2.time_t[j].type;
    ptr->time_t[j].dtype = RFglobal2.time_t[j].dtype;
    ptr->time_t[j].skip = RFglobal2.time_t[j].skip;
    ptr->time_t[j].stride = RFglobal2.time_t[j].stride;
    ptr->time_t[j].offset = RFglobal2.time_t[j].offset;
    ptr->time_t[j].tclose = RFglobal2.time_t[j].tclose;
    if (RFglobal2.time_t[j].time_ptr != 0 )
      ptr->time_t[j].time_ptr = RFglobal2.time_t[j].time_ptr ;
    else
      ptr->time_t[j].time_ptr = 0;
    ptr->time_t[j].time_ptr_end = RFglobal2.time_t[j].time_ptr_end ;

    if ( RFglobal2.time_t[j].filename != 0) {
      ptr->time_t[j].filename =
	strdup(RFglobal2.time_t[j].filename);
      free(RFglobal2.time_t[j].filename);
      RFglobal2.time_t[j].filename = NULL;
    }
    else
      ptr->time_t[j].filename = 0;

    if ( RFglobal2.time_t[j].timestring !=0 )
      strcpy(ptr->time_t[j].timestring, RFglobal2.time_t[j].timestring);
    else
      ptr->time_t[j].timestring[0] = 0;

    ptr->time_t[j].time_value = RFglobal2.time_t[j].time_value ;

    for(i=0; i<MAXVECLEN; i++) {
      ptr->min_val[j*MAXVECLEN+i] = RFglobal2.min_val[j*MAXVECLEN+i] ;
      ptr->max_val[j*MAXVECLEN+i] = RFglobal2.max_val[j*MAXVECLEN+i] ;
      ptr->data[j*MAXVECLEN+i].stride = RFglobal2.data[j*MAXVECLEN+i].stride ;
      ptr->data[j*MAXVECLEN+i].skip= RFglobal2.data[j*MAXVECLEN+i].skip ;
      ptr->data[j*MAXVECLEN+i].offset= RFglobal2.data[j*MAXVECLEN+i].offset ;
      ptr->data[j*MAXVECLEN+i].type = RFglobal2.data[j*MAXVECLEN+i].type ;
      ptr->data[j*MAXVECLEN+i].tclose = RFglobal2.data[j*MAXVECLEN+i].tclose;
      ptr->data[j*MAXVECLEN+i].file_ptr_end = RFglobal2.data[j*MAXVECLEN+i].file_ptr_end;
      ptr->data[j*MAXVECLEN+i].flag = RFglobal2.data[j*MAXVECLEN+i].flag;

      if ( RFglobal2.data[j*MAXVECLEN+i].filename != 0) {
	ptr->data[j*MAXVECLEN+i].filename=
	  strdup(RFglobal2.data[j*MAXVECLEN+i].filename );
	free(RFglobal2.data[j*MAXVECLEN+i].filename );
	RFglobal2.data[j*MAXVECLEN+i].filename = NULL;
      }
      else
	ptr->data[j*MAXVECLEN+i].filename= 0;
    }

    for(i=0; i<MAXDIM; i++) {

      ptr->min_ext[j*MAXDIM+i] = RFglobal2.min_ext[j*MAXDIM+i] ;
      ptr->max_ext[j*MAXDIM+i] = RFglobal2.max_ext[j*MAXDIM+i] ;
      ptr->points[j*MAXDIM+i].flag = RFglobal2.points[j*MAXDIM+i].flag ;
      ptr->points[j*MAXDIM+i].stride = RFglobal2.points[j*MAXDIM+i].stride ;
      ptr->points[j*MAXDIM+i].skip = RFglobal2.points[j*MAXDIM+i].skip ;
      ptr->points[j*MAXDIM+i].offset = RFglobal2.points[j*MAXDIM+i].offset ;
      ptr->points[j*MAXDIM+i].type = RFglobal2.points[j*MAXDIM+i].type ;
      ptr->points[j*MAXDIM+i].tclose = RFglobal2.points[j*MAXDIM+i].tclose ;
      ptr->points[j*MAXDIM+i].file_ptr_end = RFglobal2.points[j*MAXDIM+i].file_ptr_end;

      if ( RFglobal2.points[j*MAXDIM+i].filename != 0) {
	ptr->points[j*MAXDIM+i].filename =
	  strdup(RFglobal2.points[j*MAXDIM+i].filename);
	free(RFglobal2.points[j*MAXDIM+i].filename);
	RFglobal2.points[j*MAXDIM+i].filename = NULL;
      }
      else
	ptr->points[j*MAXDIM+i].filename = 0;
    }
  }
  *n_ptr = ptr;

  return(1);
}

/* check and return number of time steps */
static int step_count(FILE *fp, struct RFparse_results2 *s_ptr)
{
  int   no, nstep, eot;
  char  buff[MSG_LENGTH], nstep_line[1024];
  char  *cptr;

  cptr = NULL;
  nstep = -1;
  eot = no = 0;
  while(fgets(buff, MSG_LENGTH, fp) != NULL ) {
    if (strncmp(buff, "EOT", 3) == 0 ) eot++;
    if (strncmp(buff, "nstep", 4) == 0 )  {
      strncpy(nstep_line, buff, 1024);
      no = 1;
    }

    /* We should stop searching if we reached into binary area. */
    if(strncmp(buff, "\f\f", 2) == 0 ){
      break;
    }
  }
  if ( no == 1 ) {
    cptr = strstr(nstep_line, "=");
    if ( cptr == NULL ) nstep = -1;
    else {
      cptr++;
      sscanf(cptr, "%d", &nstep);
      if ( nstep <= 0 ) nstep = -1;
    }
  }
  if ((eot == 0) && (nstep == -1) )
    nstep = 1;
  else {
    if ( nstep < eot )
      nstep = eot;
  }

  s_ptr->maxtime = nstep;
  fseek(fp,0,0);

  return(nstep);
}

static void step_alloc(struct RFparse_results2 **n_ptr)
{
  struct RFparse_results2 *ptr;
  int  maxtime;

  ptr = *n_ptr;

  if (ptr->maxtime < 1)
  {
    ptr->min_ext = NULL;
    ptr->max_ext = NULL;
    ptr->min_val = NULL;
    ptr->max_val = NULL;
    ptr->name_list_count = NULL;
    ptr->unit_list_count = NULL;
    ptr->flag = NULL;
    ptr->data = NULL;
    ptr->data_in_file = NULL;
    ptr->points = NULL;
    ptr->points_in_file = NULL;
    ptr->data_ptr_file = NULL;
    ptr->points_ptr_file = NULL;
    ptr->time_t = NULL;
    ptr->time_in_file = NULL;
    return;
  }

  maxtime = ptr->maxtime;

  UALLOCN(ptr->min_ext,float, maxtime * MAXDIM,
	  "rd_fld2: Failure allocating min_ext array");

  UALLOCN(ptr->max_ext,float, maxtime * MAXDIM,
	  "rd_fld2: Failure allocating max_ext array");

  UALLOCN(ptr->min_val,float, maxtime * MAXVECLEN,
	  "rd_fld2: Failure allocating min_val array");

  UALLOCN(ptr->max_val,float, maxtime * MAXVECLEN,
	  "rd_fld2: Failure allocating max_val array");

  UALLOCN(ptr->labels,char *, MAXVECLEN,
	  "rd_fld2: Failure allocating labels array");

  UALLOCN(ptr->name_list_count,int, maxtime,
	  "rd_fld2: Failure allocating name_list_count array");

  UALLOCN(ptr->units,char *, MAXVECLEN,
	  "rd_fld2: Failure allocating units array");

  UALLOCN(ptr->unit_list_count,int, maxtime,
	  "rd_fld2: Failure allocating unit_list_count array");

  UALLOCN(ptr->flag,int, maxtime,
	  "rd_fld2: Failure allocating flag array");

  ALLOCN(ptr->data,RFfile_info , maxtime * MAXVECLEN,
	 "rd_fld2: Failure allocating data struct");

  UALLOCN(ptr->data_in_file,int, maxtime,
	  "rd_fld2: Failure allocating data_in_file  array");

  UALLOCN(ptr->data_ptr_file,FILE *, maxtime,
	  "rd_fld2: Failure allocating data_ptr_file  array");

  UALLOCN(ptr->points_ptr_file, FILE *, maxtime,
	  "rd_fld2: Failure allocating points_ptr_file  array");

  UALLOCN(ptr->points_in_file,int, maxtime,
	  "rd_fld2: Failure allocating points_in_file  array");

  ALLOCN(ptr->points, RFfile_info , maxtime * MAXDIM,
	 "rd_fld2: Failure allocating points struct");

  ALLOCN(ptr->time_t, RFtime_info, maxtime,
	 "rd_fld2: Failure allocating time_t  struct");

  UALLOCN(ptr->time_in_file,int, maxtime,
	  "rd_fld2: Failure allocating time_in_file  array");
}
