/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_seq.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>  /* time_t */

#include <avs/f_utils.h>
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/dtype.h>
#include <avs/fld.h>

#define LINE_LEN 2048

/* static const char *ui_types[MAX_TYPES] = {"byte","short","int","float","double", "long", "string"}; */
/* 64-bit changes. Modified to support 'long' */
static const char *formats[] = {"%c","%c","%h","%d","%f","%lf", "%ld", " ", "%s"};

/* 64-bit porting. Only Modified Internally */
static int get_type(const char *token)
{
	int l,i,type, real_type;

	real_type = 0;
	l = (int)strlen(token);
	for (i=0; i<l; i++) {
		if (token[i] == '\n')
			break;
		if ((token[i]>='0' && token[i]<='9') || token[i] == '+' || token[i] == '-' || token[i] == ' ')
			continue;
		else if (token[i] == '.' || token[i] =='E' || token[i] =='e')
			real_type = 1;
		else {
			type = DTYPE_STRING;
			return(type);
		}
	}
	if (real_type)
		type = DTYPE_DOUBLE;
	else
	{
		if (atol(token) <= AVS_INT_MAX)
			type = DTYPE_INT;
		else
			type = DTYPE_LONG;
	}

	return(type);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("DVread_seq_binary", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVread_seq_binary_update(OMobj_id elem_id, OMevent_mask event_mask,
                             int seq_num)
{
    OMobj_id file_id, out_id, comps_id, dims_id;
    FILE *pf = NULL;
    char *filename = NULL;
    int skip_bytes, endian, swap;
    int i;
    xp_long fld_nnodes;
    int fld_ncomps, fld_ndims;
    xp_long *fld_dims;
    float *fld_points;
    xp_long fld_pts_size;

    /* Input parameters */

    file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
    if (OMget_str_val(file_id, &filename, 0) != 1)
        return 0;
    if (filename[0] == '\0')
        return 1;

    if (OMget_name_int_val(elem_id, OMstr_to_name("skip_bytes"), &skip_bytes) != 1)
        skip_bytes = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("ndim"), &fld_ndims) != 1)
        fld_ndims = 3;

    dims_id = OMfind_subobj(elem_id, OMstr_to_name("dims"), OM_OBJ_RD);
    fld_dims = OMret_typed_array_ptr( dims_id, OM_GET_ARRAY_RD, 
                                      DTYPE_LONG, NULL);
    if( fld_dims == NULL ) {
        free(filename);
        ERR_RETURN("Can't get dimensions");
    }

    if (OMget_name_int_val(elem_id, OMstr_to_name("endian"), &endian) != 1)
        endian = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("ncomp"), &fld_ncomps) != 1)
        fld_ncomps = 1;

    comps_id = OMfind_subobj(elem_id, OMstr_to_name("comps"), OM_OBJ_RD);

    if( endian == 0 ) swap = 0; /* 0 = "same as machine" */
    else {
#ifdef AVS_LITTLEENDIAN
        swap = (endian==2);	/* 2 = "big" (thus swap if big) */
#else
        swap = (endian==1);	/* 1 = "little" (thus swap if little) */
#endif
    }

    /* Output parameters */

    out_id = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    /* Initialize output */

    FLDset_nspace( out_id, 0 );
    FLDset_ndim  ( out_id, 0 );
    FLDset_node_data_ncomp( out_id, 0 );

    if (!(pf = FILEfopen(filename, SIO_R_BIN))) {
        ERRerror("DVread_seq_binary", 1, ERR_ORIG, "Can't open data file: %s",
                 filename);
        free(filename);
        return 0;
    }
    if (filename) {
        free(filename);
        filename = NULL;
    }

    for (i=0; i<skip_bytes; i++) {
        fgetc(pf);
    }

    FLDset_nspace( out_id, fld_ndims );
    FLDset_ndim  ( out_id, fld_ndims );
    FLDset_dims  ( out_id, fld_dims );
    ARRfree( fld_dims );
    FLDget_nnodes( out_id, &fld_nnodes );
    if( fld_nnodes <= 0 ) {
        ERR_RETURN("Field nnodes is 0 or negative: check dimensions");
    }
    if (FLDget_points( out_id, &fld_points,
                       &fld_pts_size, OM_GET_ARRAY_RW ) == 1 ) {
        for (i=0; i<fld_ndims; ++i) {
            fld_points[i] = 0.0;
            fld_points[i+fld_ndims] = fld_dims[i] - 1;
        }
        ARRfree( fld_points );
    }
    else {
        ERR_RETURN("Cannot get field points");
    }
    FLDset_node_data_ncomp( out_id, fld_ncomps );

    for (i=0; i<fld_ncomps; ++i) {
        int veclen, data_type;
        xp_long size;
        char *label, *units;
        char *node_data;
        OMobj_id var_id;

        if (OMget_array_val(comps_id, i, &var_id, OM_OBJ_RD) != 1) {
            fclose(pf);
            ERR_RETURN("Cannot get per-variable information");
        }

        if (OMget_name_int_val(var_id, OMstr_to_name("data_type"), &data_type) != 1)
            data_type = 4;
        if (OMget_name_int_val(var_id, OMstr_to_name("veclen"), &veclen) != 1)
            veclen = 1;
        if (OMget_name_str_val(var_id, OMstr_to_name("label"), &label, 0) != 1) {
            char buffer[16];
            /* Use the same convention as Read_Field */
            sprintf( buffer, "DATA%d", i );
            label = strdup( buffer );
        }
        if (OMget_name_str_val(var_id, OMstr_to_name("units"), &units, 0) != 1)
            units = NULL;

        FLDset_node_data_veclen( out_id, i, veclen );
        FLDset_node_data_type  ( out_id, i, data_type );
        if( label ) {
            FLDset_node_data_label ( out_id, i, label );
            free( label );
        }
        if( units ) {
            FLDset_node_data_units ( out_id, i, units );
            free( units );
        }

        if( FLDget_node_data( out_id, i, &data_type, &node_data,
                              &size, OM_GET_ARRAY_RW ) == 1 ) {
            xp_long size_in_bytes = size * DTYPEtype_size[data_type];

            /* read node data from file */
            if( fread( node_data, size_in_bytes, 1, pf ) != 1 ) {
                fprintf( stderr, "Attempt to read %ld bytes failed\n",
                         size_in_bytes );
            }

            /* byte swapping nonsense */
            if( swap ) {
                unsigned char *p, *bdata = (unsigned char *)node_data;
                unsigned char tmp;
                xp_long j;
                switch( DTYPEtype_size[data_type] ) {
                case 1: /* no swapping needed */
                    break;
                case 2:
                    for( j = 0; j < size; ++j ) {
                        p = &(bdata[((xp_ulong)j)*2]);
                        tmp = p[1]; p[1] = p[0]; p[0] = tmp;
                    }
                    break;
                case 4:
                    for( j = 0; j < size; ++j ) {
                        p = &(bdata[((xp_ulong)j)*4]);
                        tmp = p[3]; p[3] = p[0]; p[0] = tmp;
                        tmp = p[2]; p[2] = p[1]; p[1] = tmp;
                    }
                    break;
                case 8:
                    for( j = 0; j < size; ++j ) {
                        p = &(bdata[((xp_ulong)j)*4]);
                        tmp = p[7]; p[7] = p[0]; p[0] = tmp;
                        tmp = p[6]; p[6] = p[1]; p[1] = tmp;
                        tmp = p[5]; p[5] = p[2]; p[2] = tmp;
                        tmp = p[4]; p[4] = p[3]; p[3] = tmp;
                    }
                    break;
                }
            }

            ARRfree( node_data );
        }
    }

    fclose( pf );

    return 1;
}


#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("DVread_seq_ascii", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVread_seq_ascii_update(OMobj_id elem_id, OMevent_mask event_mask,
                            int seq_num)
{
    OMobj_id file_id, out_id, comps_id, dims_id;
    FILE *pf = NULL;
    char *filename = NULL, *delim = NULL;
    int skip_lines;
    int i;

    xp_long fld_nnodes;
    int fld_ncomps, fld_ndims;
    xp_long *fld_dims;
    float *fld_points;
    xp_long fld_pts_size;

    /* Input parameters */

    file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
    if (OMget_str_val(file_id, &filename, 0) != 1)
        return 0;
    if (filename[0] == '\0')
        return 1;
    if (OMget_name_str_val(elem_id, OMstr_to_name("delimiter"), &delim, 0) != 1)
        delim = strdup(" ");
    if (OMget_name_int_val(elem_id, OMstr_to_name("skip_lines"), &skip_lines) != 1)
        skip_lines = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("ndim"), &fld_ndims) != 1)
        fld_ndims = 3;

    dims_id = OMfind_subobj(elem_id, OMstr_to_name("dims"), OM_OBJ_RD);
    fld_dims = OMret_typed_array_ptr( dims_id, OM_GET_ARRAY_RD, 
                                      DTYPE_LONG, NULL);
    if( fld_dims == NULL ) {
        free(filename);
        ERR_RETURN("Can't get dimensions");
    }

    if (OMget_name_int_val(elem_id, OMstr_to_name("ncomp"), &fld_ncomps) != 1)
        fld_ncomps = 1;

    comps_id = OMfind_subobj(elem_id, OMstr_to_name("comps"), OM_OBJ_RD);

    /* Output parameters */

    out_id = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    /* Initialize output */

    FLDset_nspace( out_id, 0 );
    FLDset_ndim  ( out_id, 0 );
    FLDset_node_data_ncomp( out_id, 0 );

    if (!(pf = FILEfopen(filename, SIO_R_BIN))) {
        ERRerror("DVread_seq_ascii", 1, ERR_ORIG, "Can't open data file: %s",
                 filename);
        free(filename);
        return 0;
    }
    if (filename) {
        free(filename);
        filename = NULL;
    }

    /* Skip initial lines. */
    if( skip_lines > 0 ) {
        char line[LINE_LEN];
        for (i=0; i<skip_lines; i++) {
            if (fgets(line, LINE_LEN, pf) == NULL) {
                fclose(pf);
                ERR_RETURN("Error skipping lines");
            }
        }
    }

    FLDset_nspace( out_id, fld_ndims );
    FLDset_ndim  ( out_id, fld_ndims );
    FLDset_dims  ( out_id, fld_dims );
    ARRfree( fld_dims );
    FLDget_nnodes( out_id, &fld_nnodes );
    if( fld_nnodes <= 0 ) {
        ERR_RETURN("Field nnodes is 0 or negative: check dimensions");
    }
    if (FLDget_points( out_id, &fld_points,
                       &fld_pts_size, OM_GET_ARRAY_RW ) == 1 ) {
        /* The same defaults that Read_Field supplies ... allow
         * user to control?
         */
        for (i=0; i<fld_ndims; ++i) {
            fld_points[i] = 0.0;
            fld_points[i+fld_ndims] = fld_dims[i] - 1;
        }
        ARRfree( fld_points );
    }
    else {
        ERR_RETURN("Cannot get field points");
    }
    FLDset_node_data_ncomp( out_id, fld_ncomps );

    for (i=0; i<fld_ncomps; ++i) {
        int veclen, data_type;
        xp_long nd_size;
        char *label, *units;
        char *node_data;
        OMobj_id var_id;

        if (OMget_array_val(comps_id, i, &var_id, OM_OBJ_RD) != 1) {
            fclose(pf);
            if(delim) free(delim);
            ERR_RETURN("Cannot get per-variable information");
        }

        if (OMget_name_int_val(var_id, OMstr_to_name("data_type"), &data_type) != 1)
            data_type = 4;
        if( data_type != DTYPE_UNSET && 
            (data_type < 0 || data_type > DTYPE_LONG) ) { 
            fclose(pf);
            if(delim) free(delim);
            ERR_RETURN("Data type must be numerical");
        }

        if (OMget_name_int_val(var_id, OMstr_to_name("veclen"), &veclen) != 1)
            veclen = 1;
        if (OMget_name_str_val(var_id, OMstr_to_name("label"), &label, 0) != 1) {
            char buffer[16];
            /* Use the same convention as Read_Field */
            sprintf( buffer, "DATA%d", i );
            label = strdup( buffer );
        }
        if (OMget_name_str_val(var_id, OMstr_to_name("units"), &units, 0) != 1)
            units = NULL;

        FLDset_node_data_veclen( out_id, i, veclen );

        if( label ) {
            FLDset_node_data_label ( out_id, i, label );
            free( label );
            label = NULL;
        }
        if( units ) {
            FLDset_node_data_units ( out_id, i, units );
            free( units );
            units = NULL;
        }

        nd_size = fld_nnodes * veclen;
        node_data = NULL;

        {
            char token[256];
            int c, eot, tok_i;
            xp_long nd_i;
            const char *format = NULL;

            tok_i = 0;
            nd_i = 0;
            while( nd_i < nd_size ) {
                c = fgetc(pf);

                if( tok_i == 0 ) {
                    /* skip leading whitespace */
                    if( c == '\n' || c == '\r' ||
                        c == ' '  || c == '\t' ) continue;
                }

                eot = 0;	/* end of token flag */
                if( c == EOF ) eot = 1;
                else if( delim[0] == ' ' ) {
                    if( c == '\n' || c == '\r' ||
                        c == ' '  || c == '\t' ) {
                        /* If delim is space, then end the token
                         * on any whitespace. */
                        eot = 1;
                    }
                }
                else if( c == delim[0] ) {
                    /* Any white space before after the number and before
                     * the delim will be ignored by sscanf.  The effect of
                     * this somewhat convoluted logic is that multiple
                     * delimiters, e.g. ",,,", will be turned into a
                     * sequence of null values, but not if the delimiter
                     * is just a space.
                     */
                    eot = 1;
                }

                if( eot ) {	/* Found end of token, now process the token */
                    token[tok_i++] = '\0';
                    if( tok_i > 1 ) {

                        /* If dtype has not been set, guess type of token */
                        if( data_type == DTYPE_UNSET ) {
                            data_type = get_type( token );
                            if( data_type < 0 || data_type > DTYPE_LONG ) { 
                                fclose(pf);
                                if(delim) free(delim);
                                ERR_RETURN("Data type must be numerical");
                            }
                        }

                        /* Only now can we be certain that the type information
                         * is established, so the first time through this loop
                         * determine the scan format and grab the node data.
                         */
                        if( node_data == NULL ) {
                            FLDset_node_data_type( out_id, i, data_type );
                            if( FLDget_node_data( out_id, i, &data_type,
                                                  &node_data, &nd_size,
                                                  OM_GET_ARRAY_RW ) == 1 ) {
                                xp_long size_in_bytes = nd_size * DTYPEtype_size[data_type];
                                format = formats[data_type];
                                memset( node_data, 0, size_in_bytes );
                            }
                            else {
                                ERR_RETURN("Could not get node data");
                            }
                        }

                        /* Scan the token directly into the node data */
                        sscanf( token, format,
                                node_data+(nd_i*DTYPEtype_size[data_type]) );
                    }
                    /* Don't allow EOF to count as a null token */
                    if( c != EOF || tok_i > 1 ) 
                        ++nd_i;
                    tok_i = 0;
                }
                else {
                    /* copy single char into token */
                    token[tok_i++] = c;
                }

                /* End of File.  Warn if we did not fill the node data. */
                if( c == EOF ) {
                    if( nd_i < nd_size ) {
                        ERRerror( "DVread_seq_ascii", 3, ERR_WARNING, 
                                  "Attempt to read %d elements failed for node"
                                  "data component %d (only %d elements read)",
                                  nd_size, i, nd_i );
                    }
                    break; /* end of file */
                }

            } /* while */
        }
        if( node_data != NULL ) ARRfree( node_data );
    } /* loop over components */

    if(delim) free(delim);
    fclose( pf );

    return 1;
}
