/*
			Copyright (c) 1995 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_tbl.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>  /* time_t */

#include <avs/f_utils.h>
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/math.h>
#include <avs/dtype.h>
#include <avs/fld.h>

#define LINE_LEN 2048
#define MAX_COL 256
#define MAX_TYPES 7
#define ALLOC_BLOCK 1024

#define ERR_RETURN(A) ERRerror("DVtable_read_init", 0, ERR_ORIG, A); return(0);

static const char *ui_types[MAX_TYPES] = {"byte","short","int","float","double", "string", "long"};
static const char *formats[] = {"%c","%c","%h","%d","%f","%lf", "%ld", " ", "%s"};

typedef struct _TBLdata {
	int use;
	int type;
	int size;
	int count;
	int null_flag;
	double null_value;
	char *values;
	char **strings;
	char *name;
} TBLdata;

static int get_type (const char *token);
static int read_token (TBLdata *data, const char *token);
static int free_data (TBLdata *data, int ndata);

int DVtable_read_init_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id file, out, val_id;
	char *filename, *delim, line[LINE_LEN], token[LINE_LEN], *names[MAX_COL], *ptr;
	int  i, j, count, ncol, ncol1, delim_len, line_len, found, type, types[MAX_COL];
	int  header, skip_lines, read_file, in_quote;
	FILE     *pf;
	double null_value;

	if (OMget_name_int_val(elem_id, OMstr_to_name("init_file"), &read_file) != 1)
		read_file = 0;

	if (read_file == 0) {
		out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
		if (OMget_name_int_val(elem_id, OMstr_to_name("ncol"), &ncol) != 1)
			ncol = 0;

		for (i=0; i<ncol; i++) {
			if (OMget_array_val(out, i, &val_id, OM_OBJ_RW) != 1) {
				ERR_RETURN("cannot get out array member");
			}
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &ptr, 0) != 1) {
				sprintf(line, "column %d", i);
				if (OMset_name_str_val(val_id, OMstr_to_name("name"), line) != 1) {
					ERR_RETURN("cannot set column name");
				}
			}
			else free(ptr);
			if (OMget_name_int_val(val_id, OMstr_to_name("type"), &type) != 1) {
				if (OMset_name_int_val(val_id, OMstr_to_name("type"), 2) != 1) {
					ERR_RETURN("cannot set column type");
				}
			}
			if (OMget_name_int_val(val_id, OMstr_to_name("use"), &type) != 1) {
				if (OMset_name_int_val(val_id, OMstr_to_name("use"), 1) != 1) {
					ERR_RETURN("cannot set column type");
				}
			}
			if (OMget_name_real_val(val_id, OMstr_to_name("null_value"), &null_value) != 1) {
				if (OMset_name_real_val(val_id, OMstr_to_name("null_value"), 0.0) != 1) {
					ERR_RETURN("cannot set null_value");
				}
			}
		}
		return(0);
	}

	filename = NULL;
	file = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
	if (OMget_str_val(file, &filename, 0) != 1)
		return(0);
	if (filename[0] == '\0')
		return(1);
	if (OMget_name_str_val(elem_id, OMstr_to_name("delimiter"), &delim, 0) != 1)
		delim = strdup(" ");
	delim_len = (int)strlen(delim);
	if (OMget_name_int_val(elem_id, OMstr_to_name("header"), &header) != 1)
		header = 0;

	if (OMget_name_int_val(elem_id, OMstr_to_name("skip_lines"), &skip_lines) != 1)
		skip_lines = 0;

	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	if (OMset_name_int_val(elem_id, OMstr_to_name("ncol"), 0 ) != 1) {
		ERR_RETURN("cannot set ncol");
	}
	OMset_array_size(out, 0);

	if (!(pf = FILEfopen(filename, SIO_R_BIN))) {
		ERRerror("Read Table",1,ERR_ORIG, "Can't open data file: %s",
			 filename);
		return(0);
	}
	for (i=0; i<skip_lines; i++) {
		if (fgets(line, LINE_LEN, pf) == NULL) {
			fclose(pf);
			ERR_RETURN("Error skipping lines");
		}
	}

	/***  first line ***/
	/***  get names  ***/
	if (fgets(line, LINE_LEN, pf) == NULL) {
		fclose(pf);
		if (header) { ERR_RETURN("Error reading header"); }
		else        { ERR_RETURN("Error reading data"); }
	}
	line_len = (int)strlen(line);

	if (line[line_len-1] == '\r') line_len--;
	if (line[line_len-1] == '\n') line_len--;
	if (line[line_len-1] == '\r') line_len--;

	ncol = 0;
	in_quote = 0;
	for (found=0, count=0, i=0; i<line_len; i++) {
		if (line[i] == ' ' && !found)
			continue;   /* skip leading spaces */
		if ((in_quote == 0) && (strncmp(line+i, delim, delim_len) == 0)) {
			token[count++] = '\0';
			if (strlen(token) == 0)
				continue;

			if (header) {
				/* Get names from the header, get the
				   data types later from next line. */
				if (count <= 1)
					sprintf(token, "column_%d", ncol);
				names[ncol++] = strdup(token);
			}
			else {
				/* Get data types from the first data line
				   and make a synthetic column name */
				types[ncol] = get_type(token);
				sprintf(token, "column_%d", ncol);
				names[ncol++] = strdup(token);
			}
			count = 0;
			i += delim_len-1;
			found = 0;
		}
		else {
			if (line[i] == '\"')
				/* simply not copying the quote drops it. */
				in_quote = !in_quote;
			else
				/* copy single char into token */
				token[count++] = line[i];
			found = 1;
			if (i==line_len-1) {
				token[count++] = '\0';
				if (header) {
					names[ncol++] = strdup(token);
				}
				else {
					types[ncol] = get_type(token);
					sprintf(token,"column_%d", ncol);
					names[ncol++] = strdup(token);
				}
			}
		}
	}

	ncol1 = ncol;
	if (header) {   /** get types from next line (first data line) **/
		if (fgets(line, LINE_LEN, pf) == NULL) {
			fclose(pf);
			ERR_RETURN("Error reading data");
		}
		line_len = (int)strlen(line);

		if (line[line_len-1] == '\r') line_len--;
		if (line[line_len-1] == '\n') line_len--;
		if (line[line_len-1] == '\r') line_len--;

		ncol = 0;
		in_quote = 0;
		for (found=0, count = 0, i=0; i<line_len; i++) {
			if (line[i] == ' ' && !found)
				continue;   /* skip leading spaces */
			if ((in_quote==0) && (strncmp(line+i, delim, delim_len) == 0)) {
				token[count++] = '\0';
				count = 0;
				types[ncol++] = get_type(token);
				i += delim_len-1;
				found = 0;
			}
			else {
				if (line[i] == '\"')
					/* not copying the quote drops it. */
					in_quote = !in_quote;
				else
					/* copy single char into token */
					token[count++] = line[i];
				found = 1;
				if (i==line_len-1) {
					token[count++] = '\0';
					types[ncol++] = get_type(token);
				}
			}
		}

		/* bogus header line - try to cope */
		if (ncol > ncol1)
			for (i=ncol1-1; i<ncol; i++) {
				sprintf(token,"column_%d", i);
				names[i] = strdup(token);
			}

		if (ncol != ncol1) {
			if (strcmp(delim, " ") == 0) {
				for (i=ncol; i<ncol1; i++)
					types[ncol++] = get_type(" ");
			}
			else {
				fclose(pf);
				if (filename)
					free(filename);
				if (delim)
					free(delim);
				ERR_RETURN("Number of columns found in first and second lines are different. Check column separator");
			}
		}
	}

	if (OMset_name_int_val(elem_id, OMstr_to_name("ncol"), ncol) != 1) {
		ERR_RETURN("cannot set ncol");
	}
	for (i=0; i<ncol; i++) {
		if (OMget_array_val(out, i, &val_id, OM_OBJ_RW) != 1) {
			ERR_RETURN("cannot get out array member");
		}
		if (OMset_name_str_val(val_id, OMstr_to_name("name"), names[i]) != 1) {
			ERR_RETURN("cannot set column name");
		}
		for (j=0; j<MAX_TYPES; j++) {
			if (DTYPEstr_to_type_id(ui_types[j]) == types[i]) {
				type = j;
				break;
			}
			if (j == MAX_TYPES-1) {
				type = j;
				ERRerror("DVtable_read_init", 0, ERR_ORIG, "unkown column type");
			}
		}
		if (OMset_name_int_val(val_id, OMstr_to_name("type"), type) != 1) {
			ERR_RETURN("cannot set column type");
		}
		if (OMset_name_int_val(val_id, OMstr_to_name("use"), 1) != 1) {
			ERR_RETURN("cannot set column use");
		}
		if (OMset_name_real_val(val_id, OMstr_to_name("null_value"), 0.0) != 1) {
			ERR_RETURN("cannot set column null_value");
		}
	}


	fclose(pf);
	if (filename)
		free(filename);
	if (delim)
		free(delim);
	for (i=0; i<ncol; i++)
		if (names[i])
			free(names[i]);
	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("DVtable_read", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVtable_read_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id file, in, out, val_id, e_id;
	char *filename, *delim, line[LINE_LEN], token[LINE_LEN];
	int  i, j, stat, count, ncol, ndata, delim_len, line_len, line_count, col_count, found, type;
	int  header, skip_lines, read_file, in_ncol, nrow, in_quote, gen_index;
	xp_long in_ncol_w;
	TBLdata *data, *pdata;
	FILE     *pf;

	if (!(event_mask & OM_EVENT_INST)) {
		if (OMget_name_int_val(elem_id, OMstr_to_name("do"), &read_file) != 1)
			read_file = 0;

		if (read_file == 0)
			return(0);
		else
			if (OMset_name_int_val(elem_id, OMstr_to_name("do"), 0) != 1) {
				ERR_RETURN("cannot set do");
			}
	}

	filename = NULL;
	file = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
	if (OMget_str_val(file, &filename, 0) != 1)
		return(0);
	if (filename[0] == '\0')
		return(1);
	if (OMget_name_str_val(elem_id, OMstr_to_name("delimiter"), &delim, 0) != 1)
		delim = strdup(" ");
	if (OMget_name_int_val(elem_id, OMstr_to_name("header"), &header) != 1)
		header = 0;
	if (OMget_name_int_val(elem_id, OMstr_to_name("index_column"), &gen_index) != 1)
		gen_index = 0;

	if (OMget_name_int_val(elem_id, OMstr_to_name("skip_lines"), &skip_lines) != 1)
		skip_lines = 0;

	in = OMfind_subobj(elem_id, OMstr_to_name("columns"), OM_OBJ_RD);
	OMget_array_size(in, &in_ncol_w);
	in_ncol = (int)in_ncol_w;

	if (OMset_name_int_val(elem_id, OMstr_to_name("tbl_nrow"), 0) != 1) {
		ERR_RETURN("cannot set tbl_nrow");
	}

	if (OMset_name_int_val(elem_id, OMstr_to_name("tbl_ncol"), 0) != 1) {
		ERR_RETURN("cannot set tbl_ncol");
	}

	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	OMset_array_size(out, 0);
	if (in_ncol == 0)
		return(0);

	data = (TBLdata *)malloc(in_ncol*sizeof(TBLdata));

	for (ndata=0,i=0; i<in_ncol; i++) {
		pdata = data+i;
		if (OMget_array_val(in, i, &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("cannot get in array member");
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("use"), &j) != 1)
			j = 0;
		if (j == 0) {
			pdata->use = 0;
			continue;
		}
		pdata->use = 1;

		if (OMget_name_str_val(val_id, OMstr_to_name("name"), &(pdata->name), 0) != 1) {
			ERR_RETURN("cannot get column name");
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("type"), &type) != 1) {
			ERR_RETURN("cannot get column type");
		}
		type = DTYPEstr_to_type_id(ui_types[type]);
		pdata->type = type;
		if (type != DTYPE_STRING)
			if (OMget_name_real_val(val_id, OMstr_to_name("null_value"), &(pdata->null_value)) != 1)
				pdata->null_value = 0;

		pdata->null_flag = 0;
		pdata->count = 0;
		pdata->size = ALLOC_BLOCK;
		if (type != DTYPE_STRING) {
			pdata->values = ARRalloc(NULL, type, pdata->size, NULL);
			if (pdata->values == NULL) {
				ERR_RETURN("cannot allocate data");
			}
		}
		else {
			pdata->strings = (char **)malloc(pdata->size*sizeof(char*));
			if (pdata->strings == NULL) {
				ERR_RETURN("cannot allocate data");
			}
		}
		ndata++;
	}

	if (!(pf = FILEfopen(filename, SIO_R_BIN))) {
		ERRerror("DVtable_read", 1, ERR_ORIG, "Can't open data file: %s",
			 filename);
		return(0);
	}
	for (i=0; i<skip_lines; i++) {
		if (fgets(line, LINE_LEN, pf) == NULL) {
			fclose(pf);
			ERR_RETURN("Error skipping lines");
		}
	}
	delim_len = (int)strlen(delim);

	if (header) {
		if (fgets(line, LINE_LEN, pf) == NULL) {
			fclose(pf);
			ERR_RETURN("Error reading header");
		}
	}


	OMset_array_size(out, ndata);

	line_count = 0;
	while (fgets(line, LINE_LEN, pf) != NULL) {
		line_len = (int)strlen(line);

		if (line[line_len-1] == '\r') line_len--;
		if (line[line_len-1] == '\n') line_len--;
		if (line[line_len-1] == '\r') line_len--;

		col_count = 0;
		ncol = 0;
		in_quote = 0;
		for (found=0, count=0, i=0; i<line_len; i++) {
			if (line[i] == ' ' && !found)
				continue;   /* skip leading spaces */
			if ((in_quote == 0) &&
			    (strncmp(line+i, delim, delim_len) == 0)) {
				/* Found end of token */
				token[count++] = '\0';
				if ((data+ncol)->use) {
					if (ncol >= in_ncol) {
						fclose(pf);
						if (delim)
							free(delim);
						free_data(data, in_ncol);
						if (data)
							free(data);
						ERRerror("DVtable_read",4,ERR_ORIG,
							 "bad number of columns in file %s on line %d: %d instead %d",
							 filename, line_count+1, ncol+1, ndata);
						if (filename)
							free(filename);
						return(0);
					}
					read_token(data+ncol, token);
					col_count++;
				}
				ncol++;
				count = 0;
				i += delim_len-1;
				found = 0;
			}
			else {
				/* Not end of token */
				if (line[i] == '\"')
					/* not copying the quote drops it. */
					in_quote = !in_quote;
				else
					/* copy single char into token */
					token[count++] = line[i];
				found = 1;
				/* End of line */
				if (i==line_len-1) {
					token[count++] = '\0';
					if ((data+ncol)->use) {
						if (ncol >= in_ncol) {
							fclose(pf);
							if (delim)
								free(delim);
							free_data(data, in_ncol);
							if (data)
								free(data);
							ERRerror("DVtable_read",4,ERR_ORIG,
								 "bad number of columns in file %s on line %d: %d instead %d",
								 filename, line_count+1, ncol+1, ndata);
							if (filename)
								free(filename);
							return(0);
						}
						read_token(data+ncol, token);
						col_count++;
					}
				}
			}
		}
		if (col_count != ndata) {
			/* add null data to the rest of the columns */
			for (i=col_count; i<ndata; i++)
				read_token(data+i, "");
		}
		line_count++;
	}
	fclose(pf);
	if (filename)
		free(filename);

	for (nrow=0, count=0, i=0; i<in_ncol; i++) {
		if ((data+i)->use == 0)
			continue;

		pdata = data+i;

		if (count) {
			if (nrow != pdata->count) {
				fclose(pf);
				if (filename)
					free(filename);
				if (delim)
					free(delim);
				free_data(data, in_ncol);
				if (data)
					free(data);
				ERR_RETURN("different number of rows in columns");
			}
			nrow = pdata->count;
		}
		else
			nrow = pdata->count;

		if (OMget_array_val(out, count, &val_id, OM_OBJ_RW) != 1) {
			ERR_RETURN("cannot get out array member");
		}
		if (OMset_name_int_val(val_id, OMstr_to_name("nvals"), pdata->count) != 1) {
			ERR_RETURN("cannot set out nvals");
		}
		if (OMset_name_str_val(val_id, OMstr_to_name("labels"), pdata->name) != 1) {
			ERR_RETURN("cannot set out labels");
		}

		e_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RW);

		stat = OMset_data_type(e_id, pdata->type);
		if (stat != 1) {
			ERR_RETURN("cannot set out values type");
		}
		if (pdata->type == DTYPE_STRING) {
			for (j=0; j<pdata->count; j++)
				if (OMset_str_array_val(e_id, j, pdata->strings[j]) != 1) {
					ERR_RETURN("cannot set string data");
				}
		}
		else {
			stat = OMset_array(e_id, pdata->type, pdata->values, pdata->count, OM_SET_ARRAY_FREE);
			if (stat != 1) {
				ERR_RETURN("cannot set out data values");
			}
		}

		if (OMset_name_int_val(val_id, OMstr_to_name("null_flag"), pdata->null_flag) != 1) {
			ERR_RETURN("cannot set out null_flag");
		}
		if (pdata->null_flag)
			if (OMset_name_real_val(val_id, OMstr_to_name("null_value"), pdata->null_value) != 1) {
				ERR_RETURN("cannot set out null_value");
			}
		count++;
	}

	/* Generate index column */

	if (gen_index) {
		int ndim, dtype;
		xp_long dims[OM_ARRAY_MAXDIM];
		float *arr;

		OMset_array_size(out, ndata+1);

		if (OMget_array_val(out, ndata, &val_id, OM_OBJ_RW) != 1) {
			ERR_RETURN("cannot get out array member");
		}
		if (OMset_name_int_val(val_id, OMstr_to_name("nvals"), nrow) != 1) {
			ERR_RETURN("cannot set out nvals");
		}
		if (OMset_name_str_val(val_id, OMstr_to_name("labels"), "index") != 1) {
			ERR_RETURN("cannot set out labels");
		}

		e_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RW);

		/* Intended to be used as coordinate data, which is always float */
		stat = OMset_data_type(e_id, DTYPE_FLOAT);
		if (stat != 1) {
			ERR_RETURN("cannot set out values type");
		}

		arr = NULL;
		ndim = 0; /* should always get back two */
		dtype = DTYPE_FLOAT;
		if(OMget_array(e_id, &dtype, (char **)&arr, &ndim, dims, OM_GET_ARRAY_RW) == 1) {
			for(i = 0; i < nrow; ++i ) arr[i] = (float)i;
			ARRfree(arr);
			ndata++;
		}
		else
			OMset_array_size(out, ndata);
	}

	if (OMset_name_int_val(elem_id, OMstr_to_name("tbl_nrow"), nrow) != 1) {
		ERR_RETURN("cannot set tbl_nrow");
	}
	if (OMset_name_int_val(elem_id, OMstr_to_name("tbl_ncol"), ndata) != 1) {
		ERR_RETURN("cannot set tbl_ncol");
	}
	if (delim)
		free(delim);

	free_data(data, in_ncol);

	if (data)
		free(data);
	return(1);
}


typedef struct _vec_ptr {
    void **vector;	/* vector of pointers */
    int  size;		/* active length */
    int  capacity;	/* allocated length */
} UTILvec_ptr_t;


void UTILvec_ptr_construct( UTILvec_ptr_t *uvec, int initial_size )
{
    uvec->vector = malloc( initial_size * sizeof(void *) );
    uvec->size = 0;
    uvec->capacity = initial_size;
}

UTILvec_ptr_t * UTILvec_ptr_new( int initial_size )
{
    UTILvec_ptr_t *uvec = malloc( sizeof(UTILvec_ptr_t) );
    UTILvec_ptr_construct( uvec, initial_size );
    return uvec;
}

void UTILvec_ptr_push_back( UTILvec_ptr_t *uvec, void *element )
{
    if( uvec->size == uvec->capacity ) {
        int new_capacity = uvec->capacity * 1.5;
        if( new_capacity == uvec->capacity ) new_capacity += 1;
        uvec->vector = realloc( uvec->vector, new_capacity * sizeof(void *) );
        uvec->capacity = new_capacity;
    }
    uvec->vector[uvec->size] = element;
    uvec->size += 1;
}

void UTILvec_ptr_destruct( UTILvec_ptr_t *uvec )
{
    if( uvec->vector != NULL ) {
        free( uvec->vector );
        uvec->vector = NULL;
    }
    uvec->size     = 0;
    uvec->capacity = 0;
}


#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("DVgrid_read", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVgrid_read_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id file_id, out_id, x_labels_id, y_labels_id;
    FILE *pf = NULL;
    char *filename = NULL, *delim = NULL, line[LINE_LEN];
    int delim_len, col_header, row_header, skip_lines;
    int i, line_len, count, in_quote;
    int data_ncol, data_nrow;	/* NOT including headers */
    int tbl_ncol;		/* including headers */
    int dtype;			/* data type of, err, the data.  The headers
                                 * are always treated as strings. */
    const char *format = NULL;	/* Use this to sscanf a data element */

    UTILvec_ptr_t col_labels;	/* headers across top of table */
    UTILvec_ptr_t row_labels;	/* headers in left column */
    UTILvec_ptr_t rows;		/* contents of the rows */

    UTILvec_ptr_construct( &col_labels, 10 );
    UTILvec_ptr_construct( &row_labels, 10 );
    UTILvec_ptr_construct( &rows, 10 );

    /* Input parameters */

    file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
    if (OMget_str_val(file_id, &filename, 0) != 1)
        return 0;
    if (filename[0] == '\0')
        return 1;

    if (OMget_name_str_val(elem_id, OMstr_to_name("delimiter"), &delim, 0) != 1)
        delim = strdup(" ");
    delim_len = (int)strlen(delim);
    if (OMget_name_int_val(elem_id, OMstr_to_name("col_header"), &col_header) != 1)
        col_header = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("row_header"), &row_header) != 1)
        row_header = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("skip_lines"), &skip_lines) != 1)
        skip_lines = 0;
    if (OMget_name_int_val(elem_id, OMstr_to_name("data_type"), &dtype) != 1)
        dtype = DTYPE_UNSET;
    else if( dtype >= 0 && dtype < DTYPE_UNSET )
        format = formats[dtype];
    if( dtype != DTYPE_UNSET && (dtype < 0 || dtype > DTYPE_LONG) ) {
        if(delim) free(delim);
        free(filename);
        ERR_RETURN("Data type must be numerical");
     }

    {
        OMobj_id do_id = OMfind_subobj(elem_id, OMstr_to_name("do"), OM_OBJ_RD);

        /* OK if missing entirely, means we are running in non-UI mode. */
        if( !OMis_null_obj(do_id) ) {
            if( !(OM_EVENT_INST & event_mask) && !OMchanged( do_id, seq_num ) ) {
                if(delim) free(delim);
                free(filename);
                return 1;
            }
        }
    }

    /* Output parameters */

    out_id = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    x_labels_id = OMfind_subobj(elem_id, OMstr_to_name("x_labels"), OM_OBJ_RW);
    y_labels_id = OMfind_subobj(elem_id, OMstr_to_name("y_labels"), OM_OBJ_RW);

    /* Initialize output */

    OMset_array_size(x_labels_id, 0);
    OMset_array_size(y_labels_id, 0);

    FLDset_nnodes( out_id, 0 );
    FLDset_nspace( out_id, 0 );
    FLDset_ndim  ( out_id, 0 );
    FLDset_node_data_ncomp( out_id, 0 );

    if (!(pf = FILEfopen(filename, SIO_R_BIN))) {
        ERRerror("DVgrid_read", 1, ERR_ORIG, "Can't open data file: %s",
                 filename);
        if(delim) free(delim);
        free(filename);
        return 0;
    }
    if (filename) {
        free(filename);
        filename = NULL;
    }

    /* Skip initial lines. */
    for (i=0; i<skip_lines; i++) {
        if (fgets(line, LINE_LEN, pf) == NULL) {
            fclose(pf);
            ERR_RETURN("Error skipping lines");
        }
    }

    data_ncol = tbl_ncol = 0;

    /* First, read the column header if present */

    if (col_header) {
        if (fgets(line, LINE_LEN, pf) == NULL) {
            fclose(pf);
            ERR_RETURN("Error reading header");
        }
        line_len = (int)strlen(line);
        if (line[line_len-1] == '\r') line_len--;
        if (line[line_len-1] == '\n') line_len--;
        if (line[line_len-1] == '\r') line_len--;

        /* I'm not going to attempt to quote line breaks */
        in_quote = 0;

        /* i is line index, count is token index */
        for (count = 0, i = 0; i <= line_len; i++) {
            char token[256];
            const int token_limit = (sizeof(token)/sizeof(char)) - 1;

            if ((line[i] == ' ') && (in_quote==0) && (count == 0))
                continue;	/* skip leading spaces */

            if ((i==line_len) && (count == 0))
                break;		/* line ends with a delimiter */

            /* found delimiter or at EOL or about to overflow token */
            if ((count == token_limit) ||
                (i==line_len) ||
                ((in_quote == 0) && (strncmp(line+i, delim, delim_len) == 0))) {
                token[count++] = '\0';

                /* If there are row headers, then the first column header
                 * is a placeholder, typically "ID", which can be skipped.
                 * But the placeholder might be empty, so we have
                 * to accept blank labels as valid.
                 */
                if (tbl_ncol > 0 || row_header == 0) {
                    UTILvec_ptr_push_back( &col_labels, strdup(token) );
                }

                ++tbl_ncol;	/* total columns in file */
                count = 0;
                i += delim_len-1;
            }
            else {
                /* strip quotes, but if two in a row, leave one in the token */
                if( line[i] == '\"' ) {
                    if( (i+1 < line_len) && (line[i+1] == '\"') ) {
                        /* but if two quotes in a row, put one in the token */
                        token[count++] = '\"';
                        ++i;
                    }
                    else {
                        /* simply not copying the quote drops it. */
                        in_quote = !in_quote;
                    }
                }
                else {
                    /* copy single char into token */
                    token[count++] = line[i];
                }
            }
        }
        if( row_header ) data_ncol = tbl_ncol - 1;
        else             data_ncol = tbl_ncol;
    }

    /* Next, read the the main body of the table */

    data_nrow  = 0;
    while (fgets(line, LINE_LEN, pf) != NULL) {
        char token[256];
        const int token_limit = (sizeof(token)/sizeof(char)) - 1;
        char *row_buffer = NULL;
        int tbl_col_i;	/* running column index, not including row header */
        int data_col_i;	/* running column index, including row header */

        line_len = (int)strlen(line);
        if (line[line_len-1] == '\r') line_len--;
        if (line[line_len-1] == '\n') line_len--;
        if (line[line_len-1] == '\r') line_len--;
        if (line_len == 0) continue;  /* Skip blank lines */

        /* Allocating buffers will be a lot simpler if we know the
         * number of columns ahead of time.  Usually we should only
         * need this when there is no column header.  We also need
         * to guess at the data type if that has not been established yet.
         */
        if( data_ncol == 0 ) {
            in_quote = 0;
            /* i is line index, count is token index */
            for (count = 0, i = 0; i <= line_len; i++) {
                if ((line[i] == ' ') && (in_quote==0) && (count == 0))
                    continue;	/* skip leading spaces */

                if ((i==line_len) && (count == 0))
                    break;		/* line ends with a delimiter */

                /* found delimiter or at EOL or about to overflow token */
                if ((count == token_limit) ||
                    (i==line_len) ||
                    ((in_quote == 0) && (strncmp(line+i, delim, delim_len) == 0))) {
                    token[count++] = '\0';

                    /* Don't use the row header to determine the data type */
                    if (count>1 &&
                        (row_header == 0) || (tbl_ncol != 0)) {
                        /* Look at type of token ... */
                        if( dtype < 0 || dtype == DTYPE_UNSET ) {
                            dtype = get_type( token );
                            if( dtype >= 0 && dtype < DTYPE_UNSET )
                                format = formats[dtype];
                            if( dtype < 0 || dtype > DTYPE_LONG ) {
                                fclose(pf);
                                if(delim) free(delim);
                                ERR_RETURN("Data type must be numerical");
                            }
                        }
                    }

                    ++tbl_ncol;	/* total columns in file */
                    count = 0;
                    i += delim_len-1;
                }
                else {
                    /* strip quotes, but if two in a row, leave one */
                    if( line[i] == '\"' ) {
                        if( (i+1 < line_len) && (line[i+1] == '\"') ) {
                            /* but if two quotes in a row, put one in the token */
                            token[count++] = '\"';
                            ++i;
                        }
                        else {
                            /* simply not copying the quote drops it. */
                            in_quote = !in_quote;
                        }
                    }
                    else {
                        /* copy single char into token */
                        token[count++] = line[i];
                    }
                }
            }
            if( row_header ) data_ncol = tbl_ncol - 1;
            else             data_ncol = tbl_ncol;
        }

        if( data_ncol == 0 ) {
            fclose(pf);
            if(delim) free(delim);
            ERR_RETURN("Cannot determine number of columns");
        }

        in_quote = 0;

        /* i is line index, count is token index */
        for (count = 0, i = 0, data_col_i = 0, tbl_col_i = 0;
             (i <= line_len) && (data_col_i < data_ncol);
             ++i) {

            if ((line[i] == ' ') && (in_quote==0) && (count == 0))
                continue;	/* skip leading spaces */

            if ((i==line_len) && (count == 0))
                break;		/* line ends with a delimiter */

            /* Found delimiter or at EOL or about to overflow token
             * - process the token
             */
            if ((count == token_limit) ||
                (i==line_len) ||
                (in_quote == 0 && (strncmp(line+i, delim, delim_len) == 0))) {
                token[count++] = '\0';

                /* First column is a row header */
                if ((tbl_col_i == 0) && (row_header != 0)) {
                    UTILvec_ptr_push_back( &row_labels,  strdup(token) );
                }
                else {
                    char *ptr;	/* just a temp to clean up the code */

                    /* Gotta establish the data type before we can allcoate
                     * buffers and start parsing.
                     */
                    if( dtype < 0 || dtype == DTYPE_UNSET ) {
                        /* Peek at the first token to determine type. */
                        if( count > 0 ) {
                            dtype = get_type( token );
                        }
                        if( dtype >= 0 && dtype < DTYPE_UNSET ) {
                            format = formats[dtype];
                        }
                        else if( dtype < 0 || dtype > DTYPE_LONG ) {
                            fclose(pf);
                            if(delim) free(delim);
                            ERR_RETURN("Data type must be numerical");
                        }
                    }

                    if( row_buffer == NULL ) {
                        size_t sz = data_ncol * DTYPEtype_size[dtype];
                        row_buffer = malloc( sz );
                        /* Crude null-data handling - you'll always get zero. */
                        memset( row_buffer, 0, sz );
                    }

                    if( count > 1 ) {
                        /* If the sscanf fails, the buffer has been pre-filled
                         * with zeros.
                         */
                        ptr = row_buffer+(data_col_i*DTYPEtype_size[dtype]);
                        sscanf( token, format, ptr );
                    }

                    ++data_col_i;
                }

                ++tbl_col_i;
                count = 0;
                i += delim_len-1;
            }
            else {
                /* strip quotes */
                if( line[i] == '\"' ) {
                    if( (i+1 < line_len) && (line[i+1] == '\"') ) {
                        /* but if two quotes in a row, leave one in the token */
                        token[count++] = '\"';
                        ++i;
                    }
                    else {
                        /* simply not copying the quote drops it. */
                        in_quote = !in_quote;
                    }
                }
                else {
                    /* copy single char into token */
                    token[count++] = line[i];
                }
            }
        } /* loop over characters in the line */

        /* finished with the line, save it away */
        if( row_buffer != NULL ) {
            UTILvec_ptr_push_back( &rows, row_buffer );
            row_buffer = NULL;
            ++data_nrow;
        }
    } /* loop over lines */

    if(delim) free(delim);
    fclose(pf);

    /* Create the output field */
    {
        xp_long nnodes = data_nrow * data_ncol;
        xp_long size, dims[OM_ARRAY_MAXDIM];
        int type_tmp = dtype, i;
        char *node_data = NULL;
        float *points = NULL;

        FLDset_nnodes( out_id, nnodes );
        FLDset_nspace( out_id, 2 );	/* Hardwired at 2D for now. */
        FLDset_ndim  ( out_id, 2 );	/* Hardwired at 2D for now. */
        dims[0] = data_ncol;
        dims[1] = data_nrow;
        FLDset_dims( out_id, dims );

        if (FLDget_points( out_id, &points, &size, OM_GET_ARRAY_RW ) == 1 ) {
            points[0] = 0.0;
            points[1] = 0.0;
            points[2] = data_ncol - 1;
            points[3] = data_nrow - 1;
            ARRfree( points );
        }

        FLDset_node_data_ncomp ( out_id, 1 );
        FLDset_node_data_veclen( out_id, 0, 1 );
        FLDset_node_data_type  ( out_id, 0, dtype );
        if (FLDget_node_data( out_id, 0, &type_tmp, &node_data, &size,
                              OM_GET_ARRAY_RW ) == 1 ) {
            for( i = 0; i < data_nrow; ++i ) {
                char *src_row = rows.vector[i];
                char *dst_row = node_data + (i*data_ncol*DTYPEtype_size[dtype]);
                int bytes_to_copy =  data_ncol * DTYPEtype_size[dtype];
                memcpy( dst_row, src_row, bytes_to_copy );
            }
            if( node_data != NULL ) ARRfree( node_data );
        }
    }

    if( col_header ) {
        OMset_array_size(x_labels_id, data_ncol);
        for( i = 0; i < data_ncol; ++i ) {
            OMset_str_array_val(x_labels_id, i, col_labels.vector[i] );
        }
    }
    if( row_header ) {
        OMset_array_size(y_labels_id, data_nrow);
        for( i = 0; i < data_nrow; ++i ) {
            OMset_str_array_val(y_labels_id, i, row_labels.vector[i] );
        }
    }

    for( i = 0; i < rows.size; ++i ) free( rows.vector[i] );
    UTILvec_ptr_destruct( &rows );
    for( i = 0; i < row_labels.size; ++i ) free( row_labels.vector[i] );
    UTILvec_ptr_destruct( &row_labels );
    for( i = 0; i < col_labels.size; ++i ) free( col_labels.vector[i] );
    UTILvec_ptr_destruct( &col_labels );

    return 1;
}

/* 64-bit porting. Only Modified Internally */
static int get_type(const char *token)
{
	int l,i,type, real_type;

	real_type = 0;
	l = (int)strlen(token);
	for (i=0; i<l; i++) {
		if (token[i] == '\n')
			break;
		if ((token[i]>='0' && token[i]<='9') || token[i] == '+' || token[i] == '-' || token[i] == ' ')
			continue;
		else if (token[i] == '.' || token[i] =='E' || token[i] =='e')
			real_type = 1;
		else {
			type = DTYPE_STRING;
			return(type);
		}
	}
	if (real_type)
		type = DTYPE_DOUBLE;
	else
	{
		if (atol(token) <= AVS_INT_MAX)
			type = DTYPE_INT;
		else
			type = DTYPE_LONG;
	}

	return(type);
}

/* 64-bit porting. Only Modified Internally */
static int read_token (TBLdata *data, const char *token)
{
	const char *format;

	format = formats[data->type];

	if ( data->type != DTYPE_STRING && data->count >= data->size) {
		data->size += ALLOC_BLOCK;
		data->values = (char *)ARRrealloc(data->values, data->type, data->size, NULL);
		if (data->values == NULL) {
			ERR_RETURN("cannot allocate data");
		}
	}
	if (strlen(token) == 0) {
		switch (data->type) {
		      case DTYPE_BYTE:
		      case DTYPE_CHAR:
		{
			char *dvalues;

			dvalues = (char *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_SHORT:
		{
			short *dvalues;

			dvalues = (short *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_INT:
		{
			int *dvalues;

			dvalues = (int *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_LONG:
		{
			xp_long *dvalues;

			dvalues = (xp_long *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_FLOAT:
		{
			float *dvalues;

			dvalues = (float *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_DOUBLE:
		{
			double *dvalues;

			dvalues = (double *)data->values;
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
			(data->count)++;
			break;
		}
		      case DTYPE_STRING:
			if (data->count >= data->size) {
				data->size += ALLOC_BLOCK;
				data->strings = (char **)realloc(data->strings, sizeof(char *)*data->size);
				if (data->strings == NULL) {
					ERR_RETURN("cannot allocate data");
				}
			}
			data->strings[(data->count)] = strdup("");
			data->null_flag = 1;
			(data->count)++;
			break;

		      default:
			return (0);
		}
		return(1);
	}

	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	{
		char *dvalues;

		dvalues = (char *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;
		break;
	}
	      case DTYPE_SHORT:
	{
		short *dvalues;

		dvalues = (short *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;

		break;
	}
	      case DTYPE_INT:
	{
		int *dvalues;

		dvalues = (int *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;

		break;
	}
	      case DTYPE_LONG:
	{
		xp_long *dvalues;

		dvalues = (xp_long *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;

		break;
	}
	      case DTYPE_FLOAT:
	{
		float *dvalues;

		dvalues = (float *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;

		break;
	}
	      case DTYPE_DOUBLE:
	{
		double *dvalues;

		dvalues = (double *)data->values;
		if (sscanf(token, format, dvalues+(data->count)) != 1) {
			data->null_flag = 1;
			dvalues[(data->count)] = data->null_value;
		}
		(data->count)++;

		break;
	}
	      case DTYPE_STRING:
		if (data->count >= data->size) {
			data->size += ALLOC_BLOCK;
			data->strings = (char **)realloc(data->strings, sizeof(char *)*data->size);
			if (data->strings == NULL) {
				ERR_RETURN("cannot allocate data");
			}
		}
/* CFS case 15853 sscanf will chop off strings with spaces
		if (sscanf(token, format, str) != 1) {
			data->strings[(data->count)] = strdup("");
			data->null_flag = 1;
		}
		else
*/
		data->strings[(data->count)] = strdup(token);
		(data->count)++;
		break;

	      default:
		return (0);
	}
	return(1);
}

static int free_data (TBLdata *data, int ndata)
{
	int i, j;
	TBLdata *pd;

	for (i=0; i<ndata; i++) {
		pd = data+i;
		if (pd->use == 0)
			continue;
		if (pd->type == DTYPE_STRING)  {
			for (j=0; j<pd->count; j++)
				free(pd->strings[j]);
			free(pd->strings);
		}
		free(pd->name);
	}
	return(1);
}
