/*
			Copyright (c) 2003 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_tri.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define LINE_LEN 1024

#define ERR_RETURN(A) {ERRerror("read_triangle", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#ifndef GD_COLOR_DATA_ID
#define GD_COLOR_DATA_ID 667
#endif

/* RAW and TIN files. */
static int FUNCread_triangle_RAW( FILE *fp, OMobj_id fld, int tin );

/* STL and SLP files */
static int FUNCread_triangle_STL( FILE *fp, OMobj_id fld );

int
DVread_triangle_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, fld_id;
    FILE *fp = NULL;
    char *temp_str = NULL, *filename = NULL, *ext = NULL;
    char file_buf[AVS_PATH_MAX];

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );

    /* out */
    fld_id = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if( temp_str ) { free (temp_str); temp_str = NULL; }

    if( filename != NULL && filename[0] != 0 )
        fp = FILEfopen( filename, SIO_R_BIN );

    if( fp == NULL ) {
        ERRerror( "read_triangle", 1, ERR_ORIG, "Can't open data file: %s",
                  filename );
        return METHOD_FAILURE;
    }

    ext = FILEget_file_extension( filename );
    ext += 1;	/* skip over '.' */

    /* Basic (shared) field setup */
    FLDset_nnodes( fld_id, 0 );
    FLDset_ncell_sets( fld_id, 0 );
    /* Set in V code */
    /* FLDset_nspace( fld_id, 3 ); */

    if( !strcmp( ext, "RAW" ) || !strcmp( ext, "raw" ) ) {
        FUNCread_triangle_RAW( fp, fld_id, 0 );
    }
    else if( !strcmp( ext, "SLP" ) || !strcmp( ext, "slp" ) ) {
        FUNCread_triangle_STL( fp, fld_id );	/* SLP mode bit? */
    }
    else if( !strcmp( ext, "STL" ) || !strcmp( ext, "stl" ) ) {
        FUNCread_triangle_STL( fp, fld_id );
    }
    /* Note that MicroAVS comes with a "sla" file that seems to be
     * an ASCII STL file.
     */
    else if( !strcmp( ext, "STLA" ) || !strcmp( ext, "stla" ) ||
             !strcmp( ext, "SLA" )  || !strcmp( ext, "sla" ) ) {
        FUNCread_triangle_STL( fp, fld_id );	/* ASCII mode bit? */
    }
    else if( !strcmp( ext, "TIN" ) || !strcmp( ext, "tin" ) ) {
        FUNCread_triangle_RAW( fp, fld_id, 1 );
    }
    else {
        ERRerror( "read_triangle", 1, ERR_ORIG,
                  "Unknown file extension: %s", ext );
    }

    fclose( fp );

    return METHOD_SUCCESS;
}


/*
 * Shared infrastructure for the triangle readers.
 * Data structures and methods to hold the verticies and
 * connection lists and more.
 */

/* 
 * The point of all these vertex data structures is to catch
 * verticies that are shared between triangles.
 *
 * In order to handle a large number of triangles without a O(n^2)
 * slowdown, this code uses randomized binary search trees
 * to implement a O(n*log(n)) alorgithm to speed the process.
 */

typedef struct _Vertex {
    float xyz[3];
    xp_long index;	/* Does double duty as the search priority */

    /* Overhead to implement the searching algorithm */
    /* int priority; */
    struct _Vertex *left;
    struct _Vertex *right;
    struct _Vertex *parent;
} Vertex;

static void vertex_init( Vertex *v, float *xyz )
{
    if( xyz != NULL ) {
        v->xyz[0] = xyz[0];
        v->xyz[1] = xyz[1];
        v->xyz[2] = xyz[2];
    }

    v->index = rand();
    v->left   = NULL;
    v->right  = NULL;
    v->parent = NULL;
}

static void vertex_tree_free( Vertex *v )
{
    Vertex *tmp;
    while( v != NULL ) {
        if( v->left )
            vertex_tree_free( v->left );
        tmp = v->right;
        free( v );
        v = tmp;	/* OK if NULL, thats the end of the loop */
    }
}

#if 0
/* 64-bit porting. Only Modified Internally */
static void vertex_tree_dump( Vertex *v )
{
    printf( "%3ld %f %d %p [l:%3ld %p, r:%3ld %p]\n",
            v->index, v->xyz[0], v->priority, v,
            v->left?v->left->index:-1, v->left,
            v->right?v->right->index:-1, v->right );

    if( v->left )
        vertex_tree_dump( v->left );

    if( v->right )
        vertex_tree_dump( v->right );
}
#endif

static void vertex_tree_rotate_right( Vertex *y )
{
    Vertex *x = y->left;
    Vertex *p = y->parent;

    y->left = x->right;

    if( y->left != NULL )
        y->left->parent = y;

    if( p != NULL ) {	/* NULL when y is (old) root */
        if( p->right == y )
            p->right = x;
        else
            p->left = x;
    }

    x->parent = p;
    x->right  = y;
    y->parent = x;	/* x might be new root */
}

static void vertex_tree_rotate_left( Vertex *x )
{
    Vertex *y = x->right;
    Vertex *p = x->parent;

    x->right = y->left;

    if( x->right != NULL )
        x->right->parent = x;

    if( p != NULL ) {	/* NULL when x is (old) root */
        if( p->left == x )
            p->left = y;
        else
            p->right = y;
    }

    y->parent = p;
    y->left   = x;
    x->parent = y;	/* y might be new root */
}

static void vertex_tree_bubble_up( Vertex *v )
{
    Vertex *p;
    while( 1 ) {
        p = v->parent;
        if( p != NULL && v->index < p->index ) {
            if( p->left == v )
                vertex_tree_rotate_right( p );
            else
                vertex_tree_rotate_left( p );
        }
        else break;
    }
}

static int vertex_compare( float *xyz1, float *xyz2 )
{
    if( xyz1[0] < xyz2[0] ) return -1;
    if( xyz1[0] > xyz2[0] ) return  1;

    if( xyz1[1] < xyz2[1] ) return -1;
    if( xyz1[1] > xyz2[1] ) return  1;

    if( xyz1[2] < xyz2[2] ) return -1;
    if( xyz1[2] > xyz2[2] ) return  1;

    return  0;
}

static Vertex* vertex_tree_add( Vertex *v, float *xyz )
{
    int compare;
    Vertex* p;

    while( v != NULL ) {	/* not expecting this test to fail */
        compare = vertex_compare( v->xyz, xyz );
        if( compare == 0 ) {
            /* Exact hit ... we've found what we are looking for */
            return v;
        }
        else if( compare > 0 ) {
            /* current is bigger then the new, so look left */
            p = v;
            v = v->left;
        }
        else if( compare < 0 ) {
            /* current is smaller than the new, so look right */
            p = v;
            v = v->right;
        }
        if( v == NULL ) {
            /* No child to search, so create a new one */
            v = malloc( sizeof(Vertex) );
            vertex_init( v, xyz );
            if( compare > 0 ) p->left  = v;
            else              p->right = v;
            v->parent = p;
            vertex_tree_bubble_up( v );
            /* After rebalancing the tree, return the new node */
            return v;
        }
    } /* while */

    return NULL;	/* not expecting this */
}

/* 
 * v_t_count and v_t_coords need to be in exact sync in the 
 * order that they visit the nodes.
 */
/* 64-bit porting. Directly Modified */
static void vertex_tree_count( Vertex *v, xp_long * count )
{
    v->index = *count;	/* used for node connect lists */
    *count += 1;	/* count myself */

    if( v->left )
        vertex_tree_count( v->left, count );
    if( v->right )
        vertex_tree_count( v->right, count );
}
/* 64-bit porting. Only Modified Internally */
static void vertex_tree_coords( Vertex *v, float *coords )
{
    xp_long where = v->index*3;
    coords[where+0] = v->xyz[0];
    coords[where+1] = v->xyz[1];
    coords[where+2] = v->xyz[2];

    if( v->left )
        vertex_tree_coords( v->left, coords );
    if( v->right )
        vertex_tree_coords( v->right, coords );
}

/*
 * Vertex Set disconnected ... we need to connect the triangles,
 * finding duplicate points as we go along.
 */

typedef struct _Vset_d {
    Vertex *root;
} Vset_d;

static void vset_d_init( Vset_d *vset )
{
    vset->root = NULL;
}

static void vset_d_free( Vset_d *vset )
{
    if( vset != NULL && vset->root != NULL )
        vertex_tree_free( vset->root );
}

/* 
 * Return the total number of nodes we have collected and
 * additionally fill in the node index counters that will
 * be needed to construct the node connect lists.
 */
/* 64-bit porting. Directly Modified */ 
static xp_long vset_d_count( Vset_d *vset )
{
    if( vset != NULL && vset->root != NULL ) {
        xp_long count = 0;
        vertex_tree_count( vset->root, &count );
        return count;
    }
    else return 0;
}

/* Fill in the coord list you get from FLDget_coords */
static void vset_d_coords( Vset_d *vset, float *coords )
{
    if( vset != NULL && vset->root != NULL ) {
        vertex_tree_coords( vset->root, coords );
    }
}

static Vertex *vset_d_add( Vset_d *vset, float xyz[3] )
{
    if( vset->root == NULL ) {
        vset->root = malloc( sizeof(Vertex) );
        vertex_init( vset->root, xyz );
        return vset->root;
    }
    else {
        Vertex *ret = vertex_tree_add( vset->root, xyz );
        /* There is a chance the new guy is the new root. */
        if( ret->parent == NULL ) vset->root = ret;
        return ret;
    }
}

/* Data structures to handle cell sets and their connection lists.
 * The RAW format can have multiple 'objects', which are mapped
 * to multple cell sets, so extra infrastucture is needed to
 * manage the cell set information.
 */

static const int cellset_chunk = 2048;

typedef struct {
    xp_long ncells;
    xp_long nnodes;		/* help keep track of incremental adds */
    char *name;

    float *colors;	/* rgb (3 floats) color per cell */
    xp_long colors_alloc;

    /* Pointers to vert objects, *not* integer connection indicies. */
    Vertex **vertex_conn;
    xp_long vertex_alloc;
} Cellset;

static void cellset_init( Cellset *cset, const char *name )
{
    cset->ncells = 0;
    cset->nnodes = 0;
    if( name != NULL && name[0] != 0 )
        cset->name = strdup( name );
    else
        cset->name = NULL;
    cset->colors_alloc = 0;
    cset->colors       = NULL;
    cset->vertex_alloc = cellset_chunk;
    cset->vertex_conn  = malloc( cset->vertex_alloc * sizeof(Vertex *) );
}

static void cellset_free( Cellset *cset )
{
    if( cset->name ) free( cset->name );
    if( cset->vertex_conn ) free( cset->vertex_conn );
    if( cset->colors ) free( cset->colors );
}

/* In some of the file formats, you see all the verticies at the same
 * time, so you can create the triangle in one step.
 */
static void cellset_add_tri(
    Cellset *cset, Vertex *v1, Vertex *v2, Vertex *v3 )
{
    if( (cset->ncells+1)*3 >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof( Vertex *) );
    }

    /* The pointers should still be valid as we shuffle around
     * the search tree.
     */
    cset->vertex_conn[(cset->ncells*3)+0] = v1;
    cset->vertex_conn[(cset->ncells*3)+1] = v2;
    cset->vertex_conn[(cset->ncells*3)+2] = v3;

    cset->ncells += 1;
    cset->nnodes += 3;
}

static void cellset_add_colors( Cellset *cset, float *rgb )
{
    if( cset->colors_alloc == 0 ) {
        cset->colors_alloc = cellset_chunk;
        cset->colors = malloc( cset->colors_alloc * sizeof( float *) );
    }
    else if( (cset->ncells+1)*3 >= cset->colors_alloc ) {
        cset->colors_alloc += cellset_chunk;
        cset->colors = realloc( cset->colors, cset->colors_alloc * sizeof(float) );
    }

    cset->colors[(cset->ncells*3)+0] = rgb[0];
    cset->colors[(cset->ncells*3)+1] = rgb[1];
    cset->colors[(cset->ncells*3)+2] = rgb[2];
}

/* Fill in the connection list you get from FLDget_node_connect */
/* 64-bit porting. Directly Modified */
static void cellset_conn( Cellset *cset, xp_long *conn_list )
{
    xp_long i;
    for( i = 0; i < cset->ncells*3; ++i ) {
        conn_list[i] = cset->vertex_conn[i]->index;
    }
}

/*
 * None of these file formats have what AVS/Express calls node data,
 * so we can regard this as a pure cell set field.
 */
typedef struct {
    int ncsets;
    Cellset *list;
} Mesh;

static void mesh_init( Mesh *mesh )
{
    mesh->list = NULL;
    mesh->ncsets = 0;
}

static int mesh_ncsets( Mesh *mesh )
{
    return mesh->ncsets;
}

static Cellset *mesh_get( Mesh *mesh, int index )
{
    return &(mesh->list[index]);
}

static Cellset *mesh_add( Mesh *mesh, const char *name )
{
    Cellset *cset = NULL;
    if( mesh == NULL ) return NULL;

    if( mesh->ncsets == 0 ) {
        mesh->list = malloc( sizeof( Cellset ) );
    }
    else {
        /* Not very efficient, but there shouldn't be many cellsets */
        mesh->list = realloc( mesh->list, (mesh->ncsets+1) * sizeof( Cellset ) );
    }
    cset = &(mesh->list[mesh->ncsets]);
    cellset_init( cset, name );
    mesh->ncsets += 1;
    return cset;
}

static void mesh_free( Mesh *mesh )
{
    if( mesh->list ) {
        int i;
        for( i = 0; i < mesh->ncsets; ++i ) {
            cellset_free( &(mesh->list[i]) );
        }
        free( mesh->list );
    }
}

#define MODE_UNKNOWN 0
#define MODE_DEFAULT 1
#define MODE_TEXTURE 2
#define MODE_COLORS  3
#define MODE_TIN     4	/* OK, this is really an entirely different format */

/*
 * "This format is used primarily for inputting triangular facet geometry
 * through the POV utility RAW2POV."
 *
 * Default  - Nine numbers per triangle (one float per vertex).
 * +color   - Twelve numbers per triangle (rgb color, then triangle verticies).
 * +texture - Nine numbers per triangle plus texture name.
 *
 * Non-numeric identifiers can be used to separate groups
 * of triangles representing different objects.
 *
 * Also support for simple TIN (triangular irregular network) files,
 * which are pretty similar to RAW files.  The term "TIN" gets applied to
 * several diferent formats; this just reads the simple kind that have lines
 * that look like ...
 *
 * t x0 y0 z0 x1 y1 z1 x2 y2 z2
 *
 * So its just like RAW format except with the leading "t".
 *
 */

/* 64-bit porting. Only Modified Internally */
static int FUNCread_triangle_RAW( FILE *fp, OMobj_id fld_id, int tin )
{
    char line_buf[LINE_LEN];
    int i, num_scanned;
    int mode = MODE_UNKNOWN;

    char obj_name[LINE_LEN], tex_name[LINE_LEN];
    float xyz1[3], xyz2[3], xyz3[3], rgb[3];

    OMobj_id cset_id;
    xp_long fld_nodes = 0;

    Vset_d vset;
    Mesh mesh;
    Cellset *cset = NULL;

    vset_d_init( &vset );
    mesh_init( &mesh );

    obj_name[0] = 0;

    if( tin ) mode = MODE_TIN;

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }

        /* Skip blank lines */
        if( line_buf[0] == 0 ) continue;
        if( line_buf[0] == 0xa ) continue;	/* LF */
        if( line_buf[0] == 0xd ) continue;	/* CR */
        /* Skip comments */
        if( line_buf[0] == 0x23 ) continue;	/* # */

        /* Do we know yet which flavor RAW file this is? */
        if( mode == MODE_UNKNOWN ) {
            /* twelve numbers means also RGB (0-1) colors */
            num_scanned = sscanf( line_buf,
                                 "%f %f %f %f %f %f %f %f %f %f %f %f",
                                 rgb,  rgb+1,  rgb+2,
                                 xyz1, xyz1+1, xyz1+2,
                                 xyz2, xyz2+1, xyz2+2,
                                 xyz3, xyz3+1, xyz3+2 );
            if( num_scanned == 12 )
                mode = MODE_COLORS;
            if( num_scanned == 9 || num_scanned == 10 ) {
                num_scanned = sscanf( line_buf,
                                      "%f %f %f %f %f %f %f %f %f %s",
                                      xyz1, xyz1+1, xyz1+2,
                                      xyz2, xyz2+1, xyz2+2,
                                      xyz3, xyz3+1, xyz3+2,
                                      tex_name );
                if( num_scanned == 10 ) {
                    /* texture mode not supported */
                    /* mode = MODE_TEXTURE; */
                    mode = MODE_DEFAULT;
                }
                else if( num_scanned == 9 )
                    mode = MODE_DEFAULT;
            }
            else {
                num_scanned = sscanf( line_buf, "%s", obj_name );
                /* should get one scanned ... */
                continue;
            }
        }

        if( mode == MODE_DEFAULT ) {
            num_scanned = sscanf( line_buf,
                                  "%f %f %f %f %f %f %f %f %f",
                                  xyz1, xyz1+1, xyz1+2,
                                  xyz2, xyz2+1, xyz2+2,
                                  xyz3, xyz3+1, xyz3+2 );
        }
        else if( mode == MODE_TIN ) {
            num_scanned = sscanf( line_buf,
                                  "t %f %f %f %f %f %f %f %f %f",
                                  xyz1, xyz1+1, xyz1+2,
                                  xyz2, xyz2+1, xyz2+2,
                                  xyz3, xyz3+1, xyz3+2 );
        }
        else if( mode == MODE_COLORS ) {
            num_scanned = sscanf( line_buf,
                                 "%f %f %f %f %f %f %f %f %f %f %f %f",
                                 rgb,  rgb+1,  rgb+2,
                                 xyz1, xyz1+1, xyz1+2,
                                 xyz2, xyz2+1, xyz2+2,
                                 xyz3, xyz3+1, xyz3+2 );
        }

        if( num_scanned == 0 || num_scanned == 1 ) {
            /* Should get one scanned ... */
            obj_name[0] = 0;
            num_scanned = sscanf( line_buf, "%s", obj_name );
            /* Force creation of new cell set with next tri. */
            cset = NULL;
            continue;
        }

        if( cset == NULL ) {
            /* Create new cell set */
            cset = mesh_add( &mesh, obj_name );
            obj_name[0] = 0;	/* Don't accidently reuse the name. */
        }

        if( mode == MODE_COLORS ) {
            cellset_add_colors( cset, rgb );
        }

        cellset_add_tri( cset,
                         vset_d_add( &vset, xyz1 ),
                         vset_d_add( &vset, xyz2 ),
                         vset_d_add( &vset, xyz3 ) );
    } /* while (loop over lines in file) */

    /* The file has been read, now put the info in the field. */

    /* TODO ... share with read_STL */

    /* This overwrites the priorities with the index counters, but
     * the priorities are no longer needed. (makes debugging harder ... )
     */
    fld_nodes = vset_d_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

#if 0
    vertex_tree_dump( vset.root );
#endif

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_d_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    FLDset_ncell_sets( fld_id, mesh_ncsets( &mesh ) );

    /* Each cell set */
    for( i = 0; i < mesh_ncsets( &mesh ); ++i ) {
        xp_long *conn_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

        /* Create field cell set */
#if 0
        /* Due to the V code, it should already be a Tri cell set. */
        FLDadd_cell_set( fld_id, "Tri" );
#endif
        FLDget_cell_set( fld_id, i, &cset_id );
        FLDset_ncells( cset_id, cset->ncells );

        /* Cell set connection list */
        FLDget_node_connect( cset_id, &conn_list, &size, OM_GET_ARRAY_RW );
        /* cset->ncells * 3 should be the same as size */
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            ARRfree( conn_list );
        }
        /* Cell set name (optional) */
        if( cset->name != NULL ) {
            FLDset_cell_set_user_name( cset_id, cset->name );
        }

        /* Cell data */
        if( cset->colors != NULL ) {
            float *cell_data = NULL;
            int type = DTYPE_FLOAT;

            FLDset_cell_data_ncomp( cset_id, 1 );
            FLDset_cell_data_veclen( cset_id, 0, 3 );
            FLDset_cell_data_id( cset_id, 0, GD_COLOR_DATA_ID );
            FLDset_cell_data_type( cset_id, 0, DTYPE_FLOAT );

            FLDget_cell_data( cset_id, 0, &type, (char **)&cell_data,
                              &size, OM_GET_ARRAY_RW );
            if( cell_data != NULL ) {
                memcpy( cell_data, cset->colors,
                        cset->ncells * 3 * sizeof(float) );
                ARRfree( cell_data );
            }
        }
        else FLDset_cell_data_ncomp( cset_id, 0 );
    }

    vset_d_free( &vset );
    mesh_free( &mesh );

    return 1;
}


/*
 * Stereo Lithography (STL) and Pro/Engineer SLP Render files,
 * which are very similar formats.  Usually these file contain triangles,
 * but the format allows any number of verticies per facet.  This
 * reader handles quads by simple on-the-fly triangulation.
 */
/* 64-bit porting. Only Modified Internally */
static int FUNCread_triangle_STL( FILE *fp, OMobj_id fld_id )
{
    char line_buf[LINE_LEN], *p;
    int i, num_scanned, verticies;
    int use_colors = 0;

    char obj_name[LINE_LEN];

#define MAX_VERTICIES 4
    float xyz[MAX_VERTICIES][3];	/* coordinates */
    float rgb[3] = { 1.0, 1.0, 1.0 };	/* cell colors */

    OMobj_id cset_id;
    xp_long fld_nodes = 0;

    Vset_d vset;
    Mesh mesh;
    Cellset *cset = NULL;

    vset_d_init( &vset );
    mesh_init( &mesh );

    obj_name[0] = 0;

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }

        p = line_buf;
        /* Skip spaces and tabs */
        while( p[0] == 0x20 || p[0] == 0x9 ) p++;

        /* Skip blank lines */
        if( p[0] == 0 ) continue;
        if( p[0] == 0xa ) continue;	/* LF */
        if( p[0] == 0xd ) continue;	/* CR */

        /* Start of object, usually first line in file.
         * Is it safe to assume there will never be more than one?
         */
        if( strncmp( p, "solid", 5 ) == 0 ) {
            num_scanned = sscanf( p, "solid %s", obj_name );
            cset = mesh_add( &mesh, obj_name );
        }
        /* End of object, usually last line in file. */
        else if( strncmp( p, "endsolid", 8 ) == 0 ) {
        }
        /* Begins triangle.  A SLP file might have just "facet" instead
         * of "facet normal".  Or it might have both ..
         */
        else if( strncmp( p, "facet", 5 ) == 0 ) {
            /* facet normal is optional (and ignored) */
        }
        /* Before triangle verticies.
         */
        else if( strncmp( p, "outer loop", 10 ) == 0 ) {
            verticies = 0;
        }
        /* Tri (sometimes Quad) Vertex */
        else if( strncmp( p, "vertex", 6 ) == 0 ) {
            if( verticies < MAX_VERTICIES ) {
                float *tmp = xyz[verticies];
                num_scanned = sscanf( p, "vertex %f %f %f",
                                      tmp, tmp+1, tmp+2 );
                if( num_scanned == 3 )
                    verticies += 1;
            }
            else {
                static int once = 0;
                if( once == 0 ) {
                    fprintf( stderr, "Warning: this reader only handles ngons up to n == 4\n" );
                    once = 1;
                }
            }
        }
        /* After triangle (or quad) verticies. */
        else if( strncmp( p, "endloop", 7 ) == 0 ) {
            if( verticies == 3 ) {
                cellset_add_colors( cset, rgb );
                cellset_add_tri( cset, 
                                 vset_d_add( &vset, xyz[0] ),
                                 vset_d_add( &vset, xyz[1] ),
                                 vset_d_add( &vset, xyz[2] ) );
            }
            else if( verticies == 4 ) {
                /* Triangulate on the fly ... */
                /* 0 1 2 3 => 0 1 2 */
                /*         => 0 2 3 */
                cellset_add_colors( cset, rgb );
                cellset_add_tri( cset, 
                                 vset_d_add( &vset, xyz[0] ),
                                 vset_d_add( &vset, xyz[1] ),
                                 vset_d_add( &vset, xyz[2] ) );
                cellset_add_colors( cset, rgb );
                cellset_add_tri( cset, 
                                 vset_d_add( &vset, xyz[0] ),
                                 vset_d_add( &vset, xyz[2] ),
                                 vset_d_add( &vset, xyz[3] ) );
            }
        }
        /* Ends triangle */
        else if( strncmp( p, "endfacet", 9 ) == 0 ) {
        }
        /* SLP */
        /* Normal for each vertex (expect 3). */
        else if( strncmp( p, "normal", 6 ) == 0 ) {
        }
        /* SLP */
        /* Color of the triangle.  The big assumption with colors is that
         * we see the "color" line before the triangle, although this code
         * will work if the line appears anyplace before the "endloop" line.
         */
        else if( strncmp( p, "color", 5 ) == 0 ) {
            num_scanned = sscanf( p, "color %f %f %f",
                                  rgb, rgb+1, rgb+2 );
            if( num_scanned == 3 ) {
                /* If we see something other than the default white */
                if( rgb[0] != 1.0 || rgb[1] != 1.0 || rgb[2] != 1.0 )
                    use_colors = 1;
            }
        }
    } /* while */

    /* The file has been read, now put the info in the field. */

    fld_nodes = vset_d_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

#if 0
    vertex_tree_dump( vset.root );
#endif

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_d_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    FLDset_ncell_sets(  fld_id, mesh_ncsets( &mesh ) );

    /* Each cell set */
    for( i = 0; i < mesh_ncsets( &mesh ); ++i ) {
        xp_long *conn_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

        /* Create field cell set */
#if 0
        FLDadd_cell_set( fld_id, "Tri" );
#endif
        FLDget_cell_set( fld_id, i, &cset_id );
        FLDset_ncells( cset_id, cset->ncells );

        /* Cell set connection list */
        FLDget_node_connect( cset_id, &conn_list, &size, OM_GET_ARRAY_RW );
        /* cset->ncells * 3 should be the same as size */
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            ARRfree( conn_list );
        }

        /* Cell set name (optional) */
        if( cset->name )
            FLDset_cell_set_user_name( cset_id, cset->name );

        /* Cell data (SLP only) */
        if( cset->colors != NULL && use_colors != 0 ) {
            float *cell_data = NULL;
            int type = DTYPE_FLOAT;

            FLDset_cell_data_ncomp( cset_id, 1 );
            FLDset_cell_data_veclen( cset_id, 0, 3 );
            FLDset_cell_data_id( cset_id, 0, GD_COLOR_DATA_ID );
            FLDset_cell_data_type( cset_id, 0, DTYPE_FLOAT );

            FLDget_cell_data( cset_id, 0, &type, (char **)&cell_data,
                              &size, OM_GET_ARRAY_RW );
            if( cell_data != NULL ) {
                memcpy( cell_data, cset->colors,
                        cset->ncells * 3 * sizeof(float) );
                ARRfree( cell_data );
            }
        }
        else FLDset_cell_data_ncomp( cset_id, 0 );
    }

    vset_d_free( &vset );
    mesh_free( &mesh );

    return 1;
}
