/*
			Copyright (c) 1997 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_ucd3.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <float.h>

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/math.h>
#include <avs/dtype.h>
#include <avs/fld.h>
#include <avs/f_utils.h>

#ifdef TIME
#include <time.h>
int stime ;
#endif

/* rd_ucd.c */
int FUNCread_ucd(char *file_name, OMobj_id out);
int FUNCcheck_ucd_file(char *file_name);

#define	BUFFER_SIZE	1024 /* max number of characters in one line */
#define	BLOCK_SIZE	512
#define	CELL_BLOCK_SIZE_INIT	0.7
#define	CELL_BLOCK_SIZE_GROW	1.5
#define	NUM_CELL_TYPE	15
#define SWAP_4_BYTE(a) ( (((a) << 24) & 0xff000000) | \
                         (((a) <<  8) & 0x00ff0000) | \
                         (((a) >>  8) & 0x0000ff00) | \
                         (((a) >> 24) & 0x000000ff)   )
#define	ERR_RETURN(A)	ERRerror("Read_UCD",0,ERR_ORIG, A); return(0);
/* binary mode */
#define BINARY_C		0
#define BINARY_FORTRAN		1
#define BINARY_C_SWAP		2
#define BINARY_FORTRAN_SWAP	3

struct cell {
  int		mat_id;
  int		type_id;
  int		con_num;
  xp_long	cell_num;
  xp_long	size;
  xp_long	*connect;
  xp_long	*cell_no;
  struct cell	*next;
};

static xp_long	*node_ids_b = NULL;
static xp_long	*cell_no_b = NULL;
static int	binary_format = 0; /* 0:false, 1:true */
static int	binary_mode = 0;
static int  format64 = 0;        /* 64-bit changes. flag to indicate 64-bit binary ucd format */
static int	number_of_file = 0;
static char	**b_file = NULL;
static FILE	*fpb = NULL;

static int get_label (const char *, char, int, char *);
static xp_long ucd_bsearchi (xp_long v,	xp_long *, xp_long);
static int ucd_qsorti(xp_long *, xp_long *indexes, xp_long num_entries);
static int delete_cell_inf(struct cell **);


/**********************************************************************
 *
 *     function name:	b_read
 *
 *     description :	read binary data
 *
 *     return :		1: success 
 *			0: failure
 *
 **********************************************************************/
size_t b_read(
void *ptr ,
size_t size ,
size_t nobj ,
int mode
)
{
	size_t	m, mf ;
	size_t	size0, i, n ;
	xp_long size1, size2;

	/* */
	switch(mode){
	case -1 :
		switch(binary_mode) {
		case BINARY_C_SWAP:
		case BINARY_FORTRAN_SWAP:
			n=size*nobj;
			m=fread(ptr,1,n,fpb);
			{
				int *idata = (int *) ptr;
				for(i=0;i<nobj;i++) {
					idata[i]=SWAP_4_BYTE(idata[i]);
				}
			}
			m=nobj;
			break;
		case BINARY_C:
		case BINARY_FORTRAN:
			m=fread(ptr,size,nobj,fpb);
			break;
		}
		break;
	case 0:
		switch(binary_mode) {
		case BINARY_C:				/* C */
		case BINARY_C_SWAP:
			m=fread(ptr,size,nobj,fpb);
			break ;
		case BINARY_FORTRAN:			/* Fortran */
		case BINARY_FORTRAN_SWAP:
			size0=size * nobj;
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size1,8,1,-1);
			else
			{
				int size1_32;
				mf=b_read(&size1_32,4,1,-1);
				size1=size1_32;
			}
#else
			mf=b_read(&size1,4,1,-1);
#endif
			if(size0 != size1) {
				ERR_RETURN("Can't read FORTRAN(1) binary file");
			}
			m=fread(ptr,size,nobj,fpb);
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size2,8,1,-1);
			else
			{
				int size2_32;
				mf=b_read(&size2_32,4,1,-1);
				size2=size2_32;
			}
#else
			mf=b_read(&size2,4,1,-1);
#endif
			if(size1 != size2) {
				ERR_RETURN("Can't read FORTRAN(2) binary file");
			}
			break;
		}
		break;
	case 1:
		switch(binary_mode) {
		case BINARY_C:				/* C */
		case BINARY_C_SWAP:
			m=b_read(ptr,size,nobj,-mode);
			break ;
		case BINARY_FORTRAN:			/* Fortran */
		case BINARY_FORTRAN_SWAP:
			size0=size * nobj;
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size1,8,1,-1);
			else
			{
				int size1_32;
				mf=b_read(&size1_32,4,1,-1);
				size1=size1_32;
			}
#else
			mf=b_read(&size1,4,1,-1);
#endif
			if(size0 != size1) {
				ERR_RETURN("Can't read FORTRAN(3) binary file");
			}
			m=b_read(ptr,size,nobj,-mode);
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size2,8,1,-1);
			else
			{
				int size2_32;
				mf=b_read(&size2_32,4,1,-1);
				size2=size2_32;
			}
#else
			mf=b_read(&size2,4,1,-1);
#endif
			if(size1 != size2) {
				ERR_RETURN("Can't read FORTRAN(4) binary file");
			}
			break;
		}
		break;
	case 32:
		switch(binary_mode) {
		case BINARY_C:
		case BINARY_C_SWAP:
			m=b_read(ptr,size,nobj,0);
			if(binary_mode == BINARY_C_SWAP) {
				{
					int *idata = (int *) ptr ;
					for(i=mode/4;i<(size/4);i++) {
						idata[i]=SWAP_4_BYTE(idata[i]);
					}
				}
			}
			break;
		case BINARY_FORTRAN:
		case BINARY_FORTRAN_SWAP:
			size0=size * nobj;
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size1,8,1,-1);
			else
			{
				int size1_32;
				mf=b_read(&size1_32,4,1,-1);
				size1=size1_32;
			}
#else
			mf=b_read(&size1,4,1,-1);
#endif
			if(size0 != size1) {
				ERR_RETURN("Can't read FORTRAN(5) binary file");
			}
			m=fread(ptr,size,nobj,fpb);
			if(binary_mode == BINARY_FORTRAN_SWAP) {
				{
					int *idata = (int *) ptr;
					for(i=mode/4;i<(size/4);i++) {
						idata[i]=SWAP_4_BYTE(idata[i]);
					}
				}
			}
#ifdef WORDLENGTH_64
			if(format64)
				mf=b_read(&size2,8,1,-1);
			else
			{
				int size2_32;
				mf=b_read(&size2_32,4,1,-1);
				size2=size2_32;
			}
#else
			mf=b_read(&size2,4,1,-1);
#endif
			if(size1 != size2) {
				ERR_RETURN("Can't read FORTRAN(6) binary file");
			}
			break;
		}
		break;
	default:
		ERR_RETURN("Can't read mode (mode)");
	break;
	}

	if(ferror(fpb)) {
		ERR_RETURN("Can't read binary file");
	}
	if(feof(fpb)) {
		return(m);
	}
	return(m);

}

/**********************************************************************
 *
 *     function name:	b_open
 *
 *     description :	open binary file of each step
 *
 *     return :		1: success 
 *			0: failure
 *
 **********************************************************************/
int b_open(
int step
)
{
	char	key[8];
	float	var;
	char	err_mes[128];
	int	i;

	/* open binary file */
	if((fpb = FILEfopen(b_file[step-1], "rb")) == NULL){
		sprintf(err_mes, "Can't open binary file [%s] : step [%d]", 
                        b_file[step-1], step);
		ERR_RETURN(err_mes) ;
	}

	/* binary type check */
	fread(&i,4,1,fpb);
	if(i == 7) {
		fread(key,7,1,fpb);
		fread(&i,4,1,fpb);
		fread(&i,4,1,fpb);
		fread(&var,4,1,fpb);
		if(var == 1.0) {
			binary_mode=BINARY_FORTRAN;
		} else {
			ERR_RETURN("Can't find binary information.") ;
		}
	} else {
		rewind(fpb);
		binary_mode=BINARY_FORTRAN_SWAP;
		b_read(&i,4,1,-1);
		if(i == 7) {
			binary_mode=BINARY_FORTRAN_SWAP;
		} else {
			fread(&i,3,1,fpb);
			fread(&var,4,1,fpb);
			if(var == 1.0) {
				binary_mode=BINARY_C;
			} else {
				binary_mode=BINARY_C_SWAP;
			}
		}
	}

	/* get binary header informaition */
	rewind(fpb) ;
	b_read(key,7,1,0) ;
	key[7]='\0' ;
	if (!strcmp(key, "AVSUC64"))
	{
		#ifdef WORDLENGTH_64
			format64 = 1;
		#else				/* 32-bit platforms */
			ERR_RETURN("Unsupported file format");
		#endif
	}
	else
		format64 = 0;

	b_read(&var,4,1,1) ;

	return 1 ;

}

/**********************************************************************
 *
 *     function name:	b_close
 *
 *     description :	close binary file
 *
 *     return :		1: success 
 *			0: failure
 *
 **********************************************************************/
int b_close(
)
{
	/* close */
	if(fpb != NULL) {
		if(fclose(fpb) == EOF){
			return 0;
		}
	}
	return 1;
}

/**********************************************************************
 *
 *	function name:	read_step_comment_binary
 *
 *	description:	check step and read comment string
 *
 *	return:		1: success 
 *			0: failure 
 *
 **********************************************************************/
static int
read_step_comment_binary(
int	step_no,
char	*title
)
{
	int	step;
	float	step_time;

	b_read(title,70,1,0);
	b_read(&step,4,1,1);
	if(step_no != step) {
		return 0 ;
	}
	b_read(&step_time,4,1,1);

	return 1;

}

/**********************************************************************
 *
 *	function name:	read_geom_data_binary
 *
 *	description:	read binary geometry data
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
read_geom_data_binary(
xp_long	*num_nodes,	/* number of nodes */
xp_long	*num_cells,	/* number of cells */
OMobj_id	out,		/* id of ucd output object */
struct cell	**Top,		/* cell information */
int		time_step,	/* time step: if 0, output is single field */
int		num_steps 	/* number of steps of time-dependent field */
)
{
	static int con_nodes[] = { 1, 2, 3, 4, 4, 5, 6, 8,
					3, 6, 8,10,13,15,20 };
	static const char *type_list[] = { "pt", "line", "tri", "quad",
						"tet", "pyr", "prism", "hex",
						"line2", "tri2", "quad2",
						"tet2", "pyr2", "prism2", "hex2"} ;
	static const char *set_type_list[] = { "Point", "Line", "Tri", "Quad",
						"Tet", "Pyr", "Prism", "Hex",
						"Line2", "Tri2", "Quad2",
						"Tet2", "Pyr2", "Prism2", "Hex2" } ;

	OMobj_id	coord_id, cell_set ;
	char		err_mes[128] ;
	char		coord_units[10] ;
	char		*type ;
	int		m, cs, size;
	xp_long i, j, k, cnt, num, node;
	xp_long first_num_nodes, alloc_cnt;
	int		*mat;
	xp_long *cell_cnt ;
	xp_long	*node_ind;
	int     n_type, ncell_sets ;
	float		*coord, *work_buf ;
	float		*props ;
	struct		cell *p, *last ;
	struct	{
		xp_long id;
		float x ;
		float y ;
		float z ;
	} node_w ;

	/* get num_nodes */
#ifdef WORDLENGTH_64
	if(format64)
	{
		if(num=b_read(num_nodes, 8, 1,1) != 1) {
			ERR_RETURN("Can't get num_nodes (binary)");
		}
	}
	else
	{
		int num_nodes_32;
		if(num=b_read(&num_nodes_32, 4, 1,1) != 1) {
			ERR_RETURN("Can't get num_nodes (binary)");
		}
		*num_nodes = num_nodes_32;
	}
#else
	if(num=b_read(num_nodes, 4, 1,1) != 1) {
		ERR_RETURN("Can't get num_nodes (binary)");
	}
#endif

	/* num_nodes check */
	if (*num_nodes < 1) {
	    ERR_RETURN("Can't get num_nodes (binary)");
	}

	/* non time-dependent */
	if(time_step < 2) {
		/* set num_nodes */
		FLDset_nnodes (out, *num_nodes);
		/* set nspace */
		FLDset_nspace(out, 3);
	}

	/* time-dependent data */
	if(time_step >= 2){
		if(FLDget_nnodes (out, &first_num_nodes) != 1) {
			ERR_RETURN("Can't get first_num_nodes.");
		}
		if(*num_nodes != first_num_nodes) {
			ERR_RETURN("Can't read this data under Store All Steps mode because numbers of nodes don't match between steps.");
		}
	}

	/*******************************************/
	/* get and set node coodinates and node id */
	/*******************************************/

	if(time_step >= 1) {
		if(FLDget_time_coordinates (out, time_step-1, &coord_id) != 1) {
			ERR_RETURN("Can't get coord_id of output time-dependent field.");
		}
	} else {
		coord_id = out;
	}

	/* set coord unit */
	strcpy(coord_units, "");
	FLDset_coord_units(coord_id, coord_units);

	/* allocation */
	node_ids_b = (xp_long *)malloc(sizeof(xp_long) * *num_nodes);
	if(node_ids_b == NULL){
		ERR_RETURN("Can't alloc memory.(node_ids_b)");
	}

	node_ind = (xp_long *)malloc(sizeof(xp_long) * *num_nodes);
	if(node_ind == NULL) {
		free( node_ids_b );
		ERR_RETURN("Can't alloc memory.(node_ind)");
	}

	coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, *num_nodes*3, NULL);
	if(coord == NULL) {
		free(node_ids_b);
		free(node_ind);
		ERR_RETURN("Can't alloc memory.(coord)");
	}

	if(b_read(&n_type,4,1,1) != 1) {
		free(node_ids_b);
		free(node_ind);
		ERR_RETURN("Can't node write type (binary)");
	}

	/* check coord type */
	if(n_type != 1 && n_type != 2) {
		ERR_RETURN("Unsupported coordinate type (binary)");
	}

#ifdef DEBUG
printf("\n");
printf("n_type=%d\n",n_type);
#endif

	/* get node id and coordinates */
	if(n_type == 1) {
		for(i=cnt=0; i< *num_nodes; i++,cnt+=3) {
#ifdef WORDLENGTH_64
			if(format64)
				num=b_read(&node_w.id,8,1,1);
			else
			{
				int node_id_32;
				num=b_read(&node_id_32,4,1,1);
				node_w.id=node_id_32;
			}
#else
			num=b_read(&node_w.id,4,1,1);
#endif
			num = num + b_read(&node_w.x,4,3,1);
			if(num != 4) {
				free(node_ids_b);
				free(node_ind);
				ERR_RETURN("Can't read node coordinate.");
			}
			node_ids_b[i]= node_w.id;
			coord[cnt]= node_w.x;
			coord[cnt+1] = node_w.y;
			coord[cnt+2] = node_w.z;
			node_ind[i] = i;
#ifdef DEBUG_DATA
printf("%ld %ld %f %f %f\n",i,node_w.id,node_w.x,node_w.y,node_w.z) ;
#endif
		}
	} else {
#ifdef WORDLENGTH_64
		if(format64)
			num=b_read(node_ids_b,8,*num_nodes,1);
		else
		{
			int *node_ids_b_32;
			node_ids_b_32 = (int *)malloc(sizeof(int) * *num_nodes);
			if(node_ids_b_32 == NULL){
				ERR_RETURN("Can't alloc memory.(node_ids_b_32)");
			}
			num=b_read(node_ids_b_32,4,*num_nodes,1);
			for(i=0;i<*num_nodes;i++) {
				node_ids_b[i] = node_ids_b_32[i];
			}
			free(node_ids_b_32);
		}
#else
		num=b_read(node_ids_b,4,*num_nodes,1);
#endif
		for(i=0;i<*num_nodes;i++) {
			node_ind[i] = i;
		}
		if(num != *num_nodes) {
			free(node_ids_b);
			free(node_ind);
			ERR_RETURN("Can't read node coordinate.");
		}

		work_buf = (float *)ARRalloc(NULL, DTYPE_FLOAT,*num_nodes, NULL) ;

		num=b_read(work_buf,4,*num_nodes,1);
		if(num != *num_nodes) {
			free(node_ids_b);
			free(node_ind);
			free(work_buf) ;
			ERR_RETURN("Can't read node coordinate.");
		}
		for(i=cnt=0; i< *num_nodes; i++,cnt+=3) {
			coord[cnt]=work_buf[i];
		}

		num=b_read(work_buf,4,*num_nodes,1);
		if(num != *num_nodes) {
			free(node_ids_b);
			free(node_ind);
			free(work_buf) ;
			ERR_RETURN("Can't read node coordinate.");
		}
		for(i=0,cnt=1; i< *num_nodes; i++,cnt+=3) {
			coord[cnt]=work_buf[i];
		}

		num=b_read(work_buf,4,*num_nodes,1);
		if(num != *num_nodes) {
			free(node_ids_b);
			free(node_ind);
			free(work_buf);
			ERR_RETURN("Can't read node coordinate.");
		}
		for(i=0,cnt=2; i< *num_nodes; i++,cnt+=3) {
			coord[cnt]=work_buf[i];
		}
		free(work_buf);
	}

	/* sort node ids */
	ucd_qsorti(node_ids_b, node_ind, *num_nodes);

	/* set coordinates */
	if(FLDset_coord(coord_id, coord, *num_nodes*3, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Can't set coordinate.") ;
	}

	/*****************************/
	/* get and set cell topology */
	/*****************************/

	if(time_step <= 1) {
		delete_cell_inf(Top);
		ncell_sets = 0;

		/* get num_cells */
#ifdef WORDLENGTH_64
		if(format64)
			b_read(num_cells,8,1,1);
		else
		{
			int num_cells_32;
			b_read(&num_cells_32,4,1,1);
			*num_cells=num_cells_32;
		}
#else
		b_read(num_cells,4,1,1);
#endif

		/* check num_cells */
		if (*num_cells < 1) {
		    ERR_RETURN("Can't get num_cells.(binary)");
		}

		/* allocation */
		cell_no_b = (xp_long *)malloc(sizeof(xp_long) * *num_cells);
		if(cell_no_b == NULL) {
			ERR_RETURN("Can't alloc memory.(cell_no_b)");
		}

		mat = (int *)malloc(sizeof(int) * *num_cells);
		if(mat == NULL) {
			free(cell_no_b);
			ERR_RETURN("Can't alloc memory.(mat)");
		}
		type = (char *)malloc(sizeof(char)* *num_cells);
		if(type == NULL) {
			free(cell_no_b);
			free(mat);
			ERR_RETURN("Can't alloc memory.(type)");
		}

		/* get cell_id, material_id and cell_type */
#ifdef WORDLENGTH_64
		if(format64)
			num=b_read(cell_no_b,8,*num_cells,1) ;
		else
		{
			int *cell_no_b_32;
			cell_no_b_32 = (int *)malloc(sizeof(int) * *num_cells);
			if(cell_no_b_32 == NULL) {
				ERR_RETURN("Can't alloc memory.(cell_no_b_32)");
			}
			num=b_read(cell_no_b_32,4,*num_cells,1);
			for(i=0;i<*num_cells;i++) {
				cell_no_b[i] = cell_no_b_32[i];
			}
			free(cell_no_b_32);
		}
#else
		num=b_read(cell_no_b,4,*num_cells,1) ;
#endif
		if(num != *num_cells) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			ERR_RETURN("Can't read cell no coordinate.");
		}

#ifdef DEBUG_DATA
for(j=0;j<*num_cells;j++) printf("%ld ",cell_no_b[j]);
printf("\n") ;
#endif

		num=b_read(mat,4,*num_cells,1);
		if(num != *num_cells) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			ERR_RETURN("Can't read mat coordinate.");
		}
		num=b_read(type,1,*num_cells,0);
		if(num != *num_cells) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			ERR_RETURN("Can't read type coordinate.");
		}

		cnt = 0;
		for(i=0; i< *num_cells; i++) {
			cnt += con_nodes[ type[i] ];
		}
		cell_cnt = (xp_long *)malloc(sizeof(xp_long) * cnt);
		if(cell_cnt == NULL) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			ERR_RETURN("Can't alloc memory.(cell_cnt)");
		}
#ifdef WORDLENGTH_64
		if(format64)
			num=b_read(cell_cnt,8,cnt,1);
		else
		{
			int *cell_cnt_32;
			cell_cnt_32 = (int *)malloc(sizeof(int) * cnt);
			if(cell_cnt_32 == NULL) {
				ERR_RETURN("Can't alloc memory.(cell_cnt_32)");
			}
			num=b_read(cell_cnt_32,4,cnt,1);
			for(i=0;i<cnt;i++) {
				cell_cnt[i] = cell_cnt_32[i];
			}
			free(cell_cnt_32);
		}
#else
		num=b_read(cell_cnt,4,cnt,1);
#endif
		if(num != cnt) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			free(cell_cnt);
			ERR_RETURN("Can't read cell connectivity");
		}

#ifdef DEBUG_DATA
for(i=0,k=0;i<*num_cells;i++){
 printf("[%ld] type=%d \"%s\" -> ", cell_no_b[i],type[i],set_type_list[type[i]]) ;
 for(j=0;j<con_nodes[type[i]];j++,k++){
  printf("%ld ",cell_cnt[k]) ;
 }
 printf("\n") ;
}
#endif

		/* check cell type */
		ncell_sets = 0;
		for(i=0,j=0; i< *num_cells; i++) {
			k=type[i] ;
			if(k >= NUM_CELL_TYPE || k < 0) {
				sprintf(err_mes, "Can't support cell_type[%s].",type_list[k]);
				free(node_ids_b);
				free(node_ind);
				free(cell_no_b);
				free(mat);
				free(type);
				free(cell_cnt);
				ERR_RETURN(err_mes);
			}
			for(p=*Top; p != NULL; p=p->next) {
				if(p->mat_id != mat[i]) {
					continue;
				}
				if(p->type_id != (int)k) {
					continue;
				}
				break;
			}

			if(p == NULL) {
				p=(struct cell *)malloc(sizeof(struct cell));
				if(p == NULL) {
					free(node_ids_b);
					free(node_ind);
					free(cell_no_b);
					free(mat);
					free(type);
					free(cell_cnt);
					ERR_RETURN("Can't alloc memory.(struct cell)");
				}
				memset(p, 0, sizeof(struct cell));
				if(*Top == NULL) {
					*Top = last = p;
				} else {
					last->next = p;
					last = p;
				}
				p->mat_id = mat[i];
				p->type_id = (int)k;
				p->con_num = con_nodes[k];
				p->cell_num = 1;
				ncell_sets++ ;
			} else {
				p->cell_num++;
			}
			if (!p->cell_no) {
				p->size = *num_cells * CELL_BLOCK_SIZE_INIT;
				p->cell_no = (xp_long *)malloc(p->size*sizeof(xp_long));
				p->connect = (xp_long *)malloc(p->con_num*p->size*sizeof(xp_long));
			} else if(p->cell_num > p->size) {
				p->size = p->size * CELL_BLOCK_SIZE_GROW;
				p->cell_no = (xp_long *)realloc(p->cell_no, p->size*sizeof(xp_long)) ;
				p->connect = (xp_long *)realloc(p->connect, p->con_num*p->size*sizeof(xp_long)) ;
			}
			if(p->cell_no == NULL || p->connect == NULL) {
				if (p->cell_no) free(p->cell_no);
				if (p->connect) free(p->connect);
				free(node_ids_b );
				free(node_ind );
				free(cell_no_b);
				free(mat);
				free(type);
				free(cell_cnt);
				ERR_RETURN("Can't alloc memory.(p->cell_no or p->connect)");
			}
			p->cell_no[p->cell_num-1] = i;

			if(p->connect == NULL) {
				free(node_ids_b);
				free(node_ind);
				free(cell_no_b);
				free(mat);
				free(type);
				free(cell_cnt);
				ERR_RETURN("Can't alloc memory.(p->connect)");
			}
			for(m=0; m < p->con_num; m++,j++) {
				node = *(cell_cnt+j);
				k=ucd_bsearchi(node, node_ids_b, *num_nodes);
				alloc_cnt = p->cell_num * p->con_num;
				p->connect[alloc_cnt-p->con_num+m] = node_ind[k];
			}

		}

		/* set cell_set */
		cell_set = OMfind_subobj(out, OMstr_to_name("cell_set"), OM_OBJ_RW);
		OMreset_obj(cell_set, 0 );
		FLDset_node_data_ncomp(out, 0);
		FLDset_ncell_sets(out, ncell_sets);

		for(p = *Top, cs=0; p != NULL; p = p->next) {
			FLDget_cell_set(out, cs, &cell_set);
			FLDset_cell_set(cell_set, set_type_list[p->type_id]);
			FLDset_ncells(cell_set, p->cell_num );
			FLDset_node_connect(cell_set, p->connect, p->cell_num*p->con_num,OM_SET_ARRAY_COPY);
			/* set material ID */
			if(FLDset_cell_nprops (cell_set, 1) != 1) {
				free(node_ids_b);
				free(node_ind);
				free(cell_no_b);
				free(mat);
				free(type);
				free(cell_cnt);
				ERR_RETURN("Can't set nprops");
			}
			if(FLDget_cell_props(cell_set, (float **)&props, &size, OM_GET_ARRAY_WR) != 1) {
				free(node_ids_b);
				free(node_ind);
				free(cell_no_b);
				free(mat);
				free(type);
				free(cell_cnt);
				ERR_RETURN("Can't get cell props");
			}
			props[0] = p->mat_id;
			ARRfree(props);
			cs++;

			/* set nsteps */
			if(time_step == 1) {  /* in case of time-dependent data */
				if(FLDset_time_nsteps(cell_set, num_steps) != 1) {
					free(node_ids_b);
					free(node_ind);
					free(cell_no_b);
					free(mat);
					free(type);
					free(cell_cnt);
					ERR_RETURN("Can't set nsteps of cell_set of time-dependent field.");
				}
			}
		}

	}

	if(time_step >= 2) {

		/* skip cell connectivity */
#ifdef WORDLENGTH_64
		if(format64)
			b_read(num_cells,8,1,1);
		else
		{
			int num_cells_32;
			b_read(&num_cells_32,4,1,1);
			*num_cells=num_cells_32;
		}
#else
		b_read(num_cells,4,1,1);
#endif

		/* check num_cells */
		if (*num_cells < 1) {
		    ERR_RETURN("Can't get num_cells.(binary)");
		}

		/* allocation */
		cell_no_b = (xp_long *)malloc(sizeof(xp_long) * *num_cells);
		if(cell_no_b == NULL) {
			ERR_RETURN("Can't alloc memory.(cell_no_b)");
		}
		mat = (int *)malloc(sizeof(int) * *num_cells);
		if(mat == NULL) {
			free(cell_no_b);
			ERR_RETURN("Can't alloc memory.(mat)");
		}
		type = (char *)malloc(sizeof(char)* *num_cells);
		if(type == NULL) {
			free(cell_no_b);
			free(mat);
			ERR_RETURN("Can't alloc memory.(type)");
		}

#ifdef WORDLENGTH_64
		if(format64)
			b_read(cell_no_b,8,*num_cells,1) ;
		else
		{
			int *cell_no_b_32;
			cell_no_b_32 = (int *)malloc(sizeof(int) * *num_cells);
			if(cell_no_b_32 == NULL) {
				ERR_RETURN("Can't alloc memory.(cell_no_b_32)");
			}
			b_read(cell_no_b_32,4,*num_cells,1);
			for(i=0;i<*num_cells;i++) {
				cell_no_b[i] = cell_no_b_32[i];
			}
			free(cell_no_b_32);
		}
#else
		b_read(cell_no_b,4,*num_cells,1) ;
#endif
		b_read(mat,4,*num_cells,1);
		b_read(type,1,*num_cells,0);

		cnt = 0;
		for(i=0; i< *num_cells; i++) {
			cnt += con_nodes[ type[i] ];
		}
		cell_cnt = (xp_long *)malloc(sizeof(xp_long) * cnt);
		if(cell_cnt == NULL) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			ERR_RETURN("Can't alloc memory.(cell_cnt)");
		}
#ifdef WORDLENGTH_64
		if(format64)
			num=b_read(cell_cnt,8,cnt,1);
		else
		{
			int *cell_cnt_32;
			cell_cnt_32 = (int *)malloc(sizeof(int) * cnt);
			if(cell_cnt_32 == NULL) {
				ERR_RETURN("Can't alloc memory.(cell_cnt_32)");
			}
			num=b_read(cell_cnt_32,4,cnt,1);
			for(i=0;i<*num_cells;i++) {
				cell_cnt[i] = cell_cnt_32[i];
			}
			free(cell_cnt_32);
		}
#else
		num=b_read(cell_cnt,4,cnt,1);
#endif
		if(num != cnt) {
			free(node_ids_b);
			free(node_ind);
			free(cell_no_b);
			free(mat);
			free(type);
			free(cell_cnt);
			ERR_RETURN("Can't read cell connectivity");
		}

	}

	if(time_step <= 1)  free(node_ind);
	free(mat);
	free(type);
	free(cell_cnt);

	return 1;

}

/**********************************************************************
 *
 *	function name:	read_data_binary
 *
 *	description: 	read node / cell data
 *
 *	return:	1: success 
 *		0: failure 
 *
 **********************************************************************/
static int
read_data_binary(
xp_long num_data,		/* number of node or cell (input) */
int	num_comp,		/* number of component (input) */
int	comp_list[],		/* component list (input) */
int	null_data_flag[],	/* null data flag (input */
float	null_data[],		/* null data (input) */
int	data_type,		/* data type (input) */
int	num_mode,		/* node=0 ,cell=1 (input) */
float	*data			/* data (output) */
)
{
	int	num_vals, i, j, id;
	xp_long num_sum, k, offset, node, num;
	int	num_node_data,r_mode ;
	xp_long cnt, cnt2, ic;
	xp_long	*id_b ;
	float	min_data, max_data;
	float	*value_b ;
	struct	{
		xp_long id;
		float data[100];
	} data_w;


	num_node_data=0 ;
	for(i=0; i<num_comp; i++) {
		num_node_data+=comp_list[i];
	}

	min_data = max_data = FLT_MAX;

#ifdef DEBUG_DATA
printf("data_type=%d num_mode=%d\n",data_type,num_mode) ;
#endif

	/* get step value */
	switch(data_type) {
	case 1:
		num_vals=num_node_data;
		value_b = (float *) malloc(sizeof(float)* num_vals);
		if(value_b == NULL) {
			free(node_ids_b);
			free(cell_no_b);
			ERR_RETURN("Can't alloc memory.(buffer 1)");
		}
		for(node=0; node<num_data; node++) {
			num = b_read(value_b,4,num_vals,1);
			if((int)num != num_vals) {
				free(value_b);
				free(node_ids_b);
				free(cell_no_b);
				ERR_RETURN("Can't read step data value (type1)");
			}
			k=0;
			for(i=0; i<num_comp; i++) {
				offset = num_data * i + node * comp_list[i];
				for(j=0;j<comp_list[i];j++) {
					cnt = offset + j;
					data[cnt] = value_b[k];
					k++;
				}
			}
		}
		free(value_b);
		break;
	case 2:
		num_vals = num_node_data;
		value_b = (float *) malloc(sizeof(float)* num_data);
		if(value_b == NULL) {
			free(node_ids_b);
			free(cell_no_b);
			ERR_RETURN("Can't alloc memory.(buffer 2)");
		}
		k=0;
		for(i=0; i<num_comp; i++) {
			for(j=0;j<comp_list[i];j++) {
				num = b_read(value_b,4,num_data,1);
				if (num != num_data) {
					free(value_b);
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't read step data value (type 2)");
				}
				offset = num_data * i * comp_list[i];
				for(node=0; node<num_data; node++) {
					cnt = node * comp_list[i] + j + offset;
					data[cnt] = value_b[node];
				}
				k++;
			}
		}
		free(value_b);
		break;
	case 3:
		cnt2=0;
		for(i=0; i<num_comp; i++) {
#ifdef WORDLENGTH_64
			if(format64)
				b_read(&num_sum,8,1,1);
			else
			{
				int num_sum_32;
				b_read(&num_sum_32,4,1,1);
				num_sum=num_sum_32;
			}
#else
			b_read(&num_sum,4,1,1);
#endif
			if(num_data != num_sum) {
				null_data_flag[i]=1;
				null_data[i]=max_data;
			}
			num_vals = comp_list[i];
			r_mode = 1;
			for(node=0; node<num_sum; node++) {
				offset = 0;
				if(r_mode == 1) {
#ifdef WORDLENGTH_64
					if(format64)
						num=b_read(&data_w.id,8,1,1);
					else
					{
						int data_id_32;
						num=b_read(&data_id_32,4,1,1);
						data_w.id=data_id_32;
					}
#else
					num=b_read(&data_w.id,4,1,1);
#endif
					num=b_read(&data_w.data,4, num_vals,1);
				}
				if((int)num != num_vals) {
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't read step data value (type 3)");
				}
				id=0;
				if(num_mode == 0) {
					for(k=0;k<num_data;k++) {
						if(node_ids_b[k] == data_w.id) {
							id=1;
							break;
						}
					}
				} else {
					for(k=0;k<num_data;k++) {
						if(cell_no_b[k] == data_w.id) {
							id=1;
							break;
						}
					}
				}
				if(id == 1) {
					for(j=0; j<num_vals; j++) {
						cnt = node * num_vals + offset + j + cnt2;
						data[cnt] = data_w.data[j];
						if(min_data > data_w.data[j]) {
							min_data=data_w.data[j];
						}
					}
					r_mode=1;
				} else {
					for(j=0; j<num_vals; j++) {
						cnt = node * num_vals + offset + j + cnt2;
						data[cnt] = max_data;
					}
					null_data_flag[i]=1;
					null_data[i]=max_data;
					r_mode=0;
				}
				offset += num_data * num_vals;
			}
			cnt2 += num_data * num_vals;
		}
		break;
	case 4:
		cnt2 = 0;
		for(i=0; i<num_comp; i++) {
			num_vals = comp_list[i];
#ifdef WORDLENGTH_64
			if(format64)
				num = b_read(&num_sum,8,1,1);
			else
			{
				int num_sum_32;
				num = b_read(&num_sum_32,4,1,1);
				num_sum=num_sum_32;
			}
#else
			num= b_read(&num_sum,4,1,1);
#endif
			if(num_data != num_sum) {
				null_data_flag[i]=1;
				null_data[i]=max_data;
			}
			if(num_sum != 0) {
				value_b = (float *) malloc(sizeof(float)* num_sum);
				if(value_b == NULL) {
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't alloc memory.(buffer 4.1)");
				}
				id_b = (xp_long *) malloc(sizeof(xp_long)* num_sum);
				if(id_b == NULL) {
					free(value_b);
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't alloc memory.(buffer 4.2)") ;
				}
#ifdef WORDLENGTH_64
				if(format64)
					num = b_read(id_b,8,num_sum,1);
				else
				{
					int *id_b_32;
					id_b_32 = (int *)malloc(sizeof(int) * num_sum);
					if(id_b_32 == NULL){
						ERR_RETURN("Can't alloc memory.(id_b_32)");
					}
					num=b_read(id_b_32,4,num_sum,1);
					for(k=0;k<num_sum;k++) {
						id_b[k] = id_b_32[k];
					}
					free(id_b_32);
				}
#else
				num = b_read(id_b,4,num_sum,1);
#endif
				if(num != num_sum) {
					free(value_b);
					free(id_b);
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't read step id data (type 4)");
				}
			} else {
				value_b = (float *) malloc(sizeof(float));
				if(value_b == NULL) {
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't alloc memory.(buffer 4.3)");
				}
				id_b = (xp_long *) malloc(sizeof(xp_long));
				if(id_b == NULL) {
					free(value_b);
					free(node_ids_b);
					free(cell_no_b);
					ERR_RETURN("Can't alloc memory.(buffer 4.4)");
				}
				id_b[0]=-1;
				null_data_flag[i]=1;
				null_data[i]=max_data;
			}
			for(j=0; j<num_vals; j++) {
				offset=0;
				if(num_sum != 0) {
					num = b_read(value_b,4,num_sum,1);
					if(num != num_sum) {
						free(value_b);
						free(id_b);
						free(node_ids_b);
						free(cell_no_b);
						ERR_RETURN("Can't read step data value (type 4)");
					}
				} else {
					value_b[0]=max_data;
				}
				ic=0;

#ifdef DEBUG_DATA
printf("num_vals=%d num_data=%d num_sum=%d\n", num_vals,num_data,num_sum) ;
#endif

				for(node=0; node<num_data; node++) {
					cnt = node * num_vals + offset + j + cnt2;
					data[cnt] = max_data ;
					for(k=0;k<num_sum;k++) {
						if(num_mode == 0) {
							if(node_ids_b[ic]==id_b[k]) {
								data[cnt] = value_b[k];
								break;
							}
						} else {
							if(cell_no_b[ic]==id_b[k]) {
								data[cnt] = value_b[k];
								break;
							}
						}
					}
					if(data[cnt] == max_data) {
						null_data_flag[i]=1;
						null_data[i]=max_data;
					} else {
						ic++;
					}
#ifdef DEBUG_DATA
printf("%d %d\n",num_sum,ic) ;
#endif

				}
				offset += num_data * num_vals;
#ifdef DEBUG_DATA
printf("null_data_flag[%d]=%d\n",i,null_data_flag[i]) ;
#endif

			}
			free(value_b);
			free(id_b);
			cnt2+=num_data*num_vals;
		}
		break;
	default:
		free(node_ids_b);
		free(cell_no_b);
		ERR_RETURN("Can't read step data value.(type)");
		return 0;
		break;
	}

	return 1 ;

}

/**********************************************************************
 *
 *	function name:  read_node_cell_data_binary
 *
 *     description:    read data of specified step
 *
 *	return:	1: success 
 *		0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
read_node_cell_data_binary(
xp_long		num_nodes,	/* number of nodes */
xp_long		num_cells,	/* number of cells */
OMobj_id	out,		/* object id of output ucd data */
OMobj_id	id,		/* object id of parent object */
struct cell	*Top,		/* cell information */
int		time_step	/* time step: if 0, output is single field */
)
{
	char		labels[BLOCK_SIZE][17], units[BLOCK_SIZE][17];
	char		label[17], unit[17];
	int		comp_list[BLOCK_SIZE] ;
	int		i, m, cs, cnt, type;
	xp_long j, k, size;
	int		num_comp, veclen;
	xp_long offset;
	int		num_node_data, num_cell_data, sum ;
	xp_long num;
	int		first_num_comp, first_cell_comp;
	int		data_type ;
	int		null_data_flag[BUFFER_SIZE] ;
	float		*data, *out_data;
	float		null_data[BUFFER_SIZE] ;
	OMobj_id	cell_set;
	OMobj_id	node_data_id, cell_data_id;
	struct cell	*p;
	struct {
		char name[16];
		char unit[16];
		int leng;
		int flag;
		float data;
	} hed1;

	/******************************************/
	/* read number of node data and cell data */
	/******************************************/
	num=b_read(&num_comp,4,1,1);
	if(num != 1) {
		ERR_RETURN("Can't read step component number") ;
	}
	if(num_comp != 0) {
		num=b_read(&data_type,4,1,1);
		if(num != 1){
			ERR_RETURN("Can't read data type");
		}
		if(data_type < 0 || data_type > 4) {
			ERR_RETURN("Can't node data type (Not 1 <= data type >= 4)");
		}

		sum=0;
		for(i=0; i<num_comp; i++) {
			if(data_type == 1 || data_type == 2) {
				num=b_read(&hed1,44,1,32);
				for(j=0;j<16;j++) {
					labels[i][j]=hed1.name[j];
					units[i][j]=hed1.unit[j];
				}
				labels[i][16]='.';
				units[i][16]='\0';
				comp_list[i]=hed1.leng;
				null_data_flag[i]=hed1.flag;
				null_data[i]=hed1.data;
			} else {
				num=b_read(&hed1,36,1,32);
				for(j=0;j<16;j++) {
					labels[i][j]=hed1.name[j];
					units[i][j]=hed1.unit[j];
				}
				labels[i][16]='\0';
				units[i][16]='\0';
				comp_list[i]=hed1.leng;
				null_data_flag[i]=0;
				null_data[i]=0.0;
			}
			sum += comp_list[i];
		}
		num_node_data=sum;

		/******************/
		/* read node data */
		/******************/

		/* allocation */
		data = (float *)malloc(sizeof(float)*num_nodes*num_node_data);
		if(data == NULL){
			ERR_RETURN("Can't alloc memory.(node_data)");
		}

		/* read node data */
		if(read_data_binary(num_nodes, num_comp, comp_list, 
			null_data_flag, null_data, data_type, 0, data) != 1) {
			free(data);
			ERR_RETURN("Can't read node data.");
		}

		/* check or set number of components */
		if(time_step >= 2) {
			if(FLDget_node_data_ncomp(out, &first_num_comp) != 1) {
				first_num_comp = 0;
			}
			if(num_comp != first_num_comp) {
				free(data);
				ERR_RETURN("Can't read this data under Store All Step mode because numbers of node components don't match between steps.")
			}
		} else {
			FLDset_node_data_ncomp(out, num_comp);
		}

		/* get time_node_data object id */
		if(time_step >= 1) {  /* in case of time-dependent field */
			if(FLDget_time_node_data(out, time_step-1, &node_data_id) != 1) {
				ERR_RETURN("Can't get time_node_data id.");
			}
		} else { /* in case of single step field */
			node_data_id = out ;
		}

		for (i = 0, offset = 0; i < num_comp; i++) {
			/* get a label of node data component */
			if(!get_label(labels[i],'.', 0, label)) {
				sprintf(label, "node_data_%d", i );
			}
			/* get a unit of node data component */
			if(!get_label(units[i],'.', 0, unit)) {
				sprintf(unit, " ") ;
			}
			/* set node data component information */
			veclen = comp_list[i] ;
			FLDset_node_data_comp(node_data_id, i, veclen, label, unit) ;
			/* set node data value */
			FLDset_node_data(node_data_id, i, (char *)&data[offset], 
				DTYPE_FLOAT, veclen*num_nodes, OM_SET_ARRAY_COPY ) ;
			/* set null data value */
			if(null_data_flag[i] == 1) {
				FLDset_node_null_data(node_data_id, i, (char *) &null_data[i], DTYPE_FLOAT);
			}
			offset += (veclen * num_nodes) ;
		}
		free(data) ;
	} else {
		if(time_step >= 2) {
			if(FLDget_node_data_ncomp(out, &first_num_comp) == 1) {
				if(first_num_comp > 0) {
					ERR_RETURN("Can't read this data under Store All Step mode because numbers of node components don't match between steps.")
				}
			}
		}
	}

	/******************/
	/* read cell data */
	/******************/
	num=b_read(&num_comp,4,1,1) ;
	if (num != 1){
		ERR_RETURN("Can't read step component number") ;
	}

	if(num_comp != 0) {  /* if cell data exist */

		num=b_read(&data_type,4,1,1);
		if(num != 1) {
			ERR_RETURN("Can't read data type");
		}
		if(data_type < 0 || data_type > 4) {
			ERR_RETURN("Can't cell data type (Not 1 < data type > 4)");
		}

		sum=0;
		for(i=0; i<num_comp; i++) {
			if(data_type == 1 || data_type == 2) {
				num=b_read(&hed1,44,1,32);
				for(j=0;j<16;j++) {
					labels[i][j]=hed1.name[j];
					units[i][j]=hed1.unit[j];
				}
				labels[i][j]='\0';
				units[i][j]='\0';
				comp_list[i]=hed1.leng;
				null_data_flag[i]=hed1.flag;
				null_data[i]=hed1.data;
			} else {
				num=b_read(&hed1,36,1,32);
				for(j=0;j<16;j++) {
					labels[i][j]=hed1.name[j];
					units[i][j]=hed1.unit[j];
				}
				labels[i][j]='\0';
				units[i][j]='\0';
				comp_list[i]=hed1.leng;
				null_data_flag[i]=0;
				null_data[i]=0.0;
			}
			sum += comp_list[i];
		}
		num_cell_data=sum ;

#ifdef DEBUG_DATA
printf("num_cells=%ld num_cell_data=%d",num_cells,num_cell_data) ;
#endif

		/* allocation */
		data = (float *)malloc(sizeof(float)*num_cells*num_cell_data);
		if(data == NULL) {
			ERR_RETURN("Can't alloc memory.(node_data)");
		}
		/* read cell data */
		if(read_data_binary(num_cells, num_comp, comp_list, 
			null_data_flag, null_data, data_type, 1, data) != 1){
			free(data);
			ERR_RETURN("Can't read cell data.");
		}

		/* set cell data */
		for(p = Top, cs = 0; p != NULL; p = p->next, cs++) {

			/* get cell_set id */
			if(FLDget_cell_set(out, cs, &cell_set) != 1) {
				ERR_RETURN("Can't get cell_set.");
			}

			/* set number of components */
			if(time_step >= 2) {
				if(FLDget_cell_data_ncomp(cell_set, &first_cell_comp) != 1) {
					first_cell_comp = 0;
				}
				if(num_comp != first_cell_comp) {
					free(data);
					ERR_RETURN("Can't read this data under Store All Step mode because numbers of cell components don't match between steps.") ;
				}
			} else {
				FLDset_cell_data_ncomp(cell_set, num_comp);
			}

			/* get time-dependent cell data id */
			if(time_step >= 1) {
				if(FLDget_time_cell_data(cell_set, time_step-1,&cell_data_id) != 1) {
					free(data);
					ERR_RETURN("Can't get time_cell_data id.");
				}
			} else {
				cell_data_id = cell_set;
			}

			for(i = 0, offset = 0; i < num_comp; i++) {
				/* get a label of cell data component */
				if(!get_label(labels[i],'.', 0, label)) {
					sprintf(label, "cell_data_%d", i);
				}
				/* get a unit of cell data component */
				if(!get_label(units[i],'.', 0, unit)) {
					sprintf(unit, " ");
				}
				/* set node data compornent information */
				veclen = comp_list[i];
				if(FLDset_cell_data_comp(cell_data_id, i, veclen, label,unit ) != 1) {
					free(data);
					ERR_RETURN("Can't set cell_data_comp.");
				}
				/* set cell data value */
				type = DTYPE_FLOAT ;
				if(FLDget_cell_data(cell_data_id, i, &type, (char **)&(out_data), &size, OM_GET_ARRAY_WR) != 1) {
					free(data);
					ERR_RETURN("Can't get cell_data array.");
				}
				/* set null data value */
				if(null_data_flag[i] == 1) {
					FLDset_cell_null_data(cell_data_id, i, (char *) &null_data[i], DTYPE_FLOAT);
				}
				for(j = cnt = 0; j < p->cell_num; j++) {
					k = p->cell_no[j];
					for(m = 0; m < veclen; m++, cnt++) {
						out_data[cnt] = data[offset+veclen*k+m];
					}
				}
				offset += (veclen * num_cells);
				ARRfree(out_data);
			}
		}
		free(data) ;
	} else {
		if(time_step >= 2) {
			if(FLDget_cell_set(out, 0, &cell_set) != 1) {
				ERR_RETURN("Can't get cell_set.");
			}
			if(FLDget_cell_data_ncomp(cell_set, &first_cell_comp) == 1) {
				if(first_cell_comp > 0) {
					ERR_RETURN("Can't read this data under Store All Step mode because numbers of cell components don't match between steps.");
				}
			}
		}
	}

	return 1 ;

}

/**********************************************************************
 *
 *	function name:	skip_geom_data_binary 
 *
 *	description:	skip geometry data
 *
 *	return:	1: success 
 *		0: failure
 *
 **********************************************************************/
static int
skip_geom_data_binary (
xp_long	*num_nodes,
xp_long	*num_cells
)
{
	static int con_nodes[] = { 1, 2, 3, 4, 4, 5, 6, 8, 
					3, 6, 8,10,13,15,20 };
	char	*type;
	int	    n_type;
	xp_long i, num, cnt;
	xp_long *ibuf, *ibuf1,*ibuf2;
	int     *mat;
	float	*fbuf;
	struct	{
		xp_long id;
		float x;
		float y;
		float z;
	} node_w;

	/* get num_nodes */
#ifdef WORDLENGTH_64
	if(format64)
		num=b_read(num_nodes,8,1,1);
	else
	{
		int num_nodes_32;
		num=b_read(&num_nodes_32, 4, 1,1);
		*num_nodes = num_nodes_32;
	}
#else
	num=b_read(num_nodes,4,1,1);
#endif

	/* get n_type */
	num=b_read(&n_type,4,1,1);

	/* allocation */
	ibuf = (xp_long *)malloc(sizeof(xp_long)* *num_nodes);
	if(ibuf == NULL){
		ERR_RETURN("Can't alloc memory.(buffer)");
	}
	fbuf = (float *)malloc(sizeof(float)* *num_nodes);
	if(fbuf == NULL){
		free(ibuf);
		ERR_RETURN("Can't alloc memory.(buffer)");
	}

	/* skip node data */
	if(n_type == 1) {
		for(i=0; i< *num_nodes; i++) {
#ifdef WORDLENGTH_64
			if(format64)
				num=b_read(&node_w.id,8,1,1);
			else
			{
				int node_id_32;
				num=b_read(&node_id_32,4,1,1);
				node_w.id=node_id_32;
			}
#else
			num=b_read(&node_w.id,4,1,1);
#endif
			num=b_read(&node_w.x,4,3,1);
		}
	} else {
#ifdef WORDLENGTH_64
		if(format64)
			num=b_read(ibuf,8,*num_nodes,1);
		else
		{
			int *ibuf_32;
			ibuf_32 = (int *)malloc(sizeof(int) * *num_nodes);
			if(ibuf_32 == NULL){
				ERR_RETURN("Can't alloc memory.(ibuf_32)");
			}
			num=b_read(ibuf_32,4,*num_nodes,1);
			for(i=0;i<*num_nodes;i++) {
				ibuf[i] = ibuf_32[i];
			}
			free(ibuf_32);
		}
#else
		num=b_read(ibuf,4,*num_nodes,1);
#endif
		num=b_read(fbuf,4,*num_nodes,1);
		num=b_read(fbuf,4,*num_nodes,1);
		num=b_read(fbuf,4,*num_nodes,1);
	}

	/* get num_cells */
#ifdef WORDLENGTH_64
	if(format64)
		num=b_read(num_cells,8,1,1);
	else
	{
		int num_cells_32;
		num=b_read(&num_cells_32, 4, 1,1);
		*num_cells = num_cells_32;
	}
#else
	num=b_read(num_cells,4,1,1);
#endif

	/* allocation */
	ibuf1 = (xp_long *)malloc(sizeof(xp_long)* *num_cells);
	if(ibuf1 == NULL) {
		free(ibuf);
		free(fbuf);
		ERR_RETURN("Can't alloc memory.(buffer)");
	}
	mat = (int *)malloc(sizeof(int)* *num_cells);
	if(mat == NULL) {
		free(ibuf);
		free(fbuf);
		ERR_RETURN("Can't alloc memory.(buffer)");
	}
	type = (char *)malloc(sizeof(char)* *num_cells);
	if(type == NULL) {
		free(ibuf);
		free(fbuf);
		free(ibuf1);
		free(mat);
		ERR_RETURN("Can't alloc memory.(type)");
	}

	/* skip cell id */
#ifdef WORDLENGTH_64
	if(format64)
		num=b_read(ibuf1,8,*num_cells,1);
	else
	{
		int *ibuf1_32;
		ibuf1_32 = (int *)malloc(sizeof(int) * *num_cells);
		if(ibuf1_32 == NULL){
			ERR_RETURN("Can't alloc memory.(ibuf1_32)");
		}
		num=b_read(ibuf1_32,4,*num_cells,1);
		for(i=0;i<*num_cells;i++) {
			ibuf1[i] = ibuf1_32[i];
		}
		free(ibuf1_32);
	}
#else
	num=b_read(ibuf1,4,*num_cells,1);
#endif

	/* skip material id */
	num=b_read(mat,4,*num_cells,1);

	/* skip cell type */
	num=b_read(type,1,*num_cells,0);

	cnt = 0 ;
	for(i=0; i< *num_cells; i++){
		cnt += con_nodes[ type[i] ];
	}

	ibuf2 = (xp_long *)malloc(sizeof(xp_long)* cnt);
	if(ibuf2 == NULL){
		free(ibuf);
		free(fbuf);
		free(ibuf1);
		free(mat);
		free(type);
		ERR_RETURN("Can't alloc memory.(buffer)");
	}
#ifdef WORDLENGTH_64
	if(format64)
		num=b_read(ibuf2,8,cnt,1);
	else
	{
		int *ibuf2_32;
		ibuf2_32 = (int *)malloc(sizeof(int) * cnt);
		if(ibuf2_32 == NULL){
			ERR_RETURN("Can't alloc memory.(ibuf2_32)");
		}
		num=b_read(ibuf2_32,4,cnt,1);
		for(i=0;i<cnt;i++) {
			ibuf2[i] = ibuf2_32[i];
		}
		free(ibuf2_32);
	}
#else
	num=b_read(ibuf2,4,cnt,1);
#endif

	free(ibuf);
	free(fbuf);
	free(ibuf1);
	free(mat);
	free(type);
	free(ibuf2);

	return 1;

}


/**********************************************************************
 *
 *	function name:	get_token
 *
 *	description:	extract the next string in tline[] separeted by
 *		 	blank and  convert it in the specified form
 *
 *	return: ntc	size of extracted string 
 * 
 **********************************************************************/
static int
get_token (
char	**tline,	/* string (input) */
char	*data,		/* extracted and converted string (output) */
const char	*control	/* format (e.g. %d, %f) (input) */
)
{
    char  *line, token[80], c;
    int   ntc;


    line = *tline;

    c = 0;
    for (ntc = 0; c != '\n'; line++) {
        if (((c  = *line) != ' ') && (c != '\t') && (c != '\f') && (c != '\n')) {
            token[ntc++] = c;
        } else if (ntc != 0) {
            token[ntc] = '\0';
            sscanf (token, control, data);
            *tline = line;
            break;
        }
    }
    return (ntc);
}

/**********************************************************************
 *
 *	function name:  get_label
 *
 *      description: 	get label and unit of data component
 *			from string separeted by delimiter
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
static int
get_label (
const char *string,		/* string (input) */
char	delimiter,		/* delimiter (input) */
int	number,			/* component number (input) */
char	*label			/* label or unit (output) */
)
{
  int   i, j, k, len;
  char  current;


  if (string == NULL)
    return 0;

  k = 0;
  len = (int)strlen (string);

  if(len == 0) return 0 ;	/* 03.04.2002 K.Hirasawa */

  for (i = 0; i <= number; i++) {
    current = string[k++];
    j = 0;
    while (current != delimiter) {
      /* build the label character by character */
      label[j++] = current;
      current = string[k++];

      /* the last character was found */
      if (k > len) {
        /* the nth label was not found, where n = number */
        if (i < number) {
          return 0;
        }
        current = delimiter;
      }

    }
    label[j] = '\0';
  }
  return 1;
}

/**********************************************************************
 *
 *	function name:  ucd_qsi
 *
 *      description:	sort data
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
static int
ucd_qsi(
int	*values,
int	*indexes,
int	left,
int	right
)
{
  register int i, j;
  int x, y;
  int z;


  x = values[(left + right) / 2];
  i = left;
  j = right;

  do {
    while (values[i] < x && i < right) i++;
    while (x < values[j] && j > left) j--;

    if (i <= j) {
      y = values[i];
      values[i] = values[j];
      values[j] = y;
      z = indexes[i];
      indexes[i] = indexes[j];
      indexes[j] = z;
      i++;
      j--;
    }
  } while (i <= j);

  if (left < j) ucd_qsi (values, indexes, left, j);
  if (i < right) ucd_qsi (values, indexes, i, right);
  return 1;
}

/**********************************************************************
 *
 *	function name:  ucd_qsl
 *
 *      description:	sort data
 *
 *	return:		1: success
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
ucd_qsl(
xp_long	*values,
xp_long	*indexes,
xp_long	left,
xp_long	right
)
{
  register xp_long i, j;
  xp_long x, y;
  xp_long z;

  x = values[(left + right) / 2];
  i = left;
  j = right;

  do {
    while (values[i] < x && i < right) i++;
    while (x < values[j] && j > left) j--;

    if (i <= j) {
      y = values[i];
      values[i] = values[j];
      values[j] = y;
      z = indexes[i];
      indexes[i] = indexes[j];
      indexes[j] = z;
      i++;
      j--;
    }
  } while (i <= j);

  if (left < j) ucd_qsl (values, indexes, left, j);
  if (i < right) ucd_qsl (values, indexes, i, right);
  return 1;
}

/**********************************************************************
 *
 *	function name:  ucd_bsearchi
 *
 *      description: 	search
 *			(calculate the index of v in list[])
 *
 *	return:		calculated index
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static xp_long
ucd_bsearchi (
xp_long	v,			/* value that will be searched (input) */
xp_long	*list,			/* dimension of values (input) */
xp_long	n			/* size of list (input) */
)
{
  int found;
  xp_long top, bot, indexer;

  n = n - 1;

  if (v <= list[0])
    return (0);

  if (v >= list[n])
    return (n);

  found = 0;
  top = 0;
  bot = n;

  while (!found) {
    indexer = (top + bot) / 2;

    if (v > list[indexer])
      top = indexer;
    else if (v < list[indexer])
      bot = indexer;
    else
      return (indexer);

    if ((bot - top) < 2)
      if (fabs(list[bot] - v + 0.0) < fabs(v - list[top] + 0.0))
        return (bot);
      else
        return (top);
  }
  return 0;
}

/**********************************************************************
 *
 *	function name:  ucd_qsorti
 *
 *      description:	sort data
 *
 *	return:		1: success
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Newly Introduced */
static int
ucd_qsorti(
xp_long	*values,
xp_long	*indexes,
xp_long	num_entries
)
{
  xp_long *x;
  xp_long *ix, mid;
  register xp_long i, j, k;

  mid = (num_entries - 1) / 2;
  if (mid == 0) mid = 1;

  x = (xp_long *)malloc(mid *sizeof(xp_long));
  ix = (xp_long *)malloc(mid * sizeof(xp_long));

  /* dir no_recurrence */
  for (i = 0; i < mid; i++) {
    x[i] = values[i];
    ix[i] = indexes[i];
  }

  /* dir fork */
  ucd_qsl (x, ix, 0, mid-1);
  ucd_qsl (values, indexes, mid, num_entries-1);
  /* dir join */

  i = 0; j = mid;

  while ((i < mid) && (j < num_entries)) {
    if (values[j] < x[i]) {
      values[i + j - mid] = values[j];
      indexes[i + j - mid] = indexes[j];
      j++;
    }
    else {
      values[i + j - mid] = x[i];
      indexes[i + j - mid] = ix[i];
      i++;
    }
  }

  if (i < mid) {
    /* dir no_recurrence */
    for (k = i; k < mid; k++) {
      values[k + j - mid] = x[k];
      indexes[k + j - mid] = ix[k];
    }
  }

  free(x);
  free(ix);
  return 1;
}

/**********************************************************************
 *
 *	function name:  delete_cell_inf
 *
 *      description:	free cell information data
 *
 *	return:		1: success 
 *
 **********************************************************************/
static int
delete_cell_inf(
struct cell	**Top
)
{
  struct cell	*p, *del;


  del = (*Top);
  while( del != NULL ) {
    if( del->cell_no != NULL ) {
      free( del->cell_no );
    }
    if( del->connect != NULL ) {
      free( del->connect );
    }
    p = del->next;
    free( del );
    del = p;
  }
  (*Top) = NULL;
  return 1;
}

/**********************************************************************
 *
 *	function name:  reset_out_data
 *
 *      description:	initialize ouput ucd data
 *
 *	return:		1: success
 *			0: failure 
 *
 **********************************************************************/
static int
reset_out_data(
OMobj_id	out,	/* id of ucd (fld) data object (input) */
int		time_flag	/* 0: single field,  
			  	   1: time-dependent field */
)
{
  int      stat, i, j, ncell_sets, ncomps;
  OMobj_id cell_set_id;


  if (time_flag == 1)
  {
    int      nsteps;
    OMobj_id time_id, time_coord_id;

    if ((FLDget_time_nsteps(out, &nsteps) != 1) || nsteps <= 0)
      return 0;

    for (i = 0; i < nsteps; i++)
    {
      if ((FLDget_cell_set(out, 0, &cell_set_id) == 1) &&
          (FLDget_cell_data_ncomp(cell_set_id, &ncomps) == 1) &&
          (FLDget_ncell_sets(out, &ncell_sets) == 1))
      {
        for (j = 0; j < ncell_sets; j++)
        {
          if (FLDget_cell_set(out, j, &cell_set_id) == 1)
            stat = FLDset_cell_data_ncomp(cell_set_id, 0);
        }
      }

      if (FLDget_time_node_data(out, i, &time_id) == 1)
      {
        stat = FLDset_nnodes(time_id, 0);
        stat = FLDset_nspace(time_id, 0);
        stat = FLDset_node_data_ncomp(time_id, 0);
      }

      if (FLDget_time_coordinates(out, i, &time_coord_id) != 1)
        stat = FLDset_coord(time_coord_id, NULL, 0, OM_SET_ARRAY_STATIC);
    }
    stat = FLDset_ncell_sets(out, 0);
    stat = FLDset_time_nsteps(out, 0);
  }
  else
  {
    if ((FLDget_cell_set(out, 0, &cell_set_id) == 1) &&
        (FLDget_cell_data_ncomp(cell_set_id, &ncomps) == 1) &&
        (FLDget_ncell_sets(out, &ncell_sets) == 1))
    {
      for (j = 0; j < ncell_sets; j++)
      {
        if (FLDget_cell_set(out, j, &cell_set_id) == 1)
          stat = FLDset_cell_data_ncomp(cell_set_id, 0);
      }
    }

    stat = FLDset_nnodes(out, 0);
    stat = FLDset_nspace(out, 0);
    stat = FLDset_coord(out, NULL, 0, OM_SET_ARRAY_STATIC);
    stat = FLDset_node_data_ncomp(out, 0);
    stat = FLDset_ncell_sets(out, 0);
  }

  return 1;
}


/**********************************************************************
 *
 *	function name:  read_header
 *
 *      description:    skip #lines and read num_steps and cycle_type
 *
 *	return:		1: success 
 *			0: failure 
 *			-1: binary, must be single-step
 *
 **********************************************************************/
/* from modules/rd_ucd.c: get it in shared header file */
#define UCD_ASCII_FILE 3

static int
read_header(
char    *filename,
FILE	*fp,
int	*num_steps,
char	*cycle_type
)
{
	char	line[BUFFER_SIZE];
	char  	n_buf[BUFFER_SIZE];
	char	filepath[BUFFER_SIZE];
	int	i, j, k, jstop;

	char	*pdest;
	int	slash_unix = '/';
	int	slash_win = '\\';
	xp_long	pathlength = 0;

	binary_format = 0;	/* binary-ascii flag */

	/* check to be sure it's ASCII */
	if (FUNCcheck_ucd_file(filename) == UCD_ASCII_FILE) {

		/* skip comment lines which starts from # */
		while (fgets(line, sizeof(line), fp) != NULL) {
			if (line[0] != '#') {
				break;
			}
		}

		/* check num_steps field (if not exists, this file is binary format) */
		if (sscanf(line, "%d", num_steps) != 1) {

			/* check cycle_type field */
			if(sscanf( line, "%s", cycle_type) != 1) {
				return 0;
			} else {

				/* pre-check the number of file (number of step) */
				number_of_file = 0;
				while(fgets(line, sizeof(line), fp) != NULL) {
					number_of_file++;
				}
				/* rewind & skip line */
				rewind(fp);
				while (fgets(line, sizeof(line), fp) != NULL) {
					if (line[0] != '#') {
						break;
					}
				}

				/* reset buffer for line and filename */
				for(k=0;k<BUFFER_SIZE;k++) {
					n_buf[k]='\0';
					filepath[k]='\0';
				}

#ifdef DEBUG
printf("Number of step (pre-check) = %d\n", number_of_file);
#endif

				/* get absolute path from ascii filename */
/*
				for(j=0;j<strlen(filename);j++) {
					if(filename[j]=='/' || filename[j]=='\\') {
						jstop = j ;
					}
					if(filename[j]=='.' && filename[j+1]=='i' && filename[j+2]=='n' && filename[j+3]=='p') {
						break ;
					}
					if(filename[j]=='.' && filename[j+1]=='I' && filename[j+2]=='N' && filename[j+3]=='P') {
						break ;
					}
				}
*/

				pdest = strrchr( filename, slash_unix );
				if (pdest != NULL) {
					pathlength = pdest - filename + 1;
				} else {
					pdest = strrchr( filename, slash_win );
					if (pdest != NULL) {
						pathlength = pdest - filename + 1;
					}
				}
				if (pathlength != 0) {
					strncpy( filepath, filename, pathlength );
				} else {
					strcpy( filepath, filename );
				}

#ifdef DEBUG
fprintf(stderr, "last=%ld, filepath=%s, filename=%s\n", 
        pathlength, filepath, filename) ;
#endif


				/* read filename information */
				b_file = (char **)malloc(number_of_file * sizeof(char*));
				if (b_file == NULL) {
					ERR_RETURN("Can't allocate space for filename");
				}

				/* */
				i=0;
				while(fgets(line, sizeof(line), fp) != NULL) {
					for(j=0;j<BUFFER_SIZE;j++) {
						n_buf[j] = '\0';
					}
					if(sscanf( line, "%s", n_buf) == 0) {
						if(i == 0) {
							return 0;
						} else {
							*num_steps=i;
							binary_format = 1;
							return 1;
						}
					}
					if(n_buf[0] == '\0' || n_buf[0] == ' ') {
						if(i == 0) {
							return 0;
						} else {
							*num_steps=i;
							binary_format = 1;
							return 1;
						}
					}

					/* create filename */
					/* (check absolute path and allocate filespace) */
					if(n_buf[0]=='/' || n_buf[1]==':') {
						b_file[i] = (char*)malloc((strlen(line)+1)*sizeof(char));
						if(b_file[i] == NULL) {
							while(--i>=0) {
								free(b_file[i]);
							}
							free(b_file);
							ERR_RETURN("Can't allocate space for binary list");
						}
						strcpy(b_file[i], n_buf);
					} else {
						b_file[i] = (char*)malloc((strlen(filepath)+strlen(line)+1)*sizeof(char));
						if(b_file[i] == NULL) {
							while(--i>=0) {
								free(b_file[i]);
							}
							free(b_file);
							ERR_RETURN("Can't allocate space for binary list");
						}
						strncpy(b_file[i], filepath, pathlength);
						b_file[i][pathlength] = 0;
						strcat(b_file[i], n_buf);
					}
#ifdef DEBUG
printf("b_file[%d] = %s\n", i, b_file[i]);
#endif
					i++;

				}
				*num_steps=i;
				binary_format = 1;

				return 1;
			}
			return 0;
		}

		if (fgets(line, sizeof(line), fp) == NULL) {
			return 0;
		}

		if (sscanf( line, "%s", cycle_type) != 1) {
			return 0;
		}
		return 1 ;

	} else {
		return -1 ;
	}

}


/**********************************************************************
 *
 *	function name:  skip_header
 *
 *      description:    skip #lines and num_steps and cycle_type
 *
 *	return:		1: success 
 *
 **********************************************************************/
static int
skip_header(
FILE	*fp
)
{
    char  line[BUFFER_SIZE];


    /* skip comment lines which starts from # */
    while (fgets(line, sizeof(line), fp) != NULL) {
        if (line[0] != '#') {
            break;
        }
    }


    fgets (line, sizeof(line), fp);

    return 1 ;
}


/**********************************************************************
 *
 *	function name:  read_step_comment
 *
 *      description:    check step and read comment string
 *
 *	return:		1: success 
 *			0: failure 
 *
 **********************************************************************/
static int
read_step_comment(
FILE	*fp,
int	step_no,	/* step number */
char	*comment
)
{
   char	line[BUFFER_SIZE];
   char	step[20];
   char check[20];
   int  len;

   if (fgets(line, sizeof(line), fp) == NULL) {
       return 0;
   }
   if (sscanf(line, "%s", step) != 1) {
       return 0;
   }

   sprintf(check, "step%d", step_no);

   if (strcmp(step, check) != 0) {
       return 0;
   }

   len = (int)strlen(step);

   sscanf(line, "%s %s", step, comment);
   return 1;
}


/**********************************************************************
 *
 *	function name:	skip_geom_data 
 *
 *	description: 	skip geometry data
 *
 *	return:		1: success 
 *			0: 
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
skip_geom_data (
FILE	*fp,
xp_long	*num_nodes,
xp_long	*num_cells
)
{
  char	line[BUFFER_SIZE], err_mes[128];
  xp_long i;


    /* get num_nodes and num_cells */
    fgets( line, sizeof( line ), fp );
    if (sscanf(line, "%ld %ld", num_nodes, num_cells) != 2) {
        sprintf( err_mes,
		"Can't get num_nodes or num_cells of the first step.");
        ERR_RETURN( err_mes );
    }
    /* skip geometry data */
    for (i = 0; i < *num_nodes + *num_cells; i++) {
        fgets( line, sizeof( line ), fp );
    }
    return 1;
}


/**********************************************************************
 *
 *	function name:	get_step_file_pos 
 *
 *	description:  	get file pointer of beginning of each 
 *                      step
 *
 *	return:		1: success 
 *
 **********************************************************************/
/* 64-bit porting. Only Modified Internally */
static int
get_step_file_pos(
FILE	*fp,
int	num_steps,
char	*cycle_type,
long	**file_pos	/* position of beginning of each step */
)
{
    char	line[BUFFER_SIZE], err_mes[128];
    int	set_num, num, num_node_data, num_cell_data, num_comp;
    xp_long	i, offset;
    xp_long	num_nodes, num_cells;


    /* allocation of file pointer array */
    *file_pos = (long *)malloc(sizeof(long)*(num_steps+1));
    if (*file_pos == NULL) {
        ERR_RETURN("Can't alloc memory.(file_pos)");
    }

    rewind(fp);

    /* skip header */
    skip_header(fp);

    /* get file pointers of beginning of all steps */
    for (set_num = 0; set_num < num_steps; set_num++) {

	/* set current file pointer */
	(*file_pos)[set_num] = ftell(fp);

	/* skip step ID */
	if (fgets(line, sizeof(line), fp) == NULL) {
	    sprintf(err_mes, "Error in step %d: Can't find step_id line.",
		 set_num);
	    ERR_RETURN(err_mes);
	}

	/* skip geometry data */
	if (set_num == 0 || strcmp(cycle_type, "geom") == 0 ||
			strcmp(cycle_type, "data_geom") == 0) {
	    if (skip_geom_data(fp, &num_nodes, &num_cells) != 1) {
		sprintf(err_mes,
			"Error in step %d: Can't skip geometry",
			set_num);
		ERR_RETURN(err_mes);
	    }
	}

        /* set position of beginning of node and cell data of first step  */
        /* to file_pos array (index: num_steps) */
	if (set_num == 0) {
	    (*file_pos)[num_steps] = ftell(fp);
	}

        /* skip node and cell data */
	if (set_num == 0 || strcmp(cycle_type, "data") == 0 ||
			strcmp(cycle_type, "data_geom") == 0) {

	    /* get num_node_data and num_cell_data */
	    if (fgets(line, sizeof(line), fp) == NULL) {
		sprintf(err_mes,
			"Error in step %d: Can't find num_node_data and num_cell_data line.",
			set_num);
		ERR_RETURN(err_mes);
	    }
	    num = sscanf(line, "%d %d", &num_node_data, &num_cell_data);
	    if (num != 2) {
		sprintf(err_mes,
			"Error in step %d: Can't read num_node_data and num_cell_data.",
			set_num);
		ERR_RETURN(err_mes);
	    }

	    /* skip node data */
	    if (num_node_data > 0) {
	        /* get num_comp */
	        if (fgets(line, sizeof(line), fp) == NULL) {
	  	    sprintf(err_mes,
			"Error in step %d: Can't find num_comp line.",
			set_num);
		    ERR_RETURN(err_mes);
	        }
	        sscanf(line, "%d", &num_comp);

	        /* skip node data and component label data */
	        offset = num_nodes + num_comp;
	        for (i = 0; i < offset; i++) {
	    	    fgets(line, sizeof(line), fp);
	        }
	    }

	    /* skip cell data */
	    if (num_cell_data > 0) {
	        /* get num_comp */
	        if (fgets(line, sizeof( line ), fp)  == NULL) {
	    	    sprintf(err_mes,
			"Error in step %d: Can't find num_comp line.",
			set_num);
		    ERR_RETURN(err_mes);
	        }
	        sscanf(line, "%d", &num_comp);

	        /* skip cell data and component label data */
	        offset = num_cells + num_comp;
	        for (i = 0; i < offset; i++) {
		    fgets(line, sizeof(line), fp);
		}
	    }
	}

    }
    return 1;
}


/**********************************************************************
 *
 *	function name:  read_geom_data
 *
 *      description:	read geometry data
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
read_geom_data(
FILE		*fp,		/* file pointer */
xp_long		*num_nodes,	/* number of nodes */
xp_long		*num_cells,	/* number of cells */
OMobj_id	out,		/* id of ucd output object */
struct cell	**Top,		/* cell information */
int		time_step,	/* time step: if 0, output is single field */
int		num_steps 	/* number of steps of time-dependent field */
)
{
    static int	con_nodes[] = { 1, 2, 3, 4,  4,  5,  6,  8,
				   3, 6, 8, 10, 13, 15, 20 };
    static const char	*type_list[] = { "pt", "line", "tri", "quad",
				 "tet", "pyr", "prism", "hex",
				       "line2", "tri2", "quad2",
				 "tet2", "pyr2", "prism2", "hex2" };
    static const char	*set_type_list[] = { "Point", "Line", "Tri", "Quad",
				     "Tet", "Pyr", "Prism", "Hex",
				             "Line2", "Tri2", "Quad2",
				     "Tet2", "Pyr2", "Prism2", "Hex2" };
    char	line[BUFFER_SIZE], *tl, err_mes[128];
    char	type[BUFFER_SIZE], coord_units[10];
    int	k, type_len, n, num;
    xp_long i, cnt, ind;
    xp_long	cell_no, alloc_cnt, node;
    int mat, m, cs;
    xp_long	*node_ids,*node_ind;
    int ncell_sets;
    float	*coord;
    OMobj_id	cell_set;
    struct cell	*p, *last;
    float	*props;
    int		size;
    OMobj_id	coord_id;
    xp_long		first_num_nodes;


    /* get num_nodes and num_cells */
    if (fgets (line, sizeof(line), fp) == NULL) {
	ERR_RETURN("Can't get num_nodes and num_cells");
    }
    if (sscanf(line, "%ld %ld", num_nodes, num_cells) != 2) {
	ERR_RETURN("Can't get num_nodes or num_cells");
    }

    /* check num_nodes, num_cells */
    if (*num_nodes < 1 || *num_cells < 1) {
        ERR_RETURN("Can't get num_nodes or num_cells");
    }

    if (time_step < 2) {
	/* set num_nodes */
	FLDset_nnodes (out, *num_nodes);

	/* set nspace */
	FLDset_nspace (out, 3);
    }

    if (time_step >= 2) {
        if (FLDget_nnodes (out, &first_num_nodes) != 1) {
	    ERR_RETURN("Can't get first_num_nodes.")
        }
	if (*num_nodes != first_num_nodes) {
	    ERR_RETURN("Can't read this data under Store All Steps mode because numbers of nodes don't match between steps.")
        }
    }

    /*******************************************/
    /* get and set node coodinates and node id */
    /*******************************************/

    if (time_step >= 1) {
        if (FLDget_time_coordinates (out, time_step-1, &coord_id) != 1) {
	    ERR_RETURN("Can't get coord_id of output time-dependent field.");
        }
    } else {
	coord_id = out;
    }


    /* set coord unit */
    strcpy(coord_units, "");
    FLDset_coord_units(coord_id, coord_units);

    /* allocation */
    node_ids = (xp_long *)malloc(sizeof(xp_long) * *num_nodes);
    if (node_ids == NULL) {
	ERR_RETURN("Can't alloc memory.(node_ids)");
    }

    node_ind = (xp_long *)malloc(sizeof(xp_long) * *num_nodes);
    if (node_ind == NULL) {
        free( node_ids );
        ERR_RETURN("Can't alloc memory.(node_ind)");
    }

    coord = (float *)ARRalloc(NULL, DTYPE_FLOAT,
				*num_nodes*3, NULL);
    if (coord == NULL) {
        free(node_ids);
        free(node_ind);
        ERR_RETURN("Can't alloc memory.(coord)");
    }

    /* get node id and coordinates */
    for (i = cnt = 0; i < *num_nodes; i++, cnt+=3) {
	fgets(line, sizeof( line ), fp);
	num = sscanf(line, "%ld %f %f %f", &node_ids[i],
		  &coord[cnt], &coord[cnt+1], &coord[cnt+2]);
	if (num != 4) {
	    free(node_ids);
            free(node_ind);
            ERR_RETURN("Can't read node coordinate.");
        }
	node_ind[i] = i;
    }

    /* sort node ids */
    ucd_qsorti(node_ids, node_ind, *num_nodes);

    /* set coordinates */
    if (FLDset_coord (coord_id, coord, *num_nodes*3, OM_SET_ARRAY_FREE) != 1) {
        ERR_RETURN("Can't set coordinate.");
    }

    /*****************************/
    /* get and set cell topology */
    /*****************************/
    if (time_step <= 1) {
        delete_cell_inf(Top);

	ncell_sets = 0;
	for (i = 0; i < *num_cells; i++) {
	    /* get cell_id, material_id and cell_type */
	    if (fgets( line, sizeof( line ), fp ) == NULL) {
		free( node_ids );
		free( node_ind );
		ERR_RETURN("Can't read cell connectivity.");
	    }
	    tl = line;
	    n = 0;

	    get_token (&tl, (void *) &cell_no, "%ld");
	    get_token (&tl, (void *) &mat, "%d");
	    type_len = get_token (&tl, (void *)type, "%s");
	    type[type_len] = '\0';

	    /* check cell type */
	    for (k = 0; k < NUM_CELL_TYPE; k++) {
		if (!strncmp( type, type_list[k], type_len)) {
		    break;
		}
	    }
	    if (k == NUM_CELL_TYPE) {
		sprintf(err_mes, "Can't support cell_type[%s].", type);
		free(node_ids);
		free(node_ind);
		ERR_RETURN(err_mes);
	    }

	    for (p = *Top; p != NULL; p = p->next) {
		if (p->mat_id != mat) {
		    continue;
		}
		if (p->type_id != k) {
		    continue;
		}
		break;
	    }
	    if (p == NULL) {
		p = (struct cell *)malloc(sizeof(struct cell));
		if (p == NULL) {
		    free (node_ids);
		    free( node_ind );
		    ERR_RETURN("Can't alloc memory.(struct cell)");
		}
		memset(p, 0, sizeof(struct cell));

		if (*Top == NULL) {
		    *Top = last = p;
		} else {
		    last->next = p;
		    last = p;
		}

		p->mat_id = mat;
		p->type_id = k;
		p->con_num = con_nodes[k];

		p->cell_num = 1;

		ncell_sets++;
	    } else {
		p->cell_num++;
	    }

	    if (!p->cell_no) {
		p->size = *num_cells * CELL_BLOCK_SIZE_INIT;
		p->cell_no = (xp_long *)malloc(p->size*sizeof(xp_long));
		p->connect = (xp_long *)malloc(p->con_num*p->size*sizeof(xp_long));
	    } else if (p->cell_num > p->size) {
		p->size = p->size * CELL_BLOCK_SIZE_GROW;
		p->cell_no = (xp_long *)realloc(p->cell_no, p->size*sizeof(xp_long));
		p->connect = (xp_long *)realloc(p->connect,
					    p->con_num*p->size*sizeof(xp_long));
	    }

	    if (p->cell_no == NULL || p->connect == NULL) {
		if (p->cell_no) free(p->cell_no);
		if (p->connect) free(p->connect);
		free( node_ids );
		free( node_ind );
		ERR_RETURN("Can't alloc memory.(p->cell_no or p->connect)");
	    }

	    p->cell_no[p->cell_num-1] = i;


	    if (p->connect == NULL) {
		free(node_ids);
		free(node_ind);
		ERR_RETURN("Can't alloc memory.(p->connect)");
	    }

	    for (m = 0; m < p->con_num; m++) {
		if (get_token(&tl, (void *)&node, "%ld") < 1) {
		    free(node_ids);
		    free(node_ind);
		    ERR_RETURN("Can't read cell connectivity");
		}

		ind = ucd_bsearchi(node, node_ids, *num_nodes);

		alloc_cnt = p->cell_num * p->con_num;
		p->connect[alloc_cnt-p->con_num+m] = node_ind[ind];
	    }
	}

	free(node_ids);
	free(node_ind);

	cell_set = OMfind_subobj(out, OMstr_to_name("cell_set"), OM_OBJ_RW);
	OMreset_obj(cell_set, 0 );

	FLDset_node_data_ncomp(out, 0);

	FLDset_ncell_sets(out, ncell_sets);

	for (p = *Top, cs=0; p != NULL; p = p->next) {
	    FLDget_cell_set(out, cs, &cell_set);

	    FLDset_cell_set(cell_set, set_type_list[p->type_id]);

	    FLDset_ncells(cell_set, p->cell_num );

	    FLDset_node_connect(cell_set, p->connect, p->cell_num*p->con_num,
			    OM_SET_ARRAY_COPY);

	    /* set material ID */
	    if (FLDset_cell_nprops (cell_set, 1) != 1) {
		ERR_RETURN("Can't set nprops");
	    }
	    if (FLDget_cell_props(cell_set, (float **)&props, &size,
		    OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Can't get cell props");
	    }
	    props[0] = p->mat_id;
	    ARRfree(props);

	    cs++;

	    /* set nsteps */
            if (time_step == 1) {  /* in case of time-dependent data */
	        if (FLDset_time_nsteps(cell_set, num_steps) != 1) {
	            ERR_RETURN("Can't set nsteps of cell_set of time-dependent field.");
                }
            }
	}
    }
    else {
        free(node_ids);
        free(node_ind);
    }

    if (time_step >= 2) {
        /* skip cell connectivity */
	for (i = 0; i < *num_cells; i++) {
	    if (fgets( line, sizeof( line ), fp ) == NULL) {
		ERR_RETURN("Can't read cell connectivity.");
            }
        }
    }
    return 1;
}


/**********************************************************************
 *
 *	function name:	read_data
 *
 *	description: 	read node / cell data
 *
 *	return:		1: success 
 *			0: failure 
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
read_data(
FILE 	*fp,			/* file pointer (input) */
xp_long num_data,		/* number of node or cell (input) */
int 	sum_veclen,		/* sum of veclen of each component */
float 	*data,			/* data (output) */
char 	*data_labels,		/* data label (ouput) */
char 	*units,			/* data unit (ouput) */
int	*num_comp,		/* number of component (ouput) */
int 	*comp_list		/* component list (output) */
)
{
  char	line[80];
  int 	num_vals, i, j, n, m, num;
  xp_long offset, node, id;
  float value;
  int 	sum;


  num = fscanf (fp, "%d", num_comp);
  if( num != 1 ) {
    ERR_RETURN("Can't read step component number");
  }
  if (*num_comp > sum_veclen) {
    ERR_RETURN("Number of components is greater than number of data.");
  }


  sum = 0;
  for (i = 0; i < *num_comp; i++) {
    num = fscanf (fp, "%d", &comp_list[i]);
    if( num != 1 ) {
      ERR_RETURN("Can't read step comp_list.");
    }
    sum += comp_list[i];

  }
  if (sum != sum_veclen) {
      ERR_RETURN("Sum of veclen is not coincident with number ob data.");
  }

  fgets (line, BUFFER_SIZE, fp);

  for (i = 0, n = 0, m = 0; i < *num_comp; i++) {

    if (fgets (line, BUFFER_SIZE, fp) == NULL) {
      ERR_RETURN("Can't read comp_label and unit_label");
    }

    for (j = 0; line[j] == ' '; j++);

    for (; line[j] != ','; j++) {
      data_labels[n++] = line[j];
      if (n >= BUFFER_SIZE - 1) {
          ERR_RETURN("Can't read comp_label.");
      }
    }

    for (++j; line[j] == ' '; j++);

    for (; line[j] != '\n'; j++) {
      units[m++] = line[j];
      if (m >= BUFFER_SIZE - 1) {
          ERR_RETURN("Can't read unit_label.");
      }
    }

    if (i != *num_comp - 1) {
      data_labels[n++] = '.';
      units[m++] = '.';
      if (m >= BUFFER_SIZE - 1) {
        ERR_RETURN("Can't read comp_label");
      }
    }
  }
  data_labels[n] = '\0';
  units[m] = '\0';

  for (node = 0; node < num_data; node++) {

    num = fscanf (fp, "%ld", &id);

    for (offset = i = 0; i < *num_comp; i++) {

      num_vals = comp_list[i];

      for (j = 0; j < num_vals; j++) {

	num = fscanf (fp, "%f",  &value);

	if (num != 1) {
	  ERR_RETURN("Can't read step data value");
	}
	data[node * num_vals + offset + j] = value;
      }

      offset += num_data * num_vals;
    }
  }

  fgets(line, BUFFER_SIZE, fp);

  return 1;
}

/**********************************************************************
 *
 *	function name:  read_node_cell_data
 *
 *      description:    read data of specified step
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
read_node_cell_data(
FILE		*fp,		/* file pointer */
xp_long		num_nodes,	/* number of nodes */
xp_long		num_cells,	/* number of cells */
OMobj_id	out,		/* object id of output ucd data */
OMobj_id	id,		/* object id of parent object */
struct cell	*Top,		/* cell information */
int		time_step	/* time step: if 0, output is single field */
)
{
    char	line[BUFFER_SIZE];
    char	labels[BUFFER_SIZE], units[BUFFER_SIZE];
    char	label[BUFFER_SIZE], unit[BUFFER_SIZE];
    int		i, m, cs, type;
    xp_long j, k, cnt, size;
    int		*comp_list, num_comp, veclen;
    xp_long offset;
    int		num_node_data, num_cell_data, num;
    float	*data, *out_data;
    struct cell	*p;
    OMobj_id	cell_set;
    OMobj_id	node_data_id, cell_data_id;
    int		first_num_comp, first_cell_comp;


    /******************************************/
    /* read number of node data and cell data */
    /******************************************/
    fgets(line, sizeof(line), fp);
    num = sscanf(line, "%d%d", &num_node_data, &num_cell_data);
    if (num != 2) {
	ERR_RETURN("Can't read num_ndata and num_cdata.");
    }

    /******************/
    /* read node data */
    /******************/
    if (num_node_data) { /* if node data exist */
	/* allocation */
	data = (float *)malloc(sizeof(float)*num_nodes*num_node_data);
        if (data == NULL) {
	    ERR_RETURN("Can't alloc memory.(node_data)");
	}
	comp_list = (int *)malloc(sizeof(int)*num_node_data);
        if (comp_list == NULL) {
	    free(data);
	    ERR_RETURN("Can't alloc memory.(comp_list)");
	}

	/* read node data */
	if (read_data(fp, num_nodes, num_node_data, data, labels,
			   units, &num_comp, comp_list) != 1) {
	    free(data);
	    free(comp_list);
	    ERR_RETURN("Can't read node data.");
	}

	/* check or set number of components */
	if (time_step >= 2) {
	    if (FLDget_node_data_ncomp(out, &first_num_comp) != 1) {
		first_num_comp = 0;
	    }
	    if (num_comp != first_num_comp) {
	        free(data);
	        free(comp_list);
	        ERR_RETURN("Can't read this data under Store All Step mode because numbers of node components don't match between steps.")
	    }
	} else {
	    FLDset_node_data_ncomp(out, num_comp);
	}

	/* get time_node_data object id */
	if (time_step >= 1) { 	/* in case of time-dependent field */
	    if (FLDget_time_node_data(out, time_step-1, &node_data_id) != 1) {
	    	ERR_RETURN("Can't get time_node_data id.");
	    }
	} else { 	/* in case of single step field */
	    node_data_id = out;
	}

	for (i = 0, offset = 0; i < num_comp; i++) {

	    /* get a label of node data component */
	    if (!get_label(labels,'.', i, label)) {
		sprintf(label, "node_data_%d", i );
	    }

	    /* get a unit of node data component */
	    if (!get_label(units,'.', i, unit)) {
		sprintf(unit, " ");
	    }

	    /* set node data component information */
	    veclen = comp_list[i];
	    FLDset_node_data_comp(node_data_id, i, veclen, label, unit);

	    /* set node data value */
	    FLDset_node_data(node_data_id, i, (char *)&data[offset],
			DTYPE_FLOAT, veclen*num_nodes, OM_SET_ARRAY_COPY );

	    offset += (veclen * num_nodes);
	}
	free(data);
	free(comp_list);

    } else {
	if (time_step >= 2) {
	    if (FLDget_node_data_ncomp(out, &first_num_comp) == 1) {
	       if (first_num_comp > 0) {
	           ERR_RETURN("Can't read this data under Store All Step mode because numbers of node components don't match between steps.")
               }
	    }
        }
    }


    /******************/
    /* read cell data */
    /******************/
    if (num_cell_data) { /* if cell data exist */
	/* allocation */
	data = (float *)malloc(sizeof(float)*num_cells*num_cell_data);
	if (data == NULL) {
	    ERR_RETURN("Can't alloc memory.(node_data)");
	}
	comp_list = (int *)malloc(sizeof(int)*num_cell_data);
	if (comp_list == NULL) {
	    free(data);
	    ERR_RETURN("Can't alloc memory.(comp_list)");
	}

	/* read cell data */
	if (read_data(fp, num_cells, num_cell_data, data, labels,
		       units, &num_comp, comp_list) != 1) {
	    free(data);
            free(comp_list);
            ERR_RETURN("Can't read cell data.");
        }

	/* set cell data */
	for (p = Top, cs = 0; p != NULL; p = p->next, cs++) {

	    /* get cell_set id */
	    if (FLDget_cell_set(out, cs, &cell_set) != 1) {
		ERR_RETURN("Can't get cell_set.")
	    }

	    /* set number of components */
	    if (time_step >= 2) {
		if (FLDget_cell_data_ncomp(cell_set, &first_cell_comp) != 1) {
		    first_cell_comp = 0;
		}
		if (num_comp != first_cell_comp) {
		    free(data);
		    free(comp_list);
		    ERR_RETURN("Can't read this data under Store All Step mode because numbers of cell components don't match between steps.")
		}
	    } else {
		FLDset_cell_data_ncomp(cell_set, num_comp);
	    }

	    /* get time-dependent cell data id */
	    if (time_step >= 1) {
	        if (FLDget_time_cell_data(cell_set, time_step-1,
			&cell_data_id) != 1) {
		    free(data);
		    free(comp_list);
		    ERR_RETURN("Can't get time_cell_data id.");
		}
	    } else {
	    	cell_data_id = cell_set;
	    }

	    for (i = 0, offset = 0; i < num_comp; i++) {

	        /* get a label of cell data component */
	        if (!get_label(labels,'.', i, label)) {
	            sprintf(label, "cell_data_%d", i);
	        }

	        /* get a unit of cell data component */
	        if (!get_label(units,'.', i, unit)) {
	            sprintf(unit, " ");
	        }

	        /* set node data compornent information */
	        veclen = comp_list[i];
		if (FLDset_cell_data_comp(cell_data_id, i, veclen, label,
			unit ) != 1) {
		    free(data);
		    free(comp_list);
		    ERR_RETURN("Can't set cell_data_comp.");
                }

		/* set cell data value */
		type = DTYPE_FLOAT;
		if (FLDget_cell_data(cell_data_id, i, &type,
			(char **)&(out_data), &size, OM_GET_ARRAY_WR) != 1) {
		    free(data);
		    free(comp_list);
		    ERR_RETURN("Can't get cell_data array.");
                }
		for (j = cnt = 0; j < p->cell_num; j++) {
		    k = p->cell_no[j];
		    for (m = 0; m < veclen; m++, cnt++) {
			out_data[cnt] = data[offset+veclen*k+m];
		    }
		}
		offset += (veclen * num_cells);
		ARRfree(out_data);
	    }
	}

	free(data);
	free(comp_list);

    } else {
	if (time_step >= 2) {
	    if (FLDget_cell_set(out, 0, &cell_set) != 1) {
		ERR_RETURN("Can't get cell_set.")
	    }
	    if (FLDget_cell_data_ncomp(cell_set, &first_cell_comp) == 1) {
		if (first_cell_comp > 0) {
		    ERR_RETURN("Can't read this data under Store All Step mode because numbers of cell components don't match between steps.")
		}
	    }
	}
    }

    return 1;
}

/**********************************************************************
 *
 *	function name:  copy_coord 
 *
 *      description:    copy coord data of the 1st step to other steps. 
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Only Modified Internally */
static int
copy_coord(
OMobj_id	out_all, 	/* ouput time-dependent field */
int		num_steps 	/* number of steps */
) {

    int		i;
    OMobj_id	coord_id;
    float	*coord;
    xp_long	size;


    /* get node data of first step */
    if (FLDget_time_coordinates (out_all, 0, &coord_id) != 1) {
	ERR_RETURN("Can't get coord_id of output time-dependent field.");
    }
    if (FLDget_coord(coord_id, &coord, &size, OM_GET_ARRAY_RD) != 1) {
        ERR_RETURN("Can't get coordinate.");
    }

    /* set coordinate of other steps */
    for (i = 1; i < num_steps; i++) {
        if (FLDget_time_coordinates (out_all, i, &coord_id) != 1) {
	    ERR_RETURN("Can't get coord_id of output time-dependent field.");
        }
        if (FLDset_coord(coord_id, coord, size, OM_SET_ARRAY_STATIC) != 1) {
            ERR_RETURN("Can't get coordinate.");
        }
    }

    ARRfree(coord);

    return(1);
}


/**********************************************************************
 *
 *	function name:  copy_node_cell 
 *
 *      description:    copy node and/or cell data of the 1st step to 
 *			other steps.
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Directly Modified */
static int
copy_node_cell(
OMobj_id	out_all, 	/* ouput time-dependent field */
int		num_steps,	/* number of steps */
xp_long	num_nodes,
xp_long	num_cells
) {
    int		node_ncomp;
    OMobj_id	node_data_id, *data_id_list;
    int		i, j;
    char	unit[BUFFER_SIZE], label[BUFFER_SIZE];
    int		veclen, type;
    xp_long data_size;
    float	*node_data, *cell_data;
    int		step;
    int		ncell_sets, cell_ncomp;
    OMobj_id	cell_set_id, cell_data_id;

    /******************/
    /* copy node data */
    /******************/
    node_ncomp = 0;
    if (FLDget_node_data_ncomp (out_all, &node_ncomp) != 1) {
        node_ncomp = 0;
    }

    if (node_ncomp > 0) {
	data_id_list = (OMobj_id *) malloc(num_steps * sizeof(OMobj_id));
	if (!data_id_list) {
	    ERR_RETURN("Can't alloc data_id_list.");
        }
        for (i = 0; i < num_steps; i++) {
	    if (FLDget_time_node_data(out_all, i, &node_data_id) != 1) {
		free(data_id_list);
		ERR_RETURN("Can't get first time_node_data id.");
	    }
	    data_id_list[i] = node_data_id;
        }

	for (i = 0; i < node_ncomp; i++) {
	    if (FLDget_node_data_units(data_id_list[0], i, unit,
			BUFFER_SIZE) != 1) {
		free(data_id_list);
		ERR_RETURN("Can't get first node_data_units.");
            }
	    if (FLDget_node_data_label(data_id_list[0], i, label,
			BUFFER_SIZE) != 1) {
		free(data_id_list);
		ERR_RETURN("Can't get first node_data_label.");
            }
	    if (FLDget_node_data_veclen(data_id_list[0], i, &veclen) != 1) {
		free(data_id_list);
		ERR_RETURN("Can't get first node_data_velcen.");
            }
	    if (FLDget_node_data(data_id_list[0], i, &type,
		    (char **)&node_data, &data_size, OM_GET_ARRAY_RD) != 1) {
		free(data_id_list);
		ERR_RETURN("Can't get first node_data.");
            }

	    for (step = 1; step < num_steps; step++) {
		if (FLDset_node_data_comp(data_id_list[step], i, veclen,
			label, unit) != 1) {
		    free(data_id_list);
		    ERR_RETURN("Can't set node_data_comp.");
                }
		if (FLDset_node_data(data_id_list[step], i, (char *)node_data,
			type, data_size, OM_SET_ARRAY_STATIC) != 1) {
		    free(data_id_list);
		    ERR_RETURN("Can't set node_data.");
                }
            }
        }
        free(data_id_list);
    }

    /******************/
    /* copy cell data */
    /******************/
    if (FLDget_ncell_sets(out_all, &ncell_sets) != 1) {
	ERR_RETURN("Can't get ncell_sets.");
    }
    if (FLDget_cell_set(out_all, 0, &cell_set_id) != 1) {
	ERR_RETURN("Can't get cell_set id.");
    }
    cell_ncomp = 0;
    if (FLDget_cell_data_ncomp(cell_set_id, &cell_ncomp) != 1) {
	return(1);	/* no cell data */
    }
    if (cell_ncomp == 0) {
	return(1);
    }

    for (j = 0; j < ncell_sets; j++) {
	if (FLDget_cell_set(out_all, j, &cell_set_id) != 1) {
	    ERR_RETURN("Can't get cell_set id.");
	}


	for (i = 0; i < cell_ncomp; i++) {
	    if (FLDget_time_cell_data(cell_set_id, 0, &cell_data_id) != 1) {
	        ERR_RETURN("Can't get cell_data id.");
	    }
	    if (FLDget_cell_data_units(cell_data_id, i, unit,
			    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get first cell_data_units.");
	    }
	    if (FLDget_cell_data_label(cell_data_id, i, label,
			    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get first cell_data_label.");
	    }
	    if (FLDget_cell_data_veclen(cell_data_id, i, &veclen) != 1) {
		ERR_RETURN("Can't get cell_data_veclen.");
	    }
	    if (FLDget_cell_data(cell_data_id, i, &type,
		    (char **)&cell_data, &data_size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Can't get first node_data.");
	    }

	    for (step = 1; step < num_steps; step++) {
		if (FLDget_time_cell_data(cell_set_id, step,
					    &cell_data_id) != 1) {
		    ERR_RETURN("Can't get cell_data_id.");
		}
		if (FLDset_cell_data_comp(cell_data_id, i, veclen,
			label, unit) != 1) {
		    ERR_RETURN("Can't set cell_data_comp.");
		}
		if (FLDset_cell_data(cell_data_id, i, (char *)cell_data,
			    type, data_size, OM_SET_ARRAY_STATIC) != 1) {
		    ERR_RETURN("Can't set cell_data.");
		}
	    }
	}
    }
    return(1);
}


/**********************************************************************
 *
 *	function name:  isdigit 
 *
 *      description:    check if input string is numerical value or not 
 *
 *	return:		1: numerical value 
 *			0: not numerical value 
 *
 **********************************************************************/

static int
isvalue(const char *str_array)
{
   int i, len, flag;

   flag = 1;
   len = (int)strlen(str_array);
   for(i = 0; i < len; i++)
   {
     if(( str_array[i] < 0x30 &&  str_array[i] != 0x2e)  ||
            ( str_array[i] > 0x39 ))
     {
       flag = 0;
       break;
     }
   }
   return(flag);
}


/**********************************************************************
 *
 *	function name:  set_time 
 *
 *      description:    set time in time-dependent field. 
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/

static int
set_time(
OMobj_id	out_all,
int	num_steps,
char	(*all_comments)[BUFFER_SIZE]	/* comments of all steps */
)
{
    int		i;
    double	*time;
    char	comment[BUFFER_SIZE];
    int		float_flag;
    OMobj_id 	set_id;
    int		ncell_sets;

    time = (double *) malloc(num_steps * sizeof(double));
    if (time == NULL) {
	ERR_RETURN("Can't alloc time.");
    }

    float_flag = 1;
    for (i = 0; i < num_steps; i++) {
	strcpy(comment, all_comments[i]);
	if (isvalue(comment) != 1) {
            float_flag = 0;
	    break;
	}
	if (sscanf(comment, "%lf", &time[i]) != 1) {
            float_flag = 0;
	    break;
        }
    }

    if (float_flag == 0) {
		/* if any comment is not numerical value, sequential
                   numbers are set into time */
	for (i = 0; i < num_steps; i++) {
	    time[i] = i;
	}
    }

    if (FLDset_time_values(out_all, time, num_steps,
				OM_SET_ARRAY_COPY) != 1) {
        free(time);
	ERR_RETURN("Can't set time.");
    }

    if (FLDget_ncell_sets(out_all, &ncell_sets) != 1) {
	free(time);
	ERR_RETURN("Can't get ncell_sets");
    }

    for (i = 0; i < ncell_sets; i++) {
	if (FLDget_cell_set(out_all, i, &set_id) != 1) {
	    free(time);
	    ERR_RETURN("Can't get cell_set id.");
	}
	if (FLDset_time_values(set_id, time, num_steps,
				    OM_SET_ARRAY_COPY) != 1) {
	    free(time);
	    ERR_RETURN("Can't set time.");
        }
    }
    free(time);

    return(1);
}


/**********************************************************************
 *
 *	function name:  read_all_step
 *
 *      description:    read and set data of all steps
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Only Modified Internally */
static int
read_all_step(
FILE 		*fp,
int		num_steps,
char		*cycle_type,
OMobj_id	out_all, 	/* ouput time-dependent field */
char		(*all_comments)[BUFFER_SIZE],
struct cell	**Top,		/* cell information */
OMobj_id	id		/* object id of parent object */
)
{
	char	comment[BUFFER_SIZE];
	xp_long	num_nodes, num_cells;
	xp_long num_nodes_b, num_cells_b;
	int	step;
	char	err_mes[128];

	/********************************/
	/* read and set first step data */
	/********************************/
	/* set num_steps */
	if (FLDset_time_nsteps(out_all, num_steps) != 1) {
		ERR_RETURN("Can't set num_steps.");
	}

	/* read and set first step */
	if(binary_format == 1){
		if (read_step_comment_binary(1, comment) != 1) {
			if (b_close() != 1) {
				ERR_RETURN("Can't close binary file.");
			}
			ERR_RETURN("Can't find 'step' or step number is incorrect.(binary)");
		}
	} else {
		if (read_step_comment(fp, 1, comment) != 1) {
			ERR_RETURN("Can't find 'step' or step number is incorrect.");
		}
	}
	strcpy(all_comments[0], comment);

	if(binary_format == 1){
		if (read_geom_data_binary(&num_nodes_b, &num_cells_b, out_all, Top, 1, num_steps) != 1) {
			delete_cell_inf(Top);
			ERR_RETURN("Can't set geometry data of the first step.");
		}
		if (read_node_cell_data_binary(num_nodes_b, num_cells_b, out_all, id, *Top, 1) != 1) {
			delete_cell_inf(Top);
			ERR_RETURN("Can't set node or cell data of the first step.");
		}
		if (b_close() != 1) {
			ERR_RETURN("Can't close binary file.");
		}
	} else {
		if (read_geom_data(fp, &num_nodes, &num_cells, out_all, Top, 1, num_steps) != 1) {
			delete_cell_inf(Top);
			ERR_RETURN("Can't set geometry data of the first step.");
		}
		if (read_node_cell_data(fp, num_nodes, num_cells, out_all, id, *Top, 1) != 1) {
			delete_cell_inf(Top);
			ERR_RETURN("Can't set node or cell data of the first step.");
		}
	}

	/**************************************************/
	/* read and set second and after second step data */
	/**************************************************/
	for (step = 2; step <= num_steps; step++) {
		if(binary_format == 1){
			if (b_open(step) != 1) {
				return 0;
			}
			if (read_step_comment_binary(step, comment) != 1) {
				sprintf(err_mes, "Error in step %d: Can't find 'step' or step number is incorrect.(binary)", step);
				ERR_RETURN(err_mes);
			}
		} else {
			if (read_step_comment(fp, step, comment) != 1) {
				sprintf(err_mes, "Error in step %d: Can't find 'step' or step number is incorrect.", step);
				ERR_RETURN(err_mes);
			}
		}
		strcpy(all_comments[step-1], comment);

		if(binary_format == 1){
			if (strcmp(cycle_type, "data_geom") == 0 || strcmp(cycle_type, "geom") == 0) {
				if (read_geom_data_binary(&num_nodes_b, &num_cells_b, out_all, Top, step, num_steps) != 1) {
					delete_cell_inf(Top);
					sprintf(err_mes, "Error in step %d: Can't set geometry data.", step);
					ERR_RETURN(err_mes);
				}
			}
			if (strcmp(cycle_type, "data_geom") == 0 || strcmp(cycle_type, "data") == 0) {
				if (read_node_cell_data_binary(num_nodes_b, num_cells_b, out_all, id, *Top, step) != 1) {
					delete_cell_inf(Top);
					sprintf(err_mes, "Error in step %d: Can't set node or cell data", step);
					ERR_RETURN(err_mes);
				}
			}
			if (b_close() != 1) {
				ERR_RETURN("Can't close binary file.");
			}
		} else {
			if (strcmp(cycle_type, "data_geom") == 0 || strcmp(cycle_type, "geom") == 0) {
				if (read_geom_data(fp, &num_nodes, &num_cells, out_all, Top, step, num_steps) != 1) {
					delete_cell_inf(Top);
					sprintf(err_mes, "Error in step %d: Can't set geometry data.", step);
					ERR_RETURN(err_mes);
				}
			}
			if (strcmp(cycle_type, "data_geom") == 0 || strcmp(cycle_type, "data") == 0) {
				if (read_node_cell_data(fp, num_nodes, num_cells, out_all, id, *Top, step) != 1) {
					delete_cell_inf(Top);
					sprintf(err_mes, "Error in step %d: Can't set node or cell data", step);
					ERR_RETURN(err_mes);
				}
			}
		}
	}

	if (strcmp(cycle_type, "data") == 0) {
		if (copy_coord(out_all, num_steps) != 1) {
			ERR_RETURN("Can't copy coordinates.");
		}
	}
	if (strcmp(cycle_type, "geom") == 0) {
		if(binary_format == 1) {
			if (copy_node_cell(out_all, num_steps, num_nodes_b, num_cells_b) != 1) {
				ERR_RETURN("Can't copy node and cell data.");
			}
		}
		else {
			if (copy_node_cell(out_all, num_steps, num_nodes, num_cells) != 1) {
				ERR_RETURN("Can't copy node and cell data.");
			}
		}
	}

	if (set_time(out_all, num_steps, all_comments) != 1) {
		ERR_RETURN("Can't set time.");
	}

	return(1);

}


/**********************************************************************
 *
 *	function name: 	extract_one_step 
 *
 *      description:    extract one step data from time-dependent field
 *			and set it into single field
 *
 *	return:		1: success 
 *			0: failure
 *
 **********************************************************************/
/* 64-bit porting. Only Modified Internally */
static int
extract_one_step(
OMobj_id	out_all,	/* time-dependent field data */
int		step,		/* step number (count from 0) */
OMobj_id	out		/* output single field */
)
{

    xp_long	nnodes, size, ncells;
    int     node_ncomp, type, ncell_sets, prop_size;
    float	*coord, *node_data, *cell_data;
    int		i, j;
    char	name[10], label[BUFFER_SIZE], units[BUFFER_SIZE];
    OMobj_id	all_coord_id, all_node_data_id, cell_set_id,
		all_cell_set_id, all_cell_data_id;
    xp_long	*node_conn;
    int		cell_ncomp, veclen;
    float	*props, *all_props;


    if (FLDget_nnodes(out_all, &nnodes) != 1) {
	ERR_RETURN("Can't get nnodes.");
    }
    if (FLDset_nnodes(out, nnodes) != 1) {
	ERR_RETURN("Can't set nnodes.");
    }

    if (FLDset_nspace(out, 3) != 1) {
	ERR_RETURN("Can't set nspace.");
    }

    if (FLDget_time_coordinates(out_all, step, &all_coord_id) != 1) {
	ERR_RETURN("Can't get coord_id.");
    }
    if (FLDget_coord(all_coord_id, &coord, &size, OM_GET_ARRAY_RD) != 1) {
	ERR_RETURN("Can't get coord.");
    }
    if (FLDset_coord(out, coord, size, OM_SET_ARRAY_STATIC) != 1) {
	ERR_RETURN("Can't set coord.");
    }
    ARRfree(coord);

    if (FLDget_node_data_ncomp(out_all, &node_ncomp) != 1) {
	node_ncomp = 0;
    }

    if (node_ncomp > 0) {
	if (FLDset_node_data_ncomp(out, node_ncomp) != 1) {
	    ERR_RETURN("Can't get node_data_ncomp.");
	}

	if (FLDget_time_node_data(out_all, step, &all_node_data_id) != 1) {
	    ERR_RETURN("Can't get node_data_id.");
	}
	if (FLDset_node_data_ncomp(out, node_ncomp) != 1) {
	    ERR_RETURN("Can't set node_data_ncomp.");
	}

	if (FLDget_time_node_data(out_all, step, &all_node_data_id) != 1) {
	    ERR_RETURN("Can't get node_data_id.");
	}

	for (i = 0; i < node_ncomp; i++) {
	    if (FLDget_node_data_veclen(all_node_data_id, i, &veclen) != 1) {
		ERR_RETURN("Can't get node_data_veclen.");
	    }
	    if (FLDget_node_data_units(all_node_data_id, i, units,
				    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get node_data_units.");
	    }
	    if (FLDget_node_data_label(all_node_data_id, i, label,
				    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get node_data_label.");
	    }

	    if (FLDset_node_data_comp(out, i, veclen, label, units) != 1) {
		ERR_RETURN("Can't set node_data_comp.");
	    }

	    if (FLDget_node_data(all_node_data_id, i, &type, (char **)&node_data,
			     &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Can't get node_data.");
	    }
	    if (FLDset_node_data(out, i, (char *)node_data, type, size,
			    OM_SET_ARRAY_STATIC) != 1) {
		ERR_RETURN("Can't set node_data.");
	    }
	    ARRfree(node_data);
	}
    }

    if (FLDget_ncell_sets(out_all, &ncell_sets) != 1) {
	ERR_RETURN("Can't get ncell_sets.");
    }
    if (FLDset_ncell_sets(out, ncell_sets) != 1) {
	ERR_RETURN("Can't set ncell_sets.");
    }

    for (i = 0; i < ncell_sets; i++) {
	if (FLDget_cell_set(out_all, i, &all_cell_set_id) != 1) {
	    ERR_RETURN("Can't get cell_set id.");
	}
	if (FLDget_cell_set(out, i, &cell_set_id) != 1) {
	    ERR_RETURN("Can't get cell_set id.");
	}

	if (FLDget_cell_set_name(all_cell_set_id, name, 10) != 1) {
	    ERR_RETURN("Can't get cell_set id");
	}
	if (FLDset_cell_set(cell_set_id, name) != 1) {
	    ERR_RETURN("Can't set cell_set.");
	}

	if (FLDget_ncells(all_cell_set_id, &ncells) != 1) {
	    ERR_RETURN("Can't get ncells.");
        }
	if (FLDset_ncells(cell_set_id, ncells) != 1) {
	    ERR_RETURN("Can't set ncells.");
	}

        if (FLDget_node_connect(all_cell_set_id, &node_conn, &size,
		OM_GET_ARRAY_RD) != 1) {
	    ERR_RETURN("Can't get node_connect.");
	}
        if (FLDset_node_connect(cell_set_id, node_conn, size,
			    OM_SET_ARRAY_STATIC) != 1) {
	    ERR_RETURN("Can't get node_connect.");
	}
	ARRfree(node_conn);

	if (FLDset_cell_nprops (cell_set_id, 1) != 1) {
	    ERR_RETURN("Can't set nprops.");
	}
	if (FLDget_cell_props(all_cell_set_id, (float **)&all_props, &prop_size,
		    OM_GET_ARRAY_RD) != 1) {
	    ERR_RETURN("Can't get cell props of time-dependent field.");
	}
	if (FLDget_cell_props(cell_set_id, (float **)&props, &prop_size,
		    OM_GET_ARRAY_WR) != 1) {
	    ERR_RETURN("Can't get cell props.");
	}
	props[0] = all_props[0];
	ARRfree(props);
	ARRfree(all_props);

	if (FLDget_time_cell_data(all_cell_set_id, step,
			&all_cell_data_id) != 1) {
	    ERR_RETURN("Can't get time_cell_data id.");
	}

	if (FLDget_cell_data_ncomp(all_cell_data_id, &cell_ncomp) != 1) {
	    cell_ncomp = 0;
	}
	else {
	    if (FLDset_cell_data_ncomp(cell_set_id, cell_ncomp) != 1) {
		ERR_RETURN("Can't set cell_data_ncomp.");
	    }
        }

        for (j = 0; j < cell_ncomp; j++) {
	    if (FLDget_cell_data_veclen(all_cell_data_id, j, &veclen) != 1) {
		ERR_RETURN("Can't get cell_data_veclen.");
	    }
	    if (FLDget_cell_data_units(all_cell_data_id, j, units,
				    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get node_cell_units.");
	    }
	    if (FLDget_cell_data_label(all_cell_data_id, j, label,
				    BUFFER_SIZE) != 1) {
		ERR_RETURN("Can't get cell_data_label.");
	    }
	    if (FLDset_cell_data_comp(cell_set_id, j, veclen, label,
				units) != 1) {
		ERR_RETURN("Can't set cell_data_comp.");
	    }

	    if (FLDget_cell_data(all_cell_data_id, j, &type,
			(char **)&cell_data, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Can't get cell_data");
	    }
	    if (FLDset_cell_data(cell_set_id, j, (char *)cell_data, type, size,
			    OM_SET_ARRAY_STATIC) != 1) {
		ERR_RETURN("Can't set cell_data");
	    }
	}
    }
    return(1);
}


/**********************************************************************
 *
 *	function name:  DVread_ucd3_update
 *
 *      description:	update function (method)
 *
 *	return:		1: 
 *			0: 
 *
 **********************************************************************/
/* 64-bit porting. Only Modified Internally */
int
DVread_ucd3_update(
OMobj_id	id,
OMevent_mask	event_mask,
int		seq_num
)
{
	char		cycle_type[BUFFER_SIZE];
	char		*cycle_ptr;
	int		new_format;
	char		comment[BUFFER_SIZE];
	int		num_steps;
	xp_long num_nodes, num_cells;
	xp_long num_nodes_b, num_cells_b;
	char		*filename = NULL, err_mes[128];
	int		current_step, chg_step;
	OMobj_id	out, chk_id;
	FILE		*FilePtr;
	long		*FilePos;
	struct cell	*Top;
	int		ncell_sets;
	xp_long ncells;
	OMobj_id	cell_set;
	int		i, L;
	int		store_all, store_all_visible;
	char		(*all_comments)[BUFFER_SIZE];
	OMobj_id	out_all;
	int		file_changed, store_changed;

	/*************************************/
	/* check if filename is input or not */
	/*************************************/
	chk_id = OMfind_subobj(id, OMstr_to_name("filename"), OM_OBJ_RD);
	if (OMis_null_obj(chk_id)) {
		ERR_RETURN("Can't get 'filename' object id.");
	}
	if (OMchanged(chk_id, seq_num)) {
		file_changed = 1;
	} else {
		file_changed = 0;
	}

	/*******************************************************/
	/* check if store_all is changed or not  and its value */
	/*******************************************************/
	chk_id = OMfind_subobj(id, OMstr_to_name("store_all"), OM_OBJ_RD);
	if (OMis_null_obj(chk_id)) {
		ERR_RETURN("Can't get 'store_all' object id.");
	}
	if (OMchanged(chk_id, seq_num)) {
		store_changed = 1;
	} else {
		store_changed = 0;
	}

	/* get store_all */
	if (OMget_name_int_val(id, OMstr_to_name("store_all"), &store_all) != 1) {
		store_all = 0;
	}

	/***************************************************/
	/* process in case filname or store_all is changed */
	/***************************************************/
	if (file_changed) {
		/* set store_all = 0 */
		OMset_name_int_val(id, OMstr_to_name("store_all"), 0);
		store_all = 0;
		OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 0);
		store_all_visible = 0;
	}

	if (store_changed && store_all == 0) {
		/* set store_all_visible = 0 */
		OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 0);
		store_all_visible = 0;
	}

	if (file_changed || store_changed) {
		/* reset filename ptr for binary data */
		if (b_file != NULL && number_of_file != 0) {
			for (L = 0; L < number_of_file; L++) {
				if(b_file[L] != NULL) {
					free(b_file[L]);
				}
			}
			free(b_file);
			number_of_file = 0;
		}

		/* reset current_step and total */
		OMset_name_int_val(id, OMstr_to_name("current_step"), 0);
		OMset_name_int_val(id, OMstr_to_name("total"), 0 );

		/* free cell information */
		if (OMget_name_ptr_val(id, OMstr_to_name("top"), (void *)&Top, 0) != 1) {
			ERR_RETURN("Can't get saved top.");
		}
		delete_cell_inf(&Top);
		if (OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0) != 1) {
			ERR_RETURN("Can't save top.");
		}

		/* get object ID of UCD ouput object */
		out = OMfind_subobj(id, OMstr_to_name("out"), OM_OBJ_RW);
		if (OMis_null_obj(out)) {
			ERR_RETURN("Can't get 'out' object id.");
		}
		out_all = OMfind_subobj(id, OMstr_to_name("out_all"), OM_OBJ_RW);
		if (OMis_null_obj(out_all)) {
			ERR_RETURN("Can't get 'out_all' object id.");
		}

		/* get input filename */
		filename = NULL;
		if (OMget_name_str_val(id, OMstr_to_name("filename"), &filename, 0) != 1) {
			ERR_RETURN("Can't get filename.");
		}
		if (filename == NULL) {
			ERR_RETURN("Please select filename.");
		}

		/* initialize output UCD data */
		reset_out_data(out, 0);
		reset_out_data(out_all, 1);

		/* open file */
		if (OMget_name_ptr_val(id, OMstr_to_name("file_ptr"), (void *) &FilePtr, 0) != 1) {
			ERR_RETURN("Can't get file_ptr");
		}
		if (FilePtr != NULL) {
			fclose(FilePtr);
			FilePtr = NULL;
			if (OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), FilePtr, 0) != 1) {
				free(filename);
				ERR_RETURN("Can't save file_ptr.");
			}
		}
		FilePtr = FILEfopen(filename, SIO_R_TXT);
		if (FilePtr == NULL) {
			sprintf(err_mes, "Can't open file[%s]", filename);
			free(filename);
			ERR_RETURN(err_mes);
		}

		/* read num_steps and cycle_type */
		cycle_type[0] = '\0';
		if (read_header(filename, FilePtr, &num_steps, cycle_type) == 0) {
			fclose(FilePtr);
			free(filename);
			/* free binary filename ptr */
			if (b_file != NULL && number_of_file != 0) {
				for (L = 0; L < number_of_file; L++) {
					if(b_file[L] != NULL) {
						free(b_file[L]);
					}
				}
				free(b_file);
				number_of_file = 0;
			}
			ERR_RETURN("Can't read header section.");
		}

		/* store cycle_type */
		if (OMset_name_str_val(id, OMstr_to_name("cycle_type"), cycle_type) != 1) {
			ERR_RETURN("Can't save cycle_type.");
		}

		/* in case of new (time-dependent) format */
		if (strcmp(cycle_type, "data") == 0 ||
			strcmp(cycle_type, "geom") == 0 ||
			strcmp(cycle_type, "data_geom") == 0) {

			new_format = 1;

#ifdef TIME
stime=clock() ;
printf("time=%d\n",stime);
#endif

			if(binary_format == 1) {
				if (b_open(1) != 1) {
					fclose(FilePtr);
					free(filename);
					/* free binary filename ptr */
					if (b_file != NULL && number_of_file != 0) {
						for (L = 0; L < number_of_file; L++) {
							if(b_file[L] != NULL) {
								free(b_file[L]);
							}
						}
						free(b_file);
						number_of_file = 0;
					}
					b_close();
					return 0;
				}
			}

			if (file_changed) {
				/* set store_all_visible = 1 */
				OMset_name_int_val(id, OMstr_to_name("store_all_visible"), 1);
				store_all_visible = 1;
			}

			if (num_steps < 1) {
				fclose(FilePtr);
				free(filename);
				/* free binary filename ptr */
				if (b_file != NULL && number_of_file != 0) {
					for (L = 0; L < number_of_file; L++) {
						if(b_file[L] != NULL) {
							free(b_file[L]);
						}
					}
					free(b_file);
					number_of_file = 0;
				}
				b_close();
				ERR_RETURN("num step must be greater than 0");
			}

			/* in case data of all steps will be stored */
			if (store_all == 1) {

				/* free and malloc all_comments */
				if (OMget_name_ptr_val(id, OMstr_to_name("all_comments"), (void *) &all_comments, 0) != 1) {
					ERR_RETURN("Can't restore all_comments.");
				}

				if (all_comments != NULL) {
					free (all_comments);
				}
				all_comments = (char (*)[BUFFER_SIZE])malloc(num_steps * BUFFER_SIZE);
				if (OMset_name_ptr_val(id, OMstr_to_name("all_comments"), all_comments, 0) != 1) {
					ERR_RETURN("Can't store all_comments.");
				}

				if (all_comments == NULL) {
					ERR_RETURN("Can't allocate all_comments");
				}

				/* read and set data of all steps */
				if (read_all_step(FilePtr, num_steps, cycle_type, out_all, all_comments, &Top, id) != 1 ) {
					fclose (FilePtr);
					free(filename);
					/* free binary filename ptr */
					if (b_file != NULL && number_of_file != 0) {
						for (L = 0; L < number_of_file; L++) {
							if(b_file[L] != NULL) {
								free(b_file[L]);
							}
						}
						free(b_file);
						number_of_file = 0;
					}
					b_close();
					return(0);
				}

				/* store Top (cell information) */
				if (OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0) != 1) {
					ERR_RETURN("Can't save cell_info");
				}

				if (extract_one_step(out_all, 0, out) != 1) {
					fclose (FilePtr);
					free(filename);
					/* free binary filename ptr */
					if (b_file != NULL && number_of_file != 0) {
						for (L = 0; L < number_of_file; L++) {
							if(b_file[L] != NULL) {
								free(b_file[L]);
							}
						}
						free(b_file);
						number_of_file = 0;
					}
					b_close();
					ERR_RETURN("Can't extract one step.");
				}

				/* set comment to object */
				if (OMset_name_str_val(id, OMstr_to_name("step_title"), all_comments[0]) != 1) {
					ERR_RETURN("Can't set step_title.");
				}

			/* in case data of single step will be stored */
			} else {

				/* check step and get step comment string */
				comment[0] = '\0';
				if(binary_format == 1){
					if (read_step_comment_binary(1, comment) != 1) {
						b_close() ;
						ERR_RETURN("Can't find 'step' or step number is incorrect.(binary)");
					}
				} else {
					if (read_step_comment(FilePtr, 1, comment) != 1) {
						fclose(FilePtr);
						free(filename);
						ERR_RETURN("Can't find 'step' or step number is incorrect.");
					}
				}

				/* set comment to object */
				if (OMset_name_str_val(id, OMstr_to_name("step_title"), comment) != 1) {
					ERR_RETURN("Can't set step_title.");
				}

				/* read and set geometry data of the first step */
				if(binary_format == 1) {
					if (read_geom_data_binary(&num_nodes_b, &num_cells_b, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						b_close() ;
						ERR_RETURN("Can't set geometry data of the first step.");
					}
				} else {
					if (read_geom_data(FilePtr, &num_nodes, &num_cells, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						fclose(FilePtr);
						free(filename);
						ERR_RETURN("Can't set geometry data of the first step.");
					}
				}

				/* store Top (cell information) */
				if (OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0) != 1) {
					ERR_RETURN("Can't save cell_info");
				}

				/* read and set node and cell data of the first step */
				if(binary_format == 1){
					if(read_node_cell_data_binary(num_nodes_b, num_cells_b, out, id, Top, 0) != 1 ){
						fclose(FilePtr);
						free(filename);
						/* free binary filename ptr */
						if (b_file != NULL && number_of_file != 0) {
							for (L = 0; L < number_of_file; L++) {
								if(b_file[L] != NULL) {
									free(b_file[L]);
								}
							}
							free(b_file);
							number_of_file = 0;
						}
						b_close();
						ERR_RETURN("Can't set node or cell data of the first step.");
					}
				} else {
					if(read_node_cell_data(FilePtr, num_nodes, num_cells, out, id, Top, 0) != 1 ){
						fclose(FilePtr);
						free(filename);
						ERR_RETURN("Can't set node or cell data of the first step.");
					}
				}

				/* get file pointer of beginning of each step */
				if (OMget_name_ptr_val(id, OMstr_to_name("file_pos"), (void *) &FilePos, 0) != 1) {
					ERR_RETURN("Can't restore file_pos.");
				}
				if (FilePos != NULL) {
					free(FilePos);
					FilePos = NULL;
					if (OMset_name_ptr_val(id, OMstr_to_name("file_pos"), FilePos, 0) != 1) {
						ERR_RETURN("Can't store file_pos.");
					}
				}

				if(binary_format == 1){
					if (b_close() != 1) {
						ERR_RETURN("Can't close binary file.");
					}
				} else {
					if (get_step_file_pos(FilePtr, num_steps, cycle_type, &FilePos) != 1) {
						delete_cell_inf(&Top);
						fclose(FilePtr);
						free(filename);
						ERR_RETURN("Can't get position of each step.");
					}
				}

				/* store file position */
				if (OMset_name_ptr_val(id, OMstr_to_name("file_pos"), FilePos, 0) != 1) {
					ERR_RETURN("Can't store file_pos.");
				}
			}

#ifdef TIME
printf("time=%d %d\n",clock(),clock()-stime);
#endif

		/* in case of old format */
		} else {
			new_format = 0;
			num_steps = 1;

			fclose(FilePtr);
			FilePtr = NULL;
			FUNCread_ucd(filename, out);
			if (OMset_name_str_val(id, OMstr_to_name("step_title"), "") != 1) {
				ERR_RETURN("Can't set step_title.");
			}
		}

		/* set total number of steps to the parameter object */
		if (OMset_name_int_val(id, OMstr_to_name("total"), num_steps) != 1) {
			ERR_RETURN("Can't set total.");
		}

		/* set 1 to current step when new file is read */
		current_step = 1;
		if (OMset_name_int_val(id, OMstr_to_name("current_step"), current_step) != 1) {
			ERR_RETURN("Can't set current_step.");
		}

		/* store file pointer */
		if (OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), FilePtr, 0) != 1) {
			ERR_RETURN("Can't store file_ptr.");
		}

		/* store new_format */
		if (OMset_name_int_val(id, OMstr_to_name("new_format"), new_format) != 1) {
			ERR_RETURN("Can't store new_format.");
		}

		free(filename);
		return(1);
	}

	/***************************************/
	/* check paramter and get current step */
	/***************************************/

	/* get store_all */
	if (OMget_name_int_val(id, OMstr_to_name("store_all"), &store_all) != 1) {
		ERR_RETURN("Can't get Store All Steps.");
	}

	/* initilize change_of_step flag */
	chg_step = 0;

	/* check file is open (selected) */
	FilePtr = NULL;
	if (OMget_name_ptr_val(id, OMstr_to_name("file_ptr"), (void *) &FilePtr, 0) != 1) {
		ERR_RETURN("Can't restore file_ptr.");
	}
	if (FilePtr == NULL) {
		return (0);
	}

	/* get current_step */
	if (OMget_name_int_val(id, OMstr_to_name("current_step"), &current_step) != 1) {
		ERR_RETURN("Can't get Current Step No.");
	}

	/* get num_steps (total steps) */
	if (OMget_name_int_val(id, OMstr_to_name("total"), &num_steps) != 1) {
		ERR_RETURN("Can't get Total Steps.");
	}

	/* check if forward button was clicked */
	chk_id = OMfind_subobj(id, OMstr_to_name("forward"), OM_OBJ_RD);
	if (OMis_null_obj(chk_id)) {
		ERR_RETURN("Can't get 'forward' object id.");
	}
	if (OMchanged(chk_id, seq_num)) { /* in case forward was clicked */
		/* set current_step (increment) */
		current_step++;
		/* in case current_step is over the total number of steps */
		if (current_step > num_steps) {
			/* return to the first step */
			current_step = 1;
		}

		/* set the value to current_step object */
		if (OMset_name_int_val(id, OMstr_to_name("current_step"), current_step) != 1) {
			ERR_RETURN("Can't set current_step.");
		}

		/* indicate step is changed */
		chg_step = 1;
	}

	/* check if backward button was clicked */
	chk_id = OMfind_subobj(id, OMstr_to_name("backward"), OM_OBJ_RD);
	if (OMis_null_obj(chk_id)) {
		ERR_RETURN("Can't get 'backward' object id.");
	}
	if (OMchanged(chk_id, seq_num)) { /* in case backward was clicked */
		/* set current_step (decrement) */
		current_step--;
		/* in case current_step is less than 1 */
		if (current_step < 1) {
			/* return to the last step */
			current_step = num_steps;
		}

		/* set the value to current_step object */
		if (OMset_name_int_val(id, OMstr_to_name("current_step"), current_step) != 1) {
			ERR_RETURN("Can't set current_step.");
		}

		/* indicate step is changed */
		chg_step = 1;
	}

	/* check if current_step parameter is changed */
	if (chg_step == 0) {
		chk_id = OMfind_subobj(id, OMstr_to_name("current_step"), OM_OBJ_RW);
		if (OMis_null_obj(chk_id)) {
			ERR_RETURN("Can't get 'current_step' object id.");
		}
		/* in case "current step" parameter is changed */
		if (OMchanged(chk_id, seq_num)) {

			/* check the validity of current_step */
			if (current_step < 1 || current_step > num_steps) {
				ERRerror("Read_UCD",1,ERR_ORIG, "Step number %d does not exist.", current_step);
				if (current_step < 1) {
					current_step = 1;
				}
				if (current_step > num_steps) {
					current_step = num_steps;
				}
				if (OMset_name_int_val(id, OMstr_to_name("current_step"), current_step) != 1) {
					ERR_RETURN("Can't set current_step.");
				}
			}

			/* indicate current step is changed */
			chg_step = 1;
		}
	}

	/* get new_format */
	if (OMget_name_int_val(id, OMstr_to_name("new_format"), &new_format) != 1) {
		ERR_RETURN("Can't restore new_format.");
	}

	/* if the format is old and filename is not changed, no data is */
	/* created.                                                     */
	if (new_format == 0) {
		return(1);
	}

	/*************************************/
	/* read and set data of current step */
	/*************************************/

	/* in case current step is changed */
	if (chg_step == 1) {
		/* in case of Store All Step */
		if (store_all == 1) {
			/* get ID of ucd output object */
			out = OMfind_subobj(id, OMstr_to_name("out"), OM_OBJ_RW);
			if (OMis_null_obj(out)) {
				ERR_RETURN("Can't get 'out' object id.");
			}
			out_all = OMfind_subobj(id, OMstr_to_name("out_all"), OM_OBJ_RW);
			if (OMis_null_obj(out_all)) {
				ERR_RETURN("Can't get 'out_all' object id.");
			}

			/* get all_comments */
			if (OMget_name_ptr_val(id, OMstr_to_name("all_comments"), (void *) &all_comments, 0) != 1) {
				ERR_RETURN("Can't restore all_comments.");
			}

			/* extract one step data from time-dependent field and set */
			/* it into single field */
			if (extract_one_step(out_all, current_step-1, out) != 1) {
				ERR_RETURN("Can't extract one step.");
			}
			/* set comment to object */
			if (OMset_name_str_val(id, OMstr_to_name("step_title"), all_comments[current_step-1]) != 1) {
				ERR_RETURN("Can't set step_title.");
			}

		/* in case of single step */
		} else {

			/* get Top (cell_info) */
			if (OMget_name_ptr_val(id, OMstr_to_name("top"), (void *) &Top, 0) != 1) {
				ERR_RETURN("Can't restore cell_info");
			}

			/* get cycle_type */
			if (OMget_name_str_val(id, OMstr_to_name("cycle_type"), (void *) &cycle_ptr, 0) != 1) {
				ERR_RETURN("Can't restore cycle_type");
			}

			if (cycle_ptr) {
				strcpy (cycle_type, cycle_ptr);
				free (cycle_ptr);
			}

			/* get FilePos */
			if (OMget_name_ptr_val(id, OMstr_to_name("file_pos"), (void *) &FilePos, 0) != 1) {
				ERR_RETURN("Can't restore file_pos");
			}

			/* set file pointer to specified step */
			if(binary_format == 1){
				if (b_open(current_step) != 1) {
					return 0;
				}
			} else {
				fseek(FilePtr, FilePos[current_step-1], SEEK_SET);
			}

			/* get ID of ucd output object */
			out = OMfind_subobj(id, OMstr_to_name("out"), OM_OBJ_RW);
			if (OMis_null_obj(out)) {
				ERR_RETURN("Can't get 'out' object id.");
			}

			/* check step and get step comment string */
			comment[0] = '\0';
			if(binary_format == 1){
				if (read_step_comment_binary(current_step, comment) != 1) {
					sprintf(err_mes, "Error in step %d: Can't find 'step' or step number is incorrect.(binary)", current_step);
					ERR_RETURN(err_mes);
				}
			} else {
				if (read_step_comment(FilePtr, current_step, comment) != 1) {
					sprintf(err_mes, "Error in step %d: Can't find 'step' or step number is incorrect.", current_step);
					ERR_RETURN(err_mes);
				}
			}

			/* set comment to object */
			if (OMset_name_str_val(id, OMstr_to_name("step_title"), comment) != 1) {
				ERR_RETURN("Can't set step_title.");
			}

			/* read geometry data, node data and cell data */
			if (strcmp(cycle_type, "data") == 0) {
				if(binary_format == 1) {
					if (current_step == 1) {
						if (skip_geom_data_binary(&num_nodes_b, &num_cells_b) != 1) {
							ERR_RETURN("Can't skip geometry of the first step.");
						}
					}
					FLDget_nnodes (out, &num_nodes);
					FLDget_ncell_sets (out, &ncell_sets);
					num_cells = 0;
					for (i = 0; i < ncell_sets; i++) {
						FLDget_cell_set (out, i, &cell_set);
						FLDget_ncells (cell_set, &ncells);
						num_cells += ncells;
					}

					if (read_node_cell_data_binary(num_nodes, num_cells, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				} else {
					if (current_step == 1) {
						if (skip_geom_data(FilePtr, &num_nodes, &num_cells) != 1) {
							ERR_RETURN("Can't skip geometry of the first step.");
						}
					}
					FLDget_nnodes (out, &num_nodes);
					FLDget_ncell_sets (out, &ncell_sets);
					num_cells = 0;
					for (i = 0; i < ncell_sets; i++) {
						FLDget_cell_set (out, i, &cell_set);
						FLDget_ncells (cell_set, &ncells);
						num_cells += ncells;
					}

					if (read_node_cell_data(FilePtr, num_nodes, num_cells, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				}

			}

			if (strcmp(cycle_type, "geom") == 0) {
				if(binary_format == 1) {
					if (read_geom_data_binary( &num_nodes_b, &num_cells_b, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						OMset_name_int_val(id, OMstr_to_name("total"), 0);
						sprintf(err_mes, "Error in step %d: Can't set geometry data.", current_step);
						ERR_RETURN(err_mes);
					}
					if (OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0) != 1) {
						ERR_RETURN("Can't store top.");
					}

					/* read node and cell data of the first step */
					if (b_close() != 1) {
						ERR_RETURN("Can't close binary file.");
					}
					if (b_open(1) != 1) {
						return 0;
					}
					read_step_comment_binary(1, comment) ;
					if (skip_geom_data_binary(&num_nodes_b, &num_cells_b) != 1) {
						ERR_RETURN("Can't skip geometry of the first step.");
					}
					if (read_node_cell_data_binary(num_nodes_b, num_cells_b, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				} else {
					if (read_geom_data(FilePtr, &num_nodes, &num_cells, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						OMset_name_int_val(id, OMstr_to_name("total"), 0);
						sprintf(err_mes, "Error in step %d: Can't set geometry data.", current_step);
						ERR_RETURN(err_mes);
					}
					if (OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0) != 1) {
						ERR_RETURN("Can't store top.");
					}

					/* read node and cell data of the first step */
					fseek(FilePtr, FilePos[num_steps], SEEK_SET);
					/* FilePos[num_steps] is the beginning */
					/* of node and cell data of the first step */
					if (read_node_cell_data(FilePtr, num_nodes, num_cells, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				}
			}

			if (strcmp(cycle_type, "data_geom") == 0) {
				if(binary_format == 1) {
					if (read_geom_data_binary( &num_nodes_b, &num_cells_b, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						OMset_name_int_val(id, OMstr_to_name("total"), 0);
						sprintf(err_mes, "Error in step %d: Can't read geometry data.", current_step);
						ERR_RETURN(err_mes);
					}
					OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
					if (read_node_cell_data_binary(num_nodes_b, num_cells_b, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				} else {
					if (read_geom_data(FilePtr, &num_nodes, &num_cells, out, &Top, 0, 0) != 1) {
						delete_cell_inf(&Top);
						OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
						OMset_name_int_val(id, OMstr_to_name("total"), 0);
						sprintf(err_mes, "Error in step %d: Can't read geometry data.", current_step);
						ERR_RETURN(err_mes);
					}
					OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);

					if (read_node_cell_data(FilePtr, num_nodes, num_cells, out, id, Top, 0) != 1) {
						sprintf(err_mes, "Error in step %d: Can't set node or cell data.", current_step);
						ERR_RETURN(err_mes);
					}
				}

			}
			if(binary_format == 1) {
				if (b_close() != 1) {
					ERR_RETURN("Can't close binary file.");
				}
			}

		}
	}
	return 1;

}

/**********************************************************************
 *
 *	function name:  DVread_ucd3_delete
 *
 *      description:	delete function (method)
 *
 *	return:		1: 
 *
 **********************************************************************/
int
DVread_ucd3_delete(
OMobj_id id
)
{
  FILE        *fp;
  long	      *FilePos;
  struct cell *Top;
  char	(*all_comments)[BUFFER_SIZE];
  int L;

  /* if we have binary file ptr */
  if (b_file != NULL && number_of_file != 0) {
    for (L = 0; L < number_of_file; L++) {
      if(b_file[L] != NULL) {
        free(b_file[L]);
      }
    }
    free(b_file);
    number_of_file = 0;
  }

  /* if we have a file open close it */
  if (OMget_name_ptr_val(id, OMstr_to_name("file_ptr"), (void *) &fp, 0)) {
      if (fp != NULL) {
	  fclose(fp);  fp = NULL;
	  OMset_name_ptr_val(id, OMstr_to_name("file_ptr"), fp, 0);
      }
  }

  if (OMget_name_ptr_val(id, OMstr_to_name("top"), (void *) &Top, 0)) {
      if (Top) {
	  delete_cell_inf(&Top);
	  free(Top);  Top = NULL;
	  OMset_name_ptr_val(id, OMstr_to_name("top"), Top, 0);
      }
  }

  if (OMget_name_ptr_val(id, OMstr_to_name("file_pos"), (void *)
			&FilePos, 0)) {
      if (FilePos != NULL) {
  	  free(FilePos);  FilePos = NULL;
	  OMset_name_ptr_val(id, OMstr_to_name("file_pos"), FilePos, 0);
      }
  }

  if (OMget_name_ptr_val(id, OMstr_to_name("all_comments"),
			(void *) &all_comments, 0)) {
      if (all_comments != NULL) {
	  free(all_comments);  all_comments = NULL;
	  OMset_name_ptr_val(id, OMstr_to_name("all_comments"),
		all_comments, 0);
      }
  }
  return(1);
}

