/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/ribbons.c#1 $
*/

#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/gd_def.h>
#include <avs/fld.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("ribbons", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024

#define  ALLOC_SIZE 1000

typedef struct _OUTmesh {
	xp_long nnodes;
	float *xyz;
	char *data;
	xp_long *pick_data;
	xp_long nquad;
	xp_long *quad_conn;
} OUTmesh;

int FUNCribbons (OMobj_id in, int h_comp, int m_comp, int dir,
		 double offset, double scale, OMobj_id out);

static int get_ribbon (OMobj_id in, OUTmesh *h, int ndim, xp_long *dims,
		       int dir, xp_long ind, int h_comp, int m_comp,
		       double scale, double offset,
		       float *x, float *y, char *dbuf);
static void get_x_coord(xp_long *dims, int nspace, float *points,
		        float *x);
static void get_y_coord(xp_long *dims, int nspace, float *points,
		        float *x);

int	DVsurf_ribbons_update(OMobj_id elem)
{
	OMobj_id src_id, dst_id, e_id;
	int h_comp, m_comp, dir;
	double offset, scale;

	src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
	dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);


	e_id = OMfind_subobj(elem,OMstr_to_name("height_comp"),OM_OBJ_RD);
	OMget_int_val(e_id, &h_comp);
	e_id = OMfind_subobj(elem,OMstr_to_name("map_comp"),OM_OBJ_RD);
	OMget_int_val(e_id, &m_comp);

	e_id = OMfind_subobj(elem,OMstr_to_name("ribbon_dir"),OM_OBJ_RD);
	OMget_int_val(e_id, &dir);

	e_id = OMfind_subobj(elem,OMstr_to_name("offset"),OM_OBJ_RD);
	OMget_real_val(e_id, &offset);

	e_id = OMfind_subobj(elem,OMstr_to_name("scale"),OM_OBJ_RD);
	OMget_real_val(e_id, &scale);

	if (FUNCribbons(src_id, h_comp, m_comp, dir, offset, scale, dst_id) == 0)
		return(0);

	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FUNCribbons (OMobj_id in, int h_comp, int m_comp, int dir,
		 double offset, double scale, OMobj_id out)
{
	OUTmesh  h;
	OMobj_id cell_set;
	int ndim, stat, ncomp, nspace, data_id, null_flag;
	xp_long *dims, i, size;
	int dims_size, veclen, dtype;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	double null_value;
	char  *dbuf;
	float *x, *y, *points;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_points (in, &points, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (h_comp >= ncomp) {
		ERRerror("ribbons", 0, ERR_ORIG,
			 "height component is out of range, assume 0-th");
		h_comp = 0;
	}
	if (m_comp >= ncomp) {
		ERRerror("ribbons", 0, ERR_ORIG,
			 "map component is out of range, assume 0-th");
		m_comp = 0;
	}
	if (FLDget_node_data_veclen(in, h_comp, &veclen) != 1) {
		ERR_RETURN("Error getting veclen");
	}
	if (veclen != 1) {
		ERR_RETURN(" height data must be a scalar");
	}
	if (FLDget_node_data_type(in, m_comp, &dtype) != 1) {
		ERR_RETURN("Error getting data type");
	}
	if (FLDget_node_data_veclen(in, m_comp, &veclen) != 1) {
		ERR_RETURN("Error getting veclen");
	}

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	h.xyz = NULL;
	h.data = NULL;

	h.nnodes = 0;
	h.xyz = (float *)ARRalloc(NULL, DTYPE_FLOAT, 3*2*dims[0]*dims[1], NULL);
	h.data = (char *)ARRalloc(NULL, DTYPE_CHAR, DTYPEtype_size[dtype]*2*dims[0]*dims[1]*veclen, NULL);
	h.pick_data = (xp_long *)ARRalloc(NULL, DTYPE_LONG, DTYPE_LONG*2*dims[0]*dims[1], NULL);
	h.nquad = 0;
	h.quad_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 4*(dims[0])*(dims[1]), NULL);

	x = (float *)malloc((dims[0]+1)*sizeof(float));
	y = (float *)malloc((dims[1]+1)*sizeof(float));
	get_x_coord(dims, nspace, points, x);
	get_y_coord(dims, nspace, points, y);

	if (dir == 0) {  /* x ribbon */
		dbuf = malloc(veclen*dims[0]*sizeof(double));
		for (i=0; i<dims[1]; i++) {
			stat = get_ribbon (in, &h, ndim, dims, dir, i, h_comp, m_comp,
					   scale, offset, x, y, dbuf);

			if (stat != 1) {
				ERR_RETURN("Error creating ribbon");
			}
		}
	} else {       /* y ribbon */
		dbuf = malloc(veclen*dims[1]*sizeof(double));
		for (i=0; i<dims[0]; i++) {
			stat = get_ribbon (in, &h, ndim, dims, dir, i, h_comp, m_comp,
					   scale, offset, x, y, dbuf);

			if (stat != 1) {
				ERR_RETURN("Error creating ribbon");
			}
		}
	}
	free(dbuf);
	free(x);
	free(y);

	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	nspace = 3;
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, h.nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (h.nnodes)
		if (FLDset_coord(out, h.xyz, nspace*h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting coord");
		}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (h.nquad) {
		if (FLDadd_cell_set(out, "Quad") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, 0, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_ncells(cell_set, h.nquad) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDset_node_connect(cell_set, h.quad_conn, 4*h.nquad,
					OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}
	}
	else {
		ARRfree(h.quad_conn);
	}
	if (h.nnodes) {
		if (FLDset_node_data_ncomp (out, 2) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		if (FLDset_node_data_veclen(out, 1, 1) !=1) {
			ERR_RETURN("Error setting veclen");
		}
		if (FLDset_node_data_id(out, 1, GD_PICK_DATA_ID) != 1) {
			ERR_RETURN("Error setting data id");
		}
		if (FLDget_node_data_units(in, m_comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, m_comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_null_data(in, m_comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (FLDset_node_data_comp (out, 0, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, m_comp, &data_id) == 1)
			FLDset_node_data_id(out, 0, data_id);
		if (FLDset_node_data(out, 0, (char *)h.data, dtype,
				     veclen*h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDset_node_data(out, 1, (char *)h.pick_data, DTYPE_INT,
				     h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDcopy_node_minmax(in, out, m_comp, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, m_comp, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, 0, (char *)&null_value, dtype) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, 0, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
	}
	else {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
	}
	ARRfree(dims);
	ARRfree(points);
	return(1);
}

/* 64-bit porting. Directly Modified */
static int get_ribbon (OMobj_id in, OUTmesh *h, int ndim, xp_long *dims,
		       int dir, xp_long ind, int h_comp, int m_comp,
		       double scale, double offset,
		       float *x, float *y, char *dbuf)
{
	int  i, veclen, mtype, htype;
	xp_long i_w, j, n, n1, n2, nnew;
	xp_long rdims[4], rmin[4], rmax[4];
	float f;

	if (FLDget_node_data_veclen(in, h_comp, &veclen) != 1) {
		ERR_RETURN("Error getting veclen");
	}
	if (FLDget_node_data_type(in, h_comp, &htype) != 1) {
		ERR_RETURN("Error getting data type");
	}
	if (FLDget_node_data_type(in, m_comp, &mtype) != 1) {
		ERR_RETURN("Error getting data type");
	}

	rmin[0]= 0;

	if (dir == 0) {  /* x */
		rmin[1] = 0;
		rmax[1] = dims[0];
		rmin[2] = ind;
		rmax[2] = ind+1;
		n1 = dims[0]-1;
		n2 = 1;
		nnew = dims[0];
	} else {         /* y */
		rmin[1] = ind;
		rmax[1] = ind+1;
		rmin[2] = 0;
		rmax[2] = dims[1];
		n1 = 1;
		n2 = dims[1]-1;
		nnew = dims[1];
	}

	rdims[0] = 1;
	rmax[0]= 1;
	for (i=0; i<ndim; i++)
		rdims[i+1] = dims[i];

	if (FLDget_sub_node_data(in, h_comp, &htype, ndim+1, rdims, rmin, rmax, dbuf) != 1) {
		ERR_RETURN("cannot get node data");
	}

	rdims[0] = veclen;
	rmax[0]= veclen;
	if (FLDget_sub_node_data(in, m_comp, &mtype, ndim+1, rdims, rmin, rmax,
				 h->data+DTYPEtype_size[mtype]*h->nnodes*veclen) != 1) {
		ERR_RETURN("cannot get node data");
	}
	memcpy(h->data+DTYPEtype_size[mtype]*(h->nnodes+nnew)*veclen,
	       h->data+DTYPEtype_size[mtype]*h->nnodes*veclen,
	       nnew*veclen*DTYPEtype_size[mtype]);

	if (dir == 0) {  /* x */
		for (i_w=0; i_w< dims[0]; i_w++) {
			UTILtype_to_float(&f, dbuf+i_w*DTYPEtype_size[htype], htype);
			h->xyz[(h->nnodes+i_w)*3] = x[i_w];
			h->xyz[(h->nnodes+i_w+nnew)*3] = x[i_w];
			h->xyz[(h->nnodes+i_w)*3+1] = y[ind];
			h->xyz[(h->nnodes+i_w+nnew)*3+1] = y[ind+1];
			h->xyz[(h->nnodes+i_w)*3+2] = f*scale+offset;
			h->xyz[(h->nnodes+i_w+nnew)*3+2] = h->xyz[(h->nnodes+i_w)*3+2];
			h->pick_data[h->nnodes+i_w] = ind*dims[0]+i_w;
			h->pick_data[h->nnodes+nnew+i_w] = ind*dims[0]+i_w;
		}

	} else {         /* y */
		for (i_w=0; i_w< dims[1]; i_w++) {
			UTILtype_to_float(&f, dbuf+i_w*DTYPEtype_size[htype], htype);
			h->xyz[(h->nnodes+i_w)*3] = x[ind];
			h->xyz[(h->nnodes+i_w+nnew)*3] = x[ind+1];
			h->xyz[(h->nnodes+i_w)*3+1] = y[i_w];
			h->xyz[(h->nnodes+i_w+nnew)*3+1] = y[i_w];
			h->xyz[(h->nnodes+i_w)*3+2] = f*scale+offset;
			h->xyz[(h->nnodes+i_w+nnew)*3+2] = h->xyz[(h->nnodes+i_w)*3+2];
			h->pick_data[h->nnodes+i_w] = ind+dims[0]*i_w;
			h->pick_data[h->nnodes+nnew+i_w] = ind+dims[0]*i_w;
		}
	}
	n = h->nnodes;
	if (dir==0) {
		for (j=0; j<n1; j++) {
			h->quad_conn[4*h->nquad] = n+j;
			h->quad_conn[4*h->nquad+1] = n+j+1;
			h->quad_conn[4*h->nquad+2] = n+j+2+n1;
			h->quad_conn[4*h->nquad+3] = n+j+n1+1;
			(h->nquad)++;
		}
	} else {
		for (j=0; j<n2; j++) {
			h->quad_conn[4*h->nquad] = n+j;
			h->quad_conn[4*h->nquad+1] = n+j+n2+1;
			h->quad_conn[4*h->nquad+2] = n+j+2+n2;
			h->quad_conn[4*h->nquad+3] = n+j+1;
			(h->nquad)++;
		}
	}
	h->nnodes += 2*nnew;
	return(1);
}

/* 64-bit porting. Directly Modified */
static void get_x_coord(xp_long *dims, int nspace, float *points,
			    float *x)
{
	xp_long i;
	float h;

	if (dims[0] > 1)
		h = (points[nspace]-points[0])/(dims[0]-1);
	else if (dims[1] > 1)
		h = (points[nspace+1]-points[1])/(dims[1]-1);
	else
		h = 0;

	for(x[0] = points[0], i=1; i<dims[0]+1; i++)
		x[i] = x[i-1]+h;
}

/* 64-bit porting. Directly Modified */
static void get_y_coord(xp_long *dims, int nspace, float *points,
			    float *x)
{
	xp_long i;
	float h;

	if (dims[1] > 1)
		h = (points[nspace+1]-points[1])/(dims[1]-1);
	else if (dims[0] > 1)
		h = (points[nspace]-points[0])/(dims[0]-1);
	else
		h = 0;

	for(x[0] = points[1], i=1; i<dims[1]+1; i++)
		x[i] = x[i-1]+h;
}
