/*
			Copyright (c) 1996 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/roi2surf.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("DVroi_to_surf_update", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_LEN 256
#define ALLOC_BLOCK 1024

/* 64-bit porting. Only Modified Internally */
int DVroi_to_surf_update(OMobj_id mod_id)
{
	OMobj_id in, out, val_id, in_set, out_set;
	int cap_surface, flip_normals, start_angle;
	xp_long nroi;

	xp_long nvert0, nvert, ntri;
	int nspace, out_nspace, ncap[2];
	xp_long *out_conn, out_ncells, nnodes, out_nnodes;
	float *coord=NULL, *out_coord=NULL, *z=NULL, *t=NULL, *t0=NULL;
	xp_long *list=NULL, *list0=NULL;

	int i, stat, first;
	xp_long i_w, n, size, sroi, rroi, tri_size = ALLOC_BLOCK;
	int poly_flag;
	xp_long count, npoly, *poly_conn;


	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	stat = OMget_array_size(in, &nroi);
	if (stat != 1 || nroi < 2) {
		/*
		 * this could be a case when module disconnected 
		 * clean the output and return
		 */
		return(0);
	}
	out = OMfind_subobj(mod_id, OMstr_to_name("out"), OM_OBJ_RW);

	if (OMget_name_int_val(mod_id, OMstr_to_name("cap_surface"), &cap_surface) != 1)
		cap_surface = 0;
	if (OMget_name_int_val(mod_id, OMstr_to_name("flip_normals"), &flip_normals) != 1)
		flip_normals = 0;
	if (OMget_name_int_val(mod_id, OMstr_to_name("start_angle"), &start_angle) != 1)
		start_angle = 0;
	z = (float *)OMret_name_array_ptr(mod_id, OMstr_to_name("z_coord"),
					    OM_GET_ARRAY_RD, &size, NULL);
	ARRset_trace( z, 1 );
	/* 
	 * clean the output
	 */
	if (FLDset_nnodes(out, 0) != 1) {
		ERR_RETURN("Cannot set nnodes");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	for (rroi=0, out_nnodes=0, i_w=0; i_w<nroi; i_w++) {
		if (OMget_array_val(in, i_w, &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("Cannot get in array member\n");
		}
		if (FLDget_nspace (val_id, &nspace) != 1) {
			continue;
		}
		if (FLDget_nnodes (val_id, &nnodes) != 1) {
			continue;
		}
		out_nnodes += nnodes;
		rroi++;
	}
	if (out_nnodes == 0) {
		if (z) ARRfree(z);
		return(0);
	}

	out_nspace = 3;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	tri_size = ALLOC_BLOCK;
	out_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 3*tri_size, NULL);

	for (count=0,first=1, out_ncells=0, out_nnodes=0, i_w=0; i_w<nroi; i_w++) {
		if (OMget_array_val(in, i_w, &val_id, OM_OBJ_RD) != 1) {
			continue;
		}
		if (FLDget_nnodes (val_id, &nnodes) != 1) {
			continue;
		}
		if (FLDget_coord(val_id, &coord, &size, OM_GET_ARRAY_RD) != 1) {
			continue;
		}
		for (n=0; n<nnodes; n++)
			memcpy(out_coord+(n+out_nnodes)*out_nspace, coord+n*nspace, nspace*sizeof(float));
		if (z) {
			for (n=0; n<nnodes; n++)
				out_coord[(n+out_nnodes)*out_nspace+2] = z[i_w];
		}
		else if (nspace < out_nspace) {
			for (n=0; n<nnodes; n++)
				out_coord[(n+out_nnodes)*out_nspace+2] = i_w;
		}

		if (FLDget_cell_set(val_id, 0, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_flag(in_set, &poly_flag) != 1 || poly_flag == 0) {
			ARRfree(coord);
			ARRfree(out_coord);
			ARRfree(out_conn);
			ERR_RETURN("Cell Set is not a Polyline, no surface created");
		}
		if (FLDget_npolys(in_set, &npoly) != 1) {
			ERR_RETURN("cannot get ncells");
		}
		if (npoly > 1) {
			ERRerror("DVroi_to_surf_update", 0, ERR_ORIG, "More than 1 Polyline found. Using first one");
		}
		if (FLDget_poly_connect(in_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Cannot get poly connectivity");
		}
		if (!first) {
			nvert0 = nvert;
			if (t0)
				free(t0);
			t0 = t;
			if (list0)
				free(list0);
			list0 = list;
		}
		nvert = poly_conn[1]-poly_conn[0]+1;
		t = (float *)malloc((nvert+1)*sizeof(float));
		list = (xp_long *)malloc((nvert+1)*sizeof(xp_long));

		if (first)
			ncap[0] = (int)nvert;
		else if (count==rroi-1)
			ncap[1] = (int)nvert;

		stat = UTILroi_sort_verts(nspace, coord, poly_conn, start_angle,
					  list, t);
		for (n=0; n<nvert+1; n++)
			list[n] += out_nnodes;
		if (stat != 1) {
			if (t)
				free(t);
			if (t0)
				free(t0);
			if (list)
				free(list);
			if (list0)
				free(list0);
			ARRfree(coord);
			ARRfree(out_coord);
			ARRfree(out_conn);
			ARRfree(poly_conn);
			ERR_RETURN("Cannot sort vertices, no surface created");
		}
		if (!first) {
			if (out_ncells+nvert0+nvert >= tri_size) {
				tri_size += ((out_ncells+nvert0+nvert)/ALLOC_BLOCK+1)*ALLOC_BLOCK;
				out_conn = (xp_long *)ARRrealloc(out_conn, DTYPE_LONG, 3*tri_size, NULL);
			}
			ntri = nvert0+nvert;
			stat = UTILroi_connect_verts(nvert0+1, nvert+1, list0, list, t0, t,
						     &ntri, out_conn+out_ncells*3);
			if (stat != 1) {
				if (t)
					free(t);
				if (t0)
					free(t0);
				if (list)
					free(list);
				if (list0)
					free(list0);
				ARRfree(coord);
				ARRfree(out_coord);
				ARRfree(out_conn);
				ARRfree(poly_conn);
				ERR_RETURN("Cannot sort vertices, no surface created");
			}
			out_ncells += ntri;

		}
		out_nnodes += nnodes;
		ARRfree(poly_conn);
		ARRfree(coord);
		if (first == 1)
			sroi = i_w;
		first = 0;
		count++;
	}
	if (t)
		free(t);
	if (t0)
		free(t0);
	if (list)
		free(list);
	if (list0)
		free(list0);
	if (flip_normals) {
		for (i_w=0; i_w<out_ncells; i_w++) {
			n = out_conn[i_w*3];
			out_conn[i_w*3] = out_conn[i_w*3+1];
			out_conn[i_w*3+1] = n;
		}
	}

	if (FLDadd_cell_set(out, "Tri") != 1) {
		ERR_RETURN("Error setting cell type");
	}
	if (FLDget_cell_set(out, 0, &out_set) != 1) {
		ERR_RETURN("Error getting cell set");
	}
	if (FLDset_ncells(out_set, out_ncells) != 1) {
		ERR_RETURN("Cannot set ncells");
	}
	if (FLDset_node_connect(out_set, out_conn, out_ncells*3, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting cell connect list");
	}


	if (cap_surface) {
		if (FLDadd_cell_set(out, "Polyhedron") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, 1, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_npolys(out_set, 2) != 1) {
			ERR_RETURN("Cannot set npolys");
		}
		if (FLDset_poly_nnodes(out_set, ncap, 2, OM_SET_ARRAY_COPY) != 1) {
			ERR_RETURN("cannot set poly_nnodes");
		}

		out_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, ncap[0]+ncap[1], NULL);

		if (OMget_array_val(in, sroi, &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("Cannot get in array member\n");
		}
		if (FLDget_cell_set(val_id, 0, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_connect(in_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Cannot get poly connectivity");
		}
		for (i=0; i<ncap[0]; i++)
			out_conn[i] = poly_conn[0]+i;
		ARRfree(poly_conn);

		if (OMget_array_val(in, rroi-1, &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("Cannot get in array member\n");
		}
		if (FLDget_cell_set(val_id, 0, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_connect(in_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Cannot get poly connectivity");
		}
		for (i=0; i<ncap[1]; i++)
			out_conn[ncap[0]+i] = out_nnodes-ncap[1]+poly_conn[1]-i;
		ARRfree(poly_conn);

		if (flip_normals) {
			for (i=0; i<ncap[0]/2; i++) {
				n = poly_conn[i];
				poly_conn[i] = poly_conn[ncap[0]-1-i];
				poly_conn[ncap[0]-1-i] = n;
			}
			for (i=0; i<ncap[1]/2; i++) {
				n = poly_conn[ncap[0]+i];
				poly_conn[ncap[0]+i] = poly_conn[ncap[0]+ncap[1]-1-i];
				poly_conn[ncap[0]+ncap[1]-1-i] = n;
			}
		}
		if (FLDset_poly_connect(out_set, out_conn, ncap[0]+ncap[1], OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Cannot get poly connectivity");
		}
	}

	ARRfree(out_coord);
	if (z) ARRfree(z);
	return(1);
}
