/*
			Copyright (c) 1996 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/sc_mod.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/smith.h>

static void UpdateGraticuleMeshes(OMobj_id DVsmith_chart_id,
				  OMobj_id min_spacing_id,
				  OMobj_id chord_factor_id,
				  OMobj_id spacing_plans_id);
static void UpdateDataMesh(OMobj_id DVsmith_chart_id, OMobj_id in_traces_id);
static xp_long GetDataCellSetInfo(OMobj_id trace, SCPoint3D *coords, char **name,
			      SCPoint3D *color);


int
CreateSmithChartMesh(OMobj_id DVsmith_chart_id, OMevent_mask event_mask, int seq_num)
{
   OMobj_id min_spacing_id;
   OMobj_id chord_factor_id;
   OMobj_id spacing_plans_id;
   OMobj_id in_traces_id;

   min_spacing_id = OMfind_subobj(DVsmith_chart_id,
				  OMstr_to_name("min_spacing"), OM_OBJ_RD);
   chord_factor_id = OMfind_subobj(DVsmith_chart_id,
				   OMstr_to_name("chord_factor"), OM_OBJ_RD);
   spacing_plans_id = OMfind_subobj(DVsmith_chart_id,
				    OMstr_to_name("spacing_plans"),
				    OM_OBJ_RD);
   in_traces_id = OMfind_subobj(DVsmith_chart_id,
				OMstr_to_name("in_traces"), OM_OBJ_RD);

   if (OMis_null_obj(min_spacing_id))
      SCerror(SC_UNABLE_TO_FIND_XP_OBJ, "Subobject name: \"min_spacing\"\n");
   else if (OMis_null_obj(chord_factor_id))
      SCerror(SC_UNABLE_TO_FIND_XP_OBJ, "Subobject name: \"chord_factor\"\n");
   else if (OMis_null_obj(spacing_plans_id))
      SCerror(SC_UNABLE_TO_FIND_XP_OBJ,
	      "Subobject name: \"spacing_plans\"\n");
   else if (OMis_null_obj(in_traces_id))
      SCerror(SC_UNABLE_TO_FIND_XP_OBJ, "Subobject name: \"in_traces\"\n");
   else
   {
      if (OMchanged(min_spacing_id, seq_num) ||
	  OMchanged(chord_factor_id, seq_num) ||
	  OMchanged(spacing_plans_id, seq_num) ||
	  (event_mask & OM_EVENT_INST))
	 UpdateGraticuleMeshes(DVsmith_chart_id, min_spacing_id,
			       chord_factor_id, spacing_plans_id);

      if (OMchanged(in_traces_id, seq_num) || (event_mask & OM_EVENT_INST))
	 UpdateDataMesh(DVsmith_chart_id, in_traces_id);

      return 1;
   }

   return 0;
}


/* 64-bit porting. Only Modified Internally */
static void UpdateGraticuleMeshes(OMobj_id DVsmith_chart_id,
				  OMobj_id min_spacing_id,
				  OMobj_id chord_factor_id,
				  OMobj_id spacing_plans_id)
{
   /***********************/
   /*  Declare variables  */
   /***********************/
   double  min_spacing;
   double  chord_factor;
   xp_long spacing_plans_size;
   int *spacing_plans; 
   OMobj_id out_id;
   int out_nspace;
   xp_long  out_nnodes;
   xp_long  out_ncells;
   xp_long  *out_node_connect;
   OMobj_id out_cell_set;
   float *out_coord;

   OMobj_id ann_out_cell_set;
   xp_long ann_out_nnodes;
   xp_long  *ann_out_node_connect;
   int ann_out_veclen;
   int ann_out_data_type;
   float *ann_out_data;
   xp_long ann_out_arr_size;

   OMobj_id line_ann_out_id;
   OMobj_id top_rim_ann_out_id;
   OMobj_id bot_rim_ann_out_id;
   float *line_ann_out_coord;
   float *top_rim_ann_out_coord;
   float *bot_rim_ann_out_coord;

   OMobj_id annotation_id;

   SCMeshInfo *mesh_info;
   int ann_str_len;
   char *ann_str;
   char *cp;
   xp_long i;

   /***********************/
   /*  Get input values   */
   /***********************/
   /* Get min_spacing's value */
   if (OMget_real_val(min_spacing_id, &min_spacing) != 1)
      min_spacing = 0.1;

   /* Get chord_factor's value */
   if (OMget_real_val(chord_factor_id, &chord_factor) != 1)
      chord_factor = 25.0;


   spacing_plans = (int *)OMret_array_ptr(spacing_plans_id, OM_GET_ARRAY_RD,
					  &spacing_plans_size, NULL);

   mesh_info = CreateGraticuleMeshInfo(chord_factor, min_spacing,
				       spacing_plans, (int)spacing_plans_size);

   if (mesh_info)
   {
      GetGraticuleMeshPoints(mesh_info, &out_nnodes, NULL,
			     &out_ncells, NULL);


      /***********************/
      /*  Set output values  */
      /***********************/
      /* set  out mesh */

      /*  Get mesh id */
      out_id = OMfind_subobj(DVsmith_chart_id, OMstr_to_name("out"),
			     OM_OBJ_RW);

      /* Set mesh nnodes */
      FLDset_nnodes (out_id, out_nnodes);

      /* Set mesh nspace */
      out_nspace = 3;
      FLDset_nspace (out_id, out_nspace);

      /* Set mesh coordinates */
      /* First allocate out_coord array */
   
      out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, 
				    out_nspace*out_nnodes, NULL);

      out_node_connect = (xp_long *)ARRalloc(NULL, DTYPE_LONG, out_ncells*2, NULL);


      GetGraticuleMeshPoints(mesh_info, NULL, (SCPoint3D *)out_coord,
			     NULL, (SCPolylineConn *)out_node_connect);

      FLDset_coord (out_id, out_coord, out_nspace*out_nnodes, 
		    OM_SET_ARRAY_FREE);


      /* Set_number of cell sets to 0 initially.
       * the real number of cell_sets should be assigned to the out_nsets
       * variable
       */
      FLDset_ncell_sets(out_id, 0);

      FLDadd_cell_set(out_id, "Polyline");

      /* Get cell set id */
      FLDget_cell_set(out_id, 0, &out_cell_set);

      /* Set number of cells */
      FLDset_npolys(out_cell_set, out_ncells);

      /* Set node connectivity list */
      FLDset_poly_connect(out_cell_set, out_node_connect, 
			  out_ncells*2, OM_SET_ARRAY_FREE);



      /* set ann_out mesh */

      GetAnnotationMeshPoints(mesh_info, &ann_out_nnodes, NULL, NULL, NULL,
			      &ann_str_len, NULL);

      /*  Get mesh ids */
      line_ann_out_id = OMfind_subobj(DVsmith_chart_id,
				      OMstr_to_name("line_ann_out"),
				      OM_OBJ_RW);
      top_rim_ann_out_id = OMfind_subobj(DVsmith_chart_id,
					 OMstr_to_name("top_rim_ann_out"),
					 OM_OBJ_RW);
      bot_rim_ann_out_id = OMfind_subobj(DVsmith_chart_id,
					 OMstr_to_name("bot_rim_ann_out"),
					 OM_OBJ_RW);

      /* Set mesh nnodes */
      FLDset_nnodes(line_ann_out_id, ann_out_nnodes);
      FLDset_nnodes(top_rim_ann_out_id, ann_out_nnodes);
      FLDset_nnodes(bot_rim_ann_out_id, ann_out_nnodes);

      /* Set mesh nspace */
      FLDset_nspace(line_ann_out_id, out_nspace);
      FLDset_nspace(top_rim_ann_out_id, out_nspace);
      FLDset_nspace(bot_rim_ann_out_id, out_nspace);

      /* Set mesh coordinates
       * First allocate the xxx_ann_out_coord arrays
       */
      ann_out_arr_size = out_nspace * ann_out_nnodes;

      line_ann_out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, 
					     ann_out_arr_size,
					     NULL);
      top_rim_ann_out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, 
						ann_out_arr_size,
						NULL);
      bot_rim_ann_out_coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, 
						ann_out_arr_size,
						NULL);

      ann_out_node_connect = (xp_long *)ARRalloc(NULL, DTYPE_LONG,
					     ann_out_nnodes, NULL);


      ann_str = (char *)malloc(sizeof(char) * ann_str_len);

      GetAnnotationMeshPoints(mesh_info, NULL,
			      (SCPoint3D *)line_ann_out_coord,
			      (SCPoint3D *)top_rim_ann_out_coord,
			      (SCPoint3D *)bot_rim_ann_out_coord,
			      NULL, ann_str);

      for (i = 0; i < ann_out_nnodes; i++)
	 ann_out_node_connect[i] = i;

      FLDset_coord(line_ann_out_id, line_ann_out_coord,
		   ann_out_arr_size, OM_SET_ARRAY_FREE);
      FLDset_coord(top_rim_ann_out_id, top_rim_ann_out_coord,
		   ann_out_arr_size, OM_SET_ARRAY_FREE);
      FLDset_coord(bot_rim_ann_out_id, bot_rim_ann_out_coord,
		   ann_out_arr_size, OM_SET_ARRAY_FREE);


      /* Set_number of cell sets to 0 initially.
       */
      FLDset_ncell_sets(line_ann_out_id, 0);
      FLDset_ncell_sets(top_rim_ann_out_id, 0);
      FLDset_ncell_sets(bot_rim_ann_out_id, 0);

      FLDadd_cell_set(line_ann_out_id, "Point");
      FLDadd_cell_set(top_rim_ann_out_id, "Point");
      FLDadd_cell_set(bot_rim_ann_out_id, "Point");

      /* Set the connectivity for all three of the annotation point meshes.
       * For each, get the cell set id, set ncells, and then set the
       * connectivity.  The connectivity is identical for all three
       * meshes, so we just use the same connectivity array.
       */
      FLDget_cell_set(line_ann_out_id, 0, &ann_out_cell_set);

      FLDset_ncells(ann_out_cell_set, ann_out_nnodes);

      FLDset_node_connect(ann_out_cell_set, ann_out_node_connect, 
			  ann_out_nnodes, OM_SET_ARRAY_COPY);


      FLDget_cell_set(top_rim_ann_out_id, 0, &ann_out_cell_set);

      FLDset_ncells(ann_out_cell_set, ann_out_nnodes);

      FLDset_node_connect(ann_out_cell_set, ann_out_node_connect, 
			  ann_out_nnodes, OM_SET_ARRAY_COPY);

      FLDget_cell_set(bot_rim_ann_out_id, 0, &ann_out_cell_set);

      FLDset_ncells(ann_out_cell_set, ann_out_nnodes);

      FLDset_node_connect(ann_out_cell_set, ann_out_node_connect, 
			  ann_out_nnodes, OM_SET_ARRAY_FREE);


      ann_out_veclen = 1;

      FLDset_node_data_veclen(line_ann_out_id, 0, ann_out_veclen);
      FLDset_node_data_veclen(top_rim_ann_out_id, 0, ann_out_veclen);
      FLDset_node_data_veclen(bot_rim_ann_out_id, 0, ann_out_veclen);

      /* Set data array.
       * data_type should be set to one of the following: 
       *  DTYPE_BYTE, DTYPE_CHAR, DTYPE_SHORT, 
       *  DTYPE_INT, DTYPE_FLOAT, DTYPE_DOUBLE, DTYPE_LONG)
       */
      ann_out_data_type = DTYPE_FLOAT;

      /* allocate ann_out_data array first */
      /* assume float array and ann_out_ndata is set to number of nodes */

      ann_out_data = (float *)ARRalloc(NULL, ann_out_data_type, 
				       ann_out_veclen*ann_out_nnodes, NULL);

      for (i = 0; i < ann_out_nnodes; i++)
	 ann_out_data[i] = 0.0;

      FLDset_node_data(line_ann_out_id, 0, (char *)ann_out_data,
		       ann_out_data_type, ann_out_nnodes*ann_out_veclen,
		       OM_SET_ARRAY_COPY);
      FLDset_node_data(top_rim_ann_out_id, 0, (char *)ann_out_data,
		       ann_out_data_type, ann_out_nnodes*ann_out_veclen,
		       OM_SET_ARRAY_COPY);
      FLDset_node_data(bot_rim_ann_out_id, 0, (char *)ann_out_data,
		       ann_out_data_type, ann_out_nnodes*ann_out_veclen,
		       OM_SET_ARRAY_FREE);

   
      /* Set annotation array
       */ 
      annotation_id = OMfind_subobj(DVsmith_chart_id,
				    OMstr_to_name("annotation"), OM_OBJ_RW);

      OMset_array_size(annotation_id, ann_out_nnodes);

      for (cp = ann_str, i = 0; i < ann_out_nnodes; i++)
      {
	 OMset_str_array_val(annotation_id, i, cp);
	 cp += strlen(cp) + 1;
      }

      if (ann_str)
	 free(ann_str);

      DestroyGraticuleMeshInfo(mesh_info);
   }

   /**************************/
   /*  Free variables        */
   /**************************/

   ARRfree(spacing_plans);
}



/* 64-bit porting. Only Modified Internally */
static void UpdateDataMesh(OMobj_id DVsmith_chart_id, OMobj_id in_traces_id)
{
   xp_long num_traces;

   if (OMget_array_size(in_traces_id, &num_traces) != OM_STAT_SUCCESS)
      SCerror(SC_UNABLE_TO_GET_ARR_SIZE, "Subobject: \"in_traces\"\n");
   else
   {
      OMobj_id data_mesh;

      data_mesh = OMfind_subobj(DVsmith_chart_id,
				OMstr_to_name("traces"), OM_OBJ_RW);

      if (OMis_null_obj(data_mesh))
	 SCerror(SC_UNABLE_TO_FIND_XP_OBJ, "Subobject: \"traces\"\n");
      else
      {
	 xp_long i;
	 OMobj_id trace;
	 xp_long data_nnodes;

	 for (data_nnodes = i = 0; i < num_traces; i++)
	 {
	    if (OMget_array_val(in_traces_id, i,
				&trace, OM_OBJ_RD) != OM_STAT_SUCCESS)
	       SCerror(SC_UNABLE_TO_FIND_XP_OBJ, "Subobject: traces[%d]\n",
		       i);
	    else
	       data_nnodes += GetDataCellSetInfo(trace, NULL, NULL, NULL);
	 }

	 if (data_nnodes > 0)
	 {
	    int data_nspace = 3;
	    int data_veclen = 3;    /* R, G, and B */
	    char *name;
	    SCPoint3D color;
	    SCPoint3D *data_coords;
	    xp_long data_node_connect[2];
	    OMobj_id data_cell_set;

	    FLDset_nnodes(data_mesh, data_nnodes);
	    FLDset_nspace(data_mesh, data_nspace);
	    FLDset_ncell_sets(data_mesh, 0);
	    FLDset_ncell_sets(data_mesh, (int)num_traces);

	    data_coords = (SCPoint3D *)ARRalloc(NULL, DTYPE_FLOAT, 
						data_nspace*data_nnodes,
						NULL);

	    for (data_nnodes = i = 0; i < num_traces; i++)
	    {
	       if (OMget_array_val(in_traces_id, i,
				   &trace, OM_OBJ_RD) == OM_STAT_SUCCESS)
	       {
		  data_node_connect[0] = data_nnodes;

		  data_nnodes += GetDataCellSetInfo(trace,
						    data_coords + data_nnodes,
						    &name, &color);

		  data_node_connect[1] = data_nnodes - 1;

		  if (FLDget_cell_set(data_mesh, (int)i,
				      &data_cell_set) == OM_STAT_SUCCESS)
		  {
		     FLDset_npolys(data_cell_set, 1);

		     FLDset_poly_connect(data_cell_set, data_node_connect, 
					 2, OM_SET_ARRAY_COPY);

		     FLDset_cell_data(data_cell_set, 0, (char *)&color,
				      DTYPE_FLOAT, data_veclen,
				      OM_SET_ARRAY_COPY);

		     if (name)
		     {
			FLDset_cell_set_user_name(data_cell_set, name);

			free(name);
		     }
		  }
		  else
		     SCerror(SC_UNABLE_TO_GET_CELL_SET,
			     "Mesh subobject: \"traces\"\n");
	       }
	    }

	    FLDset_coord(data_mesh, (float *)data_coords,
			 data_nspace*data_nnodes, OM_SET_ARRAY_FREE);
	 }
      }
   }
}


/* 64-bit porting. Directly Modified */
static xp_long GetDataCellSetInfo(OMobj_id trace, SCPoint3D *coords, char **name,
			      SCPoint3D *color)
{
   xp_long data_size = 0;

   if (name)
      if (OMget_name_str_val(trace, OMstr_to_name("name"),
			     name, 0) != OM_STAT_SUCCESS)
	 *name = NULL;

   if (color)
   {
      float *rgb;

      rgb = (float *)OMret_name_array_ptr(trace, OMstr_to_name("color"),
					  OM_GET_ARRAY_RD, NULL, NULL);

      color->x = rgb[0];
      color->y = rgb[1];
      color->z = rgb[2];

      ARRfree(rgb);
   }

   if (coords)
   {
      SCPoint2D *data;

      data = (SCPoint2D *)OMret_name_array_ptr(trace, OMstr_to_name("data"),
					       OM_GET_ARRAY_RD, &data_size,
					       NULL);
      if (data)
      {
	 xp_long i;

	 data_size /= 2;   /* convert from num components to num 2D points */

	 for (i = 0; i < data_size; i++)
	 {
	    coords[i].x = data[i].x;
	    coords[i].y = data[i].y;
	    coords[i].z = 0.0;
	 }

	 ARRfree(data);
      }
      else
	 data_size = 0;
   }
   else
   {
      OMobj_id data_id;

      data_id = OMfind_subobj(trace, OMstr_to_name("data"), OM_OBJ_RD);

      if (!OMis_null_obj(data_id))
      {
	 if (OMget_array_size(data_id, &data_size) != OM_STAT_SUCCESS)
	    data_size = 0;

	 data_size /= 2;   /* convert from num components to num 2D points */
      }
   }

   return data_size;
}
