/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/timedata.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/arr.h>
#include <avs/fld.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("extract time step", 0, ERR_ORIG, A); return(0);

int DVextract_time_step_update(OMobj_id mod_id)
{
	OMobj_id  in, out, time_id, ref_id, set_id, out_id, id;
	int       nsteps, step, nsets, set;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMis_null_obj(in))
		return(0);

	out = OMfind_subobj(mod_id, OMstr_to_name("out_step"), OM_OBJ_RW);
	if (OMis_null_obj(out))
		return(0);

	if (OMget_name_int_val(mod_id, OMstr_to_name("step"), &step) != OM_STAT_SUCCESS)
		step = 0;

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	/*---------------------------------------------*/
	/*   get number of time steps for coordinates  */
	/*   and for node_data                         */
	/*---------------------------------------------*/
	if (FLDget_time_nsteps (in, &nsteps) != OM_STAT_SUCCESS)
		nsteps = 0;

	if (nsteps <= 0) {
		FLDset_nnodes(out, 0);
		FLDset_ncell_sets(out, 0);
		FLDset_node_data_ncomp (out, 0);
		return(1);
	}
	else {
		/*-----------------------------------------------------*/
		/*   get coordinates for one time step                 */
		/*   and set the reference from in's coordinates to it */
		/*-----------------------------------------------------*/
		id = OMfind_subobj(out, OMstr_to_name("coordinates"), OM_OBJ_RW);
		if (OMis_null_obj(id)) {
			ERR_RETURN("Error getting coordinates");
		}
		if (FLDget_time_coordinates(in, step, &time_id) == OM_STAT_SUCCESS) {
			ref_id = OMfind_subobj(time_id, OMstr_to_name("coordinates"), OM_OBJ_RW);
			if (OMis_null_obj(ref_id)) {
				ERR_RETURN("Error getting coordinates");
			}
			if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting field reference");
			}

		}
		/*---------------------------------------------------*/
		/*   get node_data for one time step                 */
		/*   and set the reference from in's node_data to it */
		/*---------------------------------------------------*/
		id = OMfind_subobj(out, OMstr_to_name("node_data"), OM_OBJ_RW);
		if (OMis_null_obj(id)) {
			ERR_RETURN("Error getting node_data");
		}
		if (FLDget_time_node_data(in, step, &time_id) == OM_STAT_SUCCESS) {
			ref_id = OMfind_subobj(time_id, OMstr_to_name("node_data"), OM_OBJ_RW);
			if (OMis_null_obj(id)) {
				ERR_RETURN("Error getting node_data");
			}
			if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting field reference");
			}
		}
		else {
			if (FLDset_node_data_ncomp (out, 0) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting ncell_data");
			}
		}
	}

	if (FLDget_ncell_sets (in, &nsets) != OM_STAT_SUCCESS)
		nsets = 0;

	for (set=0; set<nsets; set++) {
		if (FLDget_cell_set (in, set, &set_id) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_cell_set (out, set, &out_id) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		/*---------------------------------------------*/
		/*   get number of time steps for cell_data    */
		/*---------------------------------------------*/
		if (FLDget_time_nsteps (set_id, &nsteps) != OM_STAT_SUCCESS)
			nsteps = 0;
		if (nsteps) {
			/*---------------------------------------------------*/
			/*   get cell_data for one time step                 */
			/*   and set the reference from in's cell_data to it */
			/*---------------------------------------------------*/
			id = OMfind_subobj(out_id, OMstr_to_name("cell_data"), OM_OBJ_RW);
			if (OMis_null_obj(id)) {
				ERR_RETURN("Error getting cell_data");
			}
			if (FLDget_time_cell_data(set_id, step, &time_id) == OM_STAT_SUCCESS) {
				ref_id = OMfind_subobj(time_id, OMstr_to_name("cell_data"), OM_OBJ_RW);
				if (OMis_null_obj(id)) {
					ERR_RETURN("Error getting cell_data");
				}
				if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting field reference");
				}
			}
			else {
				if (FLDset_cell_data_ncomp (out_id, 0) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting ncell_data");
				}
			}
		}
	}
	return(1);
}

#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("interp time step", 0, ERR_ORIG, A); return(0);

/* 64-bit porting. Only Modified Internally */
int DVinterp_time_step_update(OMobj_id mod_id)
{
	OMobj_id  in, out, time_id, set_id, out_id, ref_id, id;
	double    time, *t, tt;
	int       nspace, nsteps, step, nsets, set, ncomp, comp, dtype;
        int       time_size;
	xp_long   size;
	float     *coord1, *coord2, *out_coord;
	float     min_ext1[3], max_ext1[3], min_ext2[3], max_ext2[3], out_min_ext[3], out_max_ext[3];
	char      *data1, *data2, *out_data;
	double    min1, min2, max1, max2, out_min, out_max;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMis_null_obj(in))
		return(0);

	out = OMfind_subobj(mod_id, OMstr_to_name("out_step"), OM_OBJ_RW);
	if (OMis_null_obj(out))
		return(0);

	if (OMget_name_real_val(mod_id, OMstr_to_name("time"), &time) != OM_STAT_SUCCESS)
		time = 0.0;

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	/*---------------------------------------------*/
	/*   get number of time steps for coordinates  */
	/*   and for node_data                         */
	/*---------------------------------------------*/
	if (FLDget_time_nsteps (in, &nsteps) != OM_STAT_SUCCESS)
		nsteps = 0;
	if (nsteps) {
		/*-----------------------------------------------------*/
		/*   find two time steps: time1 <= time <= time2       */
		/*-----------------------------------------------------*/
		step=0;
		tt = 0.0;
		if (FLDget_time_values(in, &t, &time_size, OM_GET_ARRAY_RD)!= OM_STAT_SUCCESS) {
			ERR_RETURN("Error getting time array");
		}
		if (time < t[0] || time > t[nsteps-1]) {
			ERR_RETURN("time value is out of range");
		}
		for (step=0; step<nsteps-1; step++)
			if (t[step] <= time && time <= t[step+1]) {
				if (t[step] != t[step+1])
					tt = (time - t[step])/(t[step+1] - t[step]);
				break;
			}

		/*-----------------------------------------------------*/
		/*   get coordinates for two time steps                */
		/*   and interpolate between two steps                 */
		/*-----------------------------------------------------*/
		id = OMfind_subobj(out, OMstr_to_name("coordinates"), OM_OBJ_RW);
		if (OMis_null_obj(id)) {
			ERR_RETURN("Error getting node_data");
		}
		if (FLDget_time_coordinates(in, step, &time_id) == OM_STAT_SUCCESS) {
			ref_id = OMfind_subobj(time_id, OMstr_to_name("coordinates"), OM_OBJ_RW);
			if (OMis_null_obj(id)) {
				ERR_RETURN("Error getting coordinates");
			}
			if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting field reference");
			}

			if (FLDget_nspace (in, &nspace) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting nspace");
			}

			if (FLDget_coord(time_id, &coord1, &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting coordinates");
			}
			if (FLDget_coord_extent(time_id, min_ext1, max_ext1) != 1) {
				ERR_RETURN("Error getting extents");
			}

			if (FLDget_time_coordinates(in, step+1, &time_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting time coordinates");
			}
			if (FLDget_coord(time_id, &coord2, &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting coordinates");
			}
			if (FLDget_coord_extent(time_id, min_ext2, max_ext2) != 1) {
				ERR_RETURN("cannot get extents");
			}

			if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_RW) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting out coordinates");
			}
			UTILinterpolate(size, tt, DTYPE_FLOAT,
					(char *)coord1, (char *)coord2, (char *)out_coord);

			UTILinterpolate(nspace, tt, DTYPE_FLOAT,
					(char *)min_ext1, (char *)min_ext2, (char *)out_min_ext);
			UTILinterpolate(nspace, tt, DTYPE_FLOAT,
					(char *)max_ext1, (char *)max_ext2, (char *)out_max_ext);
			if (FLDset_coord_extent(out, out_min_ext, out_max_ext, nspace)!= OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting extents");
			}
			ARRfree(coord1);
			ARRfree(coord2);
			ARRfree(out_coord);
		}
		/*---------------------------------------------------*/
		/*   get node_data for one time step                 */
		/*   and set the reference from in's node_data to it */
		/*   and interpolate between two steps               */
		/*---------------------------------------------------*/
		id = OMfind_subobj(out, OMstr_to_name("node_data"), OM_OBJ_RW);
		if (OMis_null_obj(id)) {
			ERR_RETURN("Error getting node_data");
		}
		if (FLDget_time_node_data(in, step, &time_id) == OM_STAT_SUCCESS) {
			ref_id = OMfind_subobj(time_id, OMstr_to_name("node_data"), OM_OBJ_RW);
			if (OMis_null_obj(id)) {
				ERR_RETURN("Error getting node_data");
			}
			if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting field reference");
			}

			if (FLDget_node_data_ncomp (out, &ncomp) != OM_STAT_SUCCESS)
				ncomp = 0;
			for (comp = 0; comp < ncomp; comp++) {
				if (FLDget_time_node_data(in, step, &time_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting time node_data");
				}
				if (FLDget_node_data_type(time_id, comp, &dtype) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data type");
				}
				if (FLDset_node_data_type(out, comp, dtype) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting node data type");
				}


				if (FLDget_node_data(time_id, comp, &dtype, (char **)(&data1),
						     &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data");
				}
				if (FLDget_node_data_minmax(time_id, comp, (char *)&min1, (char *)&max1) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data minmax");
				}

				if (FLDget_time_node_data(in, step+1, &time_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting time node_data");
				}
				if (FLDget_node_data(time_id, comp, &dtype, (char **)(&data2),
						     &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data");
				}
				if (FLDget_node_data_minmax(time_id, comp, (char *)&min2, (char *)&max2) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data minmax");
				}

				if (FLDget_node_data(out, comp, &dtype, (char **)(&out_data),
						     &size, OM_GET_ARRAY_RW) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting node data");
				}

				UTILinterpolate(size, tt, dtype,
						(char *)data1, (char *)data2, (char *)out_data);
				UTILinterpolate(1, tt, dtype,
						(char *)&min1, (char *)&min2, (char *)&out_min);
				UTILinterpolate(1, tt, dtype,
						(char *)&max1, (char *)&max2, (char *)&out_max);

				if (FLDset_node_data_minmax(out, comp, (char *)&out_min, (char *)&out_max, dtype) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting node data minmax");
				}

				ARRfree(data1);
				ARRfree(data2);
				ARRfree(out_data);
			}
		}
		else {
			if (FLDset_node_data_ncomp (out, 0) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting ncell_data");
			}
		}
		ARRfree(t);
	}

	if (FLDget_ncell_sets (in, &nsets) != OM_STAT_SUCCESS)
		nsets = 0;

	for (set=0; set<nsets; set++) {
		if (FLDget_cell_set (in, set, &set_id) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_cell_set (out, set, &out_id) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		/*---------------------------------------------*/
		/*   get number of time steps for cell_data    */
		/*---------------------------------------------*/
		if (FLDget_time_nsteps (set_id, &nsteps) != OM_STAT_SUCCESS)
			nsteps = 0;
		if (nsteps) {
			if (FLDget_cell_data_ncomp (out_id, &ncomp) != OM_STAT_SUCCESS)
				ncomp = 0;
			/*-----------------------------------------------------*/
			/*   find two time steps: time1 <= time <= time2       */
			/*-----------------------------------------------------*/
			step=0;
			tt = 0.0;
			if (FLDget_time_values(set_id, &t, &time_size, OM_GET_ARRAY_RD)!= OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting time array");
			}
			if (time < t[0] || time > t[nsteps-1]) {
				ERR_RETURN("time value is out of range");
			}
			for (step=0; step<nsteps-1; step++)
				if (t[step] <= time && time <= t[step+1]) {
					if (t[step] != t[step+1])
						tt = (time - t[step])/(t[step+1] - t[step]);
					break;
				}
			ARRfree(t);
			/*---------------------------------------------------*/
			/*   get cell_data for one time step                 */
			/*   and set the reference from in's cell_data to it */
			/*   and interpolate between two steps               */
			/*---------------------------------------------------*/
			id = OMfind_subobj(out_id, OMstr_to_name("cell_data"), OM_OBJ_RW);
			if (OMis_null_obj(id)) {
				ERR_RETURN("Error getting cell_data");
			}
			if (FLDget_time_cell_data(set_id, step, &time_id) == OM_STAT_SUCCESS) {
				ref_id = OMfind_subobj(time_id, OMstr_to_name("cell_data"), OM_OBJ_RW);
				if (OMis_null_obj(id)) {
					ERR_RETURN("Error getting cell_data");
				}
				if (FLDcopy_subelem_ref(id, ref_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting field reference");
				}
				for (comp = 0; comp < ncomp; comp++) {
					if (FLDget_time_cell_data(set_id, step, &time_id) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting time cell_data");
					}
					if (FLDget_cell_data_type(time_id, comp, &dtype) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error setting cell data type");
					}
					if (FLDset_cell_data_type(out_id, comp, dtype) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error setting cell data type");
					}

					if (FLDget_cell_data(time_id, comp, &dtype, (char **)(&data1),
							     &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting cell data");
					}
					if (FLDget_cell_data_minmax(time_id, comp, (char *)&min1, (char *)&max1) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting cell data minmax");
					}

					if (FLDget_time_cell_data(set_id, step+1, &time_id) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting time cell_data");
					}
					if (FLDget_cell_data(time_id, comp, &dtype, (char **)(&data2),
							     &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting cell data");
					}
					if (FLDget_cell_data_minmax(time_id, comp, (char *)&min2, (char *)&max2) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting cell data minmax");
					}


					if (FLDget_cell_data(out_id, comp, &dtype, (char **)(&out_data),
							     &size, OM_GET_ARRAY_RW) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error getting cell data");
					}

					UTILinterpolate(size, tt, dtype,
							(char *)data1, (char *)data2, (char *)out_data);
					UTILinterpolate(1, tt, dtype,
							(char *)&min1, (char *)&min2, (char *)&out_min);
					UTILinterpolate(1, tt, dtype,
							(char *)&max1, (char *)&max2, (char *)&out_max);

					if (FLDset_cell_data_minmax(out_id, comp, (char *)&out_min, (char *)&out_max, dtype) != OM_STAT_SUCCESS) {
						ERR_RETURN("Error setting cell data minmax");
					}

					ARRfree(data1);
					ARRfree(data2);
					ARRfree(out_data);
				}
			}
			else {
				if (FLDset_cell_data_ncomp (out_id, 0) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting ncell_data");
				}
			}
		}
	}
	return(1);
}
#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("set_global_extent", 0, ERR_ORIG, A); return(0);

int DVset_global_extent_update(OMobj_id mod_id)
{
	OMobj_id  in, out, time_id;
	int       nsteps, step, nspace, i;
	float     min_ext[3], max_ext[3], min[3], max[3];

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMis_null_obj(in))
		return(0);

	out = OMfind_subobj(mod_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
	if (OMis_null_obj(out))
		return(0);

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	if (FLDget_nspace (out, &nspace) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error getting nspace");
	}
	/*---------------------------------------------*/
	/*   get number of time steps for coordinates  */
	/*---------------------------------------------*/
	if (FLDget_time_nsteps (out, &nsteps) != OM_STAT_SUCCESS || nsteps == 0)
		return(0);

	for (step=0; step<nsteps; step++) {
		if (FLDget_time_coordinates(out, step, &time_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error getting time coordinates");
		}
		/*-----------------------------------------------------*/
		/*   get extents for each time step                    */
		/*-----------------------------------------------------*/
		if (FLDget_coord_extent(time_id, min_ext, max_ext) != 1) {
			ERR_RETURN("Error getting extents");
		}
		/*-----------------------------------------------------*/
		/*   compute global extent for all time steps          */
		/*-----------------------------------------------------*/

		if (step == 0) {
			memcpy(min, min_ext, nspace*sizeof(float));
			memcpy(max, max_ext, nspace*sizeof(float));
		}
		else {
			for (i=0; i<nspace; i++) {
				if (min_ext[i] < min[i])
					min[i] = min_ext[i];
				if (max_ext[i] > max[i])
					max[i] = max_ext[i];
			}
		}
	}
	/*-----------------------------------------------------*/
	/*   set global extent for each  time steps            */
	/*-----------------------------------------------------*/
	for (step=0; step<nsteps; step++) {
		if (FLDget_time_coordinates(out, step, &time_id) != OM_STAT_SUCCESS) {
			ERR_RETURN("Error getting time coordinates");
		}
		if (FLDset_coord_extent(time_id, min, max, nspace) != 1) {
			ERR_RETURN("Error setting extents");
		}
	}
	return(1);
}


#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("set_global_minmax", 0, ERR_ORIG, A); return(0);

int DVset_global_minmax_update(OMobj_id mod_id)
{
	OMobj_id  in, out, time_id;
	int       nsteps, step, ncomp, comp, dtype;
	double    min, max, dmin, dmax, rmin, rmax;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMis_null_obj(in))
		return(0);

	out = OMfind_subobj(mod_id, OMstr_to_name("out_nd"), OM_OBJ_RW);
	if (OMis_null_obj(out))
		return(0);

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	if (FLDget_node_data_ncomp (out, &ncomp) != OM_STAT_SUCCESS || ncomp == 0)
		return(0);
	/*---------------------------------------------*/
	/*   get number of time steps for node data    */
	/*---------------------------------------------*/
	if (FLDget_time_nsteps (out, &nsteps) != OM_STAT_SUCCESS || nsteps == 0)
		return(0);
	for (comp = 0; comp < ncomp; comp++) {
		for (step=0; step<nsteps; step++) {
			if (FLDget_time_node_data(out, step, &time_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting time node_data");
			}
			/*-----------------------------------------------------*/
			/*   get minmax for each time step                     */
			/*-----------------------------------------------------*/
			if (FLDget_node_data_type(time_id, comp, &dtype) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting node data type");
			}
			if (FLDget_node_data_minmax(time_id, comp, (char *)&min, (char *)&max) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting node data minmax");
			}
			/*-----------------------------------------------------*/
			/*   compute global minmax for all time steps          */
			/*-----------------------------------------------------*/
			UTILtype_to_double(&dmin, (char *)&min, dtype);
			UTILtype_to_double(&dmax, (char *)&max, dtype);
			if (step == 0) {
				rmin = dmin;
				rmax = dmax;
			}
			else {
				if (dmin < rmin)
					rmin = dmin;
				if (dmax > rmax)
					rmax = dmax;
			}
		}
		UTILdouble_to_type(&dmin, rmin, dtype);
		UTILdouble_to_type(&dmax, rmax, dtype);
		/*-----------------------------------------------------*/
		/*   set global minmax for each  time steps            */
		/*-----------------------------------------------------*/
		for (step=0; step<nsteps; step++) {
			if (FLDget_time_node_data(out, step, &time_id) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error getting time node_data");
			}
			if (FLDset_node_data_minmax(time_id, comp, (char *)&dmin, (char *)&dmax, dtype) != OM_STAT_SUCCESS) {
				ERR_RETURN("Error setting node data minmax");
			}
		}
	}
	return(1);
}


#undef ERR_RETURN
#define ERR_RETURN(A) ERRerror("set_global_minmax_cell", 0, ERR_ORIG, A); return(0);

int DVset_global_minmax_cell_update(OMobj_id mod_id)
{
	OMobj_id  in, out, time_id, set_id;
	int       nsteps, step, nsets, set, ncomp, comp, dtype;
	double    min, max, dmin, dmax, rmin, rmax;

	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMis_null_obj(in))
		return(0);

	out = OMfind_subobj(mod_id, OMstr_to_name("out_cd"), OM_OBJ_RW);
	if (OMis_null_obj(out))
		return(0);

	/*---------------------------------------------*/
	/*  Make common sub-objects of 'out'  to be    */
	/*  the references to sub-objects of 'in'      */
	/*---------------------------------------------*/
	if (FLDcopy_subelem_ref(out, in) != OM_STAT_SUCCESS) {
		ERR_RETURN("Error setting field reference");
	}

	if (FLDget_ncell_sets (out, &nsets) != OM_STAT_SUCCESS || nsets == 0)
		return(0);

	if (FLDget_cell_set (out, 0, &set_id) != 1) {
		ERR_RETURN("Error getting cell set");
	}
	if (FLDget_cell_data_ncomp (set_id, &ncomp) != OM_STAT_SUCCESS || ncomp == 0)
		return(0);
	if (FLDget_time_nsteps (set_id, &nsteps) != OM_STAT_SUCCESS || nsteps == 0)
		return(0);

	for (comp = 0; comp < ncomp; comp++) {
		for (step=0; step<nsteps; step++) {
			for (set=0; set<nsets; set++) {
				if (FLDget_cell_set (out, set, &set_id) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDget_time_cell_data(set_id, step, &time_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting cell time data");
				}
				/*-----------------------------------------------------*/
				/*   get minmax for each time step                     */
				/*-----------------------------------------------------*/
				if (FLDget_cell_data_type(time_id, comp, &dtype) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting cell data type");
				}
				if (FLDget_cell_data_minmax(time_id, comp, (char *)&min, (char *)&max) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting cell data minmax");
				}
				/*-----------------------------------------------------*/
				/*   compute global minmax for all time steps          */
				/*-----------------------------------------------------*/
				UTILtype_to_double(&dmin, (char *)&min, dtype);
				UTILtype_to_double(&dmax, (char *)&max, dtype);
				if (step == 0 && set == 0) {
					rmin = dmin;
					rmax = dmax;
				}
				else {
					if (dmin < rmin)
						rmin = dmin;
					if (dmax > rmax)
						rmax = dmax;
				}
			}
		}
		UTILdouble_to_type(&dmin, rmin, dtype);
		UTILdouble_to_type(&dmax, rmax, dtype);
		/*-----------------------------------------------------*/
		/*   set global minmax for each  time steps            */
		/*-----------------------------------------------------*/
		for (step=0; step<nsteps; step++) {
			for (set=0; set<nsets; set++) {
				if (FLDget_cell_set (out, set, &set_id) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDget_time_cell_data(set_id, step, &time_id) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error getting cell time data");
				}
				if (FLDset_cell_data_minmax(time_id, comp, (char *)&dmin, (char *)&dmax, dtype) != OM_STAT_SUCCESS) {
					ERR_RETURN("Error setting cell data minmax");
				}
			}
		}
	}
	return(1);
}
