/*
                        Copyright (c) 1994 by
                        Advanced Visual Systems Inc.
                        All Rights Reserved

        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.

        This file is under Perforce control
        $Id: //depot/express/fcs70/modules/wr_image.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>
#include <avs/f_utils.h>
#include <avs/config.h>
#include "image.h"

/* Functions defined in this file */
static int FUNCconfirm_extension (char *, int, int);
static int FUNCconfirm_write (OMobj_id, char *, int, int,
                              int, int, int, int, int, int, int, int);
static int FUNCget_image_field (OMobj_id, int *, int *, char **);
static int FUNCscalar_to_argb (int, int, char *, char **);
static void FUNCerror (const char *);
static int FUNCwrite_image (void *, OMobj_id, char *,
                            int, int, int, int, int, int, int);
static int FUNCwrite_wrapup (const char *, char *, char *,
                             void *(*)(), void *, void *);


#define ERR_RETURN(A) {ERRerror( "DVwrite image", 0, ERR_ORIG, A); return(0);}

#ifndef NO_DL_LOAD

/* for dynamic, loaded at runtime, libs make and use an array of info strings
 * containing the library and function names: these get looked up
 */

/* indices into writer info tables below */
#define LIBNAME			0
#define OPEN_FUNC		1
#define CLOSE_FUNC		2
#define SETWIDTH_FUNC		3
#define SETHEIGHT_FUNC		4
#define SETARGB_FUNC		5
#define SETFILETYPE_FUNC	6
#define SETDEPTH_FUNC		7
#define SETCOLORTYPE_FUNC	8
#define SETCOMPRESSIONTYPE_FUNC	9
#define SETREDUCTIONTYPE_FUNC	10

static const char *avs_info[] =
{
#ifdef MSDOS
  "libavsx",
#else
  "libavsx." DLLIB_SUFFIX,
#endif
  "avsOpen", "avsClose", "avsSetWidth", "avsSetHeight", "avsSetARGBImage",
  "avsSetFileType", "avsSetDepth", "avsSetColorType", "avsSetCompressionType",
  "avsSetReductionType"
};

static const char *bmp_info[] =
{
#ifdef MSDOS
  "libbmp",
#else
  "libbmp." DLLIB_SUFFIX,
#endif
  "bmpOpen", "bmpClose", "bmpSetWidth", "bmpSetHeight", "bmpSetARGBImage",
  "bmpSetFileType", "bmpSetDepth", "bmpSetColorType", "bmpSetCompressionType",
  "bmpSetReductionType"
};

static const char *gif_info[] =
{
#ifdef MSDOS
  "libgif",
#else
  "libgif." DLLIB_SUFFIX,
#endif
  "GIFOpen", "GIFClose", "GIFSetWidth", "GIFSetHeight", "GIFSetARGBImage",
  "GIFSetFileType", "GIFSetDepth", "GIFSetColorType", "GIFSetCompressionType",
  "GIFSetReductionType"
};

static const char *jpeg_info[] =
{
#ifdef MSDOS
  "libjpeg",
#else
  "libjpeg." DLLIB_SUFFIX,
#endif
  "jpegOpen", "jpegClose", "jpegSetWidth", "jpegSetHeight", "jpegSetARGBImage",
  "jpegSetFileType", "jpegSetDepth", "jpegSetColorType", "jpegSetCompressionType",
  "jpegSetReductionType"
};

static const char *pbm_info[] =
{
#ifdef MSDOS
  "libpbm",
#else
  "libpbm." DLLIB_SUFFIX,
#endif
  "pbmOpen", "pbmClose", "pbmSetWidth", "pbmSetHeight", "pbmSetARGBImage",
  "pbmSetFileType", "pbmSetDepth", "pbmSetColorType", "pbmSetCompressionType",
  "pbmSetReductionType"
};

static const char *sgiRGB_info[] =
{
#ifdef MSDOS
  "libsgiim",
#else
  "libsgiim." DLLIB_SUFFIX,
#endif
  "sgiRGBOpen", "sgiRGBClose", "sgiRGBSetWidth", "sgiRGBSetHeight", "sgiRGBSetARGBImage",
  "sgiRGBSetFileType", "sgiRGBSetDepth", "sgiRGBSetColorType", "sgiRGBSetCompressionType",
  "sgiRGBSetReductionType"
};

static const char *tiff_info[] =
{
#ifdef MSDOS
  "libtif",
#else
  "libtif." DLLIB_SUFFIX,
#endif
  "TIFFBegin", "TIFFEnd", "TIFFSetWidth", "TIFFSetHeight", "TIFFSetARGBImage",
  "TIFFSetFileType", "TIFFSetDepth", "TIFFSetColorType", "TIFFSetCompressionType",
  "TIFFSetReductionType"
};

/* just here for mapping: should never get called! */
static const char *null_info[] =
{
#ifdef MSDOS
  "libnull",
#else
  "libnull." DLLIB_SUFFIX,
#endif
  "NULLOpen", "NULLClose", "NULLSetWidth", "NULLSetHeight", "NULLSetARGBImage",
  "NULLSetFileType", "NULLSetDepth", "NULLSetColorType", "NULLSetCompressionType",
  "NULLSetReductionType"
};

#else

/* for bound-at-link-time ("NO_DL_LOAD") libraries, make and use a function
 * table as the interface to FUNCwrite_image()
 */

typedef struct _wr_funcs_t
{
  void *(*Open)(char *, int);
  void (*Close)(void *);
  int  (*SetWidth)(void *, int);
  int  (*SetHeight)(void *, int);
  int  (*SetARGBImage)();
  int  (*SetFileType)(void *, int);
  int  (*SetDepth)(void *, int);
  int  (*SetColorType)(void *, int);
  int  (*SetCompressionType)(void *, int, int);
  int  (*SetReductionType)(void *, int);
} wr_funcs_t;

/* defined in rd_image.c...
void *NULLfunc_voidp() { return((void *) NULL); }
void NULLfunc_void() { return; }
int NULLfunc_int() { return(0); }
*/
extern void *NULLfunc_voidp(), NULLfunc_void();
extern int NULLfunc_int();

static wr_funcs_t null_funcs = { NULLfunc_voidp, NULLfunc_void,
			      NULLfunc_int, NULLfunc_int, NULLfunc_int,
			      NULLfunc_int, NULLfunc_int, NULLfunc_int,
			      NULLfunc_int, NULLfunc_int };

static wr_funcs_t avs_funcs = { avsOpen, avsClose,
			     avsSetWidth, avsSetHeight, avsSetARGBImage,
			     avsSetFileType, avsSetDepth, avsSetColorType,
			     avsSetCompressionType, avsSetReductionType };

static wr_funcs_t bmp_funcs = { bmpOpen, bmpClose,
			     bmpSetWidth, bmpSetHeight, bmpSetARGBImage,
			     bmpSetFileType, bmpSetDepth, bmpSetColorType,
			     bmpSetCompressionType, bmpSetReductionType };

static wr_funcs_t gif_funcs = { GIFOpen, GIFClose,
			     GIFSetWidth, GIFSetHeight, GIFSetARGBImage,
			     GIFSetFileType, GIFSetDepth, GIFSetColorType,
			     GIFSetCompressionType, GIFSetReductionType };

static wr_funcs_t jpeg_funcs = { jpegOpen, jpegClose,
			      jpegSetWidth, jpegSetHeight, jpegSetARGBImage,
			      jpegSetFileType, jpegSetDepth, jpegSetColorType,
			      jpegSetCompressionType, jpegSetReductionType };

static wr_funcs_t pbm_funcs = { pbmOpen, pbmClose,
			     pbmSetWidth, pbmSetHeight, pbmSetARGBImage,
			     pbmSetFileType, pbmSetDepth, pbmSetColorType,
			     pbmSetCompressionType, pbmSetReductionType };

static wr_funcs_t sgiRGB_funcs = { sgiRGBOpen, sgiRGBClose,
		     sgiRGBSetWidth, sgiRGBSetHeight, sgiRGBSetARGBImage,
		     sgiRGBSetFileType, sgiRGBSetDepth, sgiRGBSetColorType,
		     sgiRGBSetCompressionType, sgiRGBSetReductionType };

static wr_funcs_t tiff_funcs = { TIFFBegin, TIFFEnd,
			      TIFFSetWidth, TIFFSetHeight, TIFFSetARGBImage,
			      TIFFSetFileType, TIFFSetDepth, TIFFSetColorType,
			      TIFFSetCompressionType, TIFFSetReductionType };

#endif


typedef struct _write_map {
  int     dv_val;
#ifndef NO_DL_LOAD
  const char **info;
#else
  wr_funcs_t *info;
#endif
} write_map;

/* order of these corresponds to the UIradiobox vals...
 */
static write_map FUNCwrite_map[] = {
  { DV_IMAGE_FORMAT_AVSX,		/* 0 */
#ifndef NO_DL_LOAD
    avs_info
#else
    &avs_funcs
#endif
  },
  { DV_IMAGE_FORMAT_BMP,		/* 1 */
#ifndef NO_DL_LOAD
    bmp_info
#else
    &bmp_funcs
#endif
  },
  { DV_IMAGE_FORMAT_GIF,		/* 2 */
#ifndef NO_DL_LOAD
    gif_info
#else
    &gif_funcs
#endif
  },
  { DV_IMAGE_FORMAT_JPEG,		/* 3 */
#ifndef NO_DL_LOAD
    jpeg_info
#else
    &jpeg_funcs
#endif
  },
  { DV_IMAGE_FORMAT_PBM,		/* 4 */
#ifndef NO_DL_LOAD
    pbm_info
#else
    &pbm_funcs
#endif
  },
  { DV_IMAGE_FORMAT_SGIRGB,		/* 5 */
#ifndef NO_DL_LOAD
    sgiRGB_info
#else
    &sgiRGB_funcs
#endif
  },
  { DV_IMAGE_FORMAT_SUNRAS,		/* 6 */
#ifndef NO_DL_LOAD
    null_info
#else
    &null_funcs
#endif
  },
  { DV_IMAGE_FORMAT_TIFF,		/* 7 */
#ifndef NO_DL_LOAD
    tiff_info
#else
    &tiff_funcs
#endif
  }
};

/* private local data attached to write image module */
typedef struct _params_t {
  char *filename;
  int  in_seq;
  int  format;
  int  flip;
  int  type;
  int  depth;
  int  colortype;
  int  compresstype;
  int  quality;
  int  reducetype;
} params_t;

/*----------------------------------------------------------------------
 * the public update entry point (update method in v/dv.v)
 */
int DVwrite_image_update(OMobj_id elem_id)
{
  OMobj_id in_id, file_id, flip_id, over_id;
  char     filebuf[AVS_PATH_MAX], *filename = filebuf;
  int      result, overwrite, in_seq;
  int      flip, rb_val, format, type;
  int      depth, colortype, compresstype, reducetype, quality;
  double   dquality;

  in_id = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
  in_seq = OMget_obj_seq(in_id, OMnull_obj, OM_SEQ_VAL);

  file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
  if (OMis_null_obj(file_id))
    return(0);

  if (OMget_str_val(file_id, &filename, sizeof(filebuf)) != OM_STAT_SUCCESS)
    return(0);

  if (OMget_name_int_val(elem_id, OMstr_to_name("format"), &rb_val)
							!= OM_STAT_SUCCESS)
    rb_val = 0;

  /* map format UIradiobox val to internal def */
  if (rb_val < 0 || rb_val > sizeof(FUNCwrite_map) / sizeof(write_map))
  {
    FUNCerror("Not written: file format is not supported.");
    return(0);
  }
  else
    format = FUNCwrite_map[rb_val].dv_val;

  /* set default params */
  flip_id = OMfind_subobj(elem_id, OMstr_to_name("flip"), OM_OBJ_RD);
  if (OMget_int_val(flip_id, &flip) != OM_STAT_SUCCESS)
    flip = 1;

  if (OMget_name_int_val(elem_id, OMstr_to_name("filetype"), &type)
							!= OM_STAT_SUCCESS)
    type = DV_IMAGE_FILE_TYPE_BINARY;

  if (OMget_name_int_val(elem_id, OMstr_to_name("depth"), &depth)
							!= OM_STAT_SUCCESS)
    depth = DV_IMAGE_DEPTH_NA;

  if (OMget_name_int_val(elem_id, OMstr_to_name("colortype"), &colortype)
							!= OM_STAT_SUCCESS)
    colortype = DV_IMAGE_COLORTYPE_RGB;

  if (OMget_name_int_val(elem_id, OMstr_to_name("compresstype"), &compresstype)
							!= OM_STAT_SUCCESS)
    compresstype = DV_IMAGE_COMPRESS_NONE;

  if (OMget_name_real_val(elem_id, OMstr_to_name("compressqual"), &dquality)
							!= OM_STAT_SUCCESS)
    dquality = 0.0;
  quality = (int) dquality;

  if (OMget_name_int_val(elem_id, OMstr_to_name("reducetype"), &reducetype)
							!= OM_STAT_SUCCESS)
    reducetype = DV_IMAGE_QUANTIZE_NA;

  over_id = OMfind_subobj(elem_id, OMstr_to_name("overwrite"), OM_OBJ_RD);
  if (OMget_int_val(over_id, &overwrite) != OM_STAT_SUCCESS)
    overwrite = 0;

  /* check whether to add a file extension */
  FUNCconfirm_extension(filename, format, AVS_PATH_MAX);

  /* check if ok to overwrite existing file if we haven't got the overwrite
   * toggle in
   */
  if (!overwrite && !FUNCconfirm_write(elem_id, filename, in_seq, format,
				       flip, type, depth, colortype,
				       compresstype, quality, reducetype,
				       AVS_PATH_MAX))
    return(0);

  result = 0;
  switch (format)
  {
    case DV_IMAGE_FORMAT_AVSX:
    case DV_IMAGE_FORMAT_BMP:
    case DV_IMAGE_FORMAT_GIF:
    case DV_IMAGE_FORMAT_JPEG:
    case DV_IMAGE_FORMAT_PBM:
    case DV_IMAGE_FORMAT_SGIRGB:
    case DV_IMAGE_FORMAT_TIFF:
    {
      int i;

      for (i = 0; i < sizeof(FUNCwrite_map) / sizeof(write_map); i++)
      {
	if (format == FUNCwrite_map[i].dv_val)
	{
	  result = FUNCwrite_image(FUNCwrite_map[i].info, in_id, filename,
				   flip, type, depth, colortype,
				   compresstype, quality, reducetype);
	  break;
	}
      }
      break;
    }

    default:
      FUNCerror("Not written: file format is not supported.");
      break;
  }

  return(result);
}

/*----------------------------------------------------------------------
 * the public delete entry point (delete method in v/dv.v)
 */
int DVwrite_image_delete(OMobj_id elem_id)
{
  OMobj_id ptr_id;
  params_t *params;

  /* free any local data associated with this module */
  ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
  if (!OMis_null_obj(ptr_id) && OMget_ptr_val(ptr_id, (void *) &params, 0) == 1)
  {
    if (params)
    {
      if (params->filename)
	free(params->filename);
      free(params);
    }
    OMset_obj_val(ptr_id, OMnull_obj, 0);
  }

  return(1);
}

/*----------------------------------------------------------------------
 * checks if a filename has an extension and asks if one should be added
 */
static int FUNCconfirm_extension(char *filename, int format, int maxlen)
{
  char ext[5], msg[64];
  int lower, upper, i, len, button;

  if (strrchr(filename, '.'))
    return(1);

  switch (format)
  {
    case DV_IMAGE_FORMAT_AVSX:
      sprintf(ext, "%s", "x");
      break;
    case DV_IMAGE_FORMAT_GIF:
      sprintf(ext, "%s", "gif");
      break;
    case DV_IMAGE_FORMAT_JPEG:
      sprintf(ext, "%s", "jpg");
      break;
    case DV_IMAGE_FORMAT_PBM:
      sprintf(ext, "%s", "pxm");
      break;
    case DV_IMAGE_FORMAT_SUNRAS:
      sprintf(ext, "%s", "imx");
      break;
    case DV_IMAGE_FORMAT_SGIRGB:
      sprintf(ext, "%s", "rgb");
      break;
    case DV_IMAGE_FORMAT_TIFF:
      sprintf(ext, "%s", "tif");
      break;
    case DV_IMAGE_FORMAT_PS:
      sprintf(ext, "%s", "ps");
      break;
    case DV_IMAGE_FORMAT_BMP:
      sprintf(ext, "%s", "bmp");
      break;
    default:
      return(0);
  }

  /* if filename is all uppercase make the extension so */
  lower = upper = 0;
  len = (int)strlen(filename);
  for (i = 0; !(lower && upper) && i < len; i++)
  {
    if (!lower && islower(filename[i]))
      lower = 1;
    if (!upper && isupper(filename[i]))
      upper = 1;
  }
  if (upper && !lower)
  {
    len = (int)strlen(ext);
    for (i = 0; i < len; i++)
      ext[i] = toupper(ext[i]);
  }

  sprintf(msg, "No file extension, OK to add '.%s'?", ext);
  button = ERRsync_yesno_dialog(1, "Write Image", msg);

  if (button)
  {
    /* OK, add the extension */
    if ((int)(strlen(filename) + strlen(ext) + 2) > maxlen)
      FUNCerror("Extension is not added: filename is too long.");
    else
      sprintf(filename, "%s.%s", filename, ext);
    return(1);
  }
  else
    return(0);
}

/*----------------------------------------------------------------------
 * checks if a file exists and asks to confirm overwrite if necessary;
 * returns 1 if OK to write file, 0 if not
 */
static int FUNCconfirm_write(OMobj_id id, char *filename, int in_seq,
			     int format, int flip, int type, int depth,
			     int colortype, int compresstype, int quality,
			     int reducetype, int maxlen)
{
  FILE     *fp;
  OMobj_id ptr_id;
  int      status;
  params_t *params;

  /* file existence check */
  fp = FILEfopen(filename, SIO_R_BIN);
  if (fp)
    fclose(fp);

  /* get current state of params struct attached to module */
  ptr_id = OMfind_subobj(id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
  if (OMis_null_obj(ptr_id))
  {
    /* no local_ptr in module, a pre-3.1 condition, just return overwrite */
    return(1);
  }
  status = OMget_ptr_val(ptr_id, (void *) &params, 0);
  if (status == -1)
  {
    /* apparently ptr_id is "remote", we can't hack it, just return overwrite */
    return(1);
  }
  else if (status != 1)
  {
    /* doesn't exist, create and attach it to module */
    ALLOCN(params, params_t, 1, "DVwrite_image: failure allocating local data");
    OMset_ptr_val(ptr_id, params, 0);
  }

  /* don't ask for confirm unless a param changed */
  if (fp && status == 1 &&
      filename && params->filename && !strcmp(filename, params->filename) &&
      in_seq == params->in_seq &&
      format == params->format &&
      flip == params->flip &&
      type == params->type &&
      depth == params->depth &&
      colortype == params->colortype &&
      compresstype == params->compresstype &&
      quality == params->quality &&
      reducetype == params->reducetype)
  {
    /* still write the file in case the data, width, height changed! */
    return(1);
  }

  /* update state in local data */
  if (params->filename)
    free(params->filename);
  params->filename = strdup(filename);
  params->in_seq = in_seq;
  params->format = format;
  params->flip = flip;
  params->type = type;
  params->depth = depth;
  params->colortype = colortype;
  params->compresstype = compresstype;
  params->quality = quality;
  params->reducetype = reducetype;

  if (fp)
  {
    char *msg;
    int  button;

    msg = malloc(maxlen+64);
    if (msg)
    {
      sprintf(msg, "File '%s' exists, OK to overwrite?", filename);
      button = ERRsync_yesno_dialog(1, "Write Image", msg);
      free(msg);
    }
    else
    {
      button = ERRsync_yesno_dialog(1, "Write Image",
				    "File exists, OK to overwrite?");
    }

    if (button)
      return(1);
    else
      return(0);
  }
  else
    return(1);
}

/*----------------------------------------------------------------------
 * validates and gets image data from field
 */
/* 64-bit porting. Only Modified Internally */ 
static int FUNCget_image_field(OMobj_id in_id, int *w, int *h, char **data)
{
  xp_long size;
  int ndim, nspace, dims_size;
  xp_long nnodes, *dims;
  int ncomp, veclen, id, type;
  float *points;
  char *img_data;

  if (FLDget_nnodes(in_id, &nnodes) != 1) {
    ERR_RETURN("Can't get nnodes");
  }
  if (FLDget_ndim(in_id, &ndim) != 1) {
    ERR_RETURN("Can't get ndim");
  }
  if (ndim != 2) {
    ERR_RETURN("ndim must equal 2");
  }
  if (FLDget_dims(in_id, &dims, &dims_size) != 1) {
    ERR_RETURN("Can't get dims");
  }
  if (FLDget_nspace(in_id, &nspace) != 1) {
    ERR_RETURN("Can't get nspace");
  }
  if (nspace != 2) {
    ERR_RETURN("nspace must equal 2");
  }
  if (FLDget_points(in_id, &points, &size, OM_GET_ARRAY_RD) != 1) {
    ERR_RETURN("Can't get points");
  }
  if (FLDget_node_data_ncomp(in_id, &ncomp) != 1) {
    ERR_RETURN("Can't get ncomp in node data");
  }
  if (ncomp != 1) {
    ERR_RETURN("ncomp of node data must be 1");
  }
  if (FLDget_node_data_type(in_id, 0, &type) != 1) {
    ERR_RETURN("Can't get type in node data");
  }
  if (type != DTYPE_BYTE) {
    ERR_RETURN("Sorry, can only write BYTE data currently");
  }
  if (FLDget_node_data_veclen(in_id, 0, &veclen) != 1) {
    ERR_RETURN("Can't get veclen of node data");
  }

  /* we'll accept ARGB (id RGB) or scalar byte node data */
  if (FLDget_node_data_id(in_id, 0, &id) != 1)
  {
    if (veclen != 1) {
      ERR_RETURN("veclen of scalar data must be 1");
    }
  }
  else
  {
    if (id != GD_RGB_DATA_ID) {
      ERR_RETURN("id must be RGB when data has an id");
    }
    if (veclen != 4) {
      ERR_RETURN("veclen of RGB data must be 4");
    }
  }

  if (FLDget_node_data(in_id, 0, &type, (char **)&img_data, &size,
		       OM_GET_ARRAY_RD) != 1) {
    ERR_RETURN("Can't get node data array");
  }

  *w = (int)dims[0];  *h = (int)dims[1];
  *data = img_data;

  /* won't be referencing these... */
  ARRfree(dims);
  ARRfree(points);

  return(veclen);
}

/*----------------------------------------------------------------------
 * replicates scalar byte data into an ARGB array
 */
static int FUNCscalar_to_argb(int w, int h, char *node_data, char **img_data)
{
  byte *data, *src, *dst;
  int  i, n;

  n = w * h;
  UALLOCN(data, byte, n * 4, "DVwrite_image: failure allocating ARGB data");

  src = (unsigned char *) node_data;
  dst = data;
  for (i = 0; i < n; i++)
  {
    *dst++ = 0;  *dst++ = *src;  *dst++ = *src;  *dst++ = *src++;
  }

  *img_data = (char *) data;

  return(1);
}

/*----------------------------------------------------------------------
 * spits out an error message
 */
static void FUNCerror(const char *msg)
{
#if 0
  static int mbox = 0;
  static OMobj_id msg_id;

  if (!mbox)
  {
    msg_id = OMcreate_obj_from_path("UImessageBox", "Write Image", OMinst_obj);
    mbox = 1;
  }

  if (!(OMis_null_obj(msg_id)))
  {
    OMset_name_str_val(msg_id, OMstr_to_name("message"), msg);
    OMset_name_int_val(msg_id, OMstr_to_name("visible"), 1);
  }
#endif

  ERRerror("DVwrite image", 0, ERR_ORIG, msg);

  return;
}

/*----------------------------------------------------------------------
 * generic image writer; "NO_DL_LOAD" means we don't do the dynamic library/dll
 * open and lookup but just call the funcs from an assumed-to-be filled-in
 * table
 */
static int FUNCwrite_image(void *access, OMobj_id in, char *filename,
			   int flip, int type, int depth, int colortype,
			   int compresstype, int quality, int reducetype)
{
  char *node_data = NULL;
  char *img_data = NULL;
  int w, h, veclen;
  void *priv = NULL, *(*fptr)() = NULL, *(*close)() = NULL, *handle = NULL;
#ifndef NO_DL_LOAD
  char **info = (char **) access;
#else
  wr_funcs_t *funcs = (wr_funcs_t *) access;
#endif

  if (!filename || strlen(filename) == 0)
    return(0);

  /* open the lib */
#ifndef NO_DL_LOAD

  if (info[LIBNAME] != NULL)
    DL_OPEN(info[LIBNAME], DL_BIND, handle);
  else
    return(FUNCwrite_wrapup("No library name",
			    node_data, img_data, close, priv, handle));

  if (handle == NULL)
  {
    char err[512];
#ifdef MSDOS
    const char *libpath = "PATH";
    DWORD last_err = GetLastError();
#elif defined(hpux) || defined(__hpux)
    const char *libpath = "SHLIB_PATH";
#elif defined(__APPLE__) || defined(__MACH__)
    const char *libpath = "DYLD_LIBRARY_PATH";
#else
    const char *libpath = "LD_LIBRARY_PATH";
#endif

    sprintf(err, "Cannot open library '%s'.  Most likely cause is an unset or incorrectly set '%s' environment variable: please see the System Prerequisites Document.  The system error message is:\n'%s'\n", info[LIBNAME], libpath, DL_ERROR());

#ifdef MSDOS
    sprintf(err, "%s and GetLastError number is %d", err, last_err);
#endif

    return(FUNCwrite_wrapup(err, node_data, img_data, close, priv, handle));
  }
#endif


  /* lookup the Open function ----------------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[OPEN_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find Open function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->Open;
#endif

  /* and open the file */
  if (!(priv = (*fptr)(filename, DV_IMAGE_FILE_ACCESS_WRITE)))
    return(FUNCwrite_wrapup("File open failed",
			    node_data, img_data, close, priv, handle));

  /* lookup the Close function now for err return --------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[CLOSE_FUNC], void *(*)(), close);
  if (close == NULL)
    return(FUNCwrite_wrapup("Cannot find Close function in library",
			    node_data, img_data, close, priv, handle));
#else
  close = (void *(*)()) funcs->Close;
#endif

  /* validate and get the field info */
  if ((veclen = FUNCget_image_field(in, &w, &h, &node_data)) == 0)
    return(FUNCwrite_wrapup("Invalid field data",
			    node_data, img_data, close, priv, handle));
  if (veclen == 1)
  {
    /* replicate the scalar value into an ARGB array (wasteful, but it's
     * all the image write libs deal with currently)
     */
    if (!(FUNCscalar_to_argb(w, h, node_data, &img_data)))
      return(FUNCwrite_wrapup("Failure converting scalar data to RGB",
			      node_data, img_data, close, priv, handle));
  }

  /* lookup the SetWidth function -------------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETWIDTH_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetWidth function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetWidth;
#endif

  /* and set the width */
  (*fptr)(priv, w);

  /* lookup the SetHeight function ------------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETHEIGHT_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetHeight function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetHeight;
#endif

  /* and set the height */
  (*fptr)(priv, h);

  /* lookup the SetFileType function ----------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETFILETYPE_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetFileType function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetFileType;
#endif

  /* and set the type */
  (*fptr)(priv, type);

  /* lookup the SetDepth function ---------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETDEPTH_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetDepth function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetDepth;
#endif

  /* and set the depth */
  (*fptr)(priv, depth);

  /* lookup the SetColorType function ---------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETCOLORTYPE_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetColorType function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetColorType;
#endif

  /* and set the colortype */
  (*fptr)(priv, colortype);

  /* lookup the SetCompressionType function ---------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETCOMPRESSIONTYPE_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetCompressionType function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetCompressionType;
#endif

  /* and set the compressiontype and quality */
  (*fptr)(priv, compresstype, quality);

  /* lookup the SetReductionType function ------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETREDUCTIONTYPE_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetReductionType function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetReductionType;
#endif

  /* and set the reductiontype */
  (*fptr)(priv, reducetype);

  /* lookup the SetARGBImage function ----------------------------------------*/
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[SETARGB_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCwrite_wrapup("Cannot find SetARGBImage function in library",
			    node_data, img_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->SetARGBImage;
#endif

  /* and set the image data */
  if (img_data)
  {
    if (!((*fptr)(priv, img_data, flip)))
      return(FUNCwrite_wrapup("Failure setting ARGB data",
			      node_data, img_data, close, priv, handle));
  }
  else
  {
    if (!((*fptr)(priv, node_data, flip)))
      return(FUNCwrite_wrapup("Failure setting ARGB data",
			      node_data, img_data, close, priv, handle));
  }

  /* wrapup */
  FUNCwrite_wrapup(NULL, node_data, img_data, close, priv, handle);

  return(1);
}

/*----------------------------------------------------------------------
 * what to do at the end of (or on error when) writing the file
 */
static int FUNCwrite_wrapup(const char *msg, char *node_data, char *img_data,
                            void *(*close)(), void *priv, void *handle)
{
  /* close the file */
  if (close && priv)
    (*close)(priv);

  /* free the field data */
  if (node_data)
    ARRfree(node_data);
  if (img_data)
    free(img_data);

#ifndef NO_DL_LOAD
  /* close the library */
  if (handle)
    DL_CLOSE(handle);
#endif

  if (msg)
  {
    FUNCerror(msg);
    return(0);
  }
  else
    return(1);
}
