# cpp.awk
# v1.0 - gwl - 1 Feb 2001 - First released with AVS/Express 5.1.1 MPE
# ----------------------------------------------------------------------
# Limited version of cpp. Understands the following macros:
#
# #define (constants only, not paramterized macros)
# #undef
# #ifdef
# #ifndef
# #if (const or macro, not logical expressions)
# #elif
# #else
# #endif
#
# White space may occur between the '#' and the directive.
# ----------------------------------------------------------------------

# ----------------------------------------------------------------------
# Dev Notes:
#
# GLOBALS:
# Variables:
#
#   skip:	    Indicate whether subsequent input text should be ignored.
#		    Not all macros test this - some look at the parent block's
#		    skip value to see if the line containing the macro should be
#		    examined. All ordinary lines look at 'skip' though.
#   skip_condtest:  Indicate whether subsequent conditional tests should be ignored.
#		    This is only local to the current level of nested #ifs. 
#		    0 = previous conditionals were false so continue trying tests
#		    1 = a previous conditional was true so don't test any more
#		    2 = 1 but also reached the #else statement so no more #elifs
#   if_count:	    Current level of nested #if-type macros
#   prev_line_com_stat: Did a comment begin somewhere on the previous line
#
# Arrays:
#
#   macro_name_defined[]:	Indexed by the name of the macro. 0 or 1 indicating
#				whether the named macro has a value.
#   macro_value[]:		Indexed by the name of the macro. Gives the
#				current value of the macro (or "" if not defined.)
#
#   nested_skip[]:	Indexed by if_count - stores the skip flag for current 
#			level of nested #if-type macros.
#   nested_condtest[]:	Indexed by if_count - stores the skip_condtest flag for 
#			current level of nested #if-type macros.

BEGIN { 
  skip = 0;  
  in_comment = 0; skip_condtest = 0; 
  if_count = 0;   prev_line_com_start = 0; 
  debug = 0; }

# Initialise at the start of each line read. We save the current skip and
# skip_condtest flags for the current level of nested #if-type macros. This 
# is effectively a stack which would be pushed/popped as we move in/out of
# nested #if..#endif blocks.
{nested_skip[if_count]=skip}
{nested_condtest[if_count]=skip_condtest}
  

# ----------------------------------------------------------------------------
# Strip single line comments (from anywhere in the line)
/\/\*.*\*\// && !in_comment { gsub( /\/\*.*\*\//, "" ) }

# Start of C-style comments (multi-line)
/\/\*/ && !in_comment {
  
  # Strip text after start of comment
  gsub( /\/\*.*/, "" );

  # Can't ignore this line - there may be valid text before the comment start.
  # Indicate we should ignore the next line though.
  prev_line_com_start = 1;
}

# End of C-style comments (multi-line)
/\*\// && in_comment {

  # Strip text before start of comment
  gsub( /.*\*\//, "" );
  
  # Keep count of number of opening comments (comment is treated as a #if)
  in_comment = 0;
}

# ----------------------------------------------------------------------------

# Match #include
/^\#[\t ]*include/ && !skip && !in_comment {
  if ( $1=="#include" ) name_field = 2;
  else name_field = 3;

  # Strip the double quotes from the filename and copy in every line from file
  gsub(/\"/, "", $name_field) ;
  while ( getline include_line < $name_field > 0 )
    print include_line;
}

# ----------------------------------------------------------------------------

# Match #define
/^\#[\t ]*define/ && !skip && !in_comment {
  # Did white space occur between '#' and the directive?
  # Calc the index of the field containing the macro name.
  if ( $1=="#define" ) name_field=2;
  else name_field=3;

  if ( NF < name_field )
    ERROR("illegal macro name");
  else {
    # Indicate the macro name has been defined
    macro_name_defined[$name_field] = 1;

    # Get the value of the macro
    i = name_field+1;
    macro_value[$name_field]=$i
      for(i++; i<=NF; i++)
	macro_value[$name_field] = macro_value[$name_field] " " $i;
  }
}

# ----------------------------------------------------------------------------

# Match #undef
/^\#[\t ]*undef/ && !skip && !in_comment {
  if ( $1=="#undef" ) name_field = 2;
  else name_field = 3;

  if ( NF >= name_field ) {
    # Clear 'defined' flag and macro value
    macro_name_defined[$name_field] = 0; 
    macro_value[$name_field]="";
  }
}

# ----------------------------------------------------------------------------

# Match #if
/^\#[\t ]*if([\t ]*$|[\t ]+.*$)/ && !in_comment {
  if ( $1=="#if" ) name_field = 2;
  else name_field = 3;

  # Effectively push the current skip and skip_condtest flags onto the stack
  if_count++;

  DEBUG( if_count, skip, "#" skip_condtest );

  if ( !skip ) {
    if ( NF < name_field )
      ERROR("syntax error");
    else {
      # Eval the conditional and record whether to skip subsequent tests and text
      skip_condtest = eval_expr(name_field);
      skip = !skip_condtest;
    }
  }
}

# ----------------------------------------------------------------------------

# Match #elif
/^\#[\t ]*elif/ && !in_comment {
  if ( $1=="#elif" ) name_field = 2;
  else name_field = 3;

  DEBUG( if_count, "#" skip, skip_condtest );

  if ( if_count == 0 )
    ERROR("If-less #elif");
  else if ( !skipping_parent(if_count) ) {
    if ( skip_condtest ) {	# Previous #if/#elif tests were true so *skip* this
      if ( skip_condtest == 2 ) # Seen an #else in this block so can't use #elif 
	ERROR("#elif after #else");
      skip = 1;
    }
    else  {			# Previous #if/#elif tests were false so *try* this
      if ( NF < name_field )
	ERROR("syntax error");
      
      skip_condtest = eval_expr(name_field);
      skip = !skip_condtest;
    }
  }
}

# ----------------------------------------------------------------------------

# Match #else
/^\#[\t ]*else/ && !in_comment {

  DEBUG( if_count, "#" skip, skip_condtest );

  if ( if_count == 0 )
    ERROR("If-less #else");
  else if ( !skipping_parent(if_count) ) {
    if ( skip_condtest ) {	# Previous #if/#elif tests were true, so *skip* this
      if ( skip_condtest == 2 ) # Already seen #else in this block so can't use more
	ERROR("multiple #else");
      skip = 1;
    }
    else {			# Previous #if/#elif tests were false so *do* this
      skip = 0;			# Read input text after a #else
    }
    skip_condtest = 2;		# #else is always the last conditional in block
  }
}

# ----------------------------------------------------------------------------

# Match #ifdef
/^\#[\t ]*ifdef/ && !in_comment {
  if ( $1=="#ifdef" ) name_field = 2;
  else name_field = 3;

  # Keep count of number of #if-type macros (used for error checking #endifs)
  if_count++;

  DEBUG( if_count, skip, "#" skip_condtest );

  if ( !skip ) {
    if ( NF < name_field )
      ERROR("syntax error");
    else {

      # If the macro name is not defined, indicate we should skip subsequent input
      # otherwise indicate we should skip a subsequent #else clause.
      if ( !macro_name_defined[$name_field] )
	skip = 1;
      else
        skip_condtest = 1;
    }
  }
}

# ----------------------------------------------------------------------------

# Match #ifndef
/^\#[\t ]*ifndef/ && !in_comment {
  if ( $1=="#ifndef" ) name_field = 2;
  else name_field = 3;

  # Keep count of number of #if-type macros (used for error checking #endifs)
  if_count++;

  DEBUG( if_count, skip, "#" skip_condtest );
  
  if ( !skip ) {
    if ( NF < name_field )
      ERROR("syntax error");
    else {

      # If the macro name *is* defined, indicate we should skip subsequent input
      if ( macro_name_defined[$name_field] )
	skip = 1;
      else
        skip_condtest = 1;
    }
  }
}

# ----------------------------------------------------------------------------

# Match #endif
/^\#[\t ]*endif/ && !in_comment {

  # Check we've got matching correct number of #endifs
  if ( if_count == 0 )
    ERROR("If-less #endif");
  else {
    # Reduce number of #if-type macros (used for error checking)
    if_count--;

    # Restore current nested level's skip flags
    skip = nested_skip[if_count];
    skip_condtest = nested_condtest[if_count];
  }

  DEBUG( if_count, skip, skip_condtest );

}

# ----------------------------------------------------------------------------

# Do for every line in file *not* begining with a '#'
(/^[^\#]/ || NF == 0 ) && !in_comment {
  # Only process line if previous #if...'s indicate we should do so
  if ( !skip ) {
    expr = macro_subst($0);
    print expr;
  }
}

# Previous input line may have contained an opening C-style comment, possibly 
# with some valid text before the comment. Only now can we begin ignoring
# input lines. We can't ignore the previous input line which contained
# the comment start since the text before the comment may have required
# processing.
{ if ( prev_line_com_start ) {
    in_comment = 1;
    prev_line_com_start = 0;
  }
}

# ----------------------------------------------------------------------------

# Do a final check to see that we have enough #endifs
 END    { if ( if_count != 0 ) { ERROR("missing #endif") } }
 
# ----------------------------------------------------------------------------

# Print an error message, reporting the line number.
 function ERROR(msg) {
   print "#cpp.awk: error " FILENAME ":" NR ": "msg ;
 }

# ----------------------------------------------------------------------------

# Print info about the current line being processed
function DEBUG(level, skipline, skiptest) {
  if ( debug )
    print "Read:", $0, "\tlevel:", level, "skip line:", skipline, "\tskip test:", \
      skiptest, "\tskipping parent:", skipping_parent(level);
}

# ----------------------------------------------------------------------------

# Does the immediate outer conditional block (#if/#ifdef/#ifndef) mean
# that the current block should be ignored?

function skipping_parent(cur_level) {
  return nested_skip[cur_level-1];
}

# ----------------------------------------------------------------------------

# Parse the expression after a #if or #elif a return a 1 or 0 indicating
# the logical value of the expression. Pass in the field number where the
# expression begins. Note: we do not do any clever parsing in this version.
#
# expr and i are local vars, not to be passed in by caller.

 function eval_expr(field_num,    expr, i) {

   # Get the expression
   for ( i = field_num; i<=NF; i++ )
     expr = expr " " $i;

   # Perform macro substitution on the expression
   expr = macro_subst(expr);

   # Eval the expression. Could do with a good parser here. Currently, we
   # just coerce the expression to a numeric value and eval that.
   if ( expr + 0 == 0 )
     return 0;
   else
     return 1;
 }

# ----------------------------------------------------------------------------

# Perform macro substitution on the given expression and return the modified
# expression. Uses the current state of the arrays indicating which macros
# are defined and their value.
#
# i is a local var, not to be passed in by caller.

 function macro_subst(expr,     i ) {
   for( i in macro_name_defined ) {
     if( macro_name_defined[i] ) {
       gsub(i,macro_value[i], expr);
     }
   }

   return expr;
 }
   
