/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/fld/access.c#2 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdlib.h>
#include <string.h>

#include <avs/dll_in.h>
#include <avs/arr.h>
#include <avs/data_utils.h>
#include <avs/om.h>
#include <avs/om_type.h>
#include <avs/om_att.h>
#include <avs/fld.h>

typedef  unsigned char byte;

static float IXFORM[16] = {1.0, 0.0, 0.0, 0.0,
			   0.0, 1.0, 0.0, 0.0,
			   0.0, 0.0, 1.0, 0.0,
			   0.0, 0.0, 0.0, 1.0};
/*-------------------------------*/
/*      GENERAL FUNCTIONS        */
/*-------------------------------*/

OMobj_id FLDadd_elem(OMobj_id field, const char *temp_name,
                     const char *elem_name)
{
	OMobj_id elem;

	elem = OMfind_subobj (OMtempl_obj, OMstr_to_name(temp_name),
				 OM_OBJ_RD);
	if (OMis_null_obj(elem))
		return(OMnull_obj);
	elem = OMcopy_obj(elem, OMstr_to_name(elem_name), field,
			   OMlocal_proc_id, OM_COPY_CLOSE);
	/*
	 * The system currently does not figure out that this is not
	 * supposed to be saved so we must do it by hand.
	 */
	OMset_obj_atts(elem,OM_atts_nosave);
	return(elem);
}

int FLDadd_int(OMobj_id field, const char *name)
{
	OMobj_id elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("int"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	return(1);
}

int FLDset_int(OMobj_id  field, const char *name, int val)
{
	return
		( OMset_name_int_val (field, OMstr_to_name(name), val) );
}

int FLDget_int(OMobj_id field, const char *name, int *val)
{
	return
		( OMget_name_int_val (field, OMstr_to_name(name), val) );
}

/* 64-bit porting. Newly Introduced */
int FLDadd_long(OMobj_id field, const char *name)
{
	OMobj_id elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("long"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	return(1);
}

/* 64-bit porting. Newly Introduced */
int FLDset_long(OMobj_id  field, const char *name, xp_long val)
{
	return
		( OMset_name_long_val (field, OMstr_to_name(name), val) );
}

/* 64-bit porting. Newly Introduced */
int FLDget_long(OMobj_id field, const char *name, xp_long *val)
{
	return
		( OMget_name_long_val (field, OMstr_to_name(name), val) );
}

int FLDadd_real(OMobj_id field, const char *name)
{
	OMobj_id elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("double"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	return(1);
}

int FLDset_real(OMobj_id field, const char *name, double val)
{
	return
		(OMset_name_real_val (field, OMstr_to_name(name), (double)val));
}

int FLDget_real(OMobj_id field, const char *name, double *val)
{
	double dval;
	int    stat;
	stat = OMget_name_real_val (field, OMstr_to_name(name), &dval);
	if (stat < 1)
		return (stat);
	*val = (float)dval;
	return(1);
}

/* 64-bit porting. Directly Modified */
int FLDadd_int_array(OMobj_id field, const char *name, xp_long size)
{
	int   stat;
	OMobj_id elem, delem, dims_elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("int"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	delem = OMcreate_obj(OM_NULL_NAME, OM_type_raw_long, OMnull_obj,
				OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(delem))
		return(-1);
	OMset_long_val(delem, size);
	dims_elem = OMcreate_obj(OM_name_dims, OM_type_val_list,
				  elem, OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(dims_elem))
		return(-1);
	stat = OMadd_subobj(dims_elem, delem);
	if (stat<0)
		return(-1);
	return(1);
}

/* 64-bit porting. Directly Modified */
int FLDset_array_int (OMobj_id field, const char *name, xp_long size, int *array,
                      int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)array, size, mode);
	return(stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int  FLDget_array_int (OMobj_id field, const char *name, int **array,
                       xp_long *size, int mode)
{
	OMobj_id  elem_id;
	int        type, stat;

	*array = (int *)NULL;
	*size = 0;
	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)array, size, mode);
	return (stat);
}

#ifdef WORDLENGTH_64
int  FLDget_array_int_n (OMobj_id field, const char *name, int **array,
                       int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_array_int(field, name, array, &size_w, mode);

	*size = (int)size_w;

	return status;

}
#endif

/* 64-bit porting. Newly Introduced */
int FLDadd_long_array(OMobj_id field, const char *name, xp_long size)
{
	int   stat;
	OMobj_id elem, delem, dims_elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("long"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	delem = OMcreate_obj(OM_NULL_NAME, OM_type_raw_long, OMnull_obj,
				OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(delem))
		return(-1);
	OMset_long_val(delem, size);
	dims_elem = OMcreate_obj(OM_name_dims, OM_type_val_list,
				  elem, OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(dims_elem))
		return(-1);
	stat = OMadd_subobj(dims_elem, delem);
	if (stat<0)
		return(-1);
	return(1);
}

/* 64-bit porting. Newly Introduced */
int FLDset_array_long (OMobj_id field, const char *name, xp_long size, xp_long *array,
                      int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)array, size, mode);
	return(stat);
}

/* 64-bit porting. Newly Introduced */
int  FLDget_array_long (OMobj_id field, const char *name, xp_long **array,
                       xp_long *size, int mode)
{
	OMobj_id  elem_id;
	int        type, stat;

	*array = (xp_long *)NULL;
	*size = 0;
	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)array, size, mode);
	return (stat);
}

/* 64-bit porting. Directly Modified */
int FLDadd_float_array(OMobj_id field, const char *name, xp_long size)
{
	int   stat;
	OMobj_id elem, delem, dims_elem;

	elem = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("float"), field, OMnull_obj,
			     OMlocal_proc_id);
	if (OMis_null_obj(elem))
		return(-1);
	delem = OMcreate_obj(OM_NULL_NAME, OM_type_raw_long, OMnull_obj,
				OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(delem))
		return(-1);
	OMset_long_val(delem, size);
	dims_elem = OMcreate_obj(OM_name_dims, OM_type_val_list, elem,
				  OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(dims_elem))
		return(-1);
	stat = OMadd_subobj(dims_elem, delem);
	return(stat);
}

/* 64-bit porting. Directly Modified */
int FLDset_array_float (OMobj_id field, const char *name, xp_long size,
                        float *array, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)array, size, mode);
	return(stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int  FLDget_array_float (OMobj_id field, const char *name, float **array,
                         xp_long *size, int mode)
{
	OMobj_id  elem_id;
	int        type, stat;

	*array = (float *)0;
	*size = 0;
	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)array, size, mode);
	return (stat);
}

#ifdef WORDLENGTH_64
int  FLDget_array_float_n (OMobj_id field, const char *name, float **array,
                       int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_array_float(field, name, array, &size_w, mode);

	*size = (int)size_w;

	return status;

}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_array_size(OMobj_id e, xp_long *size)
{
	return(OMget_array_size(e, size));
}

#ifdef WORDLENGTH_64
int FLDget_array_size_n(OMobj_id e, int *size)
{
	int status;
	xp_long size_w;

	status = FLDget_array_size(e, &size_w);

	*size = (int)size_w;

	return status;
}
#endif

int  FLDequal(OMobj_id e1, OMobj_id e2)
{
	OMobj_id  v1, v2;
	int  stat;

	if ((stat=OMget_obj_val(e1, &v1)) != 1 ||
	    (stat=OMget_obj_val(e2, &v2)) != 1 )
		return (stat);
	return(OMequal_objs(v1, v2));
}

int FLDget_parent_val(OMobj_id e, OMobj_id *parent)
{
	OMobj_id  e1;
	int  stat;

	if ((stat=OMget_obj_val(e, &e1)) != 1)
		return (stat);
	return(OMget_obj_parent(e1, parent));
}

int FLDget_subelem(OMobj_id field, const char *name, OMobj_id *elem)
{

	*elem = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(*elem))
		return(0);
	return(1);
}

int FLDget_subelem_seq(OMobj_id field, const char *temp_name)
{
	OMobj_id  elem;

	elem = OMfind_subobj (OMtempl_obj, OMstr_to_name(temp_name),
				   OM_OBJ_RD);
	if (OMis_null_obj(elem))
		return(0);
	return(OMget_obj_seq (field, elem, OM_SEQ_VAL));
}

OMobj_id  FLDcreate(OMobj_id parent_id, const char *field_desc,
                    const char *name)
{
	OMobj_id  parnt, elem, elem_mrg;
	char  *desc, *token;

	if (OMis_null_obj(parent_id))
		parnt = OMinst_obj;
	else
		parnt = parent_id;
	desc = (char *)malloc((strlen(field_desc)+1)*sizeof(char));
	strcpy(desc, field_desc);
	token = strtok(desc, " ");
	if (token == NULL)
		return(OMnull_obj);
	elem = OMfind_subobj (OMtempl_obj, OMstr_to_name(token),
				   OM_OBJ_RD);
	if (OMis_null_obj(elem))
		return(OMnull_obj);
	elem = OMcopy_obj(elem, OMstr_to_name(name), parnt, OMlocal_proc_id,
			   OM_COPY_CLOSE);
	token = strtok(NULL, " ");
	while (token) {
		elem_mrg = OMfind_subobj (OMtempl_obj, OMstr_to_name(token),
					   OM_OBJ_RD);
		if (OMis_null_obj(elem_mrg))
			return(OMnull_obj);
		OMmerge_obj(elem_mrg, elem, 0);
		token = strtok(NULL, " ");
	}
	free(desc);
	return(elem);
}

int FLDdestroy(OMobj_id field)
{
	return(OMdestroy_obj(field,0));
}

int FLDdelete(OMobj_id field, const char *name)
{
	OMobj_id elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name(name), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMdestroy_obj(elem_id,0));
}

int  FLDcopy_elem_ref(OMobj_id ref, OMobj_id base)
{
	int  stat;

	stat =  OMset_obj_ref_mode(ref, OM_OBJ_REF);
	if (stat < 1)
		return(stat);
	stat = OMset_obj_ref(ref, base, 0);
	if (stat < 1)
		return(stat);
	return(1);
}

int  FLDcopy_subelem_ref(OMobj_id ref, OMobj_id base)
{
	int  stat;

	/*
	 * We want the NO_MATCH mode here both for performance and also
	 * because we can't rely upon the connections matching properly.
	 * If there is old data in the field, we may end up setting
	 * values before nvals is set... the connections will not match
	 * in this case.
	 */
	stat = OMset_obj_val(ref, base, OM_OBJ_VAL_REF_MODE |
					 OM_OBJ_REF_RDONLY |
					 OM_OBJ_REF_NO_MATCH);
	return(stat);
}

int FLDget_function(OMobj_id elem, const char *name, OMpfi *func)
{
	OMobj_id  func_id;
	int  stat;

	func_id = OMfind_subobj (elem, OMstr_to_name(name), OM_OBJ_RD);
	if (OMis_null_obj(func_id))
		return(0);
	stat = OMget_func_val(func_id, func);
	return(stat);
}

int FLDget_methods(OMobj_id elem, int n, const char *names[], OMpfi *func)
{
	OMobj_id  func_id, elem_id;
	int  i, stat;

	elem_id = OMfind_subobj (elem, OMstr_to_name("methods"), OM_OBJ_RD);
	if (OMis_null_obj(elem_id)) {
		for (i=0; i<n; i++)
			func[i] = (OMpfi)0;
		return(1);
	}
	for (i=0; i<n; i++) {
		func_id = OMfind_subobj (elem_id, OMstr_to_name(names[i]), OM_OBJ_RD);
		if (!OMis_null_obj(func_id)) {
			stat = OMget_func_val(func_id, &(func[i]));
			if (stat != 1)
				func[i] = (OMpfi)0;
		}
		else
			func[i] = (OMpfi)0;
	}
	return(1);
}

/*-----------------------------------------------*/
/*           GENERAL FIELD                       */
/*-----------------------------------------------*/

int FLDget_grid_type(OMobj_id field, int *grid_type)
{
	return
		( OMget_name_int_val (field, OMstr_to_name("grid_type"), grid_type) );
}

int FLDset_grid_type (OMobj_id field, int grid_type)
{
	return
		(OMset_name_int_val (field, OMstr_to_name("grid_type"), grid_type));
}

int FLDget_ndim(OMobj_id field, int *ndim)
{
	return
		( OMget_name_int_val (field, OMstr_to_name("ndim"), ndim) );
}

int FLDset_ndim (OMobj_id field, int ndim)
{
	return
		(OMset_name_int_val (field, OMstr_to_name("ndim"), ndim));
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_nnodes(OMobj_id field, xp_long *nnodes)
{
	return
		( OMget_name_long_val (field, OMstr_to_name("nnodes"), nnodes) );
}

#ifdef WORDLENGTH_64
int FLDget_nnodes_n(OMobj_id field, int *nnodes)
{
	int status;
	xp_long nnodes_w;

	status = FLDget_nnodes(field, &nnodes_w);

	*nnodes = (int)nnodes_w;

	return status;
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_nnodes (OMobj_id field, xp_long nnodes)
{
	int  stat;

	stat = OMset_name_long_val (field, OMstr_to_name("nnodes"), nnodes);
	return(stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int  FLDget_dims (OMobj_id field, xp_long **dims, int *ndim)
{
	OMobj_id  elem_id;
	xp_long size = 0;
	int     type, stat, mode;

	mode = OM_GET_ARRAY_RD;
	*dims = (xp_long *)NULL;
	*ndim = 0;
	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (field, OMstr_to_name("dims"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)dims, &size, mode);
	if (stat == 1) *ndim = size;
	return (stat);
}

#ifdef WORDLENGTH_64
int  FLDget_dims_n (OMobj_id field, int **dims, int *size)
{
	OMobj_id  elem_id;
	int        type, stat;
	int     mode;

	mode = OM_GET_ARRAY_RD;
	*dims = (int *)NULL;
	*size = 0;
	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (field, OMstr_to_name("dims"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz_n(elem_id, &type, (char **)dims, size, mode);
	return (stat);

}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDset_dims (OMobj_id field, xp_long *dims)
{
	int   stat, type, ndim;
	OMobj_id   elem_id;

	OMget_name_int_val (field, OMstr_to_name("ndim"), &ndim);

	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (field, OMstr_to_name("dims"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)dims, ndim, OM_SET_ARRAY_COPY);
	return(stat);
}

/* Code duplication is done since 'dims' size is not known */
#ifdef WORDLENGTH_64
int FLDset_dims_n (OMobj_id field, int *dims)
{
	int   stat, type, ndim;
	OMobj_id   elem_id;

	OMget_name_int_val (field, OMstr_to_name("ndim"), &ndim);

	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (field, OMstr_to_name("dims"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)dims, ndim, OM_SET_ARRAY_COPY);
	return(stat);
}
#endif

/*--------------------------------------------------*/
/*         COORDINATES                              */
/*--------------------------------------------------*/

int FLDget_xform(OMobj_id field, float *xform)
{

/*************************************************
	int i;
	float tmp_mat[16], tmp_x[3];
	OMobj_id  elem_id;

	memcpy(xform, IXFORM, 16*sizeof(float));
	elem_id = OMfind_subobj (field, OMstr_to_name("xform"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id)) {
		return(0);
	}
	GDxform_get_matrix(elem_id, tmp_mat);
	GDxform_get_xlate(elem_id, tmp_x);
	for (i=0; i<3;i++)
		memcpy(xform+4*i, tmp_mat+4*i, 3*sizeof(float));
	memcpy(xform+12, tmp_x, 3*sizeof(float));
************************************************/
	OMobj_id  elem_id;

	memcpy(xform, IXFORM, 16*sizeof(float));
	elem_id = OMfind_subobj (field, OMstr_to_name("xform"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id)) {
		return(1);
	}
	GDxform_get_comp_matrix(elem_id, xform);

	return(1);
}

int FLDset_xform(OMobj_id field, float *xform)
{
	OMobj_id  elem_id;
	int  i;
	float tmp_mat[16], tmp_x[3];

	elem_id = OMfind_subobj (field, OMstr_to_name("xform"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id)) {
		return(0);
	}
	memcpy(tmp_mat, IXFORM, 16*sizeof(float));
	for (i=0; i<3;i++)
		memcpy(tmp_mat+4*i, xform+4*i, 3*sizeof(float));
	memcpy(tmp_x, xform+12, 3*sizeof(float));
	GDxform_set_matrix(elem_id, tmp_mat);
	GDxform_set_xlate(elem_id, tmp_x);
	return(1);
}

int FLDget_rspace(OMobj_id field, int *rspace)
{
	OMobj_id  elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("xform"), OM_OBJ_RD);
	if (OMis_null_obj(elem_id)) {
		return(0);
	}
	return (OMget_name_int_val (elem_id, OMstr_to_name("rspace"), rspace));
}

int FLDset_rspace(OMobj_id field, int rspace)
{
	OMobj_id  elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("xform"), OM_OBJ_RW);
	if (OMis_null_obj(elem_id)) {
		return(0);
	}
	return (OMset_name_int_val (elem_id, OMstr_to_name("rspace"), rspace));
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_npoints(OMobj_id field, xp_long *npoints)
{
	return( OMget_name_long_val (field, OMstr_to_name("npoints"), npoints) );
}

#ifdef WORDLENGTH_64
int FLDget_npoints_n(OMobj_id field, int *npoints)
{
	int status;
	xp_long npoints_w;

	status = FLDget_npoints(field, &npoints_w);

	*npoints = (int)npoints_w;

	return status;

}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_npoints(OMobj_id field, xp_long npoints)
{
	return( OMset_name_long_val (field, OMstr_to_name("npoints"), npoints) );
}

int FLDget_nspace(OMobj_id field, int *nspace)
{
	return( OMget_name_int_val (field, OMstr_to_name("nspace"), nspace) );
}

int FLDset_nspace (OMobj_id field, int nspace)
{
	return (OMset_name_int_val (field, OMstr_to_name("nspace"), nspace));
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_coord(OMobj_id field, float **coord_array, xp_long *size, int mode)
{
	OMobj_id  elem_id;
	int    type, stat, nspace;
	xp_long nnodes;

	type = OM_TYPE_FLOAT;
	*coord_array = (float *)NULL;
	*size = 0;
	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if ((stat = OMget_name_long_val (field, OMstr_to_name("nnodes"), &nnodes)) != 1)
		return(stat);

	if ((stat = OMget_name_int_val (field, OMstr_to_name("nspace"), &nspace)) != 1)
		return(stat);
	elem_id = OMfind_subobj (elem_id, OMstr_to_name("values"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if ((stat=OMget_array_sz(elem_id, &type, (char **)coord_array,
				 size, mode)) != 1)
		return(stat);
	if (nnodes*nspace != *size)
		return(0);
	return(1);
}

#ifdef WORDLENGTH_64
int FLDget_coord_n(OMobj_id field, float **coord_array, int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_coord(field, coord_array, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_sub_coord(OMobj_id field, int ndim, xp_long *dims,
                     xp_long *min_rng, xp_long *max_rng, float *coord_array)
{
	OMobj_id  elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id = OMfind_subobj (elem_id, OMstr_to_name("values"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return (OMget_sub_farray(elem_id, ndim, dims, min_rng, max_rng,
				 (void *)coord_array));
}

#ifdef WORDLENGTH_64
int FLDget_sub_coord_n(OMobj_id field, int ndim, int *dims,
                     int *min_rng, int *max_rng, float *coord_array)
{
	int i;
	xp_long dims_w[OM_ARRAY_MAXDIM], min_rng_w[OM_ARRAY_MAXDIM], max_rng_w[OM_ARRAY_MAXDIM];

	for(i = 0; i < ndim; i++)
	{
	  dims_w[i] = (xp_long)dims[i];
	  min_rng_w[i] = (xp_long)min_rng[i];
	  max_rng_w[i] = (xp_long)max_rng[i];
	}

	return FLDget_sub_coord(field, ndim, dims_w, min_rng_w, max_rng_w, coord_array);
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_coord(OMobj_id field, float *coord_array, xp_long size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id = OMfind_subobj (elem_id, OMstr_to_name("values"),
				    OM_OBJ_RW);
	stat = OMset_array(elem_id, type, (char *)coord_array, size, mode);
	return(stat);
}

int FLDreset_point_extent(OMobj_id field)
{
	int   stat;
	OMobj_id   elem_id, elem_id_min, elem_id_max;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id_min = OMfind_subobj (elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min, "cache(min_array(points, 0, 0))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max, "cache(max_array(points, 0, 0))");
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDreset_coord_extent(OMobj_id field)
{
	int   stat;
	OMobj_id   elem_id, elem_id_min, elem_id_max;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id_min = OMfind_subobj (elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min, "cache(min_array(values, 0, 0))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max, "cache(max_array(values, 0, 0))");
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDset_coord_extent(OMobj_id field, float *min_extent, float *max_extent,
                        int nspace)
{
	int   stat, type;
	OMobj_id   elem_id, elem_id_min, elem_id_max;

	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id_min = OMfind_subobj (elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMset_array(elem_id_min, type, (char *)min_extent,
			   nspace, OM_SET_ARRAY_COPY);
	if (stat != 1)
		return(stat);
	stat = OMset_array(elem_id_max, type, (char *)max_extent,
			   nspace, OM_SET_ARRAY_COPY);
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDget_coord_extent(OMobj_id field, float *min_extent, float *max_extent)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max;
	int    type, mode, nspace, stat1, stat2;
	xp_long size;
	float  *tmin, *tmax;

	if (FLDget_nspace(field, &nspace) != 1)
		return(-1);
	type = OM_TYPE_FLOAT;
	size = 0;
	mode = OM_GET_ARRAY_RD;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);

	elem_id_min = OMfind_subobj (elem_id, OMstr_to_name("min_vec"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (elem_id, OMstr_to_name("max_vec"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	mode = OM_GET_ARRAY_RD;
	tmin = (float *)NULL;
	tmax = (float *)NULL;
	size = 0;
	stat1 = OMget_array_sz(elem_id_min, &type, (char **)&tmin, &size, mode);
	size = 0;
	stat2 = OMget_array_sz(elem_id_max, &type, (char **)&tmax, &size, mode);
	/*
	 * Make sure to free one of these if the other happens not to be
	 * valid
	 */
	if (stat1 == 1) {
	   memcpy(min_extent, tmin, nspace*sizeof(float));
	   if (tmin != NULL) ARRfree(tmin);
	}
	if (stat2 == 1) {
	   memcpy(max_extent, tmax, nspace*sizeof(float));
	   if (tmax != NULL) ARRfree(tmax);
	}
	if (stat1 != 1 || stat2 != 1)
		return(-1);
	return (1);
}

/* 64-bit porting. Modified with Parallel APIs */
int  FLDget_points (OMobj_id field, float **points, xp_long *size, int mode)
{
	OMobj_id  elem_id;
	int        type, stat;

 	/* XXX Hack to work around dgroup bug with npoints and rect flds. */
	(void) OMfind_subobj(field, OMstr_to_name("npoints"), OM_OBJ_RW);

	*points = (float *)NULL;
	*size = 0;
	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name("points"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);

	stat = OMget_array_sz(elem_id, &type, (char **)points, size, mode);
	
	return (stat);
}

#ifdef WORDLENGTH_64
int  FLDget_points_n (OMobj_id field, float **points, int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_points(field, points, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

int FLDset_points(OMobj_id field, float *points, xp_long size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (field, OMstr_to_name("points"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(FLDset_coord(field, points, size, mode));
	stat = OMset_array(elem_id, type, (char *)points, size, mode);
	return(stat);
}

int FLDget_coord_units(OMobj_id field, char *units, int size)
{
	OMobj_id    elem_id;

	/* Check for programmer error: */
	if (units == NULL || size == 0)
	   return 0;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);

	elem_id = OMfind_subobj (elem_id, OMstr_to_name("units"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &units, size));
}

int FLDset_coord_units (OMobj_id field, const char *units)
{
	int   stat;
	OMobj_id   elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("coordinates"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	elem_id = OMfind_subobj (elem_id, OMstr_to_name("units"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val(elem_id, units);
	return(stat);
}


/*-------------------------------------------------------------*/
/*                  NODE DATA                                  */
/*-------------------------------------------------------------*/

int FLDget_node_data_ncomp(OMobj_id field, int *nnode_comp)
{
	return
		( OMget_name_int_val (field, OMstr_to_name("nnode_data"), nnode_comp) );
}

int FLDset_node_data_ncomp (OMobj_id field, int nnode_comp)
{
	return(OMset_name_int_val (field, OMstr_to_name("nnode_data"), nnode_comp));
}

int FLDset_node_data_comp (OMobj_id  field, int comp, int veclen,
                           const char *label, const char *units)
{
	OMobj_id  elem_id, sub_elem_id;
	int   stat;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val(elem_id, label);
	if (stat < 1)
		return (0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val(elem_id, units);
	if (stat < 1)
		return (0);
	return(1);
}


int FLDget_node_data_veclen(OMobj_id field, int comp, int *veclen)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RD) <= 0)
		return(0);
	return
		( OMget_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen) );
}

int FLDset_node_data_veclen(OMobj_id field, int comp, int veclen)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	return(OMset_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen));
}

int FLDget_node_data_type(OMobj_id field, int comp, int *type)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_data_type(elem_id, type));
}

int FLDset_node_data_type(OMobj_id field, int comp, int type)
{
	int stat;
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	/* set values */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set null_value */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("null_value"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set min max */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set min_vec max_vec */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	return(1);
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_node_data(OMobj_id field, int comp, int *type, char **node_data,
                     xp_long *size, int mode)
{
	OMobj_id  elem_id, sub_elem_id;
	int   stat, veclen;
	xp_long nnodes;

	*node_data = (char *)NULL;
	*size = 0;
	if (mode != OM_GET_ARRAY_WR)
		*type = OM_TYPE_UNSET;
	if ((stat = OMget_name_long_val (field, OMstr_to_name("nnodes"), &nnodes)) != 1)
		return(stat);

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if ((stat=OMget_name_int_val (sub_elem_id, OMstr_to_name("veclen"),
				      &veclen)) != 1)
		return(stat);
	if ((stat = OMget_array_sz(elem_id, type, (char **)node_data,
				   size, mode)) != 1)
		return(stat);
	if (veclen*nnodes != *size)
		return(0);
	return(1);
}

#ifdef WORDLENGTH_64
int FLDget_node_data_n(OMobj_id field, int comp, int *type, char **node_data,
                     int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_node_data(field, comp, type, node_data, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_typed_node_data(OMobj_id field, int comp, int *type,
                           char **node_data, xp_long *size, int mode)
{
	OMobj_id  elem_id, sub_elem_id;
	int   stat, veclen;
	xp_long nnodes;

	*node_data = (char *)NULL;
	*size = 0;
	if ((stat = OMget_name_long_val (field, OMstr_to_name("nnodes"), &nnodes)) != 1)
		return(stat);

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if ((stat=OMget_name_int_val (sub_elem_id, OMstr_to_name("veclen"),
				      &veclen)) != 1)
		return(stat);
	if ((stat = OMget_array_sz(elem_id, type, (char **)node_data,
				   size, mode)) != 1)
		return(stat);
	if (veclen*nnodes != *size)
		return(0);
	return(1);
}

#ifdef WORDLENGTH_64
int FLDget_typed_node_data_n(OMobj_id field, int comp, int *type,
                           char **node_data, int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_typed_node_data(field, comp, type, node_data, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_sub_node_data(OMobj_id field, int comp, int *type, int ndim,
                         xp_long *dims, xp_long *min_rng, xp_long *max_rng,
                         char *node_data)
{
	int  stat;
	OMobj_id  elem_id, sub_elem_id;

	if (*type == OM_TYPE_UNSET) {
		stat = FLDget_node_data_type(field, comp, type);
		if (stat < 0)
			return(0);
	}
	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	switch (*type) {
	      case OM_TYPE_DOUBLE:
		return (OMget_sub_rarray(elem_id, ndim, dims, min_rng, max_rng,
					 (double *)node_data));
	      case OM_TYPE_FLOAT:
		return (OMget_sub_farray(elem_id, ndim, dims, min_rng, max_rng,
					 (float *)node_data));
	      case OM_TYPE_LONG:
		return (OMget_sub_larray(elem_id, ndim, dims, min_rng, max_rng,
					 (xp_long *)node_data));
	      case OM_TYPE_INT:
		return (OMget_sub_iarray(elem_id, ndim, dims, min_rng, max_rng,
					 (int *)node_data));
	      case OM_TYPE_SHORT:
		return (OMget_sub_sarray(elem_id, ndim, dims, min_rng, max_rng,
					 (short *)node_data));
	      case OM_TYPE_BYTE:
		return (OMget_sub_barray(elem_id, ndim, dims, min_rng, max_rng,
					 (byte *)node_data));
	      case OM_TYPE_CHAR:
		return (OMget_sub_barray(elem_id, ndim, dims, min_rng, max_rng,
					 (unsigned char *)node_data));
	      default:
		return(0);
	}
}

#ifdef WORDLENGTH_64
int FLDget_sub_node_data_n(OMobj_id field, int comp, int *type, int ndim,
                         int *dims, int *min_rng, int *max_rng,
                         char *node_data)
{
	int i;
	xp_long dims_w[OM_ARRAY_MAXDIM], min_rng_w[OM_ARRAY_MAXDIM], max_rng_w[OM_ARRAY_MAXDIM];

	for(i = 0; i < ndim; i++)
	{
		dims_w[i] = (xp_long)dims[i];
		min_rng_w[i] = (xp_long)min_rng[i];
		max_rng_w[i] = (xp_long)max_rng[i];
	}

	return FLDget_sub_node_data(field, comp, type, ndim, dims_w, min_rng_w, max_rng_w, node_data);

}
#endif

int FLDset_node_data(OMobj_id field, int comp, char *node_data, int type,
                     xp_long size, int mode)
{
	int   stat;
	OMobj_id   elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if ((stat = OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW))!= 1)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				    OM_OBJ_RW);
	stat = OMset_array(elem_id, type, (char *)node_data, size, mode);
	return(stat);
}

int FLDget_node_data_minmax_vec(OMobj_id field, int comp,
                                char **min, char **max, int *type, int mode)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int    stat1, stat2;
	xp_long size;

	if (mode != OM_GET_ARRAY_WR)
		*type = OM_TYPE_UNSET;
	*min = (char *)NULL;
	*max = (char *)NULL;
	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	size = 0;
	stat1 = OMget_array_sz(elem_id_min, type, (char **)min, &size, mode);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	size = 0;
	stat2 = OMget_array_sz(elem_id_max, type, (char **)max, &size, mode);
	if (stat1 != 1 || stat2 != 1)
		return(-1);
	return(1);
}

int FLDcopy_node_minmax_vec(OMobj_id in_field, OMobj_id out_field,
                            int in_comp, int out_comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	OMobj_id  elem_id1, elem_id_min1, elem_id_max1, sub_elem_id1;
	int stat;

	elem_id = OMfind_subobj (in_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, in_comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	elem_id1 = OMfind_subobj (out_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id1))
		return(0);
	if (OMget_array_val (elem_id1, out_comp, &sub_elem_id1, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id1))
		return(0);

	elem_id_min1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min1))
		return(0);
	elem_id_max1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max1))
		return(0);
 	/*
	 * The RDONLY flag here is necessary to prevent the "write" flag
	 * from propagating past us.  It will also prevent assignments
	 * through this connection which is probably good too...
	 */
	stat = OMset_obj_ref(elem_id_min1, elem_id_min, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	stat = OMset_obj_ref(elem_id_max1, elem_id_max, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FLDget_node_data_minmax(OMobj_id field, int comp, char *min, char *max)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int  stat, type, val_int;
	xp_long val_long;
	double val_real;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	stat = OMget_data_type(elem_id_min, &type);
	if (stat != 1)
		return(stat);
	switch (type) {
	      case OM_TYPE_CHAR:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((char *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((char *)max) = val_int;
		break;
	      case OM_TYPE_BYTE:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((unsigned char *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((unsigned char *)max) = val_int;
		break;
	      case OM_TYPE_SHORT:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((short *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((short *)max) = val_int;
		break;
	      case OM_TYPE_INT:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((int *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((int *)max) = val_int;
		break;
	      case OM_TYPE_LONG:
		stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("min"),
					   &val_long);
		*((xp_long *)min) = val_long;
		stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("max"),
					   &val_long);
		*((xp_long *)max) = val_long;
		break;
	      case OM_TYPE_FLOAT:
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    &val_real);
		/*
		 * Cannot dereference an uninitialized floating point value
		 * or a crash may result on some systems.
		 */
		if (stat != 1)
		   return(stat);
		*((float *)min) = val_real;
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    &val_real);
		if (stat != 1)
		   return(stat);
		*((float *)max) = val_real;
		break;
	      case OM_TYPE_DOUBLE:
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    &val_real);
		if (stat != 1)
		   return(stat);
		*((double *)min) = val_real;
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    &val_real);
		if (stat != 1)
		   return(stat);
		*((double *)max) = val_real;
		break;
	      default:
		stat = -1;
		break;
	}
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDset_node_data_minmax(OMobj_id field, int comp,
                            const char *min, const char *max, int dtype)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max,  sub_elem_id;
	int  stat, type, val_int;
	xp_long val_long;
	double val_real;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	stat = OMget_data_type(elem_id_min, &type);
	if (stat != 1)
		return(stat);
	if (type == OM_TYPE_UNSET || type != dtype) {
		type =dtype;
	}
	stat = OMset_data_type(elem_id_min, type);
	if (stat != 1)
		return(stat);
	stat = OMset_data_type(elem_id_max, type);
	if (stat != 1)
		return(stat);

	switch (type) {
	      case OM_TYPE_CHAR:
		val_int = *((char *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((char *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_BYTE:
		val_int = *((unsigned char *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((unsigned char *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_SHORT:
		val_int = *((short *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((short *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_INT:
		val_int = *((int *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((int *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_LONG:
		val_long = *((xp_long *)min);
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("min"),
					   val_long);
		val_long = *((xp_long *)max);
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("max"),
					   val_long);
		break;
	      case OM_TYPE_FLOAT:
		val_real = *((float *)min);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    val_real);
		val_real = *((float *)max);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    val_real);
		break;
	      case OM_TYPE_DOUBLE:
		val_real = *((double *)min);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    val_real);
		val_real = *((double *)max);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    val_real);
		break;
	      default:
		stat = -1;
		break;
	}
	return(stat);
}

int FLDcopy_node_minmax(OMobj_id in_field, OMobj_id out_field,
                        int in_comp, int out_comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	OMobj_id  elem_id1, elem_id_min1, elem_id_max1, sub_elem_id1;
	int stat;

	elem_id = OMfind_subobj (in_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, in_comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	elem_id1 = OMfind_subobj (out_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id1))
		return(0);
	if (OMget_array_val (elem_id1, out_comp, &sub_elem_id1, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id1))
		return(0);

	elem_id_min1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min1))
		return(0);
	elem_id_max1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max1))
		return(0);
	stat = OMset_obj_ref(elem_id_min1, elem_id_min, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	stat = OMset_obj_ref(elem_id_max1, elem_id_max, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDreset_node_minmax(OMobj_id in_field, int comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int stat;

	elem_id = OMfind_subobj (in_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min,
               "cache(min_array(magnitude(values), null_flag, null_value))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max,
		"cache(max_array(magnitude(values), null_flag, null_value))");
	if (stat != 1)
		return(stat);

	return(1);
}

int FLDreset_node_minmax_vec(OMobj_id in_field, int comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int stat;

	elem_id = OMfind_subobj (in_field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min,
		"cache(min_array(values, null_flag, null_value))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max,
		"cache(max_array(values, null_flag, null_value))");
	if (stat != 1)
		return(stat);

	return(1);
}

int FLDget_node_data_label(OMobj_id field, int comp, char *label, int size)
{
	OMobj_id    elem_id, sub_elem_id;

	/* Check for programmer error: */
	if (label == NULL || size == 0)
	   return 0;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &label, size));
}

int FLDset_node_data_label(OMobj_id field, int comp, const char *label)
{
	OMobj_id    elem_id, sub_elem_id;

	if (label == NULL) return 0;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMset_str_val (elem_id, label));
}

int FLDget_node_data_units(OMobj_id field, int comp, char *units, int size)
{
	OMobj_id    elem_id, sub_elem_id;

	/* Check for programmer error: */
	if (units == NULL || size == 0)
	   return 0;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &units, size));
}

int FLDset_node_data_units(OMobj_id field, int comp, const char *units)
{
	OMobj_id    elem_id, sub_elem_id;

	if (units == NULL) return 0;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMset_str_val (elem_id, units));
}

int FLDget_node_data_id(OMobj_id field, int comp, int *id)
{
	OMobj_id    elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
		( OMget_name_int_val (sub_elem_id, OMstr_to_name("id"), id) );
}

int FLDset_node_data_id(OMobj_id field, int comp, int id)
{
	OMobj_id    elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
		( OMset_name_int_val (sub_elem_id, OMstr_to_name("id"), id) );
}

/* 64-bit porting. Only Modified Internally */
int FLDget_node_null_data(OMobj_id field, int comp,
                          int *null_flag, char *null_value)
{
	OMobj_id  elem_id, sub_elem_id;
	int stat, type, null_int;
	xp_long null_long;
	double   null_real;

	null_int = 0;
	null_real = 0.0;
	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RD) <= 0)
		return(0);
	stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), null_flag);

	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				  OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);

	stat = OMget_data_type(elem_id, &type);
	if (stat != 1)
		return(stat);

	if (*null_flag) {
		switch (type) {
		      case OM_TYPE_CHAR:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((char *)null_value) = null_int;
			break;
		      case OM_TYPE_BYTE:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((unsigned char *)null_value) = null_int;
			break;
		      case OM_TYPE_SHORT:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((short *)null_value) = null_int;
			break;
		      case OM_TYPE_INT:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((int *)null_value) = null_int;
			break;
		      case OM_TYPE_LONG:
			stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_long);
			*((xp_long *)null_value) = null_long;
			break;
		      case OM_TYPE_FLOAT:
			stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
						    &null_real);
			if (stat == 1)
			    *((float *)null_value) = null_real;
			break;
		      case OM_TYPE_DOUBLE:
			stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
						    &null_real);
			if (stat == 1)
			   *((double *)null_value) = null_real;
			break;
		      default:
			stat = -1;
			break;
		}
	}
	else {
		stat = 1;
		switch (type) {
		      case OM_TYPE_CHAR:
			*((char *)null_value) = null_int;
			break;
		      case OM_TYPE_BYTE:
			*((unsigned char *)null_value) = null_int;
			break;
		      case OM_TYPE_SHORT:
			*((short *)null_value) = null_int;
			break;
		      case OM_TYPE_INT:
			*((int *)null_value) = null_int;
			break;
		      case OM_TYPE_LONG:
			*((xp_long *)null_value) = null_int;
			break;
		      case OM_TYPE_FLOAT:
			*((float *)null_value) = null_real;
			break;
		      case OM_TYPE_DOUBLE:
			*((double *)null_value) = null_real;
			break;
		      default:
			stat = -1;
			break;
		}
	}
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDset_node_null_data(OMobj_id field, int comp, char *null_value, int type)
{
	OMobj_id  elem_id, sub_elem_id;
	int stat, dtype, null_int;
	xp_long null_long;
	double   null_real;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), 1);
	if (stat != 1)
		return(stat);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("null_value"),
				    OM_OBJ_RW);


	stat = OMget_data_type(elem_id, &dtype);
	if (stat != 1)
		return(stat);
	if (dtype == OM_TYPE_UNSET) {
		stat = OMset_data_type(elem_id, type);
		if (stat != 1)
			return(stat);
	}


	switch (type) {
	      case OM_TYPE_CHAR:
		null_int = *((char *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_BYTE:
		null_int = *((unsigned char *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_SHORT:
		null_int = *((short *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_INT:
		null_int = *((int *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_LONG:
		null_long = *((xp_long *)(null_value));
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_long);
		break;
	      case OM_TYPE_FLOAT:
		null_real = *((float *)(null_value));
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
					     null_real);
		break;
	      case OM_TYPE_DOUBLE:
		null_real = *((double *)(null_value));
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
					    null_real);
		break;
	      default:
		stat = -1;
		break;
	}

	return(stat);
}

int FLDset_node_null_flag(OMobj_id field, int comp, int null_flag)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (field, OMstr_to_name("node_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
	      (OMset_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), null_flag));
}

/*-------------------------------------------------------------*/
/*                  CELL DATA                                  */
/*-------------------------------------------------------------*/

int FLDget_cell_data_ncomp(OMobj_id cell_set, int *ncell_comp)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("ncell_data"), ncell_comp) );
}

int FLDset_cell_data_ncomp (OMobj_id cell_set, int ncell_comp)
{
	return(OMset_name_int_val (cell_set, OMstr_to_name("ncell_data"), ncell_comp));
}

int FLDset_cell_data_comp (OMobj_id cell_set, int comp, int veclen,
                           const char *label, const char *units)
{
	OMobj_id  elem_id, sub_elem_id;
	int   stat;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val(elem_id, label);
	if (stat < 1)
		return (0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val(elem_id, units);
	if (stat < 1)
		return (0);
	return(1);
}


int FLDset_cell_data_veclen(OMobj_id cell_set, int comp, int veclen)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	return(OMset_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen));
}

int FLDget_cell_data_veclen(OMobj_id cell_set, int comp, int *veclen)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RD) <= 0)
		return(0);
	return
		( OMget_name_int_val (sub_elem_id, OMstr_to_name("veclen"), veclen) );
}

int FLDget_cell_data_type(OMobj_id cell_set, int comp, int *type)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_data_type(elem_id, type));
}

int FLDset_cell_data_type(OMobj_id cell_set, int comp, int type)
{
	OMobj_id  elem_id, sub_elem_id;
	int stat;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	/* set values */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set null_value */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("null_value"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set min max */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);

	/* set min_vec max_vec */
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_data_type(elem_id, type);
	if (stat != 1)
		return(stat);
	return(1);
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_cell_data(OMobj_id cell_set, int comp, int *type, char **cell_data,
                     xp_long *size, int mode)
{
	OMobj_id  elem_id, sub_elem_id;

	*cell_data = (char *)NULL;
	*size = 0;
	if (mode != OM_GET_ARRAY_WR)
		*type = OM_TYPE_UNSET;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return (OMget_array_sz(elem_id, type, (char **)cell_data, size, mode));
}

#ifdef WORDLENGTH_64
int FLDget_cell_data_n(OMobj_id cell_set, int comp, int *type, char **cell_data,
                     int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_cell_data(cell_set, comp, type, cell_data, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_typed_cell_data(OMobj_id cell_set, int comp, int *type,
                           char **cell_data, xp_long *size, int mode)
{
	OMobj_id  elem_id, sub_elem_id;

	*cell_data = (char *)NULL;
	*size = 0;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return (OMget_array_sz(elem_id, type, (char **)cell_data, size, mode));
}


#ifdef WORDLENGTH_64
int FLDget_typed_cell_data_n(OMobj_id cell_set, int comp, int *type,
                           char **cell_data, int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_typed_cell_data(cell_set, comp, type, cell_data, &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_cell_data(OMobj_id cell_set, int comp, char *cell_data, int type,
                     xp_long size, int mode)
{
	int   stat;
	OMobj_id   elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				    OM_OBJ_RW);
	stat = OMset_array(elem_id, type, (char *)cell_data, size, mode);
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDget_cell_data_minmax_vec(OMobj_id cell_set, int comp,
                                char **min, char **max, int *type, int mode)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int    stat1, stat2;
	xp_long size;

	if (mode != OM_GET_ARRAY_WR)
		*type = OM_TYPE_UNSET;
	*min = (char *)NULL;
	*max = (char *)NULL;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	size = 0;
	stat1 = OMget_array_sz(elem_id_min, type, (char **)min, &size, mode);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	size = 0;
	stat2 = OMget_array_sz(elem_id_max, type, (char **)max, &size, mode);
	if (stat1 != 1 || stat2 != 1)
		return(-1);
	return(1);
}

int FLDcopy_cell_minmax_vec(OMobj_id in_cell_set, OMobj_id out_cell_set,
                            int in_comp, int out_comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	OMobj_id  elem_id1, elem_id_min1, elem_id_max1, sub_elem_id1;
	int stat;

	elem_id = OMfind_subobj (in_cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, in_comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	elem_id1 = OMfind_subobj (out_cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id1))
		return(0);
	if (OMget_array_val (elem_id1, out_comp, &sub_elem_id1, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id1))
		return(0);

	elem_id_min1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min1))
		return(0);
	elem_id_max1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max1))
		return(0);
	stat = OMset_obj_ref(elem_id_min1, elem_id_min, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	stat = OMset_obj_ref(elem_id_max1, elem_id_max, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDreset_cell_minmax(OMobj_id cell_set, int comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int stat;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min,
               "cache(min_array(magnitude(values), null_flag, null_value))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max,
		"cache(max_array(magnitude(values), null_flag, null_value))");
	if (stat != 1)
		return(stat);

	return(1);
}

int FLDreset_cell_minmax_vec(OMobj_id cell_set, int comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int stat;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max_vec"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);
	stat = OMparse_obj_ref(elem_id_min,
		"cache(min_array(values, null_flag, null_value))");
	if (stat != 1)
		return(stat);
	stat = OMparse_obj_ref(elem_id_max,
		"cache(max_array(values, null_flag, null_value))");
	if (stat != 1)
		return(stat);

	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FLDget_cell_data_minmax(OMobj_id cell_set, int comp, char *min, char *max)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int  stat, type, val_int;
	xp_long val_long;
	double val_real;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	stat = OMget_data_type(elem_id_min, &type);
	if (stat != 1)
		return(stat);

	switch (type) {
	      case OM_TYPE_CHAR:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((char *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((char *)max) = val_int;
		break;
	      case OM_TYPE_BYTE:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((unsigned char *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((unsigned char *)max) = val_int;
		break;
	      case OM_TYPE_SHORT:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((short *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((short *)max) = val_int;
		break;
	      case OM_TYPE_INT:
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   &val_int);
		*((int *)min) = val_int;
		stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   &val_int);
		*((int *)max) = val_int;
		break;
	      case OM_TYPE_LONG:
		stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("min"),
					   &val_long);
		*((xp_long *)min) = val_long;
		stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("max"),
					   &val_long);
		*((xp_long *)max) = val_long;
		break;
	      case OM_TYPE_FLOAT:
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    &val_real);
		if (stat != 1)
		     return(stat);
		*((float *)min) = val_real;
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    &val_real);
		if (stat != 1)
		     return(stat);
		*((float *)max) = val_real;
		break;
	      case OM_TYPE_DOUBLE:
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    &val_real);
		if (stat != 1)
		     return(stat);
		*((double *)min) = val_real;
		stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    &val_real);
		if (stat != 1)
		     return(stat);
		*((double *)max) = val_real;
		break;
	      default:
		stat = -1;
		break;
	}
	return(stat);
}

int FLDget_cell_data_minmax_total(OMobj_id field, int comp,
                                  char *min, char *max)
{
	int stat, nsets, set, type;
	OMobj_id  tmp;
	double rmin, rmax, dmin, dmax;

	rmin=0.0; rmax=0.0; type = 0;

	stat = FLDget_ncell_sets(field, &nsets);
	if (stat != 1)
		return(stat);
	for (set=0; set<nsets; set++) {
		stat = FLDget_cell_set(field, set, &tmp);
		if (stat != 1)
			return(stat);
		stat = FLDget_cell_data_type(tmp, comp, &type);
		if (stat != 1)
			continue;
		stat = FLDget_cell_data_minmax(tmp, comp, min, max);
		if (stat != 1)
			return(stat);
		UTILtype_to_double(&dmax, max, type);
		UTILtype_to_double(&dmin, min, type);
		if (set == 0) {
			rmin = dmin;
			rmax = dmax;
		}
		else {
			if (dmin < rmin)
				rmin = dmin;
			if (dmax > rmax)
				rmax = dmax;
		}
	}
	UTILdouble_to_type(min, rmin, type);
	UTILdouble_to_type(max, rmax, type);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FLDset_cell_data_minmax(OMobj_id cell_set, int comp, const char *min, const char *max,
                            int dtype)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	int  stat, type, val_int;
	xp_long val_long;
	double val_real;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				  OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				      OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);

	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	stat = OMget_data_type(elem_id_min, &type);
	if (stat != 1)
		return(stat);

	if (type == OM_TYPE_UNSET || type != dtype) {
		type =dtype;
	}
	stat = OMset_data_type(elem_id_min, type);
	if (stat != 1)
		return(stat);
	stat = OMset_data_type(elem_id_max, type);
	if (stat != 1)
		return(stat);

	switch (type) {
	      case OM_TYPE_CHAR:
		val_int = *((char *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((char *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_BYTE:
		val_int = *((unsigned char *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((unsigned char *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_SHORT:
		val_int = *((short *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((short *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_INT:
		val_int = *((int *)min);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("min"),
					   val_int);
		val_int = *((int *)max);
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("max"),
					   val_int);
		break;
	      case OM_TYPE_LONG:
		val_long = *((xp_long *)min);
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("min"),
					   val_long);
		val_long = *((xp_long *)max);
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("max"),
					   val_long);
		break;
	      case OM_TYPE_FLOAT:
		val_real = *((float *)min);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    val_real);
		val_real = *((float *)max);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    val_real);
		break;
	      case OM_TYPE_DOUBLE:
		val_real = *((double *)min);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("min"),
					    val_real);
		val_real = *((double *)max);
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("max"),
					    val_real);
		break;
	      default:
		stat = -1;
		break;
	}
	return(stat);
}

int FLDcopy_cell_minmax(OMobj_id in_cell_set, OMobj_id out_cell_set,
                        int in_comp, int out_comp)
{
	OMobj_id  elem_id, elem_id_min, elem_id_max, sub_elem_id;
	OMobj_id  elem_id1, elem_id_min1, elem_id_max1, sub_elem_id1;
	int stat;

	elem_id = OMfind_subobj (in_cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, in_comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id_min = OMfind_subobj (sub_elem_id, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min))
		return(0);
	elem_id_max = OMfind_subobj (sub_elem_id, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max))
		return(0);

	elem_id1 = OMfind_subobj (out_cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id1))
		return(0);
	if (OMget_array_val (elem_id1, out_comp, &sub_elem_id1, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id1))
		return(0);

	elem_id_min1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("min"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_min1))
		return(0);
	elem_id_max1 = OMfind_subobj (sub_elem_id1, OMstr_to_name("max"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id_max1))
		return(0);
	stat = OMset_obj_ref(elem_id_min1, elem_id_min, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	stat = OMset_obj_ref(elem_id_max1, elem_id_max, OM_OBJ_REF_RDONLY);
	if (stat != 1)
		return(stat);
	return(1);
}

int FLDget_cell_data_label(OMobj_id cell_set, int comp, char *label, int size)
{
	OMobj_id    elem_id, sub_elem_id;

	/* Check for programmer error: */
	if (label == NULL || size == 0)
	   return 0;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &label, size));
}

int FLDset_cell_data_label(OMobj_id cell_set, int comp, const char *label)
{
	OMobj_id    elem_id, sub_elem_id;

	if (label == NULL) return 0;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("labels"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMset_str_val (elem_id, label));
}

int FLDget_cell_data_units(OMobj_id cell_set, int comp, char *units, int size)
{
	OMobj_id    elem_id, sub_elem_id;

	/* Check for programmer error: */
	if (units == NULL || size == 0)
	   return 0;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &units, size));
}

int FLDset_cell_data_units(OMobj_id cell_set, int comp, const char *units)
{
	OMobj_id    elem_id, sub_elem_id;

	if (units == NULL) return 0;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("units"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMset_str_val (elem_id, units));
}

int FLDget_cell_data_id(OMobj_id cell_set, int comp, int *id)
{
	OMobj_id    elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
		( OMget_name_int_val (sub_elem_id, OMstr_to_name("id"), id) );
}

int FLDset_cell_data_id(OMobj_id cell_set, int comp, int id)
{
	OMobj_id    elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
		( OMset_name_int_val (sub_elem_id, OMstr_to_name("id"), id) );
}

/* 64-bit porting. Only Modified Internally */
int FLDget_cell_null_data(OMobj_id cell_set, int comp,
                          int *null_flag, char *null_value)
{
	OMobj_id  elem_id, sub_elem_id;
	int stat, type, null_int;
	xp_long null_long;
	double   null_real;

	null_int = 0;
	null_real = 0.0;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RD) <= 0)
		return(0);
	stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), null_flag);

	if (stat != 1)
		return(stat);
	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("values"),
				  OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);

	stat = OMget_data_type(elem_id, &type);
	if (stat != 1)
		return(stat);

	if (*null_flag) {
		switch (type) {
		      case OM_TYPE_CHAR:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((char *)null_value) = null_int;
			break;
		      case OM_TYPE_BYTE:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((unsigned char *)null_value) = null_int;
			break;
		      case OM_TYPE_SHORT:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((short *)null_value) = null_int;
			break;
		      case OM_TYPE_INT:
			stat = OMget_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_int);
			*((int *)null_value) = null_int;
			break;
		      case OM_TYPE_LONG:
			stat = OMget_name_long_val (sub_elem_id, OMstr_to_name("null_value"),
						   &null_long);
			*((xp_long *)null_value) = null_long;
			break;
		      case OM_TYPE_FLOAT:
			stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
						    &null_real);
			if (stat == 1)
			   *((float *)null_value) = null_real;
			break;
		      case OM_TYPE_DOUBLE:
			stat = OMget_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
						    &null_real);
			if (stat == 1)
			   *((double *)null_value) = null_real;
			break;
		      default:
			stat = -1;
			break;
		}
	}
	else {
		stat = 1;
		switch (type) {
		      case OM_TYPE_CHAR:
			*((char *)null_value) = null_int;
			break;
		      case OM_TYPE_BYTE:
			*((unsigned char *)null_value) = null_int;
			break;
		      case OM_TYPE_SHORT:
			*((short *)null_value) = null_int;
			break;
		      case OM_TYPE_INT:
			*((int *)null_value) = null_int;
			break;
		      case OM_TYPE_LONG:
			*((xp_long *)null_value) = null_int;
			break;
		      case OM_TYPE_FLOAT:
			*((float *)null_value) = null_real;
			break;
		      case OM_TYPE_DOUBLE:
			*((double *)null_value) = null_real;
			break;
		      default:
			stat = -1;
			break;
		}
	}
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDset_cell_null_data(OMobj_id cell_set, int comp, char *null_value, int type)
{
	OMobj_id  elem_id, sub_elem_id;
	int stat, dtype, null_int;
	xp_long null_long;
	double   null_real;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), 1);
	if (stat != 1)
		return(stat);

	elem_id = OMfind_subobj (sub_elem_id, OMstr_to_name("null_value"),
				    OM_OBJ_RW);

	stat = OMget_data_type(elem_id, &dtype);
	if (stat != 1)
		return(stat);
	if (dtype == OM_TYPE_UNSET) {
		stat = OMset_data_type(elem_id, type);
		if (stat != 1)
			return(stat);
	}

	switch (type) {
	      case OM_TYPE_CHAR:
		null_int = *((char *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_BYTE:
		null_int = *((unsigned char *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_SHORT:
		null_int = *((short *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_INT:
		null_int = *((int *)(null_value));
		stat = OMset_name_int_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_int);
		break;
	      case OM_TYPE_LONG:
		null_long = *((xp_long *)(null_value));
		stat = OMset_name_long_val (sub_elem_id, OMstr_to_name("null_value"),
					   null_long);
		break;
	      case OM_TYPE_FLOAT:
		null_real = *((float *)(null_value));
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
					     null_real);
		break;
	      case OM_TYPE_DOUBLE:
		null_real = *((double *)(null_value));
		stat = OMset_name_real_val (sub_elem_id, OMstr_to_name("null_value"),
					    null_real);
		break;
	      default:
		stat = -1;
		break;
	}

	return(stat);
}

int FLDset_cell_null_flag(OMobj_id cell_set, int comp, int null_flag)
{
	OMobj_id  elem_id, sub_elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_data"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, comp, &sub_elem_id, OM_OBJ_RW) <= 0)
		return(0);
	if (OMis_null_obj(sub_elem_id))
		return(0);
	return
	      (OMset_name_int_val (sub_elem_id, OMstr_to_name("null_flag"), null_flag));
}


/*--------------------------------------------------*/
/*         CELL SET                                 */
/*--------------------------------------------------*/

int FLDget_cell_set(OMobj_id field, int i, OMobj_id *cell_set)
{
	OMobj_id  elem_id;
#if 0
	OMtype_id type;
	OMobj_name name;
#endif

	elem_id = OMfind_subobj (field, OMstr_to_name("cell_set"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	if (OMget_array_val (elem_id, i, cell_set, OM_OBJ_RW) <= 0)
		return(-1);
	if (OMget_obj_val_user(*cell_set, cell_set) != 1) return(-1);
#if 0
	/* This hack shouldn't be needed anymore; I changed the call above
	   to be OMget_obj_val_user, which never returns value objects (it
	   look upward until it finds a non-value object).  See bug 5752.
	   */
	/*
	 * This is a gross hack for bugs 3393 and 3394.  The problem is
	 * that the OMobj_val call is going through and getting the id
	 * of the merge object which is a value.  We need some mode
	 * on the get_obj_val call that will stop right before the
	 * merge for this case since the merge is not a rooted object.
	 *
	 * The attempt to add a ref to it screws up big time.
	 */
	OMget_obj_type(*cell_set,&type);
	OMget_obj_name(*cell_set,&name);
	if (!strcmp(type->name,"merge") && name == OM_NULL_NAME)
	    *cell_set = OMret_obj_parent(*cell_set);
#endif
        return(1);
}

int FLDget_cell_set_name(OMobj_id cell_set, char *name, int size)
{
	OMobj_id  elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_name"),
				    OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &name, size));
}

int FLDget_cell_set_user_name(OMobj_id cell_set, char *name, int size)
{
	OMobj_id  elem_id;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("name"),
				    OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	return(OMget_str_val (elem_id, &name, size));
}

int FLDset_cell_set_user_name(OMobj_id cell_set, const char *name)
{
	OMobj_id  elem_id;
	int  stat;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("name"),
				    OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_str_val (elem_id, name);
        return(stat);
}

int FLDset_cell_set(OMobj_id cell_set, const char *cell_set_name)
{
	OMobj_id  elem, corner_id;
	int       stat, order;


	elem = OMfind_subobj (OMtempl_obj, OMstr_to_name(cell_set_name),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem))
		return(0);

	/**************************************************/
	/* workaround of problem with breaking connection */
	/*        cell_corner_nnodes => cell_nnodes       */
	/* for second order cells                         */
	/**************************************************/
	corner_id = OMfind_subobj (cell_set, OMstr_to_name("cell_corner_nnodes"),
				   OM_OBJ_RW);
	if (!OMis_null_obj(corner_id)) {
		if (OMget_name_int_val(elem, OMstr_to_name("cell_order"), &order) != OM_STAT_SUCCESS)
			order = 0;
		if (order > 1) {
			stat = OMset_obj_ref(corner_id, OMnull_obj, 0);
			if (stat != OM_STAT_SUCCESS)
				return(0);
		}
	}
	stat = OMmerge_obj(elem, cell_set, 0);
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDadd_cell_set(OMobj_id field, const char *cell_set_name)
{
	OMobj_id  elem_id, elem, val_id, corner_id;
	int stat, order;
	xp_long size;


	elem_id = OMfind_subobj (field, OMstr_to_name("cell_set"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);

	if ((stat = OMget_array_size(elem_id, &size)) != 1) {
		size = 0;
	}
	if ((stat = OMset_array_size(elem_id,size+1)) != 1) {
		return(0);
	}

	elem = OMfind_subobj (OMtempl_obj, OMstr_to_name(cell_set_name),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem))
		return(0);

	if ((stat = OMget_array_val (elem_id, size, &val_id, OM_OBJ_RW)) != 1) {
		return(0);
	}
	/**************************************************/
	/* workaround of problem with breaking connection */
	/*        cell_corner_nnodes => cell_nnodes       */
	/* for second order cells                         */
	/**************************************************/
	corner_id = OMfind_subobj (val_id, OMstr_to_name("cell_corner_nnodes"),
				   OM_OBJ_RW);
	if (!OMis_null_obj(corner_id)) {
		if (OMget_name_int_val(elem, OMstr_to_name("cell_order"), &order) != OM_STAT_SUCCESS)
			order = 0;
		if (order > 1) {
			stat = OMset_obj_ref(corner_id, OMnull_obj, 0);
			if (stat != OM_STAT_SUCCESS)
				return(0);
		}
	}
	stat = OMmerge_obj(elem, val_id, 0);
	return(stat);
}

int FLDdel_cell_set(OMobj_id field, OMobj_id cell_set)
{
	OMobj_id  elem_id;
	int stat;

	elem_id = OMfind_subobj (field, OMstr_to_name("cell_set"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);

	stat = OMdel_array_val (elem_id, cell_set);
	if (stat != 1)
		return(stat);
	return(OMdestroy_obj(cell_set, 0));
}

int FLDget_ncell_sets(OMobj_id field, int *ncell_sets)
{
	return
		( OMget_name_int_val (field, OMstr_to_name("ncell_sets"), ncell_sets) );
}

int FLDset_ncell_sets (OMobj_id field, int ncell_sets)
{
	int   stat;

	stat = OMset_name_int_val (field, OMstr_to_name("ncell_sets"), ncell_sets);
	return(stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_ncells(OMobj_id cell_set, xp_long *ncells)
{
	return
		( OMget_name_long_val (cell_set, OMstr_to_name("ncells"), ncells) );
}

#ifdef WORDLENGTH_64
int FLDget_ncells_n(OMobj_id cell_set, int *ncells)
{
	int status;
	xp_long ncells_w;

	status = FLDget_ncells(cell_set, &ncells_w);

	*ncells = (int)ncells_w;

	return status;
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_ncells (OMobj_id cell_set, xp_long ncells)
{
	int   stat;

	stat = OMset_name_long_val (cell_set, OMstr_to_name("ncells"), ncells);
	return(stat);
}

int FLDget_cell_ndim(OMobj_id cell_set, int *ndim)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("cell_ndim"), ndim) );
}

int FLDget_cell_order(OMobj_id cell_set, int *order)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("cell_order"), order) );
}

int FLDget_cell_type(OMobj_id cell_set, int *cell_type)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("cell_type"), cell_type) );
}

int FLDset_cell_type (OMobj_id cell_set, int cell_type)
{
	int   stat;

	stat = OMset_name_int_val (cell_set, OMstr_to_name("cell_type"), cell_type);
	return(stat);
}

int FLDget_cell_set_nnodes(OMobj_id cell_set, int *cell_nnodes)
{
	return(OMget_name_int_val (cell_set, OMstr_to_name("cell_nnodes"), cell_nnodes));
}

int FLDget_cell_corner_nnodes(OMobj_id cell_set, int *cell_nnodes)
{
	if (OMget_name_int_val (cell_set, OMstr_to_name("cell_corner_nnodes"),
				cell_nnodes) < 1)
		return(OMget_name_int_val (cell_set, OMstr_to_name("cell_nnodes"), cell_nnodes));
	else
		return(1);
}

int FLDget_cell_set_nfaces(OMobj_id cell_set, int *cell_nfaces)
{
	return(OMget_name_int_val (cell_set, OMstr_to_name("cell_nfaces"), cell_nfaces));
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_cell_faces(OMobj_id cell_set, int *cell_nfaces, int **face_nnodes,
                      int **face_corner_nnodes, xp_long **node_list, int *size)
{
	OMobj_id   elem_id;
	xp_long nfaces_w, size_w;
	int  mode, type, stat;

	*node_list = (xp_long *)NULL;
	*face_nnodes = (int *)NULL;
	*face_corner_nnodes = (int *)NULL;
	*size = 0;
	*cell_nfaces = 0;
	type = OM_TYPE_INT;
	mode = OM_GET_ARRAY_RD;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_nnodes"),OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)face_nnodes, &nfaces_w, mode);
	if (stat < 1)
		return(stat);
	*cell_nfaces = nfaces_w;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_corner_nnodes"),OM_OBJ_RD);
	if (OMis_null_obj(elem_id))  {
		ARRincr_refcnt(*face_nnodes);
		*face_corner_nnodes = *face_nnodes;
	}
	else {
		stat = OMget_array_sz(elem_id, &type, (char **)face_corner_nnodes, &nfaces_w, mode);
		*cell_nfaces = nfaces_w;
		if (stat < 1)
			return(stat);
	}
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_connect"), OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	

	type = OM_TYPE_LONG;
	stat = OMget_array_sz(elem_id, &type, (char **)node_list, &size_w, mode);
	if (stat < 1)
		return(stat);
	*size = size_w;
	return(1);
}

#ifdef WORDLENGTH_64
int FLDget_cell_faces_n(OMobj_id cell_set, int *nfaces, int **face_nnodes,
                      int **face_corner_nnodes, int **node_list, int *size)
{
	OMobj_id   elem_id;
	int  mode, type, stat;

	*node_list = (int *)NULL;
	*face_nnodes = (int *)NULL;
	*face_corner_nnodes = (int *)NULL;
	*size = 0;
	*nfaces = 0;
	type = OM_TYPE_INT;
	mode = OM_GET_ARRAY_RD;

	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_nnodes"),OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz_n(elem_id, &type, (char **)face_nnodes, nfaces, mode);
	if (stat < 1)
		return(stat);
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_corner_nnodes"),OM_OBJ_RD);
	if (OMis_null_obj(elem_id))  {
		ARRincr_refcnt(*face_nnodes);
		*face_corner_nnodes = *face_nnodes;
	}
	else {
		stat = OMget_array_sz_n(elem_id, &type, (char **)face_corner_nnodes, nfaces, mode);
		if (stat < 1)
			return(stat);
	}
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("cell_face_connect"), OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz_n(elem_id, &type, (char **)node_list, size, mode);
	if (stat < 1)
		return(stat);
	return(1);
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_node_connect(OMobj_id cell_set, xp_long **node_conn_array,
                        xp_long *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;

	*node_conn_array = (xp_long *)NULL;
	*size = 0;
	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("node_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)node_conn_array, size, mode);
	return (stat);
}

/* Code duplication is done since 'node_conn_array' is very large */
#ifdef WORDLENGTH_64
/* 
 * FLDget_node_connect_n will not work in read-write or write-only mode,
 * thus we must offer wide FLDget_node_connect as an alternative that
 * can be used in narrow mode.
 */
int FLDget_node_connect_w(OMobj_id cell_set, xp_long **conn_array,
                        xp_long *size, int mode)
{
	return(FLDget_node_connect (cell_set, conn_array, size, mode));
}

int FLDget_node_connect_n(OMobj_id cell_set, int **node_conn_array,
                        int *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;

	if (mode == OM_GET_ARRAY_RW || mode == OM_GET_ARRAY_WR)
	{
		ERRerror("FLDget_node_connect", ERR_ORIG, 0,
			"The \"narrow\" version of FLDget_node_connect can not be used in write mode because the corresponding V array is now type long. Either convert your code to use the \"wide\" API or use FLDget_node_connect_w." );
		return(0);
	}
	*node_conn_array = (int *)NULL;
	*size = 0;
	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("node_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz_n(elem_id, &type, (char **)node_conn_array, size, mode);
	return (stat);
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDset_node_connect(OMobj_id cell_set, xp_long *node_conn_array,
                        xp_long size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("node_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)node_conn_array, size, mode);
	return(stat);
}

/* Code duplication is done since 'node_conn_array' is very large */
#ifdef WORDLENGTH_64
/* Symmetry with FLDget_node_connect_w */
int FLDset_node_connect_w(OMobj_id cell_set, xp_long *conn_array,
                        xp_long size, int mode)
{
	return(FLDset_node_connect(cell_set, conn_array, size, mode));
}

int FLDset_node_connect_n(OMobj_id cell_set, int *node_conn_array,
                        int size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("node_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)node_conn_array, size, mode);
	return(stat);
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_npolys(OMobj_id cell_set, xp_long *npolys)
{
	return
		( OMget_name_long_val (cell_set, OMstr_to_name("npolys"), npolys) );
}

#ifdef WORDLENGTH_64
int FLDget_npolys_n(OMobj_id cell_set, int *npolys)
{

	int status;
	xp_long npolys_w;

	status = FLDget_npolys(cell_set, &npolys_w);

	*npolys = (int)npolys_w;

	return status;
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_npolys (OMobj_id cell_set, xp_long npolys)
{
	int   stat;

	stat = OMset_name_long_val (cell_set, OMstr_to_name("npolys"), npolys);
	return(stat);
}

int FLDget_poly_flag(OMobj_id cell_set, int *poly_flag)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("poly_flag"), poly_flag) );
}

int FLDset_poly_flag (OMobj_id cell_set, int poly_flag)
{
	int   stat;

	stat = OMset_name_int_val (cell_set, OMstr_to_name("poly_flag"), poly_flag);
	return(stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_poly_connect(OMobj_id cell_set, xp_long **poly_conn_array,
                        xp_long *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;

	*poly_conn_array = (xp_long *)NULL;
	*size = 0;
	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)poly_conn_array, size, mode);
	return (stat);
}

#ifdef WORDLENGTH_64
/* FLDget_poly_connect_n will not work in read-write or write-only mode */
int FLDget_poly_connect_w(OMobj_id cell_set, xp_long **conn_array,
                        xp_long *size, int mode)
{
	return(FLDget_poly_connect(cell_set, conn_array, size, mode));
}

int FLDget_poly_connect_n(OMobj_id cell_set, int **poly_conn_array,
                        int *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;

	if (mode == OM_GET_ARRAY_RW || mode == OM_GET_ARRAY_WR)
	{
		ERRerror("FLDget_poly_connect", ERR_ORIG, 0,
			"The \"narrow\" version of FLDget_poly_connect can not be used in write mode because the corresponding V array is now type long. Either convert your code to use the \"wide\" API or use FLDget_poly_connect_w." );
		return(0);
	}

	*poly_conn_array = (int *)NULL;
	*size = 0;
	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz_n(elem_id, &type, (char **)poly_conn_array, size, mode);
	return (stat);
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDset_poly_connect(OMobj_id cell_set, xp_long *poly_conn_array,
                        xp_long size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_LONG;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)poly_conn_array, size, mode);
	return(stat);
}

#ifdef WORDLENGTH_64
/* Symmetry with FLDget_poly_connect_w */
int FLDset_poly_connect_w(OMobj_id cell_set, xp_long *conn_array,
                        xp_long size, int mode)
{
	return(FLDset_poly_connect(cell_set, conn_array, size, mode));
}

int FLDset_poly_connect_n(OMobj_id cell_set, int *poly_conn_array,
                        int size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_connect_list"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)poly_conn_array, size, mode);
	return(stat);
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDget_poly_nnodes(OMobj_id cell_set, int **poly_conn_array,
                       xp_long *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;

	*poly_conn_array = (int *)NULL;
	*size = 0;
	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_nnodes"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)poly_conn_array, size, mode);
	return (stat);
}

#ifdef WORDLENGTH_64
int FLDget_poly_nnodes_n(OMobj_id cell_set, int **poly_conn_array,
                       int *size, int mode)
{
	int status;
	xp_long size_w;

	status = FLDget_poly_nnodes(cell_set, poly_conn_array,  &size_w, mode);

	*size = (int)size_w;

	return status;
}
#endif

/* 64-bit porting. Directly Modified */
int FLDset_poly_nnodes(OMobj_id cell_set, int *poly_conn_array,
                       xp_long size, int mode)
{
	int   stat, type;
	OMobj_id   elem_id;

	type = OM_TYPE_INT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("poly_nnodes"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMset_array(elem_id, type, (char *)poly_conn_array, size, mode);
	return(stat);
}

int FLDget_cell_node_uvw(OMobj_id cell_set, float **uvw)
{
	OMobj_id   elem_id;
	int        type, stat;
	xp_long size;

	*uvw = (float *)NULL;
	size = 0;
	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("node_uvw"),
				  OM_OBJ_RD);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)uvw, &size, OM_GET_ARRAY_RD);
	return (stat);
}

int FLDget_cell_nprops(OMobj_id cell_set, int *nprops)
{
	return
		( OMget_name_int_val (cell_set, OMstr_to_name("nprops"), nprops) );
}

int FLDset_cell_nprops(OMobj_id cell_set, int nprops)
{
	return
		( OMset_name_int_val (cell_set, OMstr_to_name("nprops"), nprops) );
}

/* 64-bit porting. Only Modified Internally */
int FLDget_cell_props(OMobj_id cell_set, float **props, int *size, int mode)
{
	OMobj_id   elem_id;
	int        type, stat;
	xp_long    size_w;

	*props = (float *)NULL;
	size_w = 0;
	type = OM_TYPE_FLOAT;
	elem_id = OMfind_subobj (cell_set, OMstr_to_name("props"),
				   OM_OBJ_RW);
	if (OMis_null_obj(elem_id))
		return(0);
	stat = OMget_array_sz(elem_id, &type, (char **)props, &size_w, mode);
	*size = (int)size_w;
	return (stat);
}

/* 64-bit porting. Modified with Parallel APIs */
int FLDcheck_cell_connect(OMobj_id cell_set, xp_long nnodes,
                          xp_long *bad_index, xp_long *bad_value)
{
	int stat, poly_flag;
	xp_long *node_conn;
	xp_long size;

	if (FLDget_poly_flag(cell_set, &poly_flag) != 1) {
		poly_flag = 0;
	}

	if (poly_flag == 0) {
		stat = FLDget_node_connect(cell_set, &node_conn, &size, OM_GET_ARRAY_RD);
		if (stat != 1)
			return(0);
	}
	else {
		stat = FLDget_poly_connect(cell_set, &node_conn, &size, OM_GET_ARRAY_RD);
		if (stat != 1)
			return(0);
	}
	stat = UTILcheck_list(node_conn, size, 0, nnodes-1, bad_index, bad_value);
	ARRfree(node_conn);
	if (stat != 1)
		return(-1);
	return(1);
}

#ifdef WORDLENGTH_64
int FLDcheck_cell_connect_n(OMobj_id cell_set, int nnodes,
                          int *bad_index, int *bad_value)
{
	int status;
	xp_long bad_index_w;
	xp_long bad_value_w;

	status = FLDcheck_cell_connect(cell_set, (xp_long)nnodes, &bad_index_w, &bad_value_w);

	*bad_index = (int)bad_index_w;
	*bad_value = (int)bad_value_w;

	return status;
}
#endif

/* 64-bit porting. Modified with Parallel APIs */
int FLDcheck_mesh_connect(OMobj_id field, xp_long *bad_index, xp_long *bad_value)
{
	int stat, set, nsets, ndim;
	xp_long nnodes;
	OMobj_id  cell_set;

	*bad_index = 0;
	*bad_value = 0;
	stat = FLDget_ndim(field, &ndim);
	if (stat >= 0) /** dont check Struct meshes **/
		return(1);
	stat = FLDget_ncell_sets(field, &nsets);
	if (stat != 1)
		return(0);
	stat = FLDget_nnodes(field, &nnodes);
	if (stat != 1)
		return(0);
	for (set=0; set<nsets; set++) {
		stat = FLDget_cell_set(field, set, &cell_set);
		if (stat != 1)
			return(0);
		stat = FLDcheck_cell_connect(cell_set, nnodes, bad_index, bad_value);
		if (stat < 0)
			return(stat);
	}
	return(1);
}


#ifdef WORDLENGTH_64
int FLDcheck_mesh_connect_n(OMobj_id field, int *bad_index, int *bad_value)
{
	int status;
	xp_long bad_index_w;
	xp_long bad_value_w;

	status = FLDcheck_mesh_connect(field, &bad_index_w, &bad_value_w);

	*bad_index = (int)bad_index_w;
	*bad_value = (int)bad_value_w;

	return status;
}
#endif

/********************************/
/*  Time dependent data API     */
/********************************/

/*----------------------------------------------*/
/*  returns number of time steps in the field   */
/*----------------------------------------------*/
int FLDget_time_nsteps(OMobj_id field, int *nsteps)
{
	return (OMget_name_int_val (field, OMstr_to_name("nsteps"), nsteps) );
}

/*----------------------------------------------*/
/*  sets number of time steps in the field      */
/*----------------------------------------------*/
int FLDset_time_nsteps (OMobj_id field, int nsteps)
{
	return (OMset_name_int_val (field, OMstr_to_name("nsteps"), nsteps));
}

/*----------------------------------------------*/
/*  returns array of time values in the field   */
/*----------------------------------------------*/
/* 64-bit porting. Directly Modified */
int FLDget_time_values(OMobj_id field, double **time, int *size, int mode)
{
	OMobj_id  id;
	int    type, stat;
	xp_long size_w;

	type = OM_TYPE_DOUBLE;
	*time = (double *)NULL;
	*size = 0;
	id = OMfind_subobj (field, OMstr_to_name("time"), OM_OBJ_RW);
	if (OMis_null_obj(id))
		return(0);
	stat = OMget_array_sz(id, &type, (char **)time, &size_w, mode);
	if (stat == 1) *size = size_w;
	return(stat);
}

/*----------------------------------------------*/
/*  sets array of time values in the field      */
/*----------------------------------------------*/
int FLDset_time_values(OMobj_id field, double *time, int size, int mode)
{
	int   stat, type;
	OMobj_id   id;

	type = OM_TYPE_DOUBLE;
	id = OMfind_subobj (field, OMstr_to_name("time"), OM_OBJ_RW);
	if (OMis_null_obj(id))
		return(0);
	stat = OMset_array(id, type, (char *)time, size, mode);
	return(stat);
}

/*----------------------------------------------*/
/*  returns id of one coordinate time step      */
/*----------------------------------------------*/
int FLDget_time_coordinates(OMobj_id field, int step, OMobj_id *coordinates)
{
	OMobj_id   id;

	id = OMfind_subobj (field, OMstr_to_name("time_coordinates"), OM_OBJ_RW);
	if (OMis_null_obj(id))
		return(0);
	if (OMget_array_val (id, step, coordinates, OM_OBJ_RW) <= 0)
		return(0);
	return(1);
}

/*----------------------------------------------*/
/*  returns id of one node_data time step       */
/*----------------------------------------------*/
int FLDget_time_node_data(OMobj_id field, int step, OMobj_id *node_data)
{
	OMobj_id   id;

	id = OMfind_subobj (field, OMstr_to_name("time_node_data"), OM_OBJ_RW);
	if (OMis_null_obj(id))
		return(0);
	if (OMget_array_val (id, step, node_data, OM_OBJ_RW) <= 0)
		return(0);
	return(1);
}


/*----------------------------------------------*/
/*  returns id of one cell_data time step       */
/*----------------------------------------------*/
int FLDget_time_cell_data(OMobj_id cell_set, int step, OMobj_id *cell_data)
{
	OMobj_id   id;

	id = OMfind_subobj (cell_set, OMstr_to_name("time_cell_data"), OM_OBJ_RW);
	if (OMis_null_obj(id))
		return(0);
	if (OMget_array_val (id, step, cell_data, OM_OBJ_RW) <= 0)
		return(0);
	return(1);
}
