/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/fld/fld_util.c#1 $
*/

#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/dll_in.h>
#include <avs/port.h>
#include <avs/err.h>
#include <avs/fld_info.h>
#include <avs/fld.h>
#include <avs/arr.h>

#define ERR_RETURN(A) { ERRerror("FLDutil", 0, ERR_ORIG, A); return(0); }

#define ERR_OBJ_RETURN(A) { ERRerror("FLDutil",0, ERR_ORIG, A); \
                            return(OMnull_obj); }

static const char *FIELD_FUNC_NAME[NUM_FIELD_FUNC] = {
	"interp_init_func",
	"find_cell_func",
	"interp_cell_func",
	"interp_data_func",
	"cell_extent_func",
	"coord_list_func",
	"data_list_func",
	"isosurf_func",
	"isoline_func",
	"cut_func",
	"eval_coord_func",
	"glyph_func",
	"thresh_null_func",
	"rslice_func",
	"grad_func",
	"curl_func",
	"div_func",
	"deriv3d_func",
	"deriv2d_func",
	"contour_func",
	"cell_to_node_func"
	};

static const char *SET_FUNC_NAME[NUM_SET_FUNC] = {
	"shape_func",
	"shape_deriv_func",
	"rev_par_func",
	"slice_func",
	"cut_func",
	"isoline_func",
	"vol_integr_func",
	"contour_func",
	"tesselate_func",
	"slice_func_tesselate",
	"cut_func_tesselate",
	"isoline_func_tesselate",
	"contour_func_tesselate"
	};

/* 64-bit porting. Only Modified Internally */
int get_coord_func(char *field_table, float **coord)
{
	OMobj_id   id;
	Field_Table  *ct;
	xp_long size;
	int stat;

	ct = (Field_Table *)field_table;
	memcpy((char *)&id, ct->field_id, sizeof(OMobj_id));
	stat = FLDget_coord(id, coord, &size, OM_GET_ARRAY_RD);
	if (stat != 1)
		*coord = NULL;
	return(stat);
}


/* 64-bit porting. Only Modified Internally */
int get_points_func(char *field_table, float **points)
{
	OMobj_id   id;
	Field_Table  *ct;
	xp_long size;
	int stat;

	ct = (Field_Table *)field_table;
	memcpy((char *)&id, ct->field_id, sizeof(OMobj_id));
	stat = FLDget_points(id, points, &size, OM_GET_ARRAY_RD);
	if (stat != 1)
		*points = NULL;
	return(stat);
}

/* 64-bit porting. Directly Modified */
int get_connect_func(char *field_table, int set, xp_long **node_list)
{
	OMobj_id   id;
	OMobj_id   cell_set;
	Field_Table  *ct;
	xp_long size;

	ct = (Field_Table *)field_table;
	memcpy((char *)&id, ct->field_id, sizeof(OMobj_id));
	if (FLDget_cell_set(id, set, &cell_set) < 1) {
		*node_list = NULL;
		return(0);
	}
	if (FLDget_node_connect(cell_set, node_list, &size, OM_GET_ARRAY_RD) < 1) {
		*node_list = NULL;
		return(0);
	}
	return(1);
}

/* 64-bit porting. Directly Modified */
int get_sub_coord_func(char *field_table, xp_long *min_rng, xp_long *max_rng,
                       float *array)
{
	OMobj_id   id;
	Field_Table  *ct;
	int ndim;
	xp_long dims[16], rmin[16], rmax[16];
	int stat;

	ct = (Field_Table *)field_table;
	memcpy((char *)&id, ct->field_id, sizeof(OMobj_id));
	dims[0] = ct->nspace;
	rmin[0] = 0;
	rmax[0] = ct->nspace;
	if (ct->ndim && ct->dims) {
		ndim = ct->ndim+1;
		memcpy(dims+1, ct->dims, ct->ndim*sizeof(xp_long));
		memcpy(rmin+1, min_rng, ct->ndim*sizeof(xp_long));
		memcpy(rmax+1, max_rng, ct->ndim*sizeof(xp_long));
	}
	else {
		ndim = 2;
		dims[1] = ct->nnodes;
		rmin[1] = min_rng[0];
		rmax[1] = max_rng[0];
	}
	stat = FLDget_sub_coord(id, ndim, dims, rmin, rmax, array);

	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FLDget_mesh_info(OMobj_id fld, char **fld_tbl)
{
	int   cs, nsets, size;
	OMobj_id   cell_set;
	Field_Table *h;

	h = (Field_Table *)malloc(sizeof(Field_Table));

	h->field_id = malloc(sizeof(OMobj_id));
	memcpy(h->field_id, (char *)&fld, sizeof(OMobj_id));
	if (FLDget_ndim(fld, &(h->ndim)) < 1) {
		h->ndim = 0;
	}
	if (FLDget_dims(fld, &(h->dims), &size) <1) {
		h->dims = (xp_long *)0;
	}
	if (FLDget_nspace(fld, &(h->nspace)) < 1) {
		h->nspace = 0;
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_nnodes(fld, &(h->nnodes)) < 1) {
		h->nnodes = 0;
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_coord_extent(fld, h->min_ext, h->max_ext) != 1) {
		ERR_RETURN("cannot get extents");
	}
	h->points = (float *)0;
	h->coord = (float *)0;
	h->points_func = get_points_func;
	h->coord_func = get_coord_func;
	h->sub_coord_func = get_sub_coord_func;

	h->field_func = (PFI *)malloc(MAX_FIELD_FUNC*sizeof(PFI));
	if (FLDget_methods(fld, NUM_FIELD_FUNC, FIELD_FUNC_NAME,
			  (PFI *)(h->field_func)) < 1) {
		ERR_RETURN("cannot get field methods");
	}
	if (FLDget_ncell_sets(fld, &(h->nsets)) < 1) {
		h->nsets  = 0;
	}
	nsets = h->nsets;
	h->connect_func = get_connect_func;
	if (h->nsets) {
		h->set_beg = (xp_long *)malloc((nsets)*sizeof(xp_long));
		h->ncells = (xp_long *)malloc((nsets)*sizeof(xp_long));
		h->cell_nnodes = (int *)malloc((nsets)*sizeof(int));
		h->cell_ndim = (int *)malloc((nsets)*sizeof(int));
		h->cell_order = (int *)malloc((nsets)*sizeof(int));
		h->node_list = (xp_long **)malloc((nsets)*sizeof(xp_long *));
		h->node_uvw = (float **)malloc((nsets)*sizeof(float *));
		h->set_func = (PFI *)malloc((nsets)*MAX_SET_FUNC*sizeof(PFI));
	}
	for (h->tot_cells = 0, cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(fld, cs, &cell_set) < 1) {
			ERR_RETURN("cannot get cell set");
		}

		if (FLDget_ncells(cell_set, &(h->ncells[cs])) < 1) {
			ERR_RETURN("cannot get ncells");
		}
		if (FLDget_cell_set_nnodes(cell_set,  &(h->cell_nnodes[cs]))<1) {
			ERR_RETURN("cannot get cell nnodes");
		}
		if (FLDget_cell_ndim(cell_set,  &(h->cell_ndim[cs]))<1) {
			ERR_RETURN("cannot get cell ndim");
		}
		if (FLDget_cell_order(cell_set,  &(h->cell_order[cs]))<1) {
			ERR_RETURN("cannot get cell order");
		}
		h->tot_cells += h->ncells[cs];
		if (cs == 0)
			h->set_beg[0] = 0;
		else
			h->set_beg[cs] = h->set_beg[cs-1] + h->ncells[cs-1];
		h->node_list[cs] = (xp_long *)0;
		FLDget_cell_node_uvw(cell_set,  &(h->node_uvw[cs]));

		/*-----------------------------------*/
		/*    GET METHODS for cell set       */
		/*-----------------------------------*/
		if (FLDget_methods(cell_set, NUM_SET_FUNC, SET_FUNC_NAME,
				   (PFI *)&(h->set_func[cs*MAX_SET_FUNC])) < 1) {
			ERR_RETURN("cannot get set methods");
		}
	}
	*fld_tbl = (char *)h;
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FLDfree_mesh_info(char *fld_tbl)
{
	Field_Table *h;
	int  cs;

	if (!fld_tbl)
		return(0);
	h = (Field_Table *)fld_tbl;
	if (h->field_id)
		free(h->field_id);
	if (h->nsets) {
		if (h->set_beg)
			free(h->set_beg);
		if (h->ncells)
			free(h->ncells);
		if (h->cell_nnodes)
			free(h->cell_nnodes);
		if (h->cell_ndim)
			free(h->cell_ndim);
		if (h->cell_order)
			free(h->cell_order);
	}
	for (cs=0; cs<h->nsets; cs++) {
		if (h->node_list[cs])
			ARRfree(h->node_list[cs]);
		if (h->node_uvw[cs])
			ARRfree(h->node_uvw[cs]);
	}
	if (h->nsets) {
		if (h->node_list)
			free(h->node_list);
		if (h->node_uvw)
			free(h->node_uvw);
		if (h->set_func)
			free(h->set_func);
	}
	if (h->field_func)
		free(h->field_func);
	if (h->coord)
		ARRfree(h->coord);
	if (h->points)
		ARRfree(h->points);
	if (h->dims)
		ARRfree(h->dims);
	free(h);
	return(1);
}

OMobj_id
FLDadd_block_table(OMobj_id fld)
{
       OMobj_id bt, prt1, prt2, crd, cst;
       if (FLDget_subelem(fld, "coordinates", &crd) < 1) {
	       ERR_OBJ_RETURN("cannot get coordinates");
       }
       if (FLDget_subelem(fld, "cell_set", &cst) < 1) {
	       ERR_OBJ_RETURN("cannot get cell set");
       }
       if (OMget_obj_parent(crd, &prt1) != 1) {
	       ERR_OBJ_RETURN("cannot get coordinates parent");
       }
       if (OMget_obj_parent(cst, &prt2) != 1) {
	       ERR_OBJ_RETURN("cannot get cell set parent");
       }
       if (FLDequal(prt1, prt2)) {
	       bt = FLDadd_elem(prt1, "Block_Table", "block_table");
       }
       else if (FLDget_subelem(prt2, "coordinates", &crd) < 1)
	       /*  we are in a merge and we want obj that has no coordinates */
	       bt = FLDadd_elem(prt2, "Block_Table", "block_table");
       else if (FLDget_subelem(prt1, "cell_set", &cst) < 1)
	       /*  we are in a merge and we want obj that has no cell_set */
	       bt = FLDadd_elem(prt1, "Block_Table", "block_table");
       else {
	       char buf1[128];
	       fprintf(stderr,
		       "warning: adding block table to obj: %s\n",
		       OMret_obj_path(fld,buf1,sizeof(buf1)));
	       /* we are screwed up here */
	       bt = FLDadd_elem(fld, "Block_Table", "block_table");
       }
       return(bt);
}

/* 64-bit porting. Only Modified Internally */
int FLDinterp_init(OMobj_id fld, char *field_table, char **block_table)
{
	OMobj_id  bt;
	Block_Table *h;
	Field_Table *ct;
	int stat, i, nspace, grid_size, ms_seq, bt_seq;
	xp_long size;
	float *min_ext, *max_ext;
	PFI interp_init_func;

	ct = (Field_Table *)field_table;

	FIget_field_func(ct, FIELD_FUNC_interp_init, interp_init_func);

	if (interp_init_func == NULL) {
		*block_table = NULL;
		return(1);
	}
	if (FLDget_nspace(fld, &nspace) < 1) {
		ERR_RETURN("cannot get nspace");
	}

	if (FLDget_subelem(fld, "block_table", &bt) < 1) {
		/* figure out where to put block table            */
		/* if coordinates and cell set have common parent */
		/* put under this parent                          */
		bt = FLDadd_block_table(fld);

		if (OMis_null_obj(bt)) return(0);

		grid_size = 0;
	}
	else {
		bt_seq = FLDget_subelem_seq(bt, "Block_Table");
		ms_seq = FLDget_subelem_seq(fld, "Mesh");

		if (FLDget_int(bt, "grid_size", &grid_size) <= 0) {
			ERR_RETURN("cannot find block table");
		}

		if (grid_size && ms_seq > bt_seq) {
			if (FLDdestroy(bt) != 1) {
				ERR_RETURN("cannot destroy block table");
			}
			bt = FLDadd_block_table(fld);
			if (OMis_null_obj(bt)) return(0);
			grid_size = 0;
		}
	}
	if (grid_size) {
		h = (Block_Table *)malloc(sizeof(Block_Table));
		h->grid_size = grid_size;
		if (FLDget_long(bt, "nblocks", &(h->nblocks)) <= 0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_long(bt, "list_size", &(h->list_size)) <= 0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_int(bt, "list_incr", &(h->incr)) <= 0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_long(bt, "loc", &(h->loc)) <= 0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_array_long (bt, "blocks", &(h->blocks), &size, OM_GET_ARRAY_RD) <=0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_array_long (bt, "cell_list", &(h->cell_list), &size, OM_GET_ARRAY_RD) <=0) {
			ERR_RETURN("cannot find block table");
		}
		if (FLDget_array_long (bt, "list_link", &(h->link), &size, OM_GET_ARRAY_RD) <=0) {
			ERR_RETURN("cannot find block table");
		}
		h->nspace = nspace;
		FIget_extent(ct, min_ext, max_ext);
		for (i=0; i<3; i++) {
			h->min_xyz[i] = 0.0;
			h->max_xyz[i] = 0.0;
		}
		for (i=0; i<nspace; i++) {
			h->min_xyz[i] = min_ext[i];
			h->max_xyz[i] = max_ext[i];
		}

		h->exist = 1;
		*block_table = (char *)h;
	}
	else {
		stat = (*interp_init_func)(field_table, block_table);
		if (!stat) {
			ERR_RETURN("cannot add block table");
		}
		if (*block_table == NULL)
			return(1);
		h = (Block_Table *)(*block_table);

		if (FLDset_int(bt, "grid_size", h->grid_size) <= 0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_long(bt, "nblocks", h->nblocks) <= 0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_long(bt, "list_size", h->list_size) <= 0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_int(bt, "list_incr", h->incr) <= 0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_long(bt, "loc", h->loc) <= 0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_array_long (bt, "blocks", h->nblocks, h->blocks, OM_SET_ARRAY_FREE) <=0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_array_long (bt, "cell_list", h->list_size, h->cell_list, OM_SET_ARRAY_FREE) <=0) {
			ERR_RETURN("cannot set block table");
		}
		if (FLDset_array_long (bt, "list_link", h->list_size, h->link, OM_SET_ARRAY_FREE) <=0) {
			ERR_RETURN("cannot set block table");
		}
		h->exist = 0;
	}
	return(1);
}

int FLDinterp_end (char *block_tbl)
{
	Block_Table *h;
	if (!block_tbl)
		return(0);
	h = (Block_Table *)block_tbl;
	if (h->exist) {
		if (h->blocks)
			ARRfree(h->blocks);
		if (h->cell_list)
			ARRfree(h->cell_list);
		if (h->link)
			ARRfree(h->link);
	}
	free(h);
	return(1);
}
