/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gd/fld_util.c#1 $
*/

#include <stdio.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/math.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>
#include <avs/data_utils.h>
#include "gd_int.h"

/* Gets nspace and the coordinate extents of the
   field and puts the results into the object.
   Called by most of the render methods and by
   the auto-normalize code.
*/
/* 64-bit porting. Only Modified Internally */
int GDfld_get_coord_extents(GDobject *obj)
{
   OMobj_id tmp_id, vol_id, min_id, max_id;
   xp_long nnodes = 0;
   xp_long *dims, size;
   int dims_size, nspace = 0, type = OM_TYPE_FLOAT;
   float vert_min_ext[3], vert_max_ext[3];
   float xextent, yextent, *tmp_array;
   int width, height, border_width, cont_mode;

   /* We need to see if this a field that we have to do
      some extra work for - like a Tiled Volume.
   */
   GDobject_get_render_method(obj);
   GDobject_set_render_flags(obj);

   /* The field is NOT a tiled volume. */
   if (obj->render_flags != GD_REN_FLAG_TILE_VOL) {
      if (FLDget_nspace(obj->field_id, &nspace) != 1)
         return(0);
      /* Sanity check nspace. If it happens that someone
         built a bad field, the get coord extent routine 
         will just memcpy the data blindly.
      */
      if (nspace < 1 || nspace > 3)
         return(0);

      obj->nspace = nspace;

      /* If we don't already have the coordinate extents, get them.
         This is an expensive operation so we want to minimize it.
      */
      if (!obj->vert_extents) {
	  /* Fail if nnodes is 0 */
          if (FLDget_nnodes(obj->field_id, &nnodes) != 1)
	     return(0);
          if (nnodes == 0)
	     return(0);

          vert_min_ext[0] = 0.0; vert_min_ext[1] = 0.0; vert_min_ext[2] = 0.0;
          vert_max_ext[0] = 0.0; vert_max_ext[1] = 0.0; vert_max_ext[2] = 0.0;
	  /* Try first using the FLD API call. */
          if (FLDget_coord_extent(obj->field_id, vert_min_ext, 
		   vert_max_ext) == 1) {
              /* save the vert min/max of the field in the object struct */
              VEC_COPY(obj->vert_min, vert_min_ext);
              VEC_COPY(obj->vert_max, vert_max_ext);
              obj->vert_extents = 1;
          }
	  /* Now try accessing arrays that are direct sub-objects of the
	     field_id. This is true of text.
          */
	  else {
             /* Find the ids of the two arrays we want to get. */
             min_id = OMfind_subobj(obj->field_id, OMstr_to_name("min_vec"), OM_OBJ_RW);
             if (OMis_null_obj(min_id))
                return(0);
             max_id = OMfind_subobj(obj->field_id, OMstr_to_name("max_vec"), OM_OBJ_RW);
             if (OMis_null_obj(max_id))
                return(0);

	     size = obj->nspace;
             tmp_array = (float *)NULL;
             if ((OMget_array_sz(min_id, &type, (char **)&tmp_array, &size,
		OM_GET_ARRAY_RD_COPY) != OM_STAT_SUCCESS))
                return(0);
	     memcpy(obj->vert_min, tmp_array, obj->nspace*sizeof(float));
	     ARRfree(tmp_array);

	     size = obj->nspace;
             tmp_array = (float *)NULL;
             if ((OMget_array_sz(max_id, &type, (char **)&tmp_array, &size,
		OM_GET_ARRAY_RD_COPY) != OM_STAT_SUCCESS))
                return(0);
	     memcpy(obj->vert_max, tmp_array, obj->nspace*sizeof(float));
	     ARRfree(tmp_array);
             obj->vert_extents = 1;
	  }
      }
   }
   /* The field is a tiled volume - we have a bit of extra
      work to do. This code is basically a duplicate of what
      is in the draw_tiled_volume code. Make a utility routine
      out of it.
   */
   else {
      /* Get the input volume to be tiled */
      tmp_id = OMfind_subobj(obj->field_id, OMstr_to_name("VolIn"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      if (OMget_obj_val(tmp_id, &vol_id) != OM_STAT_SUCCESS)
         return(0);
      if (OMis_null_obj(vol_id))
         return(0);

      /* Get info from the input field. */
      if (FLDget_nspace(vol_id, &nspace) != 1)
         return(0);
      /* We can't constrain the field to be 3D & 3space
         due to render method matching issues so we have to
         check explicitly here.
      */
      if (nspace != 3)
         return(0);
      obj->nspace = nspace;

      if (FLDget_dims(vol_id, &dims, &dims_size) != 1)
         return(0);
      if (dims_size != 3) {
         ARRfree(dims);
         return(0);
      }

      if (!GDget_int_val(obj->field_id, "width", &width))
         width = 1;
      if (!GDget_int_val(obj->field_id, "height", &height))
         height = 1;
      if (!GDget_int_val(obj->field_id, "mode", &cont_mode))
         cont_mode = 0;
      if (!GDget_int_val(obj->field_id, "border_width", &border_width))
         border_width = 0;

      /* Build the new extents of the object. We will take the existing
         extents and create new extents by expanding them using the
         specified width and height. If the container mode is 0, the
         tiles are kept the same size as the container. If the container
         mode is 1, the extents are kept square - meaning the tiles are
         now not guaranteed to be the same size as the container. However,
         this mode ensures that a normalize will "fill" the view window.
       */
      xextent = dims[0] + (2 * border_width);
      yextent = dims[1] + (2 * border_width);
      ARRfree(dims);

      /* Leave the min values alone but add the delta of
         the width and height to the max values. This
         will give us an extent that will encompass the tiled
         display. This is important so when we normalize the
         tiled display will be sized to the view.
      */
      vert_min_ext[0] = 0.0; vert_min_ext[1] = 0.0; vert_min_ext[2] = 0.0;
      vert_max_ext[0] = 0.0; vert_max_ext[1] = 0.0; vert_max_ext[2] = 0.0;

      /* See if we need to keep the extents square by
         looking at the container mode.
       */
      if (cont_mode == 1) {
         if ((xextent * width) > (yextent * height)) {
            vert_max_ext[0] = vert_max_ext[1] = xextent * width;
         }
         else {
            vert_max_ext[0] = vert_max_ext[1] = yextent * height;
         }
      }
      else {
         vert_max_ext[0] = xextent * width;
         vert_max_ext[1] = yextent * height;
      }

      /* Save the vert min/max in the object struct. */
      VEC_COPY(obj->vert_min, vert_min_ext);
      VEC_COPY(obj->vert_max, vert_max_ext);
      obj->vert_extents = 1;
   }
   return(1);
}

/* Gets the node data from the field.
   Looks for node data that does NOT have the id set and
   returns the first one found.
   Caller must ARRfree the data.
*/
/* 64-bit porting. Only Modified Internally */
void GDfld_get_node_data(OMobj_id field_id, GDgeom_data *data)
{
   int ncomps, i, data_id, veclen;
   xp_long size = 0;

   /* Loop over the node data if we have any. */
   if (FLDget_node_data_ncomp(field_id, &ncomps) != 1)
      return;

   for (i=0; i<ncomps; i++) {
      if (FLDget_node_data_id(field_id, i, &data_id) != 1) {

	 /* If we can't get the id, it must not be set. We use the
	    first one of these we can find that has a veclen of 1 
	 */
         if (FLDget_node_data_veclen(field_id, i, &veclen) != 1)
	    veclen = 0;

         if (veclen == 1) {
            /* always use first component it finds */
            if (!data->node_value_alloced) {
               if (FLDget_node_data(field_id, i, &data->node_type,
			(char **)&data->node_value,
			&size, OM_GET_ARRAY_RD) == 1) {
                  data->node_value_alloced = 2;
                  data->node_size = GDdata_size(data->node_type);

                  if (FLDget_node_null_data(field_id, i, &data->null_flag,
			(char *) &data->null_value) != 1)
                     data->null_flag = 0;
               }
	       /* Once we have successfully found the node data, 
		  we can quit.
               */
	       return;
            }
         }
      }
   }
}

/* Gets the min/max of the node data from the field.
   Looks for node data that does NOT have the id set and
   returns the min/max of the first one found.
   Returns 1 if successful, 0 if failed.
*/
int GDfld_get_node_data_minmax(OMobj_id field_id, double *min, double *max)
{
   int ncomps, i, data_id;
   int veclen, type;
   double tmp_min, tmp_max;

   /* Loop over the node data if we have any. */
   if (FLDget_node_data_ncomp(field_id, &ncomps) != 1)
      return(0);

   for (i=0; i<ncomps; i++) {
      if (FLDget_node_data_id(field_id, i, &data_id) != 1) {

	 /* If we can't get the id, it must not be set. We use the
	    first one of these we can find that has a veclen of 1 
	 */
         if (FLDget_node_data_veclen(field_id, i, &veclen) != 1)
	    veclen = 0;

         if (veclen == 1) {
            /* always use first component it finds */
            if (FLDget_node_data_type(field_id, i, &type) == 1) {
	       if (FLDget_node_data_minmax(field_id, i, (char *)&tmp_min, 
		(char *)&tmp_max) == 1) {
		  UTILtype_to_double(min, &tmp_min, type);
		  UTILtype_to_double(max, &tmp_max, type);
	          /* Once we have successfully found the node data, 
		     we can quit.
                  */
	          return(1);
               }
            }
         }
      }
   }
   return(0);
}

/* Gets the node data with special ids from the field.
   Caller must ARRfree the data.
*/
/* 64-bit porting. Only Modified Internally */
void GDfld_get_node_data_ids(OMobj_id field_id, GDgeom_data *data)
{
   int ncomps, i, data_id, veclen;
   int typef = OM_TYPE_FLOAT, typeb = OM_TYPE_BYTE;
   xp_long size = 0;

   /* Loop over the node data if we have any. */
   if (FLDget_node_data_ncomp(field_id, &ncomps) != 1)
      return;

   for (i=0; i<ncomps; i++) {
      if (FLDget_node_data_id(field_id, i, &data_id) == 1) {

         /* Node data has id set - look for one of the special tags. */
         if (data_id == GD_NORMAL_DATA_ID) {
            if (FLDget_node_data_veclen(field_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                        "Cannot get normals veclen");
	       veclen = 0;
	    }
	    if (veclen == 3) {
               if (FLDget_typed_node_data(field_id, i, &typef,
			(char **)&data->normal, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get normals");
	       }
               else data->normal_alloced = 2;
            }
	    else ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                          "Normals veclen must be 3");
         }
         else if (data_id == GD_COLOR_DATA_ID) {
            if (FLDget_node_data_veclen(field_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                        "Cannot get colors veclen");
	       veclen = 0;
	    }
	    if (veclen == 3) {
               if (FLDget_typed_node_data(field_id, i, &typef,
			(char **)&data->color, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get colors");
	       }
               else data->color_alloced = 2;
            }
	    else ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                          "Colors veclen must be 3");
         }
         else if (data_id == GD_RADIUS_DATA_ID) {
            if (FLDget_typed_node_data(field_id, i, &typef,
			(char **)&data->radius, &size, OM_GET_ARRAY_RD) != 1) {
	       ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                        "Cannot get radii");
	    }
            else data->radius_alloced= 2;
         }
         else if (data_id == GD_RGB_DATA_ID) {
            if (FLDget_node_data_veclen(field_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                        "Cannot get rgb veclen");
	       veclen = 0;
	    }

            if (veclen == 4) {
               if (FLDget_typed_node_data(field_id, i, &typeb,
			(char **)&data->argb, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get argbs");
	       }
               else data->argb_alloced = 2;
            }
            else if (veclen == 3) {
               if (FLDget_typed_node_data(field_id, i, &typeb,
			(char **)&data->rgb, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get rgbs");
	       }
               else data->rgb_alloced = 2;
            }
	    else ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                          "RGB veclen must be 3 or 4");
         }
         else if (data_id == GD_UV_DATA_ID) {
            if (FLDget_node_data_veclen(field_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                        "Cannot get uv veclen");
	       veclen = 0;
	    }

            if (veclen == 3) {
               if (FLDget_typed_node_data(field_id, i, &typef,
			(char **)&data->uvw, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get uvws");
	       }
               else data->uvw_alloced = 2;
            }
            else if (veclen == 2 ) {
               if (FLDget_typed_node_data(field_id, i, &typef,
                        (char **)&data->uv, &size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                           "Cannot get uvs");
	       }
               else data->uv_alloced = 2;
            }
	    else ERRerror("GDfld_get_node_data_ids", 0, ERR_ORIG,
                          "UV veclen must be 2 or 3");
         }
      }
   }
}

/* Gets the node data with special pick id from the field.
   Caller must ARRfree the data.
*/
/* 64-bit porting. Directly Modified */
int GDfld_get_pick_node_data(OMobj_id field_id,
                             xp_long *size, int *type, char **data)
{
   int ncomps, i, data_id, stat = 0;

   /* Loop over the node data if we have any. */
   if (FLDget_node_data_ncomp(field_id, &ncomps) != 1)
      return(0);

   for (i=0; i<ncomps; i++) {
      if (FLDget_node_data_id(field_id, i, &data_id) == 1) {

         /* Node data has id set - look for special pick data tag. */
         if (data_id == GD_PICK_DATA_ID) {
	    *size = 0;
	    *data = (char *)NULL;
            if (FLDget_node_data(field_id, i, type, data,
			size, OM_GET_ARRAY_RD) != 1) {
	       ERRerror("GDfld_get_pick_node_data", 0, ERR_ORIG,
                        "Cannot get pick data");
	    }
            else stat = 1;
         }
      }
   }
   return(stat);
}

/* Gets the cell data from the cell set.
   Looks for cell data that does NOT have the id set and
   returns the first one found.
   Caller must ARRfree the data.
*/
/* 64-bit porting. Only Modified Internally */
void GDfld_get_cell_data(OMobj_id cell_set_id, GDgeom_data *data)
{
   int ncomps, i, data_id;
   int veclen;
   xp_long npolys;
   xp_long size;;

   /* Loop over the cell data if we have any. */
   if (FLDget_cell_data_ncomp(cell_set_id, &ncomps) != 1)
      return;

   for (i=0; i<ncomps; i++) {
      if (FLDget_cell_data_id(cell_set_id, i, &data_id) != 1) {

	 /* If we can't get the id, it must not be set. We use the
	    first one of these we can find that has a veclen of 1 
	 */
         if (FLDget_cell_data_veclen(cell_set_id, i, &veclen) != 1)
	    veclen = 0;

         if (veclen == 1) {
            /* always use first component it finds */
            if (!data->cell_value_alloced) {
               if (FLDget_cell_data(cell_set_id, i, &data->cell_type,
			(char **)&data->cell_value,
			&size, OM_GET_ARRAY_RD) == 1) {
                  data->cell_value_alloced = 2;
                  data->cell_size = GDdata_size(data->cell_type);

                  if (FLDget_cell_null_data(cell_set_id, i, &data->cell_null_flag,
			(char *) &data->cell_null_value) != 1)
                     data->cell_null_flag = 0;

		  /* Figure out it we have per cell or per facet data */
                  if (FLDget_npolys(cell_set_id, &npolys) == 1) {
		     if (size == npolys) 
			data->cell_color_type = GD_CELL_COLOR_CELL;
                     else data->cell_color_type = GD_CELL_COLOR_SEGMENT;
                  }

		  /* Once we have successfully found the cell data, 
		     we can quit.
                  */
		  return;
               }
            }
         }
      }
   }
}

/* Gets the min/max of all the cell data from the field.
   Loops over all the cell sets, getting the min/max
   that encompasses all the data.
   Returns 1 if successful, 0 if failed.
*/
int GDfld_get_cell_data_minmax(OMobj_id field_id, double *min, double *max)
{
   int i, j, valid = 0, found;
   OMobj_id cell_set_id;
   int ncell_sets, ncomps, data_id;
   int veclen, type;
   double tmp_min, tmp_max, cell_min, cell_max;

   /* Get the number of cell sets. */
   if (FLDget_ncell_sets(field_id, &ncell_sets) != 1)
      return(0);

   /* Loop over the cell sets if we have any. */
   for (i=0; i<ncell_sets; i++) {

      /* Get cell set id. */
      if (FLDget_cell_set(field_id, i, &cell_set_id) != 1)
         continue;
      /* Get the number of components. */
      if (FLDget_cell_data_ncomp(cell_set_id, &ncomps) != 1)
         continue;

      found = 0;
      /* Loop over the number of components. */
      for (j=0; j<ncomps; j++) {

         if (FLDget_cell_data_id(cell_set_id, j, &data_id) != 1) {
	    /* If we can't get the id, it must not be set. We use the
	       first one of these we can find that has a veclen of 1 
	    */
            if (FLDget_cell_data_veclen(cell_set_id, j, &veclen) != 1)
	       veclen = 0;

            if (veclen == 1 && !found) {
	       found = 1;

               /* always use first component it finds */
               if (FLDget_cell_data_type(cell_set_id, j, &type) == 1) {
	          if (FLDget_cell_data_minmax(cell_set_id, j,
			(char *)&tmp_min, (char *)&tmp_max) == 1) {
		     UTILtype_to_double(&cell_min, &tmp_min, type);
		     UTILtype_to_double(&cell_max, &tmp_max, type);
		     if (!valid) {
			valid = 1;
			*min = cell_min;
			*max = cell_max;
                     }
	             else {
			if (cell_min < *min)
			   *min = cell_min;
			if (cell_max > *max)
			   *max = cell_max;
		     }
                  }
               }
            }
         }
      }
   }
   if (!valid)
      return(0);
   else return(1);
}

/* Gets the cell data with special ids from the cell set.
   Caller must ARRfree the data.
*/
/* 64-bit porting. Only Modified Internally */
void GDfld_get_cell_data_ids(OMobj_id cell_set_id, GDgeom_data *data)
{
   int ncomps, i, data_id;
   xp_long npolys;
   int typef = OM_TYPE_FLOAT, veclen;
   xp_long size = 0;

   /* Loop over the cell data if we have any. */
   if (FLDget_cell_data_ncomp(cell_set_id, &ncomps) != 1)
      return;

   for (i=0; i<ncomps; i++) {
      if (FLDget_cell_data_id(cell_set_id, i, &data_id) == 1) {

         /* Cell data has id set - look for one of the special tags. */
         if (data_id == GD_NORMAL_DATA_ID) {
            if (FLDget_cell_data_veclen(cell_set_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                        "Cannot get normals veclen");
	       veclen = 0;
	    }
	    if (veclen == 3) {
               if (FLDget_typed_cell_data(cell_set_id, i, &typef,
                                          (char **)&data->cell_normal,
			&size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                           "Cannot get cell normals");
               }
               else data->cell_normal_alloced = 2;
            }
	    else ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                          "Normal veclen must be 3");
         }
         else if (data_id == GD_COLOR_DATA_ID) {
            if (FLDget_cell_data_veclen(cell_set_id, i, &veclen) != 1) {
	       ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                        "Cannot get colors veclen");
	       veclen = 0;
	    }
	    if (veclen == 3) {
               if (FLDget_typed_cell_data(cell_set_id, i, &typef,
                                          (char **)&data->cell_color,
			&size, OM_GET_ARRAY_RD) != 1) {
	          ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                           "Cannot get cell colors");
	       }
               else data->cell_color_alloced = 2;

	       /* Figure out it we have per cell or per facet data */
               if (FLDget_npolys(cell_set_id, &npolys) == 1) {
		  if (size == npolys * veclen)
		     data->cell_color_type = GD_CELL_COLOR_CELL;
                  else data->cell_color_type = GD_CELL_COLOR_SEGMENT;
               }
            }
	    else ERRerror("GDfld_get_cell_data_ids", 0, ERR_ORIG,
                          "Colors veclen must be 3");
         }
      }
   }
}
