/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gd/output.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>

/*****************************************************/
/* Field output create, delete and update functions. */
/*****************************************************/

int GDfield_output_create(OMobj_id obj_id)
{
   OMobj_id ptr_id;
   GDoutput *output;
   int tmp_int;

   /* Allocate data and initialize it. */
   ALLOCN(output, GDoutput, 1, "can't allocate GDoutput");

   /* Save element id in the local structure */
   output->out_id = obj_id;

   /* Get the value of flip if it is set at instance time. */
   if (GDget_int_val(obj_id, "flip", &tmp_int))
      output->flip = tmp_int;

   /* Set canonical formats for framebuffer and zbuffer. */
   output->fb_type = GD_FB_ARGB;
   output->zb_type = GD_ZB_FLOAT_1NEAR;

   /* Get the values of the frame buffer and z buffer formats. */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("buffers"), OM_OBJ_RW);
   if (!OMis_null_obj(ptr_id)) {
      if (GDget_int_val(ptr_id, "fb_type", &tmp_int))
         output->fb_type = tmp_int;
      /* Get the value of format if it is set at instance time. */
      if (GDget_int_val(ptr_id, "zb_type", &tmp_int))
         output->zb_type = tmp_int;
   }

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (!OMis_null_obj(ptr_id))
      OMset_ptr_val(ptr_id, output, 0);

   return(1);
}

int GDfield_output_delete(OMobj_id obj_id)
{
   OMobj_id ptr_id;
   GDoutput *output;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);

   if (OMget_ptr_val(ptr_id, (void **)&output, 0) != OM_STAT_SUCCESS)
      return(0);

   free(output);

   GDclear_local(obj_id);
   return(1);
}

int GDfield_output_update(OMobj_id obj_id)
{
   OMobj_id ptr_id, seq_id, buf_id, field_id;
   GDoutput *output;
   int seq;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);

   if (OMget_ptr_val(ptr_id, (void **)&output, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(obj_id, OMstr_to_name("update"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return(0);
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   if (!GDget_int_val(obj_id, "flip", &output->flip))
      output->flip = 0;

   /* Get the values of the frame buffer and z buffer formats. */
   buf_id = OMfind_subobj(obj_id, OMstr_to_name("buffers"), OM_OBJ_RW);
   if (OMis_null_obj(buf_id))
      return(0);
   if (!GDget_int_val(buf_id, "fb_type", &output->fb_type))
      output->fb_type = GD_FB_ARGB;
   if (!GDget_int_val(buf_id, "zb_type", &output->zb_type))
      output->zb_type = GD_ZB_FLOAT_1NEAR;

   /* If clear has changed, clear the output meshes. */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("clear"), OM_OBJ_RD);
   if (OMchanged(ptr_id, seq)) {
      field_id = OMfind_subobj(buf_id, OMstr_to_name("framebuffer"), OM_OBJ_RW);
      if (!OMis_null_obj(field_id))
	 UTILclear_mesh(field_id);
      field_id = OMfind_subobj(buf_id, OMstr_to_name("zbuffer"), OM_OBJ_RW);
      if (!OMis_null_obj(field_id))
	 UTILclear_mesh(field_id);
   }

   return(1);
}

/*****************************************************/
/* File output create, delete and update functions.  */
/*****************************************************/

int GDfile_output_create(OMobj_id obj_id)
{
   OMobj_id ptr_id;
   GDoutput *output;
   GDfileOutput *fileOutput;
   int tmp_int;
   char *tmp_str = NULL;

   /* Allocate data and initialize it. */
   ALLOCN(output, GDoutput, 1, "can't allocate GDoutput");
   ALLOCN(fileOutput, GDfileOutput, 1, "can't allocate file output");

   /* Save pointer to file specific data. */
   output->fileOutput = fileOutput;
   /* Save element id in the local structure */
   output->out_id = obj_id;

   /* Get the value of flip if it is set at instance time. */
   if (GDget_int_val(obj_id, "flip", &tmp_int))
      output->flip = tmp_int;

   /* Get the value of filename if it is set at instance time. */
   if (GDget_str_val(obj_id, "filename", &tmp_str))
      fileOutput->filename = tmp_str;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (!OMis_null_obj(ptr_id))
      OMset_ptr_val(ptr_id, output, 0);

   /* Set canonical FB format, no ZB output. These are needed 
      so the routine to get the buffers will get the right type.
   */
   output->fb_type = GD_FB_ARGB;
   output->zb_type = GD_ZB_NONE;

   return(1);
}

int GDfile_output_delete(OMobj_id obj_id)
{
   OMobj_id ptr_id;
   GDoutput *output;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);

   if (OMget_ptr_val(ptr_id, (void **)&output, 0) != OM_STAT_SUCCESS)
      return(0);

   if (output->fileOutput) {
      if (output->fileOutput->filename)
         free(output->fileOutput->filename);
      free(output->fileOutput);
   }
   free(output);


   GDclear_local(obj_id);
   return(1);
}

int GDfile_output_update(OMobj_id obj_id)
{
   OMobj_id ptr_id, seq_id;
   GDoutput *output;
   GDfileOutput *fileOutput;
   int seq;
   char *tmp_str = NULL;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(obj_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);

   if (OMget_ptr_val(ptr_id, (void **)&output, 0) != OM_STAT_SUCCESS)
      return(0);

   fileOutput = output->fileOutput;

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(obj_id, OMstr_to_name("update"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return(0);
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   if (!GDget_int_val(obj_id, "flip", &output->flip))
      output->flip = 0;

   if (GDget_str_val(obj_id, "filename", &tmp_str)) {
      if (fileOutput->filename)
	 free(fileOutput->filename);
      fileOutput->filename = tmp_str;
   }
   return(1);
}


/*****************************************************/
/* Init virtual functions for field/file output.     */
/*****************************************************/

/* For this type of output, this routine is a noop */
void GDoutput_init(GDview *view)
{
}

/*****************************************************/
/* Flush virtual function for field/file output.     */
/*****************************************************/

/* We need to take the data that has been rendered and
   convert it to a buffer of data. There are a couple 
   of cases:
     * 3d cameras
     * 2d cameras or 3d & 2d cameras

   If we have only 3d cameras, we have the data that
   appears on the screen in the 3d frame buffer in
   an argb format. This is exactly what we want.

   If we have 2d cameras in the scene, in order to
   get the 2d primitives in the output we need to
   extract an argb format from the drawable.

   This function is by both the file and field output.
   This routine goes an calls a second level of virtual
   function that actually writes the output. This gives
   the user an easy hook to create their own type of
   output by simply replacing the write function.
*/
void GDoutput_flush(GDview *view)
{
   /* Call the renderer dependent view method to do this.
      This routine then turns around and calls the
      output_write function associated with the
      type of GDoutput - typically field or file.
   */
   GDview_call_func(view, "view_get_buffers", 0);
}

/*****************************************************/
/* Write virtual functions for field/file output.    */
/*****************************************************/

/*
   The inputs to this function are as follows:
      dims     ==> xsize and ysize
      img_data ==> pointer to argb array of xsize*ysize
      z_data   ==> pointer to array of xsize*ysize
      output   ==> pointer to GDoutput struct containing
		   specific data like flip flag
*/
/* 64-bit porting. Directly Modified */
void GDfield_output_write(xp_long *dims, char *img_data, char *zbuf_data,
                          GDoutput *output)
{
   OMobj_id buf_id, field_id;
   RPIXEL *idata, *idata_ptr, *fb_ptr;
   char *zdata, *zdata_ptr, *zb_ptr;
   xp_long i, size;
   int type;

   /* Data associated with the output field */
   xp_long nnodes;
   float points[4];

   buf_id = OMfind_subobj(output->out_id, OMstr_to_name("buffers"), OM_OBJ_RW);
   if (OMis_null_obj(buf_id)) {
      ERRerror("GDfield_output_write", 0, ERR_ORIG,
               "Can't access output buffers");
      return;
   }

   /* Find the id of the output image field. It has already been classed as
      a uniform 2D, 2space field.
   */
   field_id = OMfind_subobj(buf_id, OMstr_to_name("framebuffer"), OM_OBJ_RW);
   if (OMis_null_obj(field_id)) {
      ERRerror("GDfield_output_write", 0, ERR_ORIG,
               "Can't access output field");
      return;
   }
   if (img_data && (output->fb_type != GD_FB_NONE)) {
      /* Set up the mesh - nnodes, nspace, dims and points */
      nnodes = dims[0] * dims[1];
      points[0] = 0.0;
      points[1] = 0.0;
      points[2] = (float)(dims[0] - 1);
      points[3] = (float)(dims[1] - 1);
      if (FLDset_nnodes(field_id, nnodes) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image nnodes");
         return;
      }
      if (FLDset_nspace(field_id, 2) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image nspace");
         return;
      }
      if (FLDset_dims(field_id, dims) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image dims");
         return;
      }
      if (FLDset_points(field_id, points, 4, OM_SET_ARRAY_COPY) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image points");
         return;
      }

      /* Set up the node data */
      if (FLDset_node_data_ncomp(field_id, 1) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image node data ncomps");
         return;
      }
      /* Only write the id if the type is ARGB */
      if (output->fb_type == GD_FB_ARGB) {
         if (FLDset_node_data_id(field_id, 0, GD_RGB_DATA_ID) != 1) {
            ERRerror("GDfield_output_write", 0, ERR_ORIG,
                     "Can't write image node data id");
            return;
         }
      }
      if (FLDset_node_data_veclen(field_id, 0, 4) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write image node data veclen");
         return;
      }

      /* Have the OM manage the allocation for us. We then
         just need to fill the data in. This will also be more
         efficient since if we fill in a local array, a copy will
         then have to be made. 
      */
      type = OM_TYPE_BYTE;
      if (FLDget_node_data(field_id, 0, &type, (char **)&idata,
	   &size, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't get buffer for image node data");
         return;
      }
      if (size != 4*nnodes) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Image buffer allocated has wrong size");
         return;
      }

      /* See if the user has requested that we flip the frame buffer */
      fb_ptr = (RPIXEL *)img_data;
      if (!output->flip) {
         /* We can copy the whole frame buffer in one swell foop !! */
         memcpy(idata, fb_ptr, dims[0]*dims[1]*RPIXEL_BYTES);
      }
      else {
         /* We need to flip the frame buffer or the result will be
            upside down so set the OM pointer to the last scan line.
         */
         idata_ptr = idata;
         idata_ptr += dims[0] * (dims[1] - 1);
         for (i=0; i<dims[1]; i++) {
            memcpy(idata_ptr, fb_ptr, dims[0]*RPIXEL_BYTES);
            idata_ptr -= dims[0];
            fb_ptr += dims[0];
         }
      }
      /* We need to free the data here or we will have a memory
	 leak. The OM has a reference to it so it doesn't really
	 get freed it just releases the reference that we have
	 to it.
      */
      ARRfree(idata);
   }
   else UTILclear_mesh(field_id);

   /* Find the id of the output Zbuffer field. It has already been classed as
      a uniform 2D, 2space field.
   */
   field_id = OMfind_subobj(buf_id, OMstr_to_name("zbuffer"), OM_OBJ_RW);
   if (OMis_null_obj(field_id)) {
      ERRerror("GDfield_output_write", 0, ERR_ORIG,
               "Can't access zbuffer field");
      return;
   }
   if (zbuf_data && (output->zb_type != GD_ZB_NONE)) {
      /* Set up the mesh - nnodes, nspace, dims and points */
      nnodes = dims[0] * dims[1];
      points[0] = 0.0;
      points[1] = 0.0;
      points[2] = (float)(dims[0] - 1);
      points[3] = (float)(dims[1] - 1);
      if (FLDset_nnodes(field_id, nnodes) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer nnodes");
         return;
      }
      if (FLDset_nspace(field_id, 2) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer nspace");
         return;
      }
      if (FLDset_dims(field_id, dims) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer dims");
         return;
      }
      if (FLDset_points(field_id, points, 4, OM_SET_ARRAY_COPY) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer points");
         return;
      }

      /* Set up the node data */
      if (FLDset_node_data_ncomp(field_id, 1) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer node data ncomps");
         return;
      }
      if (FLDset_node_data_veclen(field_id, 0, 1) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't write zbuffer node data veclen");
         return;
      }

      /* Have the OM manage the allocation for us. We then
         just need to fill the data in. This will also be more
         efficient since if we fill in a local array, a copy will
         then have to be made. 
      */
      type = OM_TYPE_FLOAT;
      if (output->zb_type == GD_ZB_SHORT)
	 type = OM_TYPE_SHORT;
      if (FLDget_node_data(field_id, 0, &type, (char **)&zdata,
	   &size, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Can't get buffer for zbuffer node data");
         return;
      }
      if (size != nnodes) {
         ERRerror("GDfield_output_write", 0, ERR_ORIG,
                  "Buffer allocated for zbuffer is wrong size");
         return;
      }

      /* See if the user has requested that we flip the Z buffer */
      size = GDdata_size(type);
      zb_ptr = zbuf_data;
      if (!output->flip) {
         /* We can copy the whole Z buffer in one swell foop !! */
         memcpy(zdata, zb_ptr, dims[0]*dims[1]*size);
      }
      else {
         /* We need to flip the frame buffer or the result will be
            upside down so set the OM pointer to the last scan line.
         */
         zdata_ptr = zdata;
         zdata_ptr += dims[0] * (dims[1] - 1) * size;
         for (i=0; i<dims[1]; i++) {
            memcpy(zdata_ptr, zb_ptr, dims[0]*size);
            zdata_ptr -= dims[0] * size;
            zb_ptr += dims[0] * size;
         }
      }
      /* We need to free the data here or we will have a memory
	 leak. The OM has a reference to it so it doesn't really
	 get freed it just releases the reference that we have
	 to it.
      */
      ARRfree(zdata);
   }
   else UTILclear_mesh(field_id);
}

/* We need to take the data that has been rendered and
   convert it to an AVS image file (.x).

   This routine can be replace with a user-specific routine
   to write the output format desired.

   The inputs to this function are as follows:
      dims     ==> xsize and ysize
      img_data ==> pointer to argb array of xsize*ysize
      z_data   ==> pointer to array of xsize*ysize
      output   ==> pointer to GDoutput struct containing
		   specific data like filename, flip flag
*/
void GDfile_output_write(xp_long *dims, char *img_data, char *z_data,
                         GDoutput *output)
{
   GDfileOutput *fileOutput = output->fileOutput;

   /* Make sure we have a filename !! */
   if (fileOutput->filename) {
      int i_dims[2];
      i_dims[0] = (int)dims[0];
      i_dims[1] = (int)dims[1];
      /* Format the inputs to the write image utility routine. */
      UTILwrite_image(i_dims, img_data, fileOutput->filename, output->flip);
   }
}

/*****************************************************/
/* Done virtual functions for field/file output.     */
/*****************************************************/

/* For this type of output, this routine is a noop */
void GDoutput_done(GDview  *view)
{
}
