/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gd/rd_poly.c#1 $
*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>

#define NUMPOINTS        2333
#define NUMPOLYS         2491
#define MAXPOLYVERTICES     5

/*
 ******************************************************
 *             geometric structures                   *
 ******************************************************
 */

/* Coordinate array */
static struct point_struct
{
    float x,y;
} point_array[NUMPOINTS];

/* Connectivity plus color array */
static struct poly_struct
{
    int num_vertices;
    int color_index;
    int vertices[MAXPOLYVERTICES];
} poly_array[NUMPOLYS];

static int npolys;

/* functions declared in this module */

void		Read_Poly(OMobj_id);
static int	load_poly_data (const char *);
static void	WritePolyMesh (OMobj_id);


void Read_Poly(OMobj_id elem_id)
{
  OMobj_id file_id;
  char *filename;
  int tmp_int;

  file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"),
	OM_OBJ_RD);
  if (OMget_str_val(file_id, &filename, 0) != OM_STAT_SUCCESS)
    return;

  npolys = NUMPOLYS;
  if (OMget_name_int_val(elem_id, OMstr_to_name("npolys"), &tmp_int) == OM_STAT_SUCCESS) {
     if (tmp_int > 0 && tmp_int <= NUMPOLYS)
	npolys = tmp_int;
  }

  /* Read in the data */
  if (!load_poly_data(filename)) {
     if (filename)
        free(filename);
     return;
  }

  /* Write the data to an Express field */
  WritePolyMesh(elem_id);

  if (filename)
    free(filename);
}

/* 64-bit porting. Only Modified Internally */
static void WritePolyMesh(OMobj_id elem_id)
{
   OMobj_id mesh_id, cset_id;
   xp_long coord_size;
   float *coord;
   xp_long pnsize = 0;
   int *pnnodes, *pnnodes_ptr;
   xp_long *poly_conn, *poly_conn_ptr, poly_conn_size;
   int i, j, type;
   xp_long csize = 0;
   int *cdata, *cdata_ptr;

   mesh_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
   if (OMis_null_obj(mesh_id)) {
      printf("[WritePolyMesh]: Can't find output mesh\n");
      return;
   }

   /* Set up the mesh. This is the grid plus cells.
      Set up the grid: we need nspace, the number of
      nodes and the coordinates.
   */
   if (FLDset_nspace(mesh_id, 2) != 1) {
      printf("[WritePolyMesh]: Can't write nspace\n");
      return;
   }
   if (FLDset_nnodes(mesh_id, NUMPOINTS) != 1) {
      printf("[WritePolyMesh]: Can't write nnodes\n");
      return;
   }
   if (FLDget_coord(mesh_id, (float **)&coord, &coord_size, OM_GET_ARRAY_WR) != 1) {
      printf("[WritePolyMesh]: Can't get coordinates\n");
      return;
   }
   /* Copy the buffer we have filled up to the array the OM has
      allocated for us.
   */
   memcpy(coord, point_array, NUMPOINTS*2*sizeof(float));

   if (FLDset_ncell_sets(mesh_id, 0) != 1) {
      printf("[WritePolyMesh]: Can't set number of cell sets\n");
      return;
   }

   /* Set up the cell set: We need to add the cell set with the
      type, set the number of cells and the connectivity.
   */
   if (FLDadd_cell_set(mesh_id, "Polyhedron") != 1) {
      printf("[WritePolyMesh]: Can't add cell set\n");
      return;
   }
   if (FLDget_cell_set(mesh_id, 0, &cset_id) != 1) {
      printf("[WritePolyMesh]: Can't get cell set id\n");
      return;
   }
   if (FLDset_cell_set_user_name(cset_id, "Polygons") != 1) {
      printf("[WritePolyMesh]: Can't set user cell set name\n");
      return;
   }

   if (FLDset_npolys(cset_id, npolys) != 1) {
      printf("[WritePolyMesh]: Can't set number of polys\n");
      return;
   }
   if (FLDget_poly_nnodes(cset_id, &pnnodes, &pnsize, OM_GET_ARRAY_WR) != 1) {
      printf("[WritePolyMesh]: Can't get cell set nnodes array\n");
      return;
   }

   /* Build the nnodes array. */
   pnnodes_ptr = pnnodes;
   poly_conn_size = 0;
   for (i=0; i<npolys; i++) {
      *pnnodes_ptr++ = poly_array[i].num_vertices;
      poly_conn_size += poly_array[i].num_vertices;
   }
   ARRfree(pnnodes);

   /* We had to figure out how big this array is ahead of time since
      each poly can have a different number of nodes. In other words,
      you can't do a get on the array for writing.
   */
   /* Allocate and init the poly connectivity array. */
   poly_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, poly_conn_size, NULL);
   poly_conn_ptr = poly_conn;
   for (i=0; i<npolys; i++) {
      for (j=0; j<poly_array[i].num_vertices; j++) {
         *poly_conn_ptr++ = poly_array[i].vertices[j];
      }
   }
   if (FLDset_poly_connect(cset_id, poly_conn, poly_conn_size, OM_SET_ARRAY_FREE) != 1) {
      printf("[WritePolyMesh]: Can't set cell set connectivity array\n");
      return;
   }

   /* Write the color index as cell data. */
   if (FLDset_cell_data_ncomp(cset_id, 1) != 1) {
      printf("[WritePolyMesh]: Can't write cell data ncomp\n");
      return;
   }

   /* Set up color index cell data */
   if (FLDset_cell_data_veclen(cset_id, 0, 1) != 1) {
      printf("[WritePolyMesh]: Can't write cell data veclen\n");
      return;
   }
   /* Have the OM manage the allocation for us. We then
      just need to fill the data in. This will also be more
      efficient since if we fill in a local array, a copy will
      then have to be made.
   */
   type = DTYPE_INT;
   if (FLDget_cell_data(cset_id, 0, &type, (char **)&cdata,
        &csize, OM_GET_ARRAY_WR) != 1) {
      printf("[WritePolyMesh]: Can't get buffer for cell data\n");
      return;
   }

   cdata_ptr = cdata;
   for (i=0; i<npolys; i++) {
      *cdata_ptr++ = poly_array[i].color_index;
   }
   ARRfree(cdata);

}

static int load_poly_data( const char *filename )
{
    int i, j;
    FILE *fp;

    if ((fp = FILEfopen(filename, "r")) == NULL) {
        printf("Can't open file: %s\n", filename);
	return(0);
    }

    for (i=0; i<NUMPOINTS; i++)
	fscanf(fp, "%f %f", &point_array[i].x, &point_array[i].y);

    for (i=0; i<NUMPOLYS; i++) {
	fscanf(fp, "%d", &poly_array[i].num_vertices);
	fscanf(fp, "%d", &poly_array[i].color_index);
	for ( j=0; j<poly_array[i].num_vertices; j++)
	    fscanf( fp, "%d", &poly_array[i].vertices[j]);
    }

    fclose(fp);

    return(1);
}

