/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gd/rd_uscnt.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>

#define NPOLY	3082
#define NPOINTS	88104

static struct
{
  int   state;
  int	county;
  int   num;
} poly[NPOLY];

static struct
{
  float   x;
  float   y;
} points[NPOINTS];

static int npoly, npoints;
static xp_long *poly_conn;

/* Forward declaration of functions declared in this module */

void	Read_USCounties (OMobj_id);
static int	ReadData (char *, int);
static void	WriteMesh (OMobj_id);
static int	my_min (int, int);


void Read_USCounties(OMobj_id elem_id)
{
  OMobj_id file_id;
  char *filename;
  int state_no;

  file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"),
	OM_OBJ_RD);
  if (OMget_str_val(file_id, &filename, 0) != OM_STAT_SUCCESS)
    return;
  if (OMget_name_int_val(elem_id, OMstr_to_name("state_no"),
	&state_no) != OM_STAT_SUCCESS)
    state_no = 0;

  /* Read in the data */
  if (!ReadData(filename, state_no)) {
     if (filename)
        free(filename);
     return;
  }

  /* Check to see that we have actually read
     something.
  */
  if (npoints == 0 || npoly == 0) {
     if (filename)
       free(filename);
     return;
  }

  /* Write the data to an Express field */
  WriteMesh(elem_id);

  if (filename)
    free(filename);
}

/*
 * Read the data from the datafile called uscount.
 * The file is written in a packed FORTRAN format which is rather difficult
 * to read from C. The first line for each polygon is a line containing the
 * state number, the county number (which is not used in this program) and
 * the number of points in the polygon. After this header line comes the
 * coordinates with 8x2 coordinates per line. Each coordinate uses 5
 * character positions. There is no white space between the coordinates, and
 * there is no decimal point. Thus "2713210066" means the point (27.132,
 * 10.066).
 */

static int ReadData(char *filename, int state_no)
{
  FILE*   stream;
  int     p, d, k, m;
  int	  state, county, num;
  char    str[83];
  char*   s;

  if ((stream = FILEfopen (filename, "r")) == NULL) {
      fprintf (stderr, "Cannot open data file \"%s\"\n", filename);
      return(0);
  }

  npoly = 0;
  npoints = 0;
  for (d = 0, p = 0; p<NPOLY; p++) {
      /*
       * Read header line with state number and number of points in the polygon
       */

      if (! (s = fgets (str, 83, stream)))
	  break;

      state = atoi (s);
      county = atoi (s+5);
      num = atoi (s+10);

      /* If the state number is 0 - we read all polygons.
	 If the state number matches - we read this polygon.
      */
      if ((!state_no) ||
	  (state_no == state)) {

          poly[npoly].state = state;
          poly[npoly].county = county;
          poly[npoly].num = num;

          /* Read coordinates of the polygon corners */
          for (k = 0; k<num; k+= 8) {
	      if (! (s = fgets (str, 83, stream)))
	          break;

	      m = my_min (k+8, num);
	      s = str+(m-k)*10;
	      while (--m>= k) {
	          *s = '\0';   s-= 5;
	          points[d+m].y = ((double) atoi (s)) / 1000.0;
	          *s = '\0';
	          s-= 5;
	          points[d+m].x = ((double) atoi (s)) / 1000.0;
	      }
          }
          d+= num;

          /* Increment local count of polys and points. */
          npoly++;
          npoints += num;
      }
      else {
          /* Read coordinates of the polygon corners */
          for (k = 0; k<num; k+= 8) {
	      if (! (s = fgets (str, 83, stream)))
	          break;

	      m = my_min (k+8, num);
	      s = str+(m-k)*10;
	      while (--m>= k) {
	          *s = '\0';   s-= 5;
	          *s = '\0';
	          s-= 5;
	      }
          }
      }
  }

  fclose (stream);
  return(1);
}

/* 64-bit porting. Only Modified Internally */
static void WriteMesh(OMobj_id elem_id)
{
   OMobj_id mesh_id, cset_id;
   xp_long coord_size;
   float *coord;
   xp_long pnsize = 0;
   int *pnnodes, *pnnodes_ptr;
   int i, type;
   xp_long ssize = 0, csize = 0;
   int *sdata, *sdata_ptr;
   int *cdata, *cdata_ptr;

   mesh_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
   if (OMis_null_obj(mesh_id)) {
      printf("[WriteMesh]: Can't find output mesh\n");
      return;
   }

   /* Set up the mesh. This is the grid plus cells.
      Set up the grid: we need nspace, the number of
      nodes and the coordinates.
   */
   if (FLDset_nspace(mesh_id, 2) != 1) {
      printf("[WriteMesh]: Can't write nspace\n");
      return;
   }
   if (FLDset_nnodes(mesh_id, npoints) != 1) {
      printf("[WriteMesh]: Can't write nnodes\n");
      return;
   }
   if (FLDget_coord(mesh_id, (float **)&coord, &coord_size, OM_GET_ARRAY_WR) != 1) {
      printf("[WriteMesh]: Can't get coordinates\n");
      return;
   }
   /* Copy the buffer we have filled up to the array the OM has
      allocated for us.
   */
   memcpy(coord, points, npoints*2*sizeof(float));

   if (FLDset_ncell_sets(mesh_id, 0) != 1) {
      printf("[WriteMesh]: Can't set number of cell sets\n");
      return;
   }

   /* Set up the cell set: We need to add the cell set with the
      type, set the number of cells and the connectivity.
   */
   if (FLDadd_cell_set(mesh_id, "Polyhedron") != 1) {
      printf("[WriteMesh]: Can't add cell set\n");
      return;
   }
   if (FLDget_cell_set(mesh_id, 0, &cset_id) != 1) {
      printf("[WriteMesh]: Can't get cell set id\n");
      return;
   }
   if (FLDset_cell_set_user_name(cset_id, "US Counties") != 1) {
      printf("[WriteMesh]: Can't set user cell set name\n");
      return;
   }

   if (FLDset_npolys(cset_id, npoly) != 1) {
      printf("[WriteMesh]: Can't set number of polys\n");
      return;
   }
   if (FLDget_poly_nnodes(cset_id, &pnnodes, &pnsize, OM_GET_ARRAY_WR) != 1) {
      printf("[WriteMesh]: Can't get cell set nnodes array\n");
      return;
   }

   /* Build the nnodes array. */
   pnnodes_ptr = pnnodes;
   for (i=0; i<npoly; i++) {
      *pnnodes_ptr++ = poly[i].num;
   }
   ARRfree(pnnodes);

   /* We had to figure out how big this array is ahead of time since
      each poly can have a different number of nodes. In other words,
      you can't do a get on the array for writing.
   */
   /* Allocate and init the poly connectivity array. */
   poly_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, npoints, NULL);
   for (i=0; i<npoints; i++)
      poly_conn[i] = i;
   if (FLDset_poly_connect(cset_id, poly_conn, npoints, OM_SET_ARRAY_FREE) != 1) {
      printf("[WriteMesh]: Can't set cell set connectivity array\n");
      return;
   }

   /* Write the state and county numbers as cell data. */
   if (FLDset_cell_data_ncomp(cset_id, 2) != 1) {
      printf("[WriteMesh]: Can't write cell data ncomp\n");
      return;
   }

   /* Set up state cell data */
   if (FLDset_cell_data_veclen(cset_id, 0, 1) != 1) {
      printf("[WriteMesh]: Can't write cell data veclen\n");
      return;
   }
   /* Have the OM manage the allocation for us. We then
      just need to fill the data in. This will also be more
      efficient since if we fill in a local array, a copy will
      then have to be made.
   */
   type = DTYPE_INT;
   if (FLDget_cell_data(cset_id, 0, &type, (char **)&sdata,
        &ssize, OM_GET_ARRAY_WR) != 1) {
      printf("[WriteMesh]: Can't get buffer for cell data\n");
      return;
   }

   /* Set up county cell data */
   if (FLDset_cell_data_veclen(cset_id, 1, 1) != 1) {
      printf("[WriteMesh]: Can't write cell data veclen\n");
      return;
   }
   /* Have the OM manage the allocation for us. We then
      just need to fill the data in. This will also be more
      efficient since if we fill in a local array, a copy will
      then have to be made.
   */
   type = DTYPE_INT;
   if (FLDget_cell_data(cset_id, 1, &type, (char **)&cdata,
        &csize, OM_GET_ARRAY_WR) != 1) {
      printf("[WriteMesh]: Can't get buffer for cell data\n");
      return;
   }

   sdata_ptr = sdata;
   cdata_ptr = cdata;
   for (i=0; i<npoly; i++) {
      *sdata_ptr++ = poly[i].state;
      *cdata_ptr++ = poly[i].county;
   }
   ARRfree(sdata);
   ARRfree(cdata);
   ARRfree(coord);
}

/*
 * Various comparison functions, these could have been defined as macros.
 */

static int my_min (int a, int b)
{
  return (a < b ? a : b);
}
