/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/modes.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>

#include <avs/dll_in.h>
#include <avs/util.h>
#include <avs/err.h>
#include <avs/arr.h>
#include <avs/om.h>
#include <avs/gd.h>

void GDmodes_ref_set(GDmodes *modes)
{
  if (modes)
    modes->ref_count = 1;
}

void GDmodes_ref_inc(GDmodes *modes)
{
  if (modes)
    modes->ref_count++;
}

void GDmodes_ref_dec(GDmodes *modes)
{
  if (modes && --modes->ref_count < 1)
    FREE(modes);
}

/* This function is called in a "delayed" fashion via the
   GDget_local routine. When we try to get the local modes
   structure pointer - if it doesn't exist the create routine
   is called. This will happen in two cases: either thru the
   modes update function that is called at instance time or
   via the object attach modes function that is called when the
   modes is attached to the object.
*/
int GDmodes_create(OMobj_id modes_id)
{
   GDmodes *modes;

   /* allocate a modes structure and initialize it */
   ALLOCN(modes, GDmodes, 1, "can't allocate modes");

   /* reference the newly created modes */
   GDmodes_ref_inc(modes);

   /* save OM element id in local data structures */
   modes->modes_id = modes_id;
   /* Save the id of the update function. */
   modes->upd_func = OMfind_subobj(modes_id, OMstr_to_name("upd_func"), OM_OBJ_RW);

   GDset_local(modes_id, (char *)modes);
   return(1);
}

int GDmodes_delete(OMobj_id modes_id)
{
   GDmodes *modes;

   if (OMis_null_obj(modes_id))
      return(1);
   modes = (GDmodes *)GDget_local(modes_id, (OMpfi)GDmodes_create);

   /* dereference modes */
   GDmodes_ref_dec(modes);
   GDclear_local(modes_id);
   return(1);
}

void GDmodes_reset(OMobj_id modes_id)
{
   GDmodes *modes;
   int mode[GD_MODE_SIZE];
   int normals, colors, outline;

   if (OMis_null_obj(modes_id))
      return;

   mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
   mode[GD_MODE_LINES] = GD_LINES_INHERIT;
   mode[GD_MODE_SURF] = GD_SURF_INHERIT;
   mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
   mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
   normals = GD_NORMALS_INHERIT;
   colors = GD_COLORS_INHERIT;
   outline = GD_OUTLINE_ON;

   /* initialize data base copy of modes */
   GDset_int_array(modes_id, "mode", GD_MODE_SIZE, mode);
   GDset_int_val(modes_id, "normals", normals);
   GDset_int_val(modes_id, "colors", colors);
   GDset_int_val(modes_id, "outline", outline);

   modes = (GDmodes *)GDget_local(modes_id, (OMpfi)GDmodes_create);
   memcpy(modes->mode, mode, GD_MODE_SIZE*sizeof(int));
   modes->normals = normals;
   modes->colors = colors;
   modes->outline = outline;
   modes->changed = 0;			/* clear changed flags */
}

/* This routine updates the local structure from the data base.
   This occurs whenever the modes data base element is
   written. Only the data that is changed is updated - change
   is noted by checking the sequence number */

/* 64-bit porting. Only Modified Internally */
int GDmodes_update(OMobj_id modes_id)
{
   GDmodes *modes = NULL;
   xp_long size;
   int upd_seq, tmp_val;
   int *tmp_arr = NULL;

    /* get pointer to local storage, return if NULL */
   modes = (GDmodes *)GDget_local(modes_id, (OMpfi)GDmodes_create);
   if (!modes)
      return(0);

   /* Get the sequence number of the update function. We can use
      this to determine if any sub-elements in modes have changed.
      A mode of 0 means of the element itself.
   */
   upd_seq = OMget_obj_seq(modes->upd_func, OMnull_obj, 0);

   /* update draw mode */
   if (GDget_updint_array(modes_id, "mode", upd_seq, &size,
	&tmp_arr)) {
      memcpy(modes->mode, tmp_arr, GD_MODE_SIZE*sizeof(int));
      if (tmp_arr[GD_MODE_POINTS] != GD_POINTS_INHERIT)
         modes->changed |= GD_CHANGE_MODE;
      else if (tmp_arr[GD_MODE_LINES] != GD_LINES_INHERIT)
	 modes->changed |= GD_CHANGE_MODE;
      else if (tmp_arr[GD_MODE_SURF] != GD_SURF_INHERIT)
	 modes->changed |= GD_CHANGE_MODE;
      else if (tmp_arr[GD_MODE_VOL] != GD_VOLUME_INHERIT)
	 modes->changed |= GD_CHANGE_MODE;
      else if (tmp_arr[GD_MODE_BOUNDS] != GD_BOUNDS_INHERIT)
	 modes->changed |= GD_CHANGE_MODE;
      else modes->changed &= ~GD_CHANGE_MODE;
      ARRfree(tmp_arr);
   }

   /* update normals */
   if (GDget_updint_val(modes_id, "normals", upd_seq, &tmp_val)) {
      modes->normals= tmp_val;
      if (tmp_val != GD_NORMALS_INHERIT)
         modes->changed |= GD_CHANGE_NORMALS;
      else modes->changed &= ~GD_CHANGE_NORMALS;
   }
   /* update colors */
   if (GDget_updint_val(modes_id, "colors", upd_seq, &tmp_val)) {
      modes->colors= tmp_val;
      if (tmp_val != GD_COLORS_INHERIT)
         modes->changed |= GD_CHANGE_COLORS;
      else modes->changed &= ~GD_CHANGE_COLORS;
   }
   /* update outline */
   if (GDget_updint_val(modes_id, "outline", upd_seq, &tmp_val)) {
      modes->outline= tmp_val;
      if (tmp_val != GD_OUTLINE_ON)
         modes->changed |= GD_CHANGE_OUTLINE;
      else modes->changed &= ~GD_CHANGE_OUTLINE;
   }
   return(1);
}

/* 64-bit porting. Only Modified Internally */
void GDmodes_set_points(OMobj_id modes_id, int points)
{
   int *mode = NULL;
   xp_long size;
   int tmp_mode[5], need_free = 1;

   /* currently there is no set array element - so I have to get the
      array, make the change and then set it */
   if (GDget_int_array(modes_id, "mode", &size, &mode) != 1) {
      mode = tmp_mode;
      size = 5;
      mode[GD_MODE_LINES] = GD_LINES_INHERIT;
      mode[GD_MODE_SURF] = GD_SURF_INHERIT;
      mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
      mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
      need_free = 0;
   }

   /* set new points mode */
   mode[GD_MODE_POINTS] = points;
   GDset_int_array(modes_id, "mode", size, mode);

   if (need_free)
      ARRfree(mode);
}

/* 64-bit porting. Only Modified Internally */
void GDmodes_set_lines(OMobj_id modes_id, int lines)
{
   int *mode = NULL;
   xp_long size;
   int tmp_mode[5], need_free = 1;

   /* currently there is no set array element - so I have to get the
      array, make the change and then set it */
   if (GDget_int_array(modes_id, "mode", &size, &mode) != 1) {
      mode = tmp_mode;
      size = 5;
      mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
      mode[GD_MODE_SURF] = GD_SURF_INHERIT;
      mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
      mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
      need_free = 0;
   }

   /* set new lines mode */
   mode[GD_MODE_LINES] = lines;
   GDset_int_array(modes_id, "mode", size, mode);

   if (need_free)
      ARRfree(mode);
}

/* 64-bit porting. Only Modified Internally */
void GDmodes_set_surf(OMobj_id modes_id, int surface)
{
   int *mode = NULL;
   xp_long size;
   int tmp_mode[5], need_free = 1;

   /* currently there is no set array element - so I have to get the
      array, make the change and then set it */
   if (GDget_int_array(modes_id, "mode", &size, &mode) != 1) {
      mode = tmp_mode;
      size = 5;
      mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
      mode[GD_MODE_LINES] = GD_LINES_INHERIT;
      mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
      mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
      need_free = 0;
   }

   /* set new surface mode */
   mode[GD_MODE_SURF] = surface;
   GDset_int_array(modes_id, "mode", size, mode);

   if (need_free)
      ARRfree(mode);
}

/* 64-bit porting. Only Modified Internally */
void GDmodes_set_volume(OMobj_id modes_id, int volume)
{
   int *mode = NULL;
   xp_long size;
   int tmp_mode[5], need_free = 1;

   /* currently there is no set array element - so I have to get the
      array, make the change and then set it */
   if (GDget_int_array(modes_id, "mode", &size, &mode) != 1) {
      mode = tmp_mode;
      size = 5;
      mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
      mode[GD_MODE_LINES] = GD_LINES_INHERIT;
      mode[GD_MODE_SURF] = GD_SURF_INHERIT;
      mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
      need_free = 0;
   }

   /* set new volume mode */
   mode[GD_MODE_VOL] = volume;
   GDset_int_array(modes_id, "mode", size, mode);

   if (need_free)
      ARRfree(mode);
}
/* 64-bit porting. Only Modified Internally */
void GDmodes_set_bounds(OMobj_id modes_id, int bounds)
{
   int *mode = NULL;
   xp_long size;
   int tmp_mode[5], need_free = 1;

   /* currently there is no set array element - so I have to get the
      array, make the change and then set it */
   if (GDget_int_array(modes_id, "mode", &size, &mode) != 1) {
      mode = tmp_mode;
      size = 5;
      mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
      mode[GD_MODE_LINES] = GD_LINES_INHERIT;
      mode[GD_MODE_SURF] = GD_SURF_INHERIT;
      mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
      need_free = 0;
   }

   /* set new bounds mode */
   mode[GD_MODE_BOUNDS] = bounds;
   GDset_int_array(modes_id, "mode", size, mode);

   if (need_free)
      ARRfree(mode);
}
/* 64-bit porting. Only Modified Internally */
void GDmodes_get_mode(OMobj_id modes_id, int *mode)
{
   xp_long size;
   int *curmode;

   if (GDget_int_array(modes_id, "mode", &size, &curmode) != 1) {
      mode[GD_MODE_POINTS] = GD_POINTS_INHERIT;
      mode[GD_MODE_LINES] = GD_LINES_INHERIT;
      mode[GD_MODE_SURF] = GD_SURF_INHERIT;
      mode[GD_MODE_VOL] = GD_VOLUME_INHERIT;
      mode[GD_MODE_BOUNDS] = GD_BOUNDS_INHERIT;
      return;
   }
   memcpy(mode, curmode, size*sizeof(int));
   ARRfree(curmode);
}

void GDmodes_set_normals(OMobj_id modes_id, int normals)
{
   GDset_int_val(modes_id, "normals", normals);
}

void GDmodes_get_normals(OMobj_id modes_id, int *normals)
{
   if (GDget_int_val(modes_id, "normals", normals) != 1)
      *normals = GD_NORMALS_INHERIT;
}

void GDmodes_set_colors(OMobj_id modes_id, int colors)
{
   GDset_int_val(modes_id, "colors", colors);
}

void GDmodes_get_colors(OMobj_id modes_id, int *colors)
{
   if (GDget_int_val(modes_id, "colors", colors) != 1)
      *colors = GD_COLORS_INHERIT;
}

void GDmodes_set_outline(OMobj_id modes_id, int outline)
{
   GDset_int_val(modes_id, "outline", outline);
}

void GDmodes_get_outline(OMobj_id modes_id, int *outline)
{
   if (GDget_int_val(modes_id, "outline", outline) != 1)
      *outline = GD_OUTLINE_ON;
}

