/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/props.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/dll_in.h>
#include <avs/util.h>
#include <avs/om.h>
#include <avs/gd.h>

/* Private functions. */
static void GDprops_reset_local (GDprops *);

/*
 * reference counting add to allow multiple references to GDprops in flipbook
 */
void GDprops_ref_set(GDprops *props)
{
  if (props)
    props->ref_count = 1;
}

void GDprops_ref_inc(GDprops *props)
{
  if (props)
    props->ref_count++;
}

void GDprops_ref_dec(GDprops *props)
{
  if (props && --props->ref_count < 1)
    FREE(props);
}

/* This function is called in a "delayed" fashion via the
   GDget_local routine. When we try to get the local props
   structure pointer - if it doesn't exist the create routine
   is called. This will happen in two cases: either thru the
   props update function that is called at instance time or
   via the object attach props function that is called when the
   props is attached to the object.
*/
int GDprops_create(OMobj_id props_id)
{
   GDprops *props;

   /* allocate a properties structure and initialize it */
   ALLOCN(props, GDprops, 1, "can't allocate properties");

   /* reference new properties local structure */
   GDprops_ref_inc(props);

   /* save OM element id in local data structures */
   props->props_id = props_id;

   /* Get in sync with the OM starting values.  GDprops_update will not
    * do the trick.  At instance time the sequence number of the update
    * functions and the sequence number of each slot will be the same
    * so GDprops_update will think no updates are needed.
    */
   GDprops_get_color(        props_id, props->col );
   GDprops_get_hi1_col(      props_id, props->hi1_col );
   GDprops_get_hi2_col(      props_id, props->hi2_col );
   GDprops_get_material(     props_id, props->material );
   GDprops_get_specular_color( props_id, props->spec_col );
   GDprops_get_transparency( props_id, &props->trans );
   GDprops_get_draw_mode(    props_id, &props->draw_mode );
   GDprops_get_line_width(   props_id, &props->line_width );
   GDprops_get_line_style(   props_id, &props->line_style );
   GDprops_get_line_aa(      props_id, &props->line_aa );
   GDprops_get_dir_pt_size(  props_id, &props->dir_pt_size );
   GDprops_get_subdiv(       props_id, &props->subdiv );
   GDprops_get_jitter(       props_id, &props->jitter );
   GDprops_get_cull(         props_id, &props->cull );
   GDprops_get_vox_interp(   props_id, &props->vox_interp );
   GDprops_get_ray_algo(     props_id, &props->ray_algo );
   GDprops_get_ray_norm(     props_id, &props->ray_norm );
   GDprops_get_fat_ray(      props_id, &props->fat_ray );
   GDprops_get_sfp_absorb(   props_id, &props->sfp_absorb );
   GDprops_get_sfp_emit(     props_id, &props->sfp_emit );
   /* Note that there is no GDprops_get_font */
   GDprops_get_inherit(      props_id, &props->inherit );

   /* Save the id of the update function. This will be used to
      to add dependencies - we add dependencies between update functions.
   */
   props->upd_func = OMfind_subobj(props_id, OMstr_to_name("upd_func"), OM_OBJ_RW);

   /* Save the local pointer out in the OM. */
   GDset_local(props_id, (char *)props);
   return(1);
}

/* This routine is invoked when the database element is being deleted */
int GDprops_delete(OMobj_id props_id)
{
   GDprops *props;

   if (OMis_null_obj(props_id))
      return(1);
   props = (GDprops *)GDget_local(props_id, (OMpfi)GDprops_create);

   /* de-reference local property structure */
   GDprops_ref_dec(props);

   GDclear_local(props_id);
   return(1);
}

void GDprops_reset(OMobj_id props_id)
{
   GDprops *props;
   float col[3], hi_col[3];
   float material[4];
   float trans = GD_DEFAULT_TRANS;

   if (OMis_null_obj(props_id))
      return;

   /* init arrays to perform initialization */
   col[0] = GD_DEFAULT_RED;
   col[1] = GD_DEFAULT_GREEN;
   col[2] = GD_DEFAULT_BLUE;
   hi_col[0] = GD_DEFAULT_RED;
   hi_col[1] = 0.0;
   hi_col[2] = 0.0;
   material[0] = (float)GD_DEFAULT_AMBIENT;
   material[1] = (float)GD_DEFAULT_DIFFUSE;
   material[2] = (float)GD_DEFAULT_SPECULAR;
   material[3] = GD_DEFAULT_SPEC_EXP;

   /* initialize data base copy of properties */
   GDprops_set_color(props_id, col);
   GDprops_set_hi1_col(props_id, col);
   GDprops_set_hi2_col(props_id, hi_col);
   GDprops_set_material(props_id, material);
   GDprops_set_specular_color(props_id, col);
   GDprops_set_transparency(props_id, trans);
   GDprops_set_draw_mode(props_id, GD_DEFAULT_DRAW_MODE);
   GDprops_set_line_width(props_id, GD_DEFAULT_LINE_WIDTH);
   GDprops_set_line_style(props_id, GD_DEFAULT_LINE_STYLE);
   GDprops_set_line_aa(props_id, GD_DEFAULT_LINE_AA);
   GDprops_set_dir_pt_size(props_id, GD_DEFAULT_DIR_PT_SIZE);
   GDprops_set_subdiv(props_id, GD_DEFAULT_SUBDIV);
   GDprops_set_jitter(props_id, GD_DEFAULT_JITTER);
   GDprops_set_cull(props_id, GD_DEFAULT_CULL);
   GDprops_set_vox_interp(props_id, GD_DEFAULT_VOX_INTERP);
   GDprops_set_ray_algo(props_id, GD_DEFAULT_RAY_ALGO);
   GDprops_set_ray_norm(props_id, GD_DEFAULT_RAY_NORM);
   GDprops_set_fat_ray(props_id, GD_DEFAULT_FAT_RAY);
   GDprops_set_sfp_absorb(props_id, (float)GD_DEFAULT_SFP_ABSORB);
   GDprops_set_sfp_emit(props_id, (float)GD_DEFAULT_SFP_EMIT);
   GDprops_set_inherit(props_id, 1);

   /* initialize the local copy of the data. */
   props = (GDprops *)GDget_local(props_id, (OMpfi)GDprops_create);
   GDprops_reset_local(props);
}

/* Initialize the local copy of the data. Called by the
   create and reset routines.
*/
static void GDprops_reset_local(GDprops *props)
{
   props->col[0]      = GD_DEFAULT_RED;
   props->col[1]      = GD_DEFAULT_GREEN;
   props->col[2]      = GD_DEFAULT_BLUE;
   props->hi1_col[0]  = GD_DEFAULT_RED;
   props->hi1_col[1]  = GD_DEFAULT_GREEN;
   props->hi1_col[2]  = GD_DEFAULT_BLUE;
   props->hi2_col[0]  = GD_DEFAULT_RED;
   props->hi2_col[1]  = 0.0;
   props->hi2_col[2]  = 0.0;
   props->material[0] = (float)GD_DEFAULT_AMBIENT;
   props->material[1] = (float)GD_DEFAULT_DIFFUSE;
   props->material[2] = (float)GD_DEFAULT_SPECULAR;
   props->material[3] = GD_DEFAULT_SPEC_EXP;
   props->spec_col[0] = GD_DEFAULT_RED;
   props->spec_col[1] = GD_DEFAULT_GREEN;
   props->spec_col[2] = GD_DEFAULT_BLUE;
   props->trans       = GD_DEFAULT_TRANS;
   props->draw_mode   = GD_DEFAULT_DRAW_MODE;
   props->line_width  = GD_DEFAULT_LINE_WIDTH;
   props->line_style  = GD_DEFAULT_LINE_STYLE;
   props->line_aa     = GD_DEFAULT_LINE_AA;
   props->dir_pt_size = GD_DEFAULT_DIR_PT_SIZE;
   props->subdiv      = GD_DEFAULT_SUBDIV;
   props->jitter      = GD_DEFAULT_JITTER;
   props->cull        = GD_DEFAULT_CULL;
   props->vox_interp  = GD_DEFAULT_VOX_INTERP;
   props->ray_algo    = GD_DEFAULT_RAY_ALGO;
   props->ray_norm    = GD_DEFAULT_RAY_NORM;
   props->fat_ray     = GD_DEFAULT_FAT_RAY;
   props->sfp_absorb  = (float)GD_DEFAULT_SFP_ABSORB;
   props->sfp_emit    = (float)GD_DEFAULT_SFP_EMIT;
   props->inherit     = 1;

   props->changed       = 0;
   props->saved_changed = 0;
}

/* This routine updates the local structure from the data base.
   This occurs whenever the properties data base element is
   written. Only the data that is changed is updated - change
   is noted by checking the sequence number
*/
/* 64-bit porting. Only Modified Internally */
int GDprops_update(OMobj_id props_id)
{
   GDprops *props = NULL;
   int upd_seq;
   xp_long size;
   int tmp_int;
   float *tmp_arr;
   float tmp_val;
   char *tmp_str = NULL;

   /* get pointer to local storage, return if NULL */
   props = (GDprops *)GDget_local(props_id, (OMpfi)GDprops_create);
   if (!props)
      return(0);

   /* Set the cache valid flag. */
   props->cache_valid = 1;

   /* Get the sequence number of the update function. We can use
      this to determine if any sub-elements in props have changed.
      A mode of 0 means of the element itself.
   */
   upd_seq = OMget_obj_seq(props->upd_func, OMnull_obj, 0);

   /* update color */
   if (GDget_updfloat_array(props_id, "col", upd_seq, &size, &tmp_arr)) {
      memcpy(props->col, tmp_arr, 3*sizeof(float));
      props->changed |= GD_CHANGE_PROP_COL;
      ARRfree(tmp_arr);
   }

   /* update highlight colors - treat them together since that's
      the way they are set in the state.
   */
   if (GDget_updfloat_array(props_id, "hi1_col", upd_seq, &size, &tmp_arr)) {
      props->changed |= GD_CHANGE_PROP_HI_COL;
      memcpy(props->hi1_col, tmp_arr, 3*sizeof(float));
      ARRfree(tmp_arr);
      /* Get the second hilite color. */
      GDprops_get_hi2_col(props_id, props->hi2_col);
   }
   else if (GDget_updfloat_array(props_id, "hi2_col", upd_seq, &size, &tmp_arr)) {
      props->changed |= GD_CHANGE_PROP_HI_COL;
      memcpy(props->hi2_col, tmp_arr, 3*sizeof(float));
      ARRfree(tmp_arr);
      /* Get the first hilite color. */
      GDprops_get_hi1_col(props_id, props->hi1_col);
   }

   /* update specular color */
   if (GDget_updfloat_array(props_id, "spec_col", upd_seq, &size,
	&tmp_arr)) {
      memcpy(props->spec_col, tmp_arr, 3*sizeof(float));
      props->changed |= GD_CHANGE_PROP_SCOL;
      ARRfree(tmp_arr);
   }

   /* update material */
   if (GDget_updfloat_array(props_id, "material", upd_seq, &size,
	&tmp_arr)) {
      memcpy(props->material, tmp_arr, 4*sizeof(float));
      props->changed |= GD_CHANGE_PROP_MAT;
      ARRfree(tmp_arr);
   }

   /* update transparency */
   if (GDget_updfloat_val(props_id, "trans", upd_seq, &tmp_val)) {
      props->trans = tmp_val;
      props->changed |= GD_CHANGE_PROP_TRANS;
   }

   /* update directed points size */
   if (GDget_updfloat_val(props_id, "dir_pt_size", upd_seq, &tmp_val)) {
      props->dir_pt_size = tmp_val;
      props->changed |= GD_CHANGE_PROP_DIR_PT_SIZE;
      props->cache_valid = 0;
   }

   /* update subdivision */
   if (GDget_updint_val(props_id, "subdiv", upd_seq, &tmp_int)) {
      props->subdiv = tmp_int;
      props->changed |= GD_CHANGE_PROP_SUBDIV;
      props->cache_valid = 0;
   }

   /* update jitter */
   if (GDget_updint_val(props_id, "jitter", upd_seq, &tmp_int)) {
      props->jitter = tmp_int;
      props->changed |= GD_CHANGE_PROP_JITTER;
      props->cache_valid = 0;
   }

   /* update cull */
   if (GDget_updint_val(props_id, "cull", upd_seq, &tmp_int)) {
      props->cull = tmp_int;
      props->changed |= GD_CHANGE_PROP_CULL;
   }

   /* update vox interp */
   if (GDget_updint_val(props_id, "voxel_interp", upd_seq, &tmp_int)) {
      props->vox_interp = tmp_int;
      props->changed |= GD_CHANGE_PROP_VOX_INTERP;
   }

   /* update ray algo */
   if (GDget_updint_val(props_id, "ray_algo", upd_seq, &tmp_int)) {
      props->ray_algo = tmp_int;
      props->changed |= GD_CHANGE_PROP_RAY_ALGO;
   }

   /* update ray dist norm */
   if (GDget_updint_val(props_id, "ray_norm", upd_seq, &tmp_int)) {
      props->ray_norm = tmp_int;
      props->changed |= GD_CHANGE_PROP_RAY_NORM;
   }

   /* update fat ray */
   if (GDget_updint_val(props_id, "fat_ray", upd_seq, &tmp_int)) {
      props->fat_ray = tmp_int;
      props->changed |= GD_CHANGE_PROP_FAT_RAY;
   }

   /* update sfp absorption */
   if (GDget_updfloat_val(props_id, "sfp_absorb", upd_seq, &tmp_val)) {
      props->sfp_absorb = tmp_val;
      props->changed |= GD_CHANGE_PROP_SFP_ABSORB;
   }

   /* update sfp emission */
   if (GDget_updfloat_val(props_id, "sfp_emit", upd_seq, &tmp_val)) {
      props->sfp_emit = tmp_val;
      props->changed |= GD_CHANGE_PROP_SFP_EMIT;
   }

   /* update draw mode */
   if (GDget_updint_val(props_id, "draw_mode", upd_seq, &tmp_int)) {
      props->draw_mode = tmp_int;
      props->changed |= GD_CHANGE_PROP_DRAW_MODE;
   }

   /* update font */
   if (GDget_updstr_val(props_id, "font", upd_seq, &tmp_str)) {
      strcpy(props->font, tmp_str);
      props->changed |= GD_CHANGE_PROP_FONT;
   }
   /* The get_updstr_val routine will return the value even if
      the sequence number has not changed, so we need to free it.
   */
   if (tmp_str != NULL)
      free(tmp_str);

   /* update line width */
   if (GDget_updint_val(props_id, "line_width", upd_seq, &tmp_int)) {
      props->line_width = tmp_int;
      props->changed |= GD_CHANGE_PROP_LINE_WIDTH;
   }

   /* update line style */
   if (GDget_updint_val(props_id, "line_style", upd_seq, &tmp_int)) {
      props->line_style = tmp_int;
      props->changed |= GD_CHANGE_PROP_LINE_STYLE;
   }

   /* update line aa */
   if (GDget_updint_val(props_id, "line_aa", upd_seq, &tmp_int)) {
      props->line_aa = tmp_int;
      props->changed |= GD_CHANGE_PROP_LINE_AA;
   }

   /* update inherit */
   if (GDget_updint_val(props_id, "inherit", upd_seq, &tmp_int)) {
      props->inherit = tmp_int;
      if (props->inherit) {
	 /* Just changed to inherit from parent:
	    save existing changed & set changed to 0
         */
	 props->saved_changed = props->changed;
	 props->changed = 0;
      }
      else {
	 /* Just changed to use existing properties:
	    Restore changed (catching any changes from this
	    activation )and clear save changed.
         */
	 props->changed |= props->saved_changed;
	 props->saved_changed = 0;
      }
   }
   /* inherit hasn't changed for this object but we still need to
      pay attention to its current value
   */
   else {
      if (props->inherit) {
	 /* Update saved changed with any changes from this
	    activation & clear changed.
         */
	 props->saved_changed |= props->changed;
	 props->changed = 0;
      }
   }

   return(1);
}

void GDprops_set_font(OMobj_id props_id, char *font)
{
  OMobj_id elem_id;

  /* lookup element id of requested sub-element */
  elem_id = OMfind_subobj(props_id, OMstr_to_name("font"), OM_OBJ_RW);
  /* return if null element */
  if (OMis_null_obj(elem_id))
    return;

  OMset_str_val(elem_id, font);
}


void GDprops_set_color(OMobj_id props_id, float *col)
{
   GDset_float_array(props_id, "col", 3, col);
}

/* 64-bit porting. Only Modified Internally */
void GDprops_get_color(OMobj_id props_id, float *col)
{
   xp_long size;
   float *curcol;

   if (GDget_float_array(props_id, "col", &size, &curcol) != 1) {
      col[0] = GD_DEFAULT_RED;
      col[1] = GD_DEFAULT_BLUE;
      col[2] = GD_DEFAULT_GREEN;
      return;
   }
   memcpy(col, curcol, size*sizeof(float));
   ARRfree(curcol);
}

void GDprops_set_hi1_col(OMobj_id props_id, float *col)
{
   GDset_float_array(props_id, "hi1_col", 3, col);
}
/* 64-bit porting. Only Modified Internally */
void GDprops_get_hi1_col(OMobj_id props_id, float *col)
{
   xp_long size;
   float *curcol;

   if (GDget_float_array(props_id, "hi1_col", &size, &curcol) != 1) {
      col[0]  = GD_DEFAULT_RED;
      col[1]  = GD_DEFAULT_GREEN;
      col[2]  = GD_DEFAULT_BLUE;
      return;
   }
   memcpy(col, curcol, size*sizeof(float));
   ARRfree(curcol);
}

void GDprops_set_hi2_col(OMobj_id props_id, float *col)
{
   GDset_float_array(props_id, "hi2_col", 3, col);
}

/* 64-bit porting. Only Modified Internally */
void GDprops_get_hi2_col(OMobj_id props_id, float *col)
{
   xp_long size;
   float *curcol;

   if (GDget_float_array(props_id, "hi2_col", &size, &curcol) != 1) {
      col[0]  = GD_DEFAULT_RED;
      col[1]  = 0.0;
      col[2]  = 0.0;
      return;
   }
   memcpy(col, curcol, size*sizeof(float));
   ARRfree(curcol);
}

void GDprops_set_material(OMobj_id props_id, float *material)
{
   GDset_float_array(props_id, "material", 4, material);
}

/* 64-bit porting. Only Modified Internally */
void GDprops_get_material(OMobj_id props_id, float *material)
{
   xp_long size;
   float *curmaterial;

   if (GDget_float_array(props_id, "material", &size, &curmaterial) != 1) {
      material[0] = (float)GD_DEFAULT_AMBIENT;
      material[1] = (float)GD_DEFAULT_DIFFUSE;
      material[2] = (float)GD_DEFAULT_SPECULAR;
      material[3] = GD_DEFAULT_SPEC_EXP;
      return;
   }
   memcpy(material, curmaterial, size*sizeof(float));
   ARRfree(curmaterial);
}

void GDprops_set_specular_color(OMobj_id props_id, float *spec_col)
{
   GDset_float_array(props_id, "spec_col", 3, spec_col);
}

/* 64-bit porting. Only Modified Internally */
void GDprops_get_specular_color(OMobj_id props_id, float *spec_col)
{
   xp_long size;
   float *curspec_col;

   if (GDget_float_array(props_id, "spec_col", &size, &curspec_col) != 1) {
      spec_col[0] = GD_DEFAULT_RED;
      spec_col[1] = GD_DEFAULT_GREEN;
      spec_col[2] = GD_DEFAULT_BLUE;
      return;
   }
   memcpy(spec_col, curspec_col, size*sizeof(float));
   ARRfree(curspec_col);
}

void GDprops_set_transparency(OMobj_id props_id, float trans)
{
   GDset_float_val(props_id, "trans", trans);
}

void GDprops_get_transparency(OMobj_id props_id, float *trans)
{
   if (GDget_float_val(props_id, "trans", trans) != 1)
      *trans = GD_DEFAULT_TRANS;
}

void GDprops_set_line_width(OMobj_id props_id, int width)
{
   GDset_int_val(props_id, "line_width", width);
}

void GDprops_get_line_width(OMobj_id props_id, int *width)
{
   if (GDget_int_val(props_id, "line_width", width) != 1)
     *width = GD_DEFAULT_LINE_WIDTH;
}

void GDprops_set_line_aa(OMobj_id props_id, int aa)
{
   GDset_int_val(props_id, "line_aa", aa);
}

void GDprops_get_line_aa(OMobj_id props_id, int *aa)
{
   if (GDget_int_val(props_id, "line_aa", aa) != 1)
     *aa = GD_DEFAULT_LINE_AA;
}

void GDprops_set_draw_mode(OMobj_id props_id, int mode)
{
   GDset_int_val(props_id, "draw_mode", mode);
}

void GDprops_get_draw_mode(OMobj_id props_id, int *mode)
{
   if (GDget_int_val(props_id, "draw_mode", mode) != 1)
      *mode = GD_DEFAULT_DRAW_MODE;
}

void GDprops_set_line_style(OMobj_id props_id, int style)
{
   GDset_int_val(props_id, "line_style", style);
}

void GDprops_get_line_style(OMobj_id props_id, int *style)
{
   if (GDget_int_val(props_id, "line_style", style) != 1)
      *style = GD_DEFAULT_LINE_STYLE;
}

void GDprops_set_inherit(OMobj_id props_id, int inherit)
{
   GDset_int_val(props_id, "inherit", inherit);
}

void GDprops_get_inherit(OMobj_id props_id, int *inherit)
{
   if (GDget_int_val(props_id, "inherit", inherit) != 1)
      *inherit = 1;
}

void GDprops_set_dir_pt_size(OMobj_id props_id, float size)
{
   GDset_float_val(props_id, "dir_pt_size", size);
}

void GDprops_get_dir_pt_size(OMobj_id props_id, float *size)
{
   if (GDget_float_val(props_id, "dir_pt_size", size) != 1)
      *size = GD_DEFAULT_DIR_PT_SIZE;
}

void
GDprops_set_subdiv(OMobj_id props_id, int subdiv)
{
   GDset_int_val(props_id, "subdiv", subdiv);
}

void
GDprops_get_subdiv(OMobj_id props_id, int *subdiv)
{
   if (GDget_int_val(props_id, "subdiv", subdiv) != 1)
      *subdiv = GD_DEFAULT_SUBDIV;
}

void
GDprops_set_jitter(OMobj_id props_id, int jitter)
{
   GDset_int_val(props_id, "jitter", jitter);
}

void
GDprops_get_jitter(OMobj_id props_id, int *jitter)
{
   if (GDget_int_val(props_id, "jitter", jitter) != 1)
      *jitter = GD_DEFAULT_JITTER;
}

void
GDprops_set_cull(OMobj_id props_id, int cull)
{
   GDset_int_val(props_id, "cull", cull);
}

void
GDprops_get_cull(OMobj_id props_id, int *cull)
{
   if (GDget_int_val(props_id, "cull", cull) != 1)
      *cull = GD_DEFAULT_CULL;
}

void
GDprops_set_vox_interp(OMobj_id props_id, int vox)
{
   GDset_int_val(props_id, "voxel_interp", vox);
}

void
GDprops_get_vox_interp(OMobj_id props_id, int *vox)
{
   if (GDget_int_val(props_id, "voxel_interp", vox) != 1)
      *vox = GD_DEFAULT_VOX_INTERP;
}

void
GDprops_set_ray_algo(OMobj_id props_id, int algo)
{
   GDset_int_val(props_id, "ray_algo", algo);
}

void
GDprops_get_ray_algo(OMobj_id props_id, int *algo)
{
   if (GDget_int_val(props_id, "ray_algo", algo) != 1)
      *algo = GD_DEFAULT_RAY_ALGO;
}

void
GDprops_set_ray_norm(OMobj_id props_id, int norm)
{
   GDset_int_val(props_id, "ray_norm", norm);
}

void
GDprops_get_ray_norm(OMobj_id props_id, int *norm)
{
   if (GDget_int_val(props_id, "ray_norm", norm) != 1)
      *norm= GD_DEFAULT_RAY_NORM;
}

void
GDprops_set_fat_ray(OMobj_id props_id, int fat)
{
   GDset_int_val(props_id, "fat_ray", fat);
}

void
GDprops_get_fat_ray(OMobj_id props_id, int *fat)
{
   if (GDget_int_val(props_id, "fat_ray", fat) != 1)
      *fat = GD_DEFAULT_FAT_RAY;
}

void
GDprops_set_sfp_absorb(OMobj_id props_id, float sfp_absorb)
{
   GDset_float_val(props_id, "sfp_absorb", sfp_absorb);
}

void
GDprops_get_sfp_absorb(OMobj_id props_id, float *sfp_absorb)
{
   if (GDget_float_val(props_id, "sfp_absorb", sfp_absorb) != 1)
      *sfp_absorb = (float)GD_DEFAULT_SFP_ABSORB;
}

void
GDprops_set_sfp_emit(OMobj_id props_id, float sfp_emit)
{
   GDset_float_val(props_id, "sfp_emit", sfp_emit);
}

void
GDprops_get_sfp_emit(OMobj_id props_id, float *sfp_emit)
{
   if (GDget_float_val(props_id, "sfp_emit", sfp_emit) != 1)
      *sfp_emit = (float)GD_DEFAULT_SFP_EMIT;
}
