/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/xform.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/dll_in.h>
#include <avs/util.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/gd.h>

void GDxform_ref_set(GDxform *xform)
{
  if (xform)
    xform->ref_count = 1;
}

void GDxform_ref_inc(GDxform *xform)
{
  if (xform)
    xform->ref_count++;
}

void GDxform_ref_dec(GDxform *xform)
{
  if (xform && --xform->ref_count < 1)
    FREE(xform);
}

GDxform *GDxform_alloc(OMobj_id xform_id)
{
   GDxform *xform;

   ALLOCN(xform, GDxform, 1, "can't allocate xform");
   xform->xform_id = xform_id;

   /* Make sure the matricies are set to the identity. */
   MATmat4_identity(xform->xform);
   MATmat4_identity(xform->matrix);

   return(xform);
}

/* This routine resets the values in the OM. */
void GDxform_reset(OMobj_id xform_id)
{
   float mat[4][4], xlate[3], center[3];
   int rspace = 3;

   if (OMis_null_obj(xform_id))
      return;

   MATmat4_identity(mat);
   xlate[0] = 0.0; xlate[1] = 0.0; xlate[2] = 0.0;
   center[0] = 0.0; center[1] = 0.0; center[2] = 0.0;

   /* Make sure this stuff gets saved in the v file. */
   OMset_state_to_usr(1);
   /* Initialize copy in the OM. */
   GDset_float_array(xform_id, "mat", 16, &mat[0][0]);
   GDset_float_array(xform_id, "xlate", 3, xlate);
   GDset_float_array(xform_id, "center", 3, center);
   GDset_float_array(xform_id, "ocenter", 3, center);
   GDset_float_array(xform_id, "dcenter", 3, center);
   OMset_state_to_usr(0);
   GDset_int_val(xform_id, "rspace", rspace);
}

/* This routine is called to update the GDxform group.
   This is done whenever the group has been written in the
   database - typically by a GDxform_set or GDxform_concat
   call. This causes an event to be propagated to the
   GDobject_update routine. This routine is called to update
   the local transformation of an object if it has been
   modified. If it has, this routine reads the matrix from the
   database and loads it into local storage. */
/* 64-bit porting. Only Modified Internally */
void GDxform_update(OMobj_id xform_id, GDxform *xform)
{
   int changed_mat = 0, changed_xlate = 0, changed_cent = 0;
   xp_long size;
   int rspace;
   float *mat, *xlate, *cent, *tmpocent, *tmpdcent;
   float dcent[3], ocent[3], odcent[3];

   /* Make sure this stuff gets saved in the v file. */
   OMset_state_to_usr(1);

   if (GDget_newfloat_array(xform_id, "mat", &xform->mat_seq, &size, &mat)) {
      changed_mat = 1;
      memcpy(xform->xform, mat, size*sizeof(float));
      ARRfree(mat);
   }
   if (GDget_newfloat_array(xform_id, "xlate", &xform->xlate_seq, &size, &xlate)) {
      changed_xlate = 1;
      memcpy(xform->xlate, xlate, size*sizeof(float));
      ARRfree(xlate);
   }
   if (GDget_newfloat_array(xform_id, "center", &xform->center_seq, &size, &cent)) {
      /* Get the value of the previous center. */
      if (!GDget_float_array(xform_id, "ocenter", &size, &tmpocent)) {
         ocent[0] = 0.0; ocent[1] = 0.0; ocent[2] = 0.0;
      }
      else {
         memcpy(ocent, tmpocent, sizeof(float)*3);
         ARRfree(tmpocent);
      }
      /* Calculate change in center */
      dcent[0] = cent[0] - ocent[0];
      dcent[1] = cent[1] - ocent[1];
      dcent[2] = cent[2] - ocent[2];
      changed_cent = 1;

      /* Update both the local copy of center and the old
	 center value kept in the OM.
      */
      memcpy(xform->center, cent, size*sizeof(float));
      GDset_float_array(xform_id, "ocenter", 3, cent);
      ARRfree(cent);
   }
   if (GDget_newint_val(xform_id, "rspace", &xform->rspace_seq, &rspace)) {
      xform->rspace = rspace;
      xform->changed |= GD_CHANGE_XFORM_RSPACE;
   }

   /* Translate the center to the origin, apply rotates & scale, translate
      back to center, then apply the position vector. Done in a fashion
      so the object doesn't move if the center changes.
   */
   if (changed_mat || changed_xlate || changed_cent) {
      /* Get the value of delta center. */
      if (!GDget_float_array(xform_id, "dcenter", &size, &tmpdcent)) {
         odcent[0] = 0.0; odcent[1] = 0.0; odcent[2] = 0.0;
      }
      else {
         memcpy(odcent, tmpdcent, sizeof(float)*3);
         ARRfree(tmpdcent);
      }

      if (changed_cent) {
         MATvec3_mat4_multiply(dcent, xform->xform);
	 odcent[0] += dcent[0];
	 odcent[1] += dcent[1];
	 odcent[2] += dcent[2];

         /* Update both the local copy of delta center and the 
	    delta center value kept in the OM.
         */
         memcpy(xform->dcenter, odcent, size*sizeof(float));
         GDset_float_array(xform_id, "dcenter", 3, odcent);
      }

      MATmat4_translate(xform->matrix, -xform->center[0], -xform->center[1], -xform->center[2]);
      MATmat4_multiply(xform->matrix, xform->xform, xform->matrix);
      xform->matrix[3][0] += xform->xlate[0] + odcent[0];
      xform->matrix[3][1] += xform->xlate[1] + odcent[1];
      xform->matrix[3][2] += xform->xlate[2] + odcent[2];
   }

   /* Restore OM save flags */
   OMset_state_to_usr(0);
}

/* This routine sets the transformation matrix in the data base */
void GDxform_set_matrix(OMobj_id xform_id, float mat[4][4])
{
   GDset_float_array(xform_id, "mat", 16, &mat[0][0]);
}

/* This routine concatenates the matrix with the existing matrix
   in the data base. This routine assumes the matrix does not
   contain any translation information. */
/* 64-bit porting. Only Modified Internally */
void GDxform_concat_matrix(OMobj_id xform_id, float mat[4][4])
{
   float *oldmat;
   float newmat[4][4];
   xp_long size;

   /* Go get the current matrix */
   if (!GDget_float_array(xform_id, "mat", &size, &oldmat)) {
      /* just set passed matrix if we fail */
      GDset_float_array(xform_id, "mat", 16, &mat[0][0]);
   }
   else {
      /* post-concatenate the new matrix */
      MATmat4_multiply((Matr4 *)oldmat, mat, newmat);
      /* put new matrix into database */
      GDset_float_array(xform_id, "mat", 16, &newmat[0][0]);

      ARRfree(oldmat);
   }
}

/* This routine gets the transformation matrix from the database */
/* 64-bit porting. Only Modified Internally */
void GDxform_get_matrix(OMobj_id xform_id, float *mat)
{
   xp_long size;
   float *curmat;

   if (!GDget_float_array(xform_id, "mat", &size, &curmat))
      MATmat4_identity((Matr4 *)mat);
   else {
      memcpy(mat, curmat, size*sizeof(float));
      ARRfree(curmat);
   }

}

/* This routine gets the composite transformation matrix 
   which is calculated using center, xform and xlate
   from the database.
   It seems like this routine ought to be able
   to just lookup the local xform information and
   return the composite matrix stored there. But this
   assumes that xform_update runs before any module
   that gets activated because the GDxform changed.
   At this point, this is not a valid assumption
   to make.
*/
/* 64-bit porting. Only Modified Internally */
void GDxform_get_comp_matrix(OMobj_id xform_id, float *mat)
{
   xp_long size;
   float *matrix, *center, *dcenter, *xlate;
   float tmpmat[4][4], tmpcent[3], tmpdcent[3], tmpxlate[3];
   float resmat[4][4];

   /* Get current values from OM.  If we can't get 
      any use a set of default values.
   */
   if (!GDget_float_array(xform_id, "mat", &size, &matrix)) {
      MATmat4_identity(tmpmat);
   }
   else {
      memcpy(tmpmat,matrix,sizeof(float)*16);
      ARRfree(matrix);
   }
   if (!GDget_float_array(xform_id, "center", &size, &center)) {
      tmpcent[0] = 0.0; tmpcent[1] = 0.0; tmpcent[2] = 0.0;
   }
   else {
      memcpy(tmpcent,center,sizeof(float)*3);
      ARRfree(center);
   }
   if (!GDget_float_array(xform_id, "dcenter", &size, &dcenter)) {
      tmpdcent[0] = 0.0; tmpdcent[1] = 0.0; tmpdcent[2] = 0.0;
   }
   else {
      memcpy(tmpdcent,dcenter,sizeof(float)*3);
      ARRfree(dcenter);
   }

   if (!GDget_float_array(xform_id, "xlate", &size, &xlate)) {
      tmpxlate[0] = 0.0; tmpxlate[1] = 0.0; tmpxlate[2] = 0.0;
   }
   else {
      memcpy(tmpxlate,xlate,sizeof(float)*3);
      ARRfree(xlate);
   }

   /* Translate the center to the origin, apply rotates & scale, 
      translate back to center, then apply the position vector. 
      Done in a fashion so the object doesn't move if the center changes.
   */
   MATmat4_translate(resmat, -tmpcent[0], -tmpcent[1], -tmpcent[2]);
   MATmat4_multiply(resmat, tmpmat, resmat);
   resmat[3][0] += tmpxlate[0] + tmpdcent[0];
   resmat[3][1] += tmpxlate[1] + tmpdcent[1];
   resmat[3][2] += tmpxlate[2] + tmpdcent[2];

   /* return composite matrix */
   memcpy(mat, resmat, 16*sizeof(float));
}

void GDxform_set_xlate(OMobj_id xform_id, float xlate[3])
{
   GDset_float_array(xform_id, "xlate", 3, xlate);
}

/* This routine concatenates the matrix with the existing matrix
   in the data base. This routine assumes the matrix does not
   contain any translation information. */
/* 64-bit porting. Only Modified Internally */
void GDxform_concat_xlate(OMobj_id xform_id, float xlate[3])
{
   float *oldxlate;
   xp_long size;

   /* go get the current translate */
   if (GDget_float_array(xform_id, "xlate", &size, &oldxlate)) {
      /* add the new translate */
      xlate[0] += oldxlate[0];
      xlate[1] += oldxlate[1];
      xlate[2] += oldxlate[2];

      ARRfree(oldxlate);
   }
   /* put new xlate into database */
   GDset_float_array(xform_id, "xlate", 3, xlate);
}

/* 64-bit porting. Only Modified Internally */
void GDxform_get_xlate(OMobj_id xform_id, float *xlate)
{
   xp_long size;
   float *curxlate;

   if (!GDget_float_array(xform_id, "xlate", &size, &curxlate)) {
      xlate[0] = 0.0; xlate[1] = 0.0; xlate[2] = 0.0;
   }
   else {
      memcpy(xlate, curxlate, size*sizeof(float));
      ARRfree(curxlate);
   }
}

void GDxform_set_center(OMobj_id xform_id, float center[3])
{
   GDset_float_array(xform_id, "center", 3, center);
}

/* 64-bit porting. Only Modified Internally */
void GDxform_concat_center(OMobj_id xform_id, float center[3])
{
   float *oldcent;
   xp_long size;

   /* go get the current center */
   if (GDget_float_array(xform_id, "center", &size, &oldcent)) {
      /* add the new center */
      center[0] += oldcent[0];
      center[1] += oldcent[1];
      center[2] += oldcent[2];

      ARRfree(oldcent);
   }
   /* put new center into database */
   GDset_float_array(xform_id, "center", 3, center);
}

/* 64-bit porting. Only Modified Internally */
void GDxform_get_center(OMobj_id xform_id, float *center)
{
   xp_long size;
   float *curcenter;

   if (!GDget_float_array(xform_id, "center", &size, &curcenter)) {
      center[0] = 0.0; center[1] = 0.0; center[2] = 0.0;
   }
   else {
      memcpy(center, curcenter, size*sizeof(float));
      ARRfree(curcenter);
   }
}
