/*
        This file is under Perforce control
        $Id: //depot/express/fcs70/hdf5/examp/write_cells.c#1 $

Example Unix command line to compile (if HDF5 is installed in /usr/local):
cc write_cells.c -I/usr/local/include /usr/local/lib/libhdf5.a -lm -lz
*/

#include <stdio.h>

#ifdef WIN32
/* #define _HDF5USEDLL_ */ /* Define this if using HDF5 DLL */
#endif
#include <hdf5.h>

static void write_int_attr( hid_t parent_id, const char *name, int value )
{
    hid_t h5_dspace_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if( h5_dspace_id > 0 ) {
        h5_attr_id = H5Acreate( parent_id, name, H5T_NATIVE_INT_g,
                                h5_dspace_id, H5P_DEFAULT );
        if (h5_attr_id > 0) {
            H5Awrite( h5_attr_id, H5T_NATIVE_INT_g, &value );
            H5Aclose( h5_attr_id );	/* close attribute */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

static void write_string_attr( hid_t parent_id, const char *name,
                               const char *value )
{
    hid_t h5_dspace_id, h5_dtype_id, h5_attr_id;

    h5_dspace_id = H5Screate( H5S_SCALAR );
    if( h5_dspace_id > 0 ) {
        h5_dtype_id   = H5Tcopy( H5T_C_S1_g );
        if( h5_dtype_id > 0 ) {
            H5Tset_size( h5_dtype_id, strlen(value)+1 );
            h5_attr_id = H5Acreate( parent_id, name,
                                    h5_dtype_id, h5_dspace_id, H5P_DEFAULT );
            if( h5_attr_id > 0 ) {
                H5Awrite( h5_attr_id, h5_dtype_id, (char *)value );
                H5Aclose( h5_attr_id );	/* close attribute */
            }
            H5Tclose( h5_dtype_id );	/* close datatype */
        }
        H5Sclose( h5_dspace_id );	/* close dataspace */
    }
    return;
}

int main( int argc, char *argv[] )
{
    hid_t h5_file_id, h5_root_id, h5_crd_id, h5_cset_id, h5_cdat_id;
    hid_t h5_dspace_id, h5_attr_id, h5_dset_id;
    hsize_t h5_dims[3];

    char *filename;

    int fld_nspace = 2;
    int fld_ndim   = 2;
    int fld_nnodes = 12;
    /* fld_points[fld_nnodes][fld_nspace] */
    float fld_coords[12][2] = { { 0.0, 0.0 },
                                { 1.0, 0.0 },
                                { 2.0, 0.0 },
                                { 3.5, 0.0 },
                                { 0.0, 1.0 }, 
                                { 1.0, 1.0 }, 
                                { 2.0, 1.0 }, 
                                { 3.0, 1.0 }, 
                                { 0.0, 2.0 },
                                { 1.0, 2.0 },
                                { 2.0, 2.0 },
                                { 3.0, 2.5 } };

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        fprintf( stderr, "Could not initialize HDF5 library." );
        return -1;
    }

    if( argc > 1 )
        filename = argv[1];
    else
        return -1;

    /* Create a new HDF5 file using H5_ACC_TRUNC access, default
     * file creation properties, and default file access properties.
     */
    h5_file_id = H5Fcreate( filename,
                            H5F_ACC_TRUNC,  /* overwrite existing files */
                            H5P_DEFAULT,    /* default file creation    */
                            H5P_DEFAULT     /* default file access      */
                            );
    if( h5_file_id < 0 ) return -1;

    /* Grab handle to the HDF5 root group, which exists by default
     * and does not have to be created.
     */
    h5_root_id = H5Gopen( h5_file_id, "/" );
    if( h5_root_id < 0 ) {
        H5Fclose( h5_file_id );
        return -1;
    }

    /* Optional, but recommended.
     */
    write_string_attr( h5_root_id, "XP_CLASS", "Mesh+Cell_Data" );

    /* nnodes is required for an unstructured field. */
    write_int_attr( h5_root_id, "nnodes", fld_nnodes );

    /* 2D field, no ndims for an unstructured field. */
    write_int_attr( h5_root_id, "nspace", fld_nspace );


    /*
     * coordinates.
     */

    /* Create a group to hold the coordinates */
    h5_crd_id = H5Gcreate( h5_root_id, "coordinates", 0 );
    if( h5_crd_id < 0 ) {
        H5Gclose( h5_root_id );
        H5Fclose( h5_file_id );
        return -1;
    }
    else {
        h5_dims[0] = fld_nnodes;
        h5_dims[1] = fld_nspace;  /* veclen of coordinates */

        /* Create a simple 3D dataspace */
        h5_dspace_id = H5Screate_simple( 2, h5_dims, NULL );
        if( h5_dspace_id > 0 ) {
            /* Create the dataset for coordinates.values.
             */
            h5_dset_id = H5Dcreate( h5_crd_id, "values",
                                    H5T_NATIVE_FLOAT_g,
                                    h5_dspace_id,  H5P_DEFAULT );
            if( h5_dset_id > 0 ) {
                H5Dwrite( h5_dset_id, H5T_NATIVE_FLOAT,
                          h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                          fld_coords );
                H5Dclose( h5_dset_id );	/* close dataset */
            }
            H5Sclose( h5_dspace_id );	/* close dataspace */    
        }
        H5Gclose( h5_crd_id );   /* close coordinates group */
    }

    /*
     * cells.
     */

    /* Required for unstructured fields. */
    write_int_attr( h5_root_id, "ncell_sets", 1 );

    /* Create a group to hold the cell set.
     * The next cell set would be "cell_set[1]", etc.
     */
    h5_cset_id = H5Gcreate( h5_root_id, "cell_set[0]", 0 );
    if( h5_cset_id > 0 ) {
        int conn_list[] = { 0,1,5,4,1,2, 6,5,2,3, 7, 6,
                            4,5,9,8,5,6,10,9,6,7,11,10 };
        int cset_ncells = 6;   /* number of cells in this cell set */ 
        int cset_nnodes = 4;   /* property of the cell type */
        int cset_cell_data[6] = { 0,  1,  2,  3,  4,  5};

        write_int_attr( h5_cset_id, "ncells", cset_ncells );

        write_string_attr( h5_cset_id, "cell_name", "Quad" );

        h5_dims[0] = cset_ncells;
        h5_dims[1] = cset_nnodes;
        h5_dspace_id = H5Screate_simple( 2, h5_dims, NULL );
        if( h5_dspace_id > 0 ) {
            h5_dset_id   = H5Dcreate( h5_cset_id, "node_connect_list",
                                      H5T_NATIVE_INT_g, h5_dspace_id,
                                      H5P_DEFAULT );
            if( h5_dset_id > 0 ) {
                H5Dwrite( h5_dset_id, H5T_NATIVE_INT_g,
                          h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                          conn_list );
                H5Dclose( h5_dset_id );
            }
            H5Sclose( h5_dspace_id );
        }

        /* One cell data component. */
        write_int_attr( h5_cset_id, "ncell_data", 1 );

        /* Create a group to hold the cell data component.
         * An additional cell data component would be "cell_data[1]", etc.
         */
        h5_cdat_id = H5Gcreate( h5_cset_id, "cell_data[0]", 0 );
        if( h5_cdat_id > 0 ) {
            /* Optional, but recommended.  The reader will default the
             * veclen to 1 if it can't find it in the HDF5 file.
             */
            write_int_attr( h5_cdat_id, "veclen", 1 );

            /* Optional, but recommended since the labels
             * are used in the UI.
             */
            write_string_attr( h5_cdat_id, "labels", "velocity" );

            /* Optional */
            write_string_attr( h5_cdat_id, "units", "furlongs per fortnight" );

            /*
             *  cell data values
             */

            /* Create a simple 1D dataspace. */
            h5_dims[0] = cset_ncells;
            /* veclen of 1 should be omitted from the dataspace dimensions */
            h5_dspace_id = H5Screate_simple( 1, h5_dims, NULL );
            if( h5_dspace_id > 0 ) {
                /* Create the dataset for cell_data[0].values, remember
                 * its a child of the cell data group, not the cell set group.
                 */
                h5_dset_id = H5Dcreate( h5_cdat_id, "values",
                                        H5T_NATIVE_INT_g,
                                        h5_dspace_id,  H5P_DEFAULT );
                if( h5_dset_id > 0 ) {
                    H5Dwrite( h5_dset_id, H5T_NATIVE_INT,
                              h5_dspace_id, h5_dspace_id, H5P_DEFAULT,
                              cset_cell_data );
                    H5Dclose( h5_dset_id );	/* close dataset */
                }
                H5Sclose( h5_dspace_id );	/* close dataspace */    
            }
            H5Gclose( h5_cdat_id );	/* close cell data group */
        }
        H5Gclose( h5_cset_id );	/* close cell set group */
    }

    H5Gclose( h5_root_id ); /* close root group */

    H5Fclose( h5_file_id ); /* close file */

    return 0;
}
