/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/hdf5/xp_mods/h5_list_vars.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_type.h>
#include <avs/fld.h>

#include "h5_xp.h"

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("HDF5 list vars", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#define INVALID_TIME_STEP -1

typedef struct field_info {
    int ndim;
    xp_long *dims;
    int n_nd;
    char **node_vars;
    int n_cd;
    char **cell_vars;
    int nsteps;
} field_info_t;

int HDF5list_vars_update( OMobj_id, OMevent_mask, int );

static int HDF5list_vars( hid_t h5_root_id, field_info_t *info );

int HDF5list_vars_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id;
    OMobj_id ndim_id, dims_id, nd_vars_id, cd_vars_id, ts_id;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    char *root_str = NULL;

    field_info_t f_info;
    int i;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    /* out */
    ndim_id    = OMfind_subobj( mod_id, OMstr_to_name("ndim"), OM_OBJ_RW );
    dims_id    = OMfind_subobj( mod_id, OMstr_to_name("dims"), OM_OBJ_RW );
    nd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("nodeVars"),
                                OM_OBJ_RW );
    cd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("cellVars"),
                                OM_OBJ_RW );
    ts_id      = OMfind_subobj( mod_id, OMstr_to_name("timeSteps"),
                                OM_OBJ_RW );

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Open a existing HDF5 file.
     */
    h5_file_id = H5Fopen( filename,
                          H5F_ACC_RDONLY, /* read-only access to file */
                          H5P_DEFAULT );  /* default file access      */

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't open input file" );
    }

    /* Grab root group */
    if( OMget_name_str_val( mod_id, OMstr_to_name("h5root"), &root_str, 0 )
        == OM_STAT_SUCCESS ) {
        h5_root_id = H5Gopen( h5_file_id, root_str );
        free( root_str );
    }
    else
        h5_root_id = H5Gopen(h5_file_id, "/");

    HDF5list_vars( h5_root_id, &f_info );

    H5Gclose( h5_root_id );

    /* Set module outputs */

    FLDset_ndim( mod_id, f_info.ndim );
    if( f_info.dims != NULL ) {
        FLDset_dims( mod_id, f_info.dims );
        free( f_info.dims );
    }

#if 0
    fprintf( stderr, "HDF5_list_vars node_size:%d, cell_size:%d\n",
             f_info.n_nd, f_info.n_cd );
#endif

    OMset_array_size( nd_vars_id, f_info.n_nd );

    if( f_info.n_nd == 0 ) {
        OMobj_id tmp_id;
        /* Sheesh, why is this so hard?  Setting the array size to 0
         * does *not* blow away the old value!
         */
        /* Skip over any links first */
        if( OMget_obj_val_user( nd_vars_id, &tmp_id ) == OM_STAT_SUCCESS )
            if( OMobj_is_type_of( tmp_id, OM_type_prim ) )
                OMset_obj_val( tmp_id, OMnull_obj, 0 );
    }
    else {
        for( i = 0; i < f_info.n_nd; ++i ) {
            OMset_str_array_val( nd_vars_id, i, f_info.node_vars[i] );
            if( f_info.node_vars[i] ) free( f_info.node_vars[i] );
        }
    }
    if( f_info.node_vars ) free( f_info.node_vars );

    OMset_array_size( cd_vars_id, f_info.n_cd );
    if( f_info.n_cd == 0 ) {
        OMobj_id tmp_id;
        if( OMget_obj_val_user( cd_vars_id, &tmp_id ) == OM_STAT_SUCCESS )
            if( OMobj_is_type_of( tmp_id, OM_type_prim ) )
                OMset_obj_val( tmp_id, OMnull_obj, 0 );
    }
    else {
        for( i = 0; i < f_info.n_cd; ++i ) {
            OMset_str_array_val( cd_vars_id, i, f_info.cell_vars[i] );
            if( f_info.cell_vars[i] ) free( f_info.cell_vars[i] );
        }
    }
    if( f_info.cell_vars ) free( f_info.cell_vars );

    OMset_int_val( ts_id, f_info.nsteps );

    /* Output trigger */
    OMset_name_int_val( mod_id, OMstr_to_name("do"), 1 );

    h5_stat = H5Fclose( h5_file_id );
    if( h5_stat < 0 ) {
        ERRerror( "HDF5 list vars", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int HDF5list_vars( hid_t h5_root, field_info_t *info )
{
    hid_t h5_grp;
    char buf1[64], buf2[64];
    int i;

    int fld_ndim, fld_ncomp, fld_nsets, cset_ncomp;
    xp_long  *fld_dims;
    int fld_nsteps;

    if( (probe_attribute( h5_root, "nsteps", 1 ) > 0) ) {
        HDF5read_int_attr( h5_root, "nsteps", &fld_nsteps );
        info->nsteps = fld_nsteps;
    }
    else info->nsteps = INVALID_TIME_STEP;

    if( (probe_attribute( h5_root, "ndim", 1 ) > 0) &&
        (probe_attribute( h5_root, "dims", 1 ) > 0) ) {
        HDF5read_int_attr( h5_root, "ndim", &fld_ndim );
        HDF5read_dims( h5_root, &fld_dims );
        info->ndim = fld_ndim;
        info->dims = fld_dims;
    }
    else {
        info->ndim = 0;
        info->dims = NULL;
    }

    if( probe_attribute( h5_root, "nnode_data", 1 ) > 0 ) {
        HDF5read_int_attr( h5_root, "nnode_data", &fld_ncomp );

        info->n_nd = fld_ncomp;
        if( fld_ncomp > 0 )
            info->node_vars = malloc( fld_ncomp * sizeof(char *) );
        else
            info->node_vars = NULL;

        for( i=0; i<fld_ncomp; ++i ) {
            sprintf( buf1, "node_data[%d]", i );
            sprintf( buf2, "time_node_data[0]/node_data[%d]", i );
            if( ((h5_grp = probe_group( h5_root, buf1, 0)) > 0) ||
                ((h5_grp = probe_group( h5_root, buf2, 0)) > 0)) {
                if( probe_attribute( h5_grp, "labels", 1 ) > 0 ) {
                    char *str = NULL;
                    HDF5read_string_attr( h5_grp, "labels", &str );
                    info->node_vars[i] = str;
                }
                else {
                    sprintf( buf1, "node_data_%d", i );
                    info->node_vars[i] = strdup( buf1 );
                }
                H5Gclose( h5_grp );
            }
            else {
                /* Trouble.  At least get the ptr into a known state. */
                info->node_vars[i] = NULL;
            }
        }
    }
    else {
        info->n_nd = 0;
        info->node_vars = NULL;
    }


    info->n_cd = 0;
    info->cell_vars = NULL;

    if( probe_attribute( h5_root, "ncell_sets", 1 ) > 0 ) {

        HDF5read_int_attr( h5_root, "ncell_sets", &fld_nsets );
        if( fld_nsets > 0 ) {
            hid_t h5_cdat;
            /* Look for cell_set[0] */
            if( (h5_grp = probe_group(h5_root, "cell_set[0]", 0)) > 0 ) {
                if( probe_attribute( h5_grp, "ncell_data", 1 ) > 0 ) {

                    HDF5read_int_attr( h5_grp, "ncell_data", &cset_ncomp );
                    info->n_cd = cset_ncomp;
                    if( cset_ncomp > 0 )
                        info->cell_vars = malloc( cset_ncomp * sizeof(char *) );
                    else
                        info->cell_vars = NULL;

                    for( i=0; i<cset_ncomp; ++i ) {
                        /* For each cell data component,
                         * look for a group called cell_data[n].
                         */
                        sprintf( buf1, "cell_data[%d]", i );
                        sprintf( buf2, "time_cell_data[0]/cell_data[%d]", i );
                        if( ((h5_cdat = probe_group( h5_grp, buf1, 0)) > 0) ||
                            ((h5_cdat = probe_group( h5_grp, buf2, 0)) > 0) ) {
                            if( probe_attribute( h5_cdat, "labels", 1 ) > 0 ) {
                                char *s = NULL;
                                HDF5read_string_attr( h5_cdat, "labels", &s );
                                info->cell_vars[i] = s;
                            }
                            else {
                                sprintf( buf1, "cell_data_%d", i );
                                info->cell_vars[i] = strdup( buf1 );
                            }
                            H5Gclose( h5_cdat );
                        }
                        else {
                            /* Trouble.  Get the ptr into a known state. */
                            info->cell_vars[i] = NULL;
                        }
                    }
                }
                H5Gclose( h5_grp );
            }
        }
    }

    return METHOD_SUCCESS;
}
