/*
			Copyright (c) 2002 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.
	This file is under Perforce control
	$Id: //depot/express/fcs70/hdf5/xp_mods/h5_rd_fld.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_type.h>
#include <avs/fld.h>

#include "h5_xp.h"

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("HDF5 read field", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#define INVALID_TIME_STEP -1

typedef enum { FLD_Unspecified = 0,
               FLD_UNIF,
               FLD_RECT,
               FLD_STRUCT,
               FLD_UNSTRUCT
} FLD_Type;

typedef struct vars_info {
    int n_node_vars;
    int *node_vars;
    int n_cell_vars;
    int *cell_vars;
} vars_info_t;

typedef struct subset_info {
    int ndim;
    xp_long min[OM_ARRAY_MAXDIM];
    xp_long max[OM_ARRAY_MAXDIM];
    int stride[OM_ARRAY_MAXDIM];
    xp_long count[OM_ARRAY_MAXDIM]; /* New dimensions */
    xp_long orig[OM_ARRAY_MAXDIM];  /* Original dimensions */
} subset_info_t;

int HDF5read_field_update( OMobj_id, OMevent_mask, int );
int HDF5read_time_field_update( OMobj_id, OMevent_mask, int );

static int HDF5read_field( hid_t h5_root_id, OMobj_id parent_id, int step,
                           vars_info_t *vars, subset_info_t *subset,
                           OMobj_id *out_fld_ptr );

static int HDF5read_time_field( hid_t h5_root_id, OMobj_id parent_id,
                                vars_info_t *vars, subset_info_t *subset,
                                OMobj_id *out_fld_ptr );

static void HDF5read_data_array( hid_t h5_grp_id, OMobj_id data_array_id,
                                 subset_info_t *subset );

static int HDF5set_dims( OMobj_id fld_id, int fld_type,
                         int fld_ndim, xp_long *fld_dims );

static int HDF5read_coords( hid_t h5_root_id, OMobj_id fld_id, int fld_type,
                            subset_info_t *subset );

static int HDF5read_points( hid_t h5_root_id, OMobj_id fld_id, int fld_type,
                            subset_info_t *subset );

static int HDF5read_xform( hid_t h5_root_id, OMobj_id fld_id );

static int HDF5read_cells( hid_t h5_root_id, OMobj_id fld_id );

static int HDF5read_node_data( hid_t h5_root_id, OMobj_id fld_id,
                               vars_info_t *vars, subset_info_t *subset );

static int HDF5read_cell_data( hid_t h5_root_id, OMobj_id fld_id,
                               vars_info_t *vars );

static int HDF5read_time_cell_data( hid_t h5_root_id, OMobj_id fld_id,
                                    int time_step, vars_info_t *vars );


/* 64-bit porting. Only Modified Internally */
int
HDF5read_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, out_id, ts_id, fld_id;
    OMobj_id nd_vars_id, cd_vars_id, min_id, max_id, stride_id;
    vars_info_t vars;
    subset_info_t subset;
    int time_step, copy_flag;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    char *root_str = NULL;

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );

    nd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("nodeVars"),
                                OM_OBJ_RD );
    cd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("cellVars"),
                                OM_OBJ_RD );

    min_id  = OMfind_subobj( mod_id, OMstr_to_name("min"), OM_OBJ_RD );
    max_id  = OMfind_subobj( mod_id, OMstr_to_name("max"), OM_OBJ_RD );
    stride_id  = OMfind_subobj( mod_id, OMstr_to_name("factor"), OM_OBJ_RD );

    ts_id = OMfind_subobj( mod_id, OMstr_to_name("timeStep"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* out */
    out_id  = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RW );

    copy_flag = 0;
    fld_id = OMnull_obj;
    if( OMget_name_int_val( mod_id, OMstr_to_name("copyIntoExisting"), &copy_flag ) == OM_STAT_SUCCESS ) {
        OMobj_id tmp_id;
        if( copy_flag != 0 && OMget_obj_val( out_id, &tmp_id ) == OM_STAT_SUCCESS ) {
            /* Assume outFld is connected to something useful */
            fld_id = out_id;
            out_id = OMnull_obj;
        }
    }

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) &&
            !OMchanged( trigger_id, seq_num ) &&
            !OMchanged( ts_id, seq_num ) ) /* OK to fire if only time-step changes */
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Open a existing HDF5 file.
     */
    h5_file_id = H5Fopen( filename,
                          H5F_ACC_RDONLY, /* read-only access to file */
                          H5P_DEFAULT );  /* default file access      */

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't open input file" );
    }

    /* Grab root group */
    if( OMget_name_str_val( mod_id, OMstr_to_name("h5root"), &root_str, 0 )
        == OM_STAT_SUCCESS ) {
        h5_root_id = H5Gopen( h5_file_id, root_str );
        free( root_str );
    }
    else
        h5_root_id = H5Gopen(h5_file_id, "/");

    if( OMget_int_val( ts_id, &time_step ) != OM_STAT_SUCCESS )
        time_step = 0;

    /* Get vars information */
    {
        xp_long array_size;
        int *array;

        if( OMget_array_size( nd_vars_id, &array_size ) == OM_STAT_SUCCESS ) {
            array = OMret_typed_array_ptr( nd_vars_id, OM_GET_ARRAY_RD,
                                           DTYPE_INT, &array_size );
            if( array != NULL ) {
                vars.n_node_vars = (int)array_size;
                vars.node_vars   = array;
                /* Do not ARRfree array here (see below) */
            }
            /* node_vars array is 0 length, don't read any */
            else {
                vars.n_node_vars = 0;
                vars.node_vars   = NULL;
            }
        }
        /* node_vars array is unset, read them all */
        else {
            vars.n_node_vars = -1; /* means read them all */
            vars.node_vars   = NULL;
        }

        if( OMget_array_size( cd_vars_id, &array_size ) == OM_STAT_SUCCESS ) {
            array = OMret_typed_array_ptr( cd_vars_id, OM_GET_ARRAY_RD,
                                           DTYPE_INT, &array_size );
            if( array != NULL ) {
                vars.n_cell_vars = (int)array_size;
                vars.cell_vars   = array;
                /* Do not ARRfree array here  (see below) */
            }
            /* cell_vars array is 0 length, don't read any */
            else {
                vars.n_cell_vars = 0;
                vars.cell_vars   = NULL;
            }
        }
        /* cell_vars array is unset, read them all */
        else {
            vars.n_cell_vars = -1; /* means read them all */
            vars.cell_vars   = NULL;
        }
    }

    /* Get subset information */
    {
        int i, *array;
        xp_long *larray, array_size;

        subset.ndim = -1;
        for( i = 0; i < OM_ARRAY_MAXDIM; ++i ) {
            subset.orig[i] = -1;
            subset.min[i] = 0;
            subset.max[i] = -1;
            subset.stride[i] = 1;
        }

        larray = OMret_typed_array_ptr( min_id, OM_GET_ARRAY_RD, DTYPE_LONG,
                                       &array_size );

        if( larray != NULL ) {
            if( array_size > OM_ARRAY_MAXDIM ) array_size = OM_ARRAY_MAXDIM;
            subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.min[i] = larray[i];
            ARRfree( larray );
        }
        larray = OMret_typed_array_ptr( max_id, OM_GET_ARRAY_RD, DTYPE_LONG,
                                       &array_size );
        if( larray != NULL ) {
            if( array_size > OM_ARRAY_MAXDIM ) array_size = OM_ARRAY_MAXDIM;
            if( subset.ndim < (int)array_size ) subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.max[i] = larray[i];
            ARRfree( larray );
        }
        array = OMret_typed_array_ptr( stride_id, OM_GET_ARRAY_RD, DTYPE_INT,
                                       &array_size );
        if( array != NULL ) {
            if( array_size > OM_ARRAY_MAXDIM ) array_size = OM_ARRAY_MAXDIM;
            if( subset.ndim < (int)array_size ) subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.stride[i] = array[i];
            ARRfree( array );
        }
    }

    if( HDF5read_field( h5_root_id, mod_id, time_step, &vars, &subset,
                        &fld_id ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while reading file" );
    }

    if( vars.node_vars ) ARRfree( vars.node_vars );
    if( vars.cell_vars ) ARRfree( vars.cell_vars );

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( !OMis_null_obj(out_id) && !OMis_null_obj(fld_id) ) {
        if( OMset_obj_ref_mode( out_id, OM_OBJ_REF ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not set outFld reference mode" );
        }
        if( OMset_obj_ref( out_id, fld_id, 0 ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not set outFld reference to Output_Field" );
        }
    }

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 read_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
HDF5read_time_field_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, trigger_id, fld_id;
    OMobj_id out_all_id;
    OMobj_id nd_vars_id, cd_vars_id, min_id, max_id, stride_id;
    vars_info_t vars;
    subset_info_t subset;

    hid_t h5_file_id, h5_root_id;
    herr_t h5_stat;

    char *temp_str = NULL, *filename = NULL;
    char file_buf[AVS_PATH_MAX];
    /* char *root_str = NULL; */

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );
    nd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("nodeVars"),
                                OM_OBJ_RD );
    cd_vars_id = OMfind_subobj( mod_id, OMstr_to_name("cellVars"),
                                OM_OBJ_RD );

    min_id  = OMfind_subobj( mod_id, OMstr_to_name("min"), OM_OBJ_RD );
    max_id  = OMfind_subobj( mod_id, OMstr_to_name("max"), OM_OBJ_RD );
    stride_id  = OMfind_subobj( mod_id, OMstr_to_name("factor"), OM_OBJ_RD );

    trigger_id  = OMfind_subobj( mod_id, OMstr_to_name("trigger"),
                                 OM_OBJ_RD );

    /* out */

    out_all_id  = OMfind_subobj( mod_id, OMstr_to_name("outFld"),
                                 OM_OBJ_RW );

    /* OK if missing entirely, means we are running in non-UI mode. */
    if( !OMis_null_obj( trigger_id ) ) {
        if( !(OM_EVENT_INST & mask) && !OMchanged( trigger_id, seq_num ) )
            return METHOD_SUCCESS;
    }

    if( OMget_str_val( file_id, &temp_str, 0 ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if (temp_str) { free (temp_str); temp_str = NULL; }

    if (!filename) {
        ERR_RETURN( "Bad filename" );
    }

    /* Initialize HDF5 library.  OK to call multiple times. */
    if( H5open() < 0 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }
    /* Detect a problem with global variables and DLL settings. */
    if( H5T_NATIVE_INT_g == 0 || H5T_NATIVE_INT_g == -1 ) {
        ERR_RETURN( "Could not initialize HDF5 library." );
    }

    /*
     * Open a existing HDF5 file.
     */
    h5_file_id = H5Fopen( filename,
                          H5F_ACC_RDONLY,  /* read-only access to file */
                          H5P_DEFAULT      /* default file access      */
                          );

    if( h5_file_id < 0 ) {
        ERR_RETURN( "Couldn't open input file" );
    }

    /* Grab root group */
    h5_root_id = H5Gopen(h5_file_id, "/");


    /* Get vars information */
    {
        int *array;
        xp_long array_size;

        if( OMget_array_size( nd_vars_id, &array_size ) == OM_STAT_SUCCESS ) {
            array = OMret_typed_array_ptr( nd_vars_id, OM_GET_ARRAY_RD,
                                           DTYPE_INT, &array_size );
            if( array != NULL ) {
                vars.n_node_vars = (int)array_size;
                vars.node_vars   = array;
            }
            /* node_vars array is 0 length, don't read any */
            else {
                vars.n_node_vars = 0;
                vars.node_vars   = NULL;
            }
        }
        /* node_vars array is unset, read them all */
        else {
            vars.n_node_vars = -1; /* means read them all */
            vars.node_vars   = NULL;
        }

        if( OMget_array_size( cd_vars_id, &array_size ) == OM_STAT_SUCCESS ) {
            array = OMret_typed_array_ptr( cd_vars_id, OM_GET_ARRAY_RD,
                                           DTYPE_INT, &array_size );
            if( array != NULL ) {
                vars.n_cell_vars = (int)array_size;
                vars.cell_vars   = array;
            }
            /* cell_vars array is 0 length, don't read any */
            else {
                vars.n_cell_vars = 0;
                vars.cell_vars   = NULL;
            }
        }
        /* cell_vars array is unset, read them all */
        else {
            vars.n_cell_vars = -1; /* means read them all */
            vars.cell_vars   = NULL;
        }
    }

    /* Get subset information */
    {
        int i, *array;
        xp_long array_size, *larray;

        subset.ndim = -1;
        for( i = 0; i < OM_ARRAY_MAXDIM; ++i ) {
            subset.orig[i] = -1;
            subset.min[i] = 0;
            subset.max[i] = -1;
            subset.stride[i] = 1;
        }

        larray = OMret_typed_array_ptr( min_id, OM_GET_ARRAY_RD, DTYPE_LONG,
                                       &array_size );

        if( array_size > OM_ARRAY_MAXDIM )
            array_size = OM_ARRAY_MAXDIM;

        if( larray != NULL ) {
            subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.min[i] = larray[i];
            ARRfree( larray );
        }
        larray = OMret_typed_array_ptr( max_id, OM_GET_ARRAY_RD, DTYPE_LONG,
                                       &array_size );
        if( larray != NULL ) {
            if( subset.ndim < (int)array_size ) subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.max[i] = larray[i];
            ARRfree( larray );
        }
        array = OMret_typed_array_ptr( stride_id, OM_GET_ARRAY_RD, DTYPE_INT,
                                       &array_size );
        if( array != NULL ) {
            if( subset.ndim < (int)array_size ) subset.ndim = (int)array_size;
            for( i = 0; i < (int)array_size; ++i ) subset.stride[i] = array[i];
            ARRfree( array );
        }
    }

    if( HDF5read_time_field( h5_root_id, mod_id, &vars, &subset,
                             &fld_id ) != METHOD_SUCCESS ) {
        ERR_RETURN( "Error while reading file" );
    }

    if( vars.node_vars ) ARRfree( vars.node_vars );
    if( vars.cell_vars ) ARRfree( vars.cell_vars );

    H5Gclose( h5_root_id );

    h5_stat = H5Fclose( h5_file_id );

    if( !OMis_null_obj(fld_id) ) {
        if( OMset_obj_ref_mode( out_all_id, OM_OBJ_REF ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not set outFld reference mode" );
        }
        if( OMset_obj_ref( out_all_id, fld_id, 0 ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not set outFld reference to Output_Field" );
        }
    }

    if( h5_stat < 0 ) {
        ERRerror( "HDF5 read_time_field", 1, ERR_ORIG,
                  "Error %d while closing HDF5 file", h5_stat );
      return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
static int
HDF5read_prim_attr( hid_t h5_parent_id, OMobj_id prim_id )
{
    const char *name = NULL;
    OMobj_name obj_name;
    hid_t h5_attr_id;

    /* Assuming we want to use the name of the prim as
       the name of the HDF5 attribute.
    */
    if( OMget_obj_name( prim_id, &obj_name ) != OM_STAT_SUCCESS )
        return METHOD_FAILURE;
    name = OMname_to_str( obj_name );

    if( (h5_attr_id = probe_attribute( h5_parent_id, name, 0 )) > 0 ) {
        hid_t h5_dtype_id, h5_dtype_app;
        int dtype, ptype;

        h5_dtype_id  = H5Aget_type( h5_attr_id );

        /* Assume its a scalar */
        /* h5_dspace_id = H5Aget_space( h5_attr_id ); */

        HDF5map_dtypes_rd( h5_dtype_id, &h5_dtype_app, &dtype );

        if( OMget_data_type( prim_id, &ptype ) != OM_STAT_SUCCESS ||
            ptype == OM_TYPE_UNSET || ptype != dtype ) {
            ptype = dtype;
        }
        /* Seems kind of redundant, but this helps the OMset_XX_val call
         * below from morphing the type, e.g. float to double.
         * Some of the routines in FLD also do this.
         */
        OMset_data_type( prim_id, ptype );

        switch( dtype ) {
        case OM_TYPE_CHAR:
        case OM_TYPE_BYTE:
        case OM_TYPE_SHORT:
        case OM_TYPE_INT:
        {
            int value;
            H5Aread( h5_attr_id, H5T_NATIVE_INT_g, &value );
            OMset_int_val( prim_id, value );
            break;
        }
        case OM_TYPE_LONG:
        {
            xp_long value;
            H5Aread( h5_attr_id, H5T_NATIVE_XP_LONG_g, &value );
            OMset_long_val( prim_id, value );
            break;			
		}
        case OM_TYPE_FLOAT:
        case OM_TYPE_DOUBLE:
        {
            double value;
            H5Aread( h5_attr_id, H5T_NATIVE_DOUBLE_g, &value );
            OMset_real_val( prim_id, value );
        }
        }

        H5Tclose( h5_dtype_id );
        H5Aclose( h5_attr_id );
    }
    /* Could not find attribute */
    else return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int
HDF5read_prim_array_attr( hid_t h5_parent_id, OMobj_id prim_id )
{
    char *array = NULL;
    int dtype;

    const char *name = NULL;
    OMobj_name obj_name;
    hid_t h5_attr_id;

    /* Assuming we want to use the name of the prim as
       the name of the H5 attribute.
    */
    if( OMget_obj_name( prim_id, &obj_name ) != OM_STAT_SUCCESS )
        return METHOD_FAILURE;
    name = OMname_to_str( obj_name );

    if( (h5_attr_id = probe_attribute( h5_parent_id, name, 0 )) > 0 ) {
        hid_t /* h5_dspace_id, */ h5_dtype_id, h5_dtype_app;
        int ndim;
        xp_long dims[OM_ARRAY_MAXDIM];

        h5_dtype_id  = H5Aget_type( h5_attr_id );

        HDF5map_dtypes_rd( h5_dtype_id, &h5_dtype_app, &dtype );

        /* h5_dspace_id = H5Aget_space( h5_attr_id ); */
        /* ndim = H5Sget_simple_extent_ndims( h5_dspace_id ); */

        /* This is OK as long as the array size is specified
           by other variables.
        */
        ndim = 0;
        if( OMget_array( prim_id, &dtype, &array, &ndim, dims,
                         OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
            H5Aread( h5_attr_id, h5_dtype_app, array );
            ARRfree( array );
        }
        H5Tclose( h5_dtype_id );
        H5Aclose( h5_attr_id );
    }
    /* Could not find attribute */
    else return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

static int
merge_node_data( OMobj_id fld_id )
{
    static int init = 0;
    static OMobj_id node_data_tmpl;
    if( init == 0 ) {
        node_data_tmpl =
            OMfind_subobj( OMtempl_obj, OMstr_to_name("Node_Data"),
                           OM_OBJ_RD );
        init = 1;
    }
    return OMmerge_obj(node_data_tmpl, fld_id, 0);
}

static int
merge_time_node_data( OMobj_id fld_id )
{
    static int init = 0;
    static OMobj_id node_data_tmpl;
    if( init == 0 ) {
        node_data_tmpl =
            OMfind_subobj( OMtempl_obj, OMstr_to_name("Time_Node_Data"),
                           OM_OBJ_RD );
        init = 1;
    }
    return OMmerge_obj(node_data_tmpl, fld_id, 0);
}

static int
merge_cell_data( OMobj_id fld_id )
{
    static int init = 0;
    static OMobj_id cell_data_tmpl;
    if( init == 0 ) {
        cell_data_tmpl =
            OMfind_subobj( OMtempl_obj, OMstr_to_name("Cell_Data"),
                           OM_OBJ_RD );
        init = 1;
    }
    return OMmerge_obj(cell_data_tmpl, fld_id, 0);
}

static int
merge_time_cell_data( OMobj_id fld_id )
{
    static int init = 0;
    static OMobj_id cell_data_tmpl;
    if( init == 0 ) {
        cell_data_tmpl =
            OMfind_subobj( OMtempl_obj, OMstr_to_name("Time_Cell_Data"),
                           OM_OBJ_RD );
        init = 1;
    }
    return OMmerge_obj(cell_data_tmpl, fld_id, 0);
}

/* 64-bit porting. Only Modified Internally */
static int
HDF5read_field( hid_t h5_root_id, OMobj_id parent_id,
                int time_step, vars_info_t *vars, subset_info_t *subset,
                OMobj_id *out_fld_ptr )
{
    hid_t h5_grp_id;

    int fld_type = FLD_Unspecified, fld_nsteps;
    int fld_nspace, fld_ndim;
    xp_long fld_nnodes, fld_npoints, *fld_dims = NULL;
    int i;
    char *class_str = NULL;

    OMobj_id fld_id;

    if( probe_attribute( h5_root_id, "XP_CLASS", 1 ) > 0 ) {
        HDF5read_string_attr( h5_root_id, "XP_CLASS", &class_str );
#if 0
        fprintf( stderr, "HDF5read_field %s\n", class_str );
#endif
    }
#if 0
    else
        fprintf( stderr, "HDF5read_field\n" );
#endif

    /* ALL fields must have nspace */
    if( probe_attribute( h5_root_id, "nspace", 1 ) > 0 ) {
        HDF5read_int_attr( h5_root_id, "nspace", &fld_nspace );
    }
    else ERR_RETURN("Can't find nspace in file");

    /* ALL time-dependent fields must have nsteps. */
    if( probe_attribute( h5_root_id, "nsteps", 1 ) > 0 )
        HDF5read_int_attr( h5_root_id, "nsteps", &fld_nsteps );
    else if( class_str != NULL && strstr( class_str, "Time_" ) != NULL ) {
        ERR_RETURN("Can't find nsteps in file");
    }
    else fld_nsteps = INVALID_TIME_STEP;

    if( probe_attribute( h5_root_id, "ndim", 1 ) > 0 ) {
        HDF5read_int_attr( h5_root_id, "ndim", &fld_ndim );

        /* Presence of "ndim" and "dims" implies Unif/Rect/Struct */

        if( probe_attribute( h5_root_id, "dims", 1 ) > 0 ) {
            HDF5read_dims( h5_root_id, &fld_dims );
        }
        /* else error, we expect dims if ndim is present */
        else ERR_RETURN("Can't find dims in file");

        if( probe_attribute( h5_root_id, "npoints", 1 ) > 0 ) {
            HDF5read_long_attr( h5_root_id, "npoints", &fld_npoints );
            /* npoints = 2         means Unif */
            if( fld_npoints == 2 ) {
                fld_type = FLD_UNIF;
            }
            /* npoints = sum(dims) means Rect */
            else if( fld_dims != NULL ) {
                xp_long dims_sum = 0;
                for( i = 0; i < fld_ndim; ++i ) dims_sum += fld_dims[i];
                if( dims_sum == fld_npoints ) fld_type = FLD_RECT;
            }
        }

        if( (fld_type == FLD_Unspecified) &&
            probe_attribute( h5_root_id, "grid_type", 1 ) > 0 ) {
            int grid_type;
            HDF5read_int_attr( h5_root_id, "grid_type", &grid_type );
            if     ( grid_type == 1 ) fld_type = FLD_UNIF;
            else if( grid_type == 2 ) fld_type = FLD_RECT;
        }

        if( (fld_type == FLD_Unspecified) && (class_str != NULL) ) {
            if     ( strstr( class_str, "_Unif" ) != NULL )
                fld_type = FLD_UNIF;
            else if( strstr( class_str, "_Rect" ) != NULL )
                fld_type = FLD_RECT;
        }

        /* else Struct, in which case coordinates need to be present */
        if( fld_type == FLD_Unspecified ) {
            if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
                fld_type = FLD_STRUCT;
            else if( fld_nsteps != INVALID_TIME_STEP &&
                     probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 )
                fld_type = FLD_STRUCT;
        }

        /* Error check the subset information */
        if( subset->ndim != fld_ndim )
            subset = NULL;
    }
    else {
        /* No "ndim" or "dims" means unstructured */
        fld_type = FLD_UNSTRUCT;
        /* Can't subset an unstructured field */
        subset = NULL;

        /* Unstructured field must have nnodes. */
        if( probe_attribute( h5_root_id, "nnodes", 1 ) > 0 ) {
            HDF5read_long_attr( h5_root_id, "nnodes", &fld_nnodes );
        }
        else {
            ERR_RETURN("Can't find nnodes in file");
        }

        /* Unstructured field must have coordinates. */
        if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
            ;
        else if( fld_nsteps != INVALID_TIME_STEP &&
                 probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 )
            ;
        else ERR_RETURN("Can't find coordinates or time_coordinates in file");
    }

    /* Oops */
    if( fld_type == FLD_Unspecified ) {
        ERR_RETURN("Can't determine field type");
    }

    /* Create the output field */

    if( OMis_null_obj( *out_fld_ptr ) ) {
        /* This is the usual case.  We create an output field
         * of the correct type.
         */
        if( fld_type == FLD_UNIF )
            fld_id = FLDcreate( parent_id, "Mesh_Unif", "Output_Field" );
        else if( fld_type == FLD_RECT )
            fld_id = FLDcreate( parent_id, "Mesh_Rect", "Output_Field" );
        else if( fld_type == FLD_STRUCT )
            fld_id = FLDcreate( parent_id, "Mesh_Struct", "Output_Field" );
        else if( fld_type == FLD_UNSTRUCT )
            fld_id = FLDcreate( parent_id, "Mesh", "Output_Field" );

        /* Also, worry if it has node data, cell data */

        if( (probe_attribute( h5_root_id, "nnode_data", 1 ) > 0 ) ||
            ((class_str != NULL) && (strstr( class_str, "Node_Data" ) != NULL))) {
            merge_node_data( fld_id );
    }

        if( (class_str != NULL) && (strstr( class_str, "Cell_Data" ) != NULL)) {
            merge_cell_data( fld_id );
    }
        else if( probe_group( h5_root_id, "cell_set[0]/cell_data[0]", 1 ) > 0 ) {
            merge_cell_data( fld_id );
        }
        if( fld_nsteps != INVALID_TIME_STEP &&
            probe_group( h5_root_id, "cell_set[0]/time_cell_data[0]", 1 ) > 0 ) {
            merge_cell_data( fld_id );
        }
    }
    else {
        /* Output field pre-created by application.  */
        int tmp;
        OMobj_id elem_id;
        fld_id = *out_fld_ptr;

        /* This is kind of a kludge.  There is more than one place in
         * this reader can set nnode_data, and there is the assumption
         * that if nnode_data is set, it is set to the value that the
         * reader wants to use.  So start by unsetting the value in the
         * pre-created field.
         */
	elem_id = OMfind_subobj( fld_id, OMstr_to_name("nnode_data"),
                                 OM_OBJ_RW );
        if( !OMis_null_obj(elem_id) && 
            OMget_int_val( elem_id, &tmp ) == OM_STAT_SUCCESS ) {
            OMset_obj_val( elem_id, OMnull_obj, 0 );
        }
    }

    if( FLDset_nspace( fld_id, fld_nspace ) != OM_STAT_SUCCESS )
      ERR_RETURN( "Cannot set out fld nspace" );

    if( fld_type == FLD_UNSTRUCT ) {
        /* nnodes needs to be specified explicitly for unstruct fields,
         * is computed from dims for all struct fields.
         */
        if( FLDset_nnodes( fld_id, fld_nnodes ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot set out fld nnodes" );
    }
    else {
        /* dims for all struct fields, never time-dependent
         *
         * Reading the dims from the HDF5 file is done earlier,
         * when trying to figure out what kind of grid type we have.
         */

        if( subset ) {
            /* In case the original arrays were the wrong length! */
            subset->ndim = fld_ndim;
            /* Map our crop/downsize parameters into
               HDF5 hyperslab parameters.
            */
            for( i = 0; i < fld_ndim; ++i ) {
                xp_long count;

                subset->orig[i] = fld_dims[i];

                if( (subset->max[i] >= fld_dims[i]) ||
                    (subset->max[i] < 0) )
                    subset->max[i] = fld_dims[i] - 1;

                if( (subset->min[i] < 0) ||
                    (subset->min[i] > subset->max[i]) )
                    subset->min[i] = 0;

                /* apply crop */
                count = subset->max[i] - subset->min[i] + 1;
                /* apply downsize */
                if( subset->stride[i] > 1 )
                    count = 1 + ((count-1)/subset->stride[i]);

                subset->count[i] = count;
                fld_dims[i]      = count;
#if 0
                fprintf( stderr, "subset[%d]: min:%ld, stride:%d, count:%ld\n",
                         i, subset->min[i], subset->stride[i],
                         subset->count[i] );
#endif
            }

        }

        if( HDF5set_dims( fld_id, fld_type, fld_ndim, fld_dims )
            != METHOD_SUCCESS )
            return METHOD_FAILURE;
    }

    if( fld_dims != NULL ) {
        free( fld_dims );
        fld_dims = NULL;
    }

    /* Read one value out of the time array. */
    if( fld_nsteps != INVALID_TIME_STEP ) {
        hid_t h5_dspace_id, h5_attr_id;
        if( (h5_attr_id = probe_attribute( h5_root_id, "time", 0 )) > 0 ) {
            OMobj_id time_id;
            hsize_t h5_dims[1];
            int ndim;
            double *time_array;

            /* Pick off the time value of this single time-step. */

            h5_dspace_id = H5Aget_space( h5_attr_id );
            ndim = H5Sget_simple_extent_ndims( h5_dspace_id );
            /* assuming ndim == 1 */
            H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
            time_array = malloc( h5_dims[0] * sizeof(double) );
            if( time_array != NULL && time_step < (int)h5_dims[0] ) {
                /* Read the whole array.  We could do that subset thang,
                 * but is it worth the bother for a smallish 1D array ?
                 */
                H5Aread( h5_attr_id, H5T_NATIVE_DOUBLE_g, time_array );

                time_id = OMfind_subobj( fld_id, OMstr_to_name( "time" ),
                                         OM_OBJ_RW );
                if( OMis_null_obj( time_id ) ) {
                    /* Attempt to create the variable if it does not exist. */
                    OMparse_buffer( fld_id, "double time;", 0 );
                    time_id = OMfind_subobj( fld_id, OMstr_to_name( "time" ),
                                             OM_OBJ_RW );
                }
                if( !OMis_null_obj( time_id ) ) {
                    /* The single time value. */
                    OMset_real_val( time_id, time_array[time_step] );
                }

                free( time_array );
            }
            H5Sclose( h5_dspace_id );
            H5Aclose( h5_attr_id );
        }
    }

    if( fld_type == FLD_UNSTRUCT || fld_type == FLD_STRUCT ) {
        /* coordinates for struct and unstruct fields. */
        if( fld_nsteps != INVALID_TIME_STEP &&
            probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 ) {
            /* coordinates are time-dependent, but we are reading just one
               time step into a regular field.
            */
            char buffer[64];
            sprintf( buffer, "time_coordinates[%d]", time_step );
            if( (h5_grp_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {
#if 0
                fprintf( stderr, "Found %s\n", buffer );
#endif
                HDF5read_coords( h5_grp_id, fld_id, fld_type, subset );
                H5Gclose( h5_grp_id );
            }
        }
        else {
            /* coordinates not time-dependent */
            HDF5read_coords( h5_root_id, fld_id, fld_type, subset );
        }
    }
    else { /* Rect, Unif */
        /* points for rect and unif fields, never time-dependent */
        HDF5read_points( h5_root_id, fld_id, fld_type, subset );
    }

    /* Xform (never time-dependent) */
    HDF5read_xform( h5_root_id, fld_id );

    /* Cells (never time-dependent) */
    HDF5read_cells( h5_root_id, fld_id );

    /* Node_Data */
    if( fld_nsteps != INVALID_TIME_STEP &&
        probe_group( h5_root_id, "time_node_data[0]", 1 ) > 0 ) {
        /* Node data is time-dependent. */
        char buffer[64];
        sprintf( buffer, "time_node_data[%d]", time_step );
        if( (h5_grp_id = probe_group( h5_root_id, buffer, 0 )) > 0 ) {
            HDF5read_node_data( h5_grp_id, fld_id, vars, subset );
            H5Gclose( h5_grp_id );
        }
    }
    else if( probe_group( h5_root_id, "node_data[0]", 1 ) > 0 ) {
        /* Node data is not time-dependent. */
        HDF5read_node_data( h5_root_id, fld_id, vars, subset );
    }

    /* Cell_Data */
    if( (class_str != NULL) &&
        (strstr( class_str, "Time_Cell_Data" ) != NULL) ) {
        /* Cell data is time-dependent. */
        HDF5read_time_cell_data( h5_root_id, fld_id, time_step, vars );
    }
    else if( fld_nsteps != INVALID_TIME_STEP &&
             probe_group( h5_root_id, "cell_set[0]/time_cell_data[0]", 1 ) > 0 ) {
        /* Cell data is time-dependent. */
        HDF5read_time_cell_data( h5_root_id, fld_id, time_step, vars );
    }
    else {
        /* Cell data is not time-dependent. */
        HDF5read_cell_data( h5_root_id, fld_id, vars );
    }

    if( class_str ) { free( class_str ); class_str = NULL; }

    *out_fld_ptr = fld_id;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int
HDF5read_time_field( hid_t h5_root_id, OMobj_id parent_id,
                     vars_info_t *vars, subset_info_t *subset,
                     OMobj_id *out_fld_ptr )
{
    hid_t h5_grp_id;

    int fld_type = FLD_Unspecified, fld_nsteps;
    int fld_nspace, fld_ndim;
    xp_long  fld_nnodes, fld_npoints, *fld_dims = NULL;
    int i;
    char *class_str = NULL;

    OMobj_id fld_id;

    if( probe_attribute( h5_root_id, "XP_CLASS", 1 ) > 0 ) {
        HDF5read_string_attr( h5_root_id, "XP_CLASS", &class_str );
#if 0
        fprintf( stderr, "HDF5read_time_field %s\n", class_str );
#endif
    }
#if 0
    else
        fprintf( stderr, "HDF5read_time_field\n" );
#endif

    /* ALL fields must have nspace. */
    if( probe_attribute( h5_root_id, "nspace", 1 ) > 0 )
        HDF5read_int_attr( h5_root_id, "nspace", &fld_nspace );
    else ERR_RETURN("Can't find nspace in file");

    /* ALL time-dependent fields must have nsteps. */
    if( probe_attribute( h5_root_id, "nsteps", 1 ) > 0 )
        HDF5read_int_attr( h5_root_id, "nsteps", &fld_nsteps );
    else ERR_RETURN("Can't find nsteps in file");

    if( probe_attribute( h5_root_id, "ndim", 1 ) > 0 ) {
        HDF5read_int_attr( h5_root_id, "ndim", &fld_ndim );

        /* Presence of "ndim" and "dims" implies Unif/Rect/Struct. */

        if( probe_attribute( h5_root_id, "dims", 1 ) > 0 ) {
            HDF5read_dims( h5_root_id, &fld_dims );
        }
        /* else error, we expect dims if ndim is present */
        else ERR_RETURN("Can't find dims in file");

        if( probe_attribute( h5_root_id, "npoints", 1 ) > 0 ) {
            HDF5read_long_attr( h5_root_id, "npoints", &fld_npoints );
            /* npoints = 2         means Unif */
            if( fld_npoints == 2 ) {
                fld_type = FLD_UNIF;
            }
            /* npoints = sum(dims) means Rect */
            else if( fld_dims != NULL ) {
                xp_long dims_sum = 0;
                for( i = 0; i < fld_ndim; ++i ) dims_sum += fld_dims[i];
                if( dims_sum == fld_npoints ) fld_type = FLD_RECT;
            }
        }
        if( (fld_type == FLD_Unspecified) &&
            probe_attribute( h5_root_id, "grid_type", 1 ) > 0 ) {
            int grid_type;
            HDF5read_int_attr( h5_root_id, "grid_type", &grid_type );
            if     ( grid_type == 1 ) fld_type = FLD_UNIF;
            else if( grid_type == 2 ) fld_type = FLD_RECT;
        }
        if( (fld_type == FLD_Unspecified) && (class_str != NULL) ) {
            if     ( strstr( class_str, "_Unif" ) != NULL )
                fld_type = FLD_UNIF;
            else if( strstr( class_str, "_Rect" ) != NULL )
                fld_type = FLD_RECT;
        }

        /* else Struct, in which case coordinates need to be present */
        if( fld_type == FLD_Unspecified ) {
            if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
                fld_type = FLD_STRUCT;
            else if( probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 )
                fld_type = FLD_STRUCT;
        }

        /* Error check the subset information */
        if( subset->ndim != fld_ndim )
            subset = NULL;
    }
    else {
        /* No "ndim" or "dims" means unstructured */
        fld_type = FLD_UNSTRUCT;

        /* Unstructured field must have nnodes. */
        if( probe_attribute( h5_root_id, "nnodes", 1 ) > 0 )
            HDF5read_long_attr( h5_root_id, "nnodes", &fld_nnodes );
        else ERR_RETURN("Can't find nnodes in file");

        /* Unstructured field must have coordinates. */
        if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
            ;
        else if( probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 )
            ;
        else ERR_RETURN("Can't find coordinates or time_coordinates in file");
    }

    /* Oops */
    if( fld_type == FLD_Unspecified ) {
        ERR_RETURN("Can't determine field type");
    }

    if( fld_type == FLD_UNIF )
        fld_id = FLDcreate( parent_id, "Mesh_Unif", "Output_Field" );
    else if( fld_type == FLD_RECT )
        fld_id = FLDcreate( parent_id, "Mesh_Rect", "Output_Field" );
    else if( fld_type == FLD_STRUCT ) {
        if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
            fld_id = FLDcreate( parent_id, "Mesh_Struct", "Output_Field" );
        else if( probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 ) {
            fld_id = FLDcreate( parent_id, "Time_Mesh_Struct", "Output_Field" );
        }
    }
    else if( fld_type == FLD_UNSTRUCT ) {
        if( probe_group( h5_root_id, "coordinates", 1 ) > 0 )
            fld_id = FLDcreate( parent_id, "Mesh", "Output_Field" );
        else if( probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 ) {
            fld_id = FLDcreate( parent_id, "Time_Mesh", "Output_Field" );
        }
    }

    /* Also, worry if it has node data, cell data */

    if( probe_group( h5_root_id, "node_data[0]", 1 ) > 0 )
        merge_node_data( fld_id );
    else if( probe_group( h5_root_id, "time_node_data[0]", 1 ) > 0 )
            merge_time_node_data( fld_id );
    else {
        if( class_str != NULL ) {
            if( strstr( class_str, "Time_Node_Data" ) != NULL ) {
                merge_time_node_data( fld_id );
            }
            else if( strstr( class_str, "Node_Data" ) != NULL ) {
                merge_node_data( fld_id );
            }
        }
    }

    if( probe_group( h5_root_id, "cell_set[0]/time_cell_data[0]", 1 ) > 0 ) {
        merge_time_cell_data( fld_id );
    }
    else if( probe_group( h5_root_id, "cell_set[0]/cell_data[0]", 1 ) > 0 ) {
        merge_cell_data( fld_id );
    }
    else if( class_str != NULL ) {
        if( strstr( class_str, "Time_Cell_Data" ) != NULL )
            merge_time_cell_data( fld_id );
        else if( strstr( class_str, "Cell_Data" ) != NULL )
            merge_cell_data( fld_id );
    }

    if( FLDset_nspace( fld_id, fld_nspace ) != OM_STAT_SUCCESS )
      ERR_RETURN( "Cannot set out fld nspace" );
    if( FLDset_time_nsteps( fld_id, fld_nsteps ) != OM_STAT_SUCCESS )
        ERR_RETURN( "Cannot set out fld nsteps" );

    if( fld_type == FLD_UNSTRUCT ) {
        /* nnodes needs to be specified explicitly for unstruct fields,
         * is computed from dims for all struct fields.
         */
        if( FLDset_nnodes( fld_id, fld_nnodes ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot set out fld nnodes" );
    }
    else {
        /* dims for all struct fields, never time-dependent
         *
         * Reading the dims from the HDF5 file is done earlier,
         * when trying to figure out what kind of grid type we have.
         */

        if( subset ) {
            /* In case the original arrays were the wrong length! */
            subset->ndim = fld_ndim;
            /* Map our crop/downsize parameters into
               HDF5 hyperslab parameters.
            */
            for( i = 0; i < fld_ndim; ++i ) {
                xp_long count;

                subset->orig[i] = fld_dims[i];

                if( (subset->max[i] >= fld_dims[i]) ||
                    (subset->max[i] < 0) )
                    subset->max[i] = fld_dims[i] - 1;

                if( (subset->min[i] < 0) ||
                    (subset->min[i] > subset->max[i]) )
                    subset->min[i] = 0;

                /* apply crop */
                count = subset->max[i] - subset->min[i] + 1;
                /* apply downsize */
                if( subset->stride[i] > 1 )
                    count = 1 + ((count-1)/subset->stride[i]);

                subset->count[i] = count;
                fld_dims[i]      = count;
#if 0
                fprintf( stderr, "subset[%d]: min:%ld, stride:%ld, count:%ld\n",
                         i, subset->min[i], subset->stride[i],
                         subset->count[i] );
#endif
            }

#if 0
#ifdef WORDLENGTH_64
            {
                /* Note that this check does not include veclen */
                long total_elements = 1;
                for( i = 0; i < fld_ndim; ++i ) {
                    total_elements = fld_dims[i];
                }
                if( total_elements > 0x8FFFFFFF ) {
                    ERR_RETURN( "Attempt to read array larger than 2 giga elements" );
                }
            }
#endif
#endif
        }

        if( HDF5set_dims( fld_id, fld_type, fld_ndim, fld_dims )
            != METHOD_SUCCESS )
            return METHOD_FAILURE;
    }

    /* time array */

    /* Read all time steps. */
    if( probe_attribute( h5_root_id, "time", 1 ) > 0 ) {
        OMobj_id elem_id = OMfind_subobj( fld_id, OMstr_to_name("time"),
                                          OM_OBJ_RW );
        /* Read the whole time array into the top level of
         * the time-dependent field.
         */
        HDF5read_prim_array_attr( h5_root_id, elem_id );
    }
    else ERR_RETURN( "Cannot find time array in file" );

    if( fld_type == FLD_UNSTRUCT || fld_type == FLD_STRUCT ) {
        /* coordinates for struct and unstruct fields. */
        if( probe_group( h5_root_id, "coordinates", 1 ) > 0 ) {
            /* coordinates not time-dependent */
            HDF5read_coords( h5_root_id, fld_id, fld_type, subset );
        }
        else if( probe_group( h5_root_id, "time_coordinates[0]", 1 ) > 0 ) {
            /* coordinates are time-dependent, read all the time steps
               into a time-dependent field.
            */
            char buffer[64];
            OMobj_id c_id;
            int t;
            for( t = 0; t < fld_nsteps; ++t ) {
                sprintf( buffer, "time_coordinates[%d]", t );
                if( (h5_grp_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {
#if 0
                    fprintf( stderr, "Found %s\n", buffer );
#endif
                    if( FLDget_time_coordinates( fld_id, t, &c_id )
                        == OM_STAT_SUCCESS ) {
                        HDF5read_coords( h5_grp_id, c_id, fld_type, subset );
                    }
                    H5Gclose( h5_grp_id );
                }
            }
        } /* found time_coordinates[0] */
    }
    else { /* Rect, Unif */
        /* points for rect and unif fields, never time-dependent */
        HDF5read_points( h5_root_id, fld_id, fld_type, NULL );
    }

    /* Xform (never time-dependent) */
    HDF5read_xform( h5_root_id, fld_id );

    /* Cells (never time-dependent) */
    HDF5read_cells( h5_root_id, fld_id );

    /* Node_Data */
    if( probe_group( h5_root_id, "node_data[0]", 1 ) > 0 ) {
        /* Node data is not time-dependent. */
        HDF5read_node_data( h5_root_id, fld_id, vars, subset );
    }
    else if( probe_group( h5_root_id, "time_node_data[0]", 1 ) > 0 ) {
        /* Node data is time-dependent, read all the time steps
           into a time-dependent field.
        */
        char buffer[64];
        OMobj_id n_id;
        int t;

        if( probe_attribute( h5_root_id, "nnode_data", 1 ) > 0 ) {
            int file_ncomp, fld_ncomp;
            HDF5read_int_attr( h5_root_id, "nnode_data", &file_ncomp );

            if( (vars != NULL) && (vars->n_node_vars >= 0)  &&
                (vars->n_node_vars < file_ncomp) ) {
                /* a subset of the vars were selected */
                fld_ncomp = vars->n_node_vars;
            }
            else
                fld_ncomp = file_ncomp;

            if( FLDset_node_data_ncomp( fld_id, fld_ncomp )
                != OM_STAT_SUCCESS ) {
                ERR_RETURN( "Cannot set out fld nnode_data" );
            }
        }

        for( t = 0; t < fld_nsteps; ++t ) {
            sprintf( buffer, "time_node_data[%d]", t );
            if( (h5_grp_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {
#if 0
                fprintf( stderr, "Found %s\n", buffer );
#endif
                if( FLDget_time_node_data( fld_id, t, &n_id )
                    == OM_STAT_SUCCESS ) {
                    HDF5read_node_data( h5_grp_id, n_id, vars, subset );
                }
                H5Gclose( h5_grp_id );
            }
        }
    }

    /* Cell_Data */
    if( (class_str != NULL) &&
        (strstr( class_str, "Time_Cell_Data" ) != NULL) ) {
        /* Cell data is time-dependent. */
        HDF5read_time_cell_data( h5_root_id, fld_id, INVALID_TIME_STEP, vars );
    }
    else if( probe_group( h5_root_id, "cell_set[0]/time_cell_data[0]", 1 ) > 0 ) {
        /* Cell data is time-dependent. */
        HDF5read_time_cell_data( h5_root_id, fld_id, INVALID_TIME_STEP, vars );
    }
    else {
        /* Cell data is not time-dependent. */
        HDF5read_cell_data( h5_root_id, fld_id, vars );
    }

    if( class_str ) { free( class_str ); class_str = NULL; }

    *out_fld_ptr = fld_id;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static void
HDF5read_data_array(
    hid_t h5_grp_id,		/* Group in which to write the data */
    OMobj_id data_array_id ,	/* OM id of an object of type 'Data_Array' */
    subset_info_t *subset )
{
    hid_t h5_attr_id;

    OMobj_id elem_id;
    int veclen;

    if( h5_grp_id < 0 ) return;
    if( OMis_null_obj( data_array_id ) ) return;

    /* nvals */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("nvals"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        xp_long nvals;
        /* Usually this will be missing from the file. */
        if( probe_attribute( h5_grp_id, "nvals", 1 ) > 0 ) {
            HDF5read_long_attr( h5_grp_id, "nvals", &nvals );
            OMset_long_val( elem_id, nvals );
        }
    }

    /* veclen */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("veclen"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( probe_attribute( h5_grp_id, "veclen", 1 ) > 0 ) {
            HDF5read_int_attr( h5_grp_id, "veclen", &veclen );
            OMset_int_val( elem_id, veclen );
        }
        else {
            /* In some cases (coordinates.veclen => nspace) we
               can get veclen from a connection inside the field.
            */
            if( OMget_int_val( elem_id, &veclen ) != OM_STAT_SUCCESS ) {
                /* With Node_Data and Cell_Data, 1 is a reasonable default */
                OMset_int_val( elem_id, 1 );
            }
        }
    }

    /* id */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("id"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        int id;
        if( probe_attribute( h5_grp_id, "id", 1 ) > 0 ) {
            HDF5read_int_attr( h5_grp_id, "id", &id );
            OMset_int_val( elem_id, id );
        }
    }

    /* values */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("values"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        hid_t h5_dset_id, h5_dspace_id, h5_dtype_id;
        hid_t h5_dtype_app;
        hsize_t h5_dims[H5S_MAX_RANK];
        int h5_ndim;

        int dtype;
        int ndim;
        xp_long dims[OM_ARRAY_MAXDIM];
        char *array = NULL;

        if( (h5_dset_id = probe_dataset( h5_grp_id, "values", 0 )) > 0 ) {
            h5_dtype_id  = H5Dget_type( h5_dset_id );
            h5_dspace_id = H5Dget_space( h5_dset_id );

            HDF5map_dtypes_rd( h5_dtype_id, &h5_dtype_app, &dtype );

            if( subset == NULL ) {
                /* We don't use this value for much ... */
                h5_ndim = H5Sget_simple_extent_ndims( h5_dspace_id );
                if( h5_ndim > 0 ) {
                    /* We don't need to worry very much about the details
                     * of the dimensions as long as the total sizes
                     * match up.
                     */
                    H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );
                    ndim = 0;
                    /* values[nvals][veclen] */
                    if( OMget_array( elem_id, &dtype, &array, &ndim, dims,
                                     OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
                        H5Dread( h5_dset_id, h5_dtype_app,
                                 H5S_ALL, H5S_ALL,
                                 H5P_DEFAULT, array );
                        ARRfree( array );
                    }
                }
            }
            else if( h5_dspace_id > 0 ) {
                /* HDF5 hyperslab parameters. */
                hssize_t h5_start[H5S_MAX_RANK];
                hsize_t h5_stride[H5S_MAX_RANK];
                hsize_t h5_count[H5S_MAX_RANK];
                int i;

                hid_t h5_dspace_app;

                h5_ndim = H5Sget_simple_extent_ndims( h5_dspace_id );
                H5Sget_simple_extent_dims( h5_dspace_id, h5_dims, NULL );

                /* The fld_dims (and the stuff in subset) are in
                 * column-major order. We need to flip 'em around to
                 * match HDF5's row-major ordering.  Using HDF5 hyperslabs
                 * to implement crop and downsize won't work unless we
                 * use HDF5's row-major ordering.
                 */
                for( i = 0; i <subset->ndim; ++i ) {
                    int ii = subset->ndim - (i + 1);
#if 0
                    fprintf( stderr, "i:%d, ii:%d\n", i, ii );
#endif
                    h5_start[i]  = subset->min[ii];
                    h5_stride[i] = subset->stride[ii];
                    h5_count[i]  = subset->count[ii];
                }

                /* Add the veclen dim if present (it will always be last
                 * in the HDF5 dims list); always select it all.
                 */
                for( ; i < h5_ndim; ++i ) {
#if 0
                    fprintf( stderr, "i:%d (veclen)\n", i );
#endif
                    h5_start[i]  = 0;
                    h5_stride[i] = 1;
                    h5_count[i]  = h5_dims[i];
                }

                H5Sselect_hyperslab( h5_dspace_id, H5S_SELECT_SET,
                                     h5_start, h5_stride, h5_count, NULL );

                /* Layout of data in the OM, which is straightforward. */
                h5_dspace_app = H5Screate_simple( h5_ndim, h5_count, NULL );

                ndim = 0;
                /* values[nvals][veclen] */
                if( OMget_array( elem_id, &dtype, &array, &ndim, dims,
                                 OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
                    H5Dread( h5_dset_id, h5_dtype_app,
                             h5_dspace_app, h5_dspace_id,
                             H5P_DEFAULT, array );
                    ARRfree( array );
                }
                H5Sclose( h5_dspace_app );
            }

            H5Sclose( h5_dspace_id );
            H5Tclose( h5_dtype_id );
            H5Dclose( h5_dset_id );
        }
    }

    /* null_flag */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("null_flag"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        int null_flag;
        if( probe_attribute( h5_grp_id, "null_flag", 1 ) > 0 ) {
            HDF5read_int_attr( h5_grp_id, "null_flag", &null_flag );
            if( null_flag != 0 ) OMset_int_val( elem_id, null_flag );
        }
    }

    /* null_value */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("null_value"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( (h5_attr_id = probe_attribute( h5_grp_id, "null_value", 1 )) > 0 ) {
            HDF5read_prim_attr( h5_grp_id, elem_id );
        }
    }

    /* min */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("min"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( (h5_attr_id = probe_attribute( h5_grp_id, "min", 1 )) > 0 ) {
            HDF5read_prim_attr( h5_grp_id, elem_id );
        }
    }

    /* max */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("max"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( (h5_attr_id = probe_attribute( h5_grp_id, "max", 1 )) > 0 ) {
            HDF5read_prim_attr( h5_grp_id, elem_id );
        }
    }

    /* min_vec */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("min_vec"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( (h5_attr_id = probe_attribute( h5_grp_id, "min_vec", 1 )) > 0 ) {
            HDF5read_prim_array_attr( h5_grp_id, elem_id );
        }
    }

    /* max_vec */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("max_vec"),
                             OM_OBJ_RW);

    if (!OMis_null_obj(elem_id)) {
        if( (h5_attr_id = probe_attribute( h5_grp_id, "max_vec", 1 )) > 0 ) {
            HDF5read_prim_array_attr( h5_grp_id, elem_id );
        }
    }

    /* units */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("units"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( probe_attribute( h5_grp_id, "units", 1 ) > 0 ) {
            char *str = NULL;
            HDF5read_string_attr( h5_grp_id, "units", &str );
            if( str ) {
                OMset_str_val( elem_id, str );
                free( str );
            }
        }
    }

    /* labels */
    elem_id = OMfind_subobj (data_array_id, OMstr_to_name("labels"),
                             OM_OBJ_RW);
    if (!OMis_null_obj(elem_id)) {
        if( probe_attribute( h5_grp_id, "labels", 1 ) > 0 ) {
            char *str = NULL;
            HDF5read_string_attr( h5_grp_id, "labels", &str );
            if( str ) {
                OMset_str_val( elem_id, str );
                free( str );
            }
        }
    }
}

/* 64-bit porting. Directly Modified */
static int
HDF5set_dims( OMobj_id fld_id, int fld_type,
              int fld_ndim, xp_long *fld_dims )
{
    if( fld_type != FLD_UNSTRUCT ) {
        if( FLDset_ndim( fld_id, fld_ndim ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot set out fld ndim" );
        if( FLDset_dims( fld_id, fld_dims ) != OM_STAT_SUCCESS )
            ERR_RETURN( "Cannot set out fld dims" );
    }
    return METHOD_SUCCESS;
}

static int
HDF5read_coords( hid_t h5_root_id, OMobj_id fld_id, int fld_type,
                 subset_info_t *subset )
{
    hid_t h5_grp_id;

    if( fld_type == FLD_UNSTRUCT || fld_type == FLD_STRUCT ) {
        OMobj_id coord_id;
        /* coordinates */
        if( (h5_grp_id = probe_group( h5_root_id, "coordinates", 0 )) > 0 ) {
            coord_id = OMfind_subobj (fld_id, OMstr_to_name("coordinates"),
                                      OM_OBJ_RW);
            if( !OMis_null_obj(coord_id) ) {
                HDF5read_data_array( h5_grp_id, coord_id, subset );
            }
            H5Gclose( h5_grp_id );
        }
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
static int
HDF5read_points( hid_t h5_root_id, OMobj_id fld_id, int fld_type,
                 subset_info_t *subset )
{
    hid_t h5_dset_id;

    if( fld_type == FLD_UNIF || fld_type == FLD_RECT ) {
        xp_long fld_npoints;
        int fld_nspace, fld_ndim;
        int i, k;
        xp_long j, jj;

        if( probe_attribute( h5_root_id, "npoints", 1 ) > 0 ) {
            HDF5read_long_attr( h5_root_id, "npoints", &fld_npoints );
            FLDset_npoints( fld_id, fld_npoints );
        }
        else if( FLDget_npoints( fld_id, &fld_npoints ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Cannot determine npoints" );
        }

        /* Should already be set. */
        FLDget_nspace( fld_id, &fld_nspace );
        FLDget_ndim( fld_id, &fld_ndim );

        /* points */
        if( (h5_dset_id = probe_dataset( h5_root_id, "points", 0 )) > 0 ) {
            float *fld_points = NULL;
            xp_long pts_size = 0;

            if( subset != NULL && fld_type == FLD_RECT ) {
                /* This is the nasty subsetting case, because HDF5 won't
                   be able to grok the way we do rectilinear points.
                */
                hid_t h5_dspace_id = H5Dget_space( h5_dset_id );
                hid_t h5_dspace_app;
                hssize_t *h5_coord = NULL;
                hsize_t h5_dims[2];
                xp_long dim_sum_subset = 0;
                xp_long dim_sum_full   = 0;

                for( i = 0; i < fld_ndim; ++i ) {
                    dim_sum_subset += subset->count[i];
                }

                FLDset_npoints( fld_id, dim_sum_subset );

                /* If stride < fld_nspace, this could be larger than
                   the original points array!  It might be better to
                   read the entire array out of the file into a temp
                   buffer and directly grab the points subset.
                */
                h5_coord = malloc( dim_sum_subset * fld_nspace * 2 *
                                   sizeof( hssize_t ) );
#if 0
                fprintf( stderr, "selection points %d\n",
                         dim_sum_subset * fld_nspace );
#endif
                dim_sum_subset = 0;
                dim_sum_full = 0;
                for( i = 0; i < fld_ndim; ++i ) {

                    for( j = subset->min[i], jj = 0;
                         jj < subset->count[i];
                         j += subset->stride[i], jj += 1 ) {
                        xp_long idx1 = jj+dim_sum_subset;

                        for( k = 0; k < fld_nspace; ++k ) {
                            xp_long idx2 = idx1*fld_nspace + k;
                            h5_coord[ 2*idx2 ]   = j + dim_sum_full;
                            h5_coord[(2*idx2)+1] = k;
#if 0
                            fprintf( stderr, "[%d %d]:%d, [%d %d]:%d\n",
                                     idx2, 0, (int)h5_coord[ 2*idx2 ],
                                     idx2, 1, (int)h5_coord[(2*idx2)+1]
                                     );
#endif
                        }
                    }
#if 0
                    fprintf( stderr, "dim_sum_subset %d + %d\n",
                             dim_sum_subset, subset->count[i] );
                    fprintf( stderr, "dim_sum_full   %d + %d\n",
                             dim_sum_full, subset->orig[i] );
#endif

                    dim_sum_subset += subset->count[i];
                    dim_sum_full   += subset->orig[i];
                }

                /* The func prototype for the last element is misleading, */
                /* its just a plain old 2d array. */
                H5Sselect_elements( h5_dspace_id, H5S_SELECT_SET,
                                    dim_sum_subset * fld_nspace,
                                    (const hssize_t **)h5_coord );

                /* Could I free h5_coord here? */

                /* layout of data in the OM, which is straightforward. */
                h5_dims[0] = dim_sum_subset;
                h5_dims[1] = fld_nspace;
                h5_dspace_app = H5Screate_simple( 2, h5_dims, NULL );

                if (FLDget_points(fld_id, &fld_points, &pts_size,
                                  OM_GET_ARRAY_WR) == OM_STAT_SUCCESS) {

#if 0
                    fprintf( stderr, "Got points %d %p\n",
                             pts_size, fld_points );
#endif

                    /* Size should be already established as npoints*nspace */
                    H5Dread( h5_dset_id, H5T_NATIVE_FLOAT_g,
                             h5_dspace_app, h5_dspace_id,
                             H5P_DEFAULT, fld_points );

#if 0
                    fprintf( stderr, "Read points\n" );
#endif

                    ARRfree( fld_points );
                }
                /* else error could not get points */

                free( h5_coord );

                H5Sclose( h5_dspace_app );
                H5Sclose( h5_dspace_id );
            }
            else if (FLDget_points(fld_id, &fld_points, &pts_size,
                                   OM_GET_ARRAY_WR) == OM_STAT_SUCCESS) {
                float *tmp_points = malloc( pts_size * sizeof(float) );

                /* Size should be already established as npoints*nspace */
                H5Dread( h5_dset_id, H5T_NATIVE_FLOAT_g,
                         H5S_ALL, H5S_ALL,
                         H5P_DEFAULT, tmp_points );

                /* If we are cropping, adjust the points to match.
                 * The number of points stays unchanged, we are just
                 * tweaking the values.  E.g., if we crop the original
                 * region in half, we want the extent to be halved as well.
                 */
                if( subset != NULL && fld_type == FLD_UNIF ) {
                    for( i = 0; i < fld_ndim; ++i ) {
                        int ii = i+fld_nspace;
                        /* original extent range = pnts max - pnts min */
                        float e_range = tmp_points[ii] - tmp_points[i];
                        /* original dims range */
                        float d_range = (float)subset->orig[i]-1;
                        /* new extent min[i] */
                        fld_points[i]  = tmp_points[i]
                            + e_range*subset->min[i]/d_range;
                        /* new extent max[i] */
                        fld_points[ii] = tmp_points[i]
                            + e_range*subset->max[i]/d_range;
                    }
                    /* In case nspace > ndim */
                    for( ; i < fld_nspace; ++i ) {
                        int ii = i+fld_nspace;
                        fld_points[i]  = tmp_points[i];  /* min extent */
                        fld_points[ii] = tmp_points[ii]; /* max extent */
                    }
                }
                else {
                    /* If no need to alter the points */
                    memcpy( fld_points, tmp_points, pts_size * sizeof(float) );
                }

                free( tmp_points );
                ARRfree( fld_points );
            }
            /* else error could not get points */

            H5Dclose( h5_dset_id );
        } /* if points is present */

        /* Can't get points.  Try plan b for Unif ... */

        /* Fill in defaults for Unif => {{0,0}, {dims[0]-1,dims[1]-1}} */
        else if ( fld_type == FLD_UNIF ) {
            xp_long *fld_dims = NULL;
            int dims_size;
            if( FLDget_dims( fld_id, &fld_dims, &dims_size )
                == OM_STAT_SUCCESS ) {
                float *fld_points = NULL;
                xp_long pts_size = 0;
                if (FLDget_points(fld_id, &fld_points, &pts_size,
                                  OM_GET_ARRAY_WR) == OM_STAT_SUCCESS) {
                    for( i = 0; i < fld_ndim; ++i ) {
                        int ii = i+fld_nspace;
                        fld_points[i]  = 0.0;
                        fld_points[ii] = fld_dims[i] - 1;
                    };
                    /* In case nspace > ndim */
                    for( ; i < fld_nspace; ++i ) {
                        int ii = i+fld_nspace;
                        fld_points[i] = fld_points[ii] = 0.0;
                    }
                    ARRfree(fld_points);
                }
                /* Could not get points */
                else {
                    ARRfree(fld_dims);
                    return METHOD_FAILURE;
                }

                ARRfree(fld_dims);
            }
            /* Could not get dims */
            else return METHOD_FAILURE;
        }
        /* Points are required for Rect */
        else return METHOD_FAILURE;
    }

    return METHOD_SUCCESS;
}

/* Xform */
static int
HDF5read_xform( hid_t h5_root_id, OMobj_id fld_id )
{
    hid_t h5_grp_id, h5_dset_id;

    /* Generally OK if this is missing.  You'll get the default
       identity transform.
    */

    if( (h5_grp_id = probe_group( h5_root_id, "xform", 0 )) > 0 ) {
        if( (h5_dset_id = probe_dataset( h5_grp_id, "mat", 0 )) > 0 ) {
            float xform[16];

            H5Dread( h5_dset_id, H5T_NATIVE_FLOAT_g,
                     H5S_ALL, H5S_ALL,
                     H5P_DEFAULT, xform );

            FLDset_xform( fld_id, xform );
            H5Dclose( h5_dset_id );
        }
        H5Gclose( h5_grp_id );
    }

    return METHOD_SUCCESS;
}


/* Cells */
/* 64-bit porting. Only Modified Internally */
static int
HDF5read_cells(
    hid_t h5_root_id,	/* Parent H5 group, usually root of field. */
    OMobj_id fld_id )	/* OM id of an object of type 'Cells' */
{
    hid_t h5_dset_id, h5_grp_id, h5_attr_id;

    if( probe_attribute( h5_root_id, "ncell_sets", 1 ) > 0 ) {
        OMobj_id cset_array_id, cset_id;
        int fld_nsets;
        int i;
        xp_long array_size;
        char buffer[64];

        cset_array_id = OMfind_subobj (fld_id, OMstr_to_name("cell_set"),
                                       OM_OBJ_RW);

        if (OMis_null_obj(cset_array_id)) {
            ERR_RETURN( "Cannot access out fld cell_set" );
        }

        /* A Struct field will come with this already set to 1,
           otherwise make sure this is in a known state.
        */
        if( OMget_array_size(cset_array_id, &array_size) != OM_STAT_SUCCESS ) {
            /* Start out with 0 cell_sets 'cuz we will be adding
               them incrementally.
            */
            if( OMset_array_size(cset_array_id, 0) != OM_STAT_SUCCESS ) {
                /* Adding cell sets won't work very well if we
                   can't set the size of the cell_set array.
                */
                ERR_RETURN( "Cannot set size of out fld cell_set" );
            }
        }

        HDF5read_int_attr( h5_root_id, "ncell_sets", &fld_nsets );
        for( i = 0; i < fld_nsets; ++i ) {

            /* For each cell set, look for a group called cell_set[n] */
            sprintf( buffer, "cell_set[%d]", i );
            if( (h5_grp_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {
                xp_long cset_ncells = 0, cset_npolys = 0;
                int cset_poly_flag, cset_nprops, *conn_iarray;
                xp_long *conn_larray, conn_size;
                char *str = NULL;

                OMget_array_size(cset_array_id, &array_size);

                HDF5read_string_attr( h5_grp_id, "cell_name", &str );
                if( str != NULL ) {
                    /* When the cell set is created like this, using the
                       cell_name, then we should be able to query the
                       virtual information.
                    */
                    /* array size should now == i, add one more */
                    if( (int)array_size == i )
                        FLDadd_cell_set( fld_id, str );
                    /* else, merge in type information? ...
                       probably a rare situation.
                    */
                    free( str );
                }
                else {
                    /* The natural thing to do is to just resize the
                       cell_set array once up front, but this
                       incremental style matches FLDadd_cell_set.
                    */

                    /* array size should now == i, add one more */
                    if( (int)array_size != i+1 )
                        OMset_array_size(cset_array_id, i+1);
                }

                if( FLDget_cell_set( fld_id, i, &cset_id ) != OM_STAT_SUCCESS )
                    continue;

                /* ncells */
                if( probe_attribute( h5_grp_id, "ncells", 1 ) > 0 ) {
                    HDF5read_long_attr( h5_grp_id, "ncells", &cset_ncells );
                    FLDset_ncells( cset_id, cset_ncells );
                }

                /* cell_ndim (usually can be derived from cell_name) */
                if( probe_attribute( h5_grp_id, "cell_ndim", 1 ) > 0 ) {
                    int cell_ndim;
                    HDF5read_int_attr( h5_grp_id, "cell_ndim", &cell_ndim );
                    FLDset_ncells( cset_id, cell_ndim );
                }

                /* cell_nnodes (usually can be derived from cell_name) */
                if( probe_attribute( h5_grp_id, "cell_nnodes", 1 ) > 0 ) {
                    int cell_nnodes;
                    HDF5read_int_attr( h5_grp_id, "cell_nnodes",
                                       &cell_nnodes );
                    FLDset_nnodes( cset_id, cell_nnodes );
                }

                /* cell_corner_nnodes (usually can be derived from cell_name) */

                /* cell_order (usually can be derived from cell_name) */
                if( probe_attribute( h5_grp_id, "cell_order", 1 ) > 0 ) {
                    int cell_order;
                    HDF5read_int_attr( h5_grp_id, "cell_order",
                                       &cell_order );
                    FLDset_ncells( cset_id, cell_order );
                }

                /* poly_flag (usually can be derived from cell_name) */
                if( probe_attribute( h5_grp_id, "poly_flag", 1 ) > 0 ) {
                    HDF5read_int_attr( h5_grp_id, "poly_flag",
                                       &cset_poly_flag );
                    FLDset_poly_flag( cset_id, cset_poly_flag );
                }

                FLDget_poly_flag( cset_id, &cset_poly_flag );

                /* node_connect_list */
                h5_dset_id = probe_dataset( h5_grp_id,
                                            "node_connect_list", 0 );
                if( (cset_poly_flag == 0) && (h5_dset_id > 0 )) {
                    /* node_connect_list[ncells*cell_nnodes] */
                    if( (FLDget_node_connect( cset_id, &conn_larray, &conn_size,
                             OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS) ) {
                        H5Dread( h5_dset_id, H5T_NATIVE_XP_LONG_g,
                                 H5S_ALL, H5S_ALL,
                                 H5P_DEFAULT, conn_larray );
                        ARRfree( conn_larray );
                    }
                    H5Dclose(h5_dset_id);
                }

                /* name (optional) */
                if( probe_attribute( h5_grp_id, "name", 1 ) > 0 ) {
                    HDF5read_string_attr( h5_grp_id, "name", &str );
                    if( str != NULL ) {
                        FLDset_cell_set_user_name( cset_id, str );
                        free( str );
                    }
                }

                /* nprops (optional) */
                if( probe_attribute( h5_grp_id, "nprops", 1 ) > 0 ) {
                    HDF5read_int_attr( h5_grp_id, "nprops", &cset_nprops );
                    FLDset_cell_nprops( cset_id, cset_nprops );
                }

                /* props (optional) */
                if( (h5_attr_id = probe_attribute( h5_grp_id, "props", 0 )) > 0 ) {
                    int props_size;
                    float *props = NULL;
                    if( FLDget_cell_props( cset_id, &props, &props_size,
                            OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS ) {
                        H5Aread( h5_attr_id, H5T_NATIVE_FLOAT_g, props );
                        ARRfree( props );
                    }
                    H5Aclose( h5_attr_id );
                }

                /* poly_type (usually can be derived from cell_name)/ */
                if( probe_attribute( h5_grp_id, "poly_type", 1 ) > 0 ) {
                    int poly_type;
                    HDF5read_int_attr( h5_grp_id, "poly_type", &poly_type );
                    OMset_name_int_val( cset_id,
                                        OMstr_to_name("poly_type"),
                                        poly_type );
                }

                /* npolys */
                if( probe_attribute( h5_grp_id, "npolys", 1 ) > 0 ) {
                    HDF5read_long_attr( h5_grp_id, "npolys", &cset_npolys );
                    FLDset_npolys( cset_id, cset_npolys );
                }

                /* poly_nnodes (just for polyhedron) */
                h5_dset_id = probe_dataset( h5_grp_id, "poly_nnodes", 0 );
                if( h5_dset_id  > 0 ) {
                    /* poly_nnodes[npolys] */
                    if( (FLDget_poly_nnodes( cset_id, &conn_iarray, &conn_size,
                             OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS) ) {
                        H5Dread( h5_dset_id, H5T_NATIVE_INT_g,
                                 H5S_ALL, H5S_ALL,
                                 H5P_DEFAULT, conn_iarray );
                        ARRfree( conn_iarray );
                    }
                    H5Dclose(h5_dset_id);
                }

                /* poly_connect_list */
                if( (cset_poly_flag == 1) &&
                    ((h5_dset_id = probe_dataset( h5_grp_id,
                                      "poly_connect_list", 0 )) > 0 )) {
                    /* poly_connect_list[npolys*2] or [] */
                    /* Might not know size for polyhedron. */
                    if( (FLDget_poly_connect( cset_id, &conn_larray, &conn_size,
                             OM_GET_ARRAY_WR ) == OM_STAT_SUCCESS) ) {
#if 0
                        fprintf( stderr, "poly_connect_list size:%d\n",
                                 conn_size );
#endif
                        H5Dread( h5_dset_id, H5T_NATIVE_XP_LONG_g,
                                 H5S_ALL, H5S_ALL,
                                 H5P_DEFAULT, conn_larray );
                        ARRfree( conn_larray );
                    }
                    H5Dclose(h5_dset_id);
                }

                /*  concave_flag (just for polyhedron) */
                if( probe_attribute( h5_grp_id, "concave_flags", 1 ) > 0 ) {
                    int concave;
                    HDF5read_int_attr( h5_grp_id, "concave_flag", &concave );
                    OMset_name_int_val( cset_id,
                                        OMstr_to_name("concave_flag"),
                                        concave );
                }

                H5Gclose( h5_grp_id );
            } /* if cell set group is in file */
        } /* if file has ncell_sets */
    } /* if file has ncell_sets */

    return METHOD_SUCCESS;
}


/* Node_Data */
static int
HDF5read_node_data( hid_t h5_root_id, OMobj_id fld_id,
                    vars_info_t *vars, subset_info_t *subset )
{
    hid_t h5_grp_id;
    OMobj_id node_data_id, comp_id;
    char buffer[64];
    int i, ii, file_ncomp, fld_ncomp;

    /* Might already be set by higher level code */
    if( FLDget_node_data_ncomp( fld_id, &fld_ncomp ) != OM_STAT_SUCCESS ) {
        /* OK if it not set yet, it might be in the file. */
        if( probe_attribute( h5_root_id, "nnode_data", 1 ) > 0 ) {
            HDF5read_int_attr( h5_root_id, "nnode_data", &file_ncomp );

            if( (vars != NULL) && (vars->n_node_vars >= 0)  &&
                (vars->n_node_vars < file_ncomp) ) {
                /* a subset of the vars were selected */
                fld_ncomp = vars->n_node_vars;
            }
            else
                fld_ncomp = file_ncomp;

            if( FLDset_node_data_ncomp( fld_id, fld_ncomp )
                != OM_STAT_SUCCESS ) {
                ERR_RETURN( "Cannot set out fld nnode_data" );
            }
        }
        else {
            /* ERR_RETURN( "Cannot find nnode_data in file" ); */

            /* OK for this call to fail if Node_Data was not merged in. */
            FLDset_node_data_ncomp( fld_id, 0 );
            return METHOD_SUCCESS;
        }
    }

    node_data_id = OMfind_subobj( fld_id, OMstr_to_name("node_data"),
                                  OM_OBJ_RW );

    for( i=0; i<fld_ncomp; ++i ) {

        if( (vars != NULL) &&  (vars->node_vars != NULL) &&
            (vars->n_node_vars > 0) &&
            (vars->node_vars[i] < file_ncomp) ) {
            ii = vars->node_vars[i];
        }
        else if( (vars != NULL) && (vars->n_node_vars == 0) )
            break;
        else ii = i;

        if( OMget_array_val(node_data_id, i, &comp_id, OM_OBJ_RW) !=
            OM_STAT_SUCCESS )
            return METHOD_FAILURE;  /* Don't expect this error */

        /* For each node data component, look for group called
         * node_data[n]
         */
        sprintf( buffer, "node_data[%d]", ii );
        if( (h5_grp_id = probe_group( h5_root_id, buffer, 0)) > 0) {
#if 0
            fprintf( stderr, "Found %s\n", buffer );
#endif
            /* read Data_Array */
            HDF5read_data_array( h5_grp_id, comp_id, subset );
            H5Gclose( h5_grp_id );
        } /* if component */
    } /* loop over components */

    return METHOD_SUCCESS;
}


static int
HDF5read_cell_data_set( hid_t h5_cset_id, OMobj_id cset_id, vars_info_t *vars )
{
    hid_t h5_cdata_id;
    OMobj_id cdata_id, comp_id;
    char buffer[64];
    int j, jj, file_ncomp, cset_ncomp;

    if( probe_attribute( h5_cset_id, "ncell_data", 1 ) > 0 ) {

        HDF5read_int_attr( h5_cset_id, "ncell_data", &file_ncomp );

        if( (vars != NULL) && (vars->n_cell_vars >= 0) &&
            (vars->n_cell_vars < file_ncomp) ) {
            cset_ncomp = vars->n_cell_vars;
        }
        else
            cset_ncomp = file_ncomp;

        if( FLDset_cell_data_ncomp( cset_id, cset_ncomp ) != OM_STAT_SUCCESS )
            return METHOD_FAILURE;

        cdata_id = OMfind_subobj( cset_id, OMstr_to_name("cell_data"),
                                  OM_OBJ_RW);

        if( OMis_null_obj( cdata_id ) )
            /* Attempt to merge in Cell_Data? */
            return METHOD_FAILURE;

        /* Loop over the cell data components. */
        for( j=0; j<cset_ncomp; ++j ) {

            if( (vars != NULL) &&  (vars->cell_vars != NULL) &&
                (vars->n_cell_vars > 0) &&
                (vars->cell_vars[j] < file_ncomp) ) {
                jj = vars->cell_vars[j];
            }
            else if( (vars != NULL) && (vars->n_cell_vars == 0) )
                break;
            else jj = j;

            if( OMget_array_val(cdata_id, j, &comp_id, OM_OBJ_RW)
                !=  OM_STAT_SUCCESS )
                return METHOD_FAILURE;

            /* For each cell data component,
             * look for a group called cell_data[n].
             */
            sprintf( buffer, "cell_data[%d]", jj );
            if( (h5_cdata_id = probe_group(h5_cset_id, buffer, 0)) > 0 ) {
#if 0
                fprintf( stderr, "Found %s\n", buffer );
#endif
                /* read Data_Array */
                HDF5read_data_array( h5_cdata_id, comp_id, NULL );
                H5Gclose( h5_cdata_id );
            } /* if cell data comp is in file */

        } /* loop over cell data components */

    } /* if ncell_data in file */

    return METHOD_SUCCESS;
}


/* Cell_Data */
static int
HDF5read_cell_data( hid_t h5_root_id, OMobj_id fld_id,
                    vars_info_t *vars)
{
    if( probe_attribute( h5_root_id, "ncell_sets", 1 ) > 0 ) {
        hid_t h5_cset_id;
        OMobj_id cset_array_id, cset_id;
        int i, fld_nsets;
        char buffer[64];

        cset_array_id = OMfind_subobj (fld_id, OMstr_to_name("cell_set"),
                                       OM_OBJ_RW);

        if (OMis_null_obj(cset_array_id)) {
            ERR_RETURN( "Cannot access out fld cell_set" );
        }

        HDF5read_int_attr( h5_root_id, "ncell_sets", &fld_nsets );
        for( i = 0; i < fld_nsets; ++i ) {

            /* For each cell set, look for a group called cell_set[n] */
            sprintf( buffer, "cell_set[%d]", i );
            if( (h5_cset_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {

                if( FLDget_cell_set( fld_id, i, &cset_id ) != OM_STAT_SUCCESS )
                    continue;

                /* Logic to add cell sets if they don't exist?  Not needed.
                   read_cells will create the cell sets, even if there
                   is no cell information aside from cell data.
                 */

                if( probe_attribute( h5_cset_id, "ncell_data", 1 ) > 0 ) {
                    HDF5read_cell_data_set( h5_cset_id, cset_id, vars );
                }

                H5Gclose( h5_cset_id );
            }  /* if cell set group is in file */
        }  /* if file has ncell_sets */
    } /* if file has ncell_sets */

    return METHOD_SUCCESS;
}

/* Cell_Data */
static int
HDF5read_time_cell_data( hid_t h5_root_id, OMobj_id fld_id,
                         int time_step, vars_info_t *vars )
{
    int fld_nsteps;

    if( time_step == INVALID_TIME_STEP ) {
        if( FLDget_time_nsteps( fld_id, &fld_nsteps ) != OM_STAT_SUCCESS ) {
            ERR_RETURN( "Could not get field nsteps" );
        }
    }

    if( probe_attribute( h5_root_id, "ncell_sets", 1 ) > 0 ) {
        hid_t h5_cset_id, h5_grp_id;
        OMobj_id cset_array_id, cset_id;
        OMobj_id fld_nsteps_id, cset_nsteps_id;
        int fld_nsets;
        char buffer[64];
        int i, j, tmp;

        cset_array_id = OMfind_subobj (fld_id, OMstr_to_name("cell_set"),
                                       OM_OBJ_RW);

        if (OMis_null_obj(cset_array_id)) {
            ERR_RETURN( "Cannot access out fld cell_set" );
        }

        /* We need for cell_set.nsteps to have a valid value, but what
           seems to be the logical default connections (see below) are
           not supplied by the definitions in the FLD library.

           cell_set {
               nsteps => <-.nsteps;
               time   => <-.time;
           };
        */
        fld_nsteps_id = OMfind_subobj (fld_id, OMstr_to_name("nsteps"),
                                       OM_OBJ_RD);
        OMget_int_val( fld_nsteps_id, &tmp );

        HDF5read_int_attr( h5_root_id, "ncell_sets", &fld_nsets );
        for( i = 0; i < fld_nsets; ++i ) {

            /* For each cell set, look for a group called cell_set[n] */
            sprintf( buffer, "cell_set[%d]", i );
            if( (h5_cset_id = probe_group(h5_root_id, buffer, 0)) > 0 ) {

                if( FLDget_cell_set( fld_id, i, &cset_id ) != OM_STAT_SUCCESS )
                    continue;

                if( time_step != INVALID_TIME_STEP ) {
                    sprintf( buffer, "time_cell_data[%d]", time_step );
                    h5_grp_id = probe_group( h5_cset_id, buffer, 0 );
                    if( h5_grp_id > 0 ) {
                        HDF5read_cell_data_set( h5_grp_id, cset_id, vars );
                        H5Gclose( h5_grp_id );
                    }
                }
                else {
                cset_nsteps_id = OMfind_subobj (cset_id,
                                                OMstr_to_name("nsteps"),
                                                OM_OBJ_RW);

                if( OMget_int_val( cset_nsteps_id, &tmp )
                    != OM_STAT_SUCCESS ) {
                    /* nsteps => <-.nsteps; */
#if 0
                    /* Does not work for some unknown reason. */
                    OMlink_objs( cset_nsteps_id, fld_nsteps_id );
#else
                    /* This works, although I'd prefer the soft link. */
                    OMset_obj_ref( cset_nsteps_id, fld_nsteps_id, 0 );
#endif
                    OMget_int_val( cset_nsteps_id, &tmp );
                }
                /* else if it does have a valid value, then don't worry */

                for( j = 0; j < fld_nsteps; ++j ) {
                    sprintf( buffer, "time_cell_data[%d]", j );
                    h5_grp_id = probe_group( h5_cset_id, buffer, 0 );
                    if( h5_grp_id > 0 ) {
                        OMobj_id cdata_id;
                        if( FLDget_time_cell_data( cset_id, j, &cdata_id )
                            == OM_STAT_SUCCESS ) {
                            HDF5read_cell_data_set( h5_grp_id, cdata_id, vars );
                        }
                        H5Gclose( h5_grp_id );
                    }
                }
                H5Gclose( h5_cset_id );
                }
            }  /* if cell set group is in file */
        }  /* if file has ncell_sets */
    } /* if file has ncell_sets */

    return METHOD_SUCCESS;
}
